#!/usr/bin/env python3
"""
Documentation Synchronization for TuskTSK CLI
=============================================
Automated tools to keep documentation in sync with code
"""

import sys
import os
import json
import re
import argparse
from pathlib import Path
from typing import Dict, List, Any, Optional, Set
from datetime import datetime
import ast
import inspect
import subprocess

# Add parent directory to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from cli.main import TuskLangCLI


class DocumentationSync:
    """Synchronizes documentation with code changes"""
    
    def __init__(self, verbose: bool = False):
        self.verbose = verbose
        self.cli = TuskLangCLI()
        self.sync_results = {
            'timestamp': datetime.now().isoformat(),
            'files_updated': 0,
            'commands_synced': 0,
            'examples_updated': 0,
            'errors': [],
            'warnings': []
        }
        
    def extract_commands_from_code(self) -> Dict[str, Any]:
        """Extract all command information from CLI code"""
        commands = {}
        
        # Get all subparsers
        for action in self.cli.parser._subparsers._group_actions:
            if hasattr(action, 'choices'):
                for command_name, subparser in action.choices.items():
                    commands[command_name] = {
                        'help': subparser.description or subparser.help,
                        'epilog': subparser.epilog,
                        'subcommands': {},
                        'arguments': [],
                        'examples': self.extract_examples_from_epilog(subparser.epilog)
                    }
                    
                    # Get subcommands
                    for subaction in subparser._subparsers._group_actions:
                        if hasattr(subaction, 'choices'):
                            for subcommand_name, subsubparser in subaction.choices.items():
                                commands[command_name]['subcommands'][subcommand_name] = {
                                    'help': subsubparser.description or subsubparser.help,
                                    'arguments': [],
                                    'examples': self.extract_examples_from_epilog(subsubparser.epilog)
                                }
                                
                                # Get arguments
                                for arg in subsubparser._actions:
                                    if arg.dest != 'help':
                                        commands[command_name]['subcommands'][subcommand_name]['arguments'].append({
                                            'name': arg.dest,
                                            'type': type(arg.type).__name__ if arg.type else 'str',
                                            'required': arg.required,
                                            'default': arg.default,
                                            'help': arg.help,
                                            'choices': arg.choices if hasattr(arg, 'choices') else None
                                        })
        
        return commands
    
    def extract_examples_from_epilog(self, epilog: str) -> List[str]:
        """Extract examples from command epilog"""
        examples = []
        if epilog:
            # Look for example patterns
            example_patterns = [
                r'Examples?:\s*\n((?:\s+[^\n]+\n?)+)',
                r'Usage:\s*\n((?:\s+[^\n]+\n?)+)',
                r'Example:\s*([^\n]+)'
            ]
            
            for pattern in example_patterns:
                matches = re.findall(pattern, epilog, re.IGNORECASE | re.MULTILINE)
                for match in matches:
                    if isinstance(match, tuple):
                        match = match[0]
                    examples.extend([ex.strip() for ex in match.split('\n') if ex.strip()])
        
        return examples
    
    def generate_command_documentation(self, output_file: str = None) -> str:
        """Generate comprehensive command documentation"""
        if not output_file:
            output_file = "commands_documentation.md"
        
        commands = self.extract_commands_from_code()
        
        # Generate markdown documentation
        md_content = """# TuskTSK CLI Commands Documentation

This document is automatically generated from the CLI code. Last updated: {timestamp}

## Overview

The TuskTSK CLI provides {total_commands} main command categories with comprehensive functionality for development, testing, and deployment.

## Command Categories

""".format(
            timestamp=datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            total_commands=len(commands)
        )
        
        # Add command categories
        for command_name, command_info in commands.items():
            md_content += f"### {command_name.title()}\n\n"
            md_content += f"{command_info['help'] or 'No description available'}\n\n"
            
            if command_info['examples']:
                md_content += "**Examples:**\n"
                for example in command_info['examples']:
                    md_content += f"```bash\n{example}\n```\n"
                md_content += "\n"
            
            # Add subcommands
            if command_info['subcommands']:
                md_content += "**Subcommands:**\n\n"
                for subcommand_name, subcommand_info in command_info['subcommands'].items():
                    md_content += f"#### {subcommand_name}\n\n"
                    md_content += f"{subcommand_info['help'] or 'No description available'}\n\n"
                    
                    # Add arguments
                    if subcommand_info['arguments']:
                        md_content += "**Arguments:**\n\n"
                        md_content += "| Argument | Type | Required | Default | Description |\n"
                        md_content += "|----------|------|----------|---------|-------------|\n"
                        
                        for arg in subcommand_info['arguments']:
                            required = "Yes" if arg['required'] else "No"
                            default = str(arg['default']) if arg['default'] is not None else "None"
                            help_text = arg['help'] or "No description"
                            
                            md_content += f"| {arg['name']} | {arg['type']} | {required} | {default} | {help_text} |\n"
                        
                        md_content += "\n"
                    
                    # Add examples
                    if subcommand_info['examples']:
                        md_content += "**Examples:**\n"
                        for example in subcommand_info['examples']:
                            md_content += f"```bash\n{example}\n```\n"
                        md_content += "\n"
            
            md_content += "---\n\n"
        
        # Add footer
        md_content += """## Global Options

All commands support the following global options:

- `--version, -v`: Show version information
- `--verbose`: Enable verbose output
- `--quiet, -q`: Suppress non-error output
- `--json`: Output in JSON format
- `--config`: Use alternate config file

## Getting Help

For detailed help on any command:

```bash
tsk <command> --help
tsk <command> <subcommand> --help
```

## Examples

### Database Operations
```bash
tsk db status                    # Check database connection
tsk db init                      # Initialize database
tsk db migrate migration.sql     # Run migration
tsk db backup backup.db          # Create backup
tsk db restore backup.db         # Restore from backup
```

### Development
```bash
tsk serve 3000                   # Start development server
tsk build                        # Build project
tsk test all                     # Run all tests
```

### Configuration
```bash
tsk config get server.port       # Get configuration value
tsk config set server.port 9090  # Set configuration value
tsk config list                  # List all configuration
```

### Services
```bash
tsk service start myapp          # Start service
tsk service stop myapp           # Stop service
tsk service status myapp         # Check service status
```

### Cache Management
```bash
tsk cache clear                  # Clear all cache
tsk cache status                 # Show cache status
```

### Binary Operations
```bash
tsk binary compile app.tsk       # Compile to binary
tsk binary decompile app.bin     # Decompile binary
```

### AI Features
```bash
tsk ai analyze "text to analyze" # Analyze text
tsk ai generate "prompt"         # Generate content
```

### Utilities
```bash
tsk utility format file.tsk      # Format code
tsk utility lint file.tsk        # Lint code
tsk utility validate file.tsk    # Validate code
```

### Peanuts Management
```bash
tsk peanuts create mypeanut      # Create peanut
tsk peanuts list                 # List peanuts
```

### CSS Operations
```bash
tsk css compile style.css        # Compile CSS
tsk css validate style.css       # Validate CSS
```

### License Management
```bash
tsk license check                # Check license
tsk license generate user        # Generate license
```

### Dependency Management
```bash
tsk dependency install package   # Install dependency
tsk dependency update package    # Update dependency
tsk dependency list              # List dependencies
```

## Contributing

This documentation is automatically generated from the CLI code. To update it:

1. Modify the command implementations in `cli/commands/`
2. Update help text and examples in the command definitions
3. Run the documentation sync script to regenerate this file

## Support

For issues and questions:
- Check the help system: `tsk --help`
- Review command-specific help: `tsk <command> --help`
- Report bugs through the project issue tracker
"""
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        with open(output_path, 'w') as f:
            f.write(md_content)
        
        self.sync_results['files_updated'] += 1
        self.sync_results['commands_synced'] = len(commands)
        
        if self.verbose:
            print(f"📝 Generated documentation: {output_path}")
        
        return output_path
    
    def update_command_counts(self, documentation_file: str) -> bool:
        """Update command counts in documentation"""
        try:
            commands = self.extract_commands_from_code()
            
            # Count total commands and subcommands
            total_commands = len(commands)
            total_subcommands = sum(len(cmd['subcommands']) for cmd in commands.values())
            total_arguments = sum(
                len(cmd['arguments']) + sum(len(sub['arguments']) for sub in cmd['subcommands'].values())
                for cmd in commands.values()
            )
            
            # Read existing documentation
            with open(documentation_file, 'r') as f:
                content = f.read()
            
            # Update counts in documentation
            content = re.sub(
                r'provides \d+ main command categories',
                f'provides {total_commands} main command categories',
                content
            )
            
            # Add detailed counts if not present
            if 'Total Commands:' not in content:
                # Find a good place to insert counts
                insert_pos = content.find('## Overview')
                if insert_pos != -1:
                    count_section = f"""
## Command Statistics

- **Main Commands**: {total_commands}
- **Subcommands**: {total_subcommands}
- **Total Arguments**: {total_arguments}
- **Last Updated**: {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}

"""
                    content = content[:insert_pos] + count_section + content[insert_pos:]
            
            # Write updated documentation
            with open(documentation_file, 'w') as f:
                f.write(content)
            
            self.sync_results['files_updated'] += 1
            
            if self.verbose:
                print(f"📊 Updated command counts in: {documentation_file}")
            
            return True
            
        except Exception as e:
            self.sync_results['errors'].append(f"Error updating command counts: {e}")
            return False
    
    def sync_command_examples(self, documentation_file: str) -> bool:
        """Ensure all examples in documentation work correctly"""
        try:
            commands = self.extract_commands_from_code()
            
            # Read existing documentation
            with open(documentation_file, 'r') as f:
                content = f.read()
            
            # Extract examples from documentation
            example_pattern = r'```bash\n(.*?)\n```'
            documented_examples = re.findall(example_pattern, content, re.DOTALL)
            
            # Validate each example
            valid_examples = []
            invalid_examples = []
            
            for example in documented_examples:
                example = example.strip()
                if self.validate_command_example(example):
                    valid_examples.append(example)
                else:
                    invalid_examples.append(example)
            
            # Update documentation with validation results
            if invalid_examples:
                warning_section = f"""
## ⚠️ Example Validation Warnings

The following examples may not work correctly:

"""
                for example in invalid_examples:
                    warning_section += f"- `{example}`\n"
                
                warning_section += "\nPlease verify these examples and update as needed.\n\n"
                
                # Insert warning at the beginning
                content = warning_section + content
            
            # Write updated documentation
            with open(documentation_file, 'w') as f:
                f.write(content)
            
            self.sync_results['examples_updated'] = len(valid_examples)
            
            if invalid_examples:
                self.sync_results['warnings'].extend(invalid_examples)
            
            if self.verbose:
                print(f"✅ Validated {len(valid_examples)} examples")
                if invalid_examples:
                    print(f"⚠️  Found {len(invalid_examples)} potentially invalid examples")
            
            return True
            
        except Exception as e:
            self.sync_results['errors'].append(f"Error syncing examples: {e}")
            return False
    
    def validate_command_example(self, example: str) -> bool:
        """Validate that a command example works"""
        try:
            # Parse the example command
            parts = example.split()
            if not parts or parts[0] != 'tsk':
                return False
            
            # Remove 'tsk' and test the rest
            args = parts[1:]
            
            # Skip examples with file paths or complex arguments
            for arg in args:
                if '/' in arg or '\\' in arg or arg.startswith('.'):
                    return True  # Assume file paths are valid
            
            # Test command parsing
            try:
                parsed_args = self.cli.parser.parse_args(args)
                return True
            except SystemExit:
                return False
            except Exception:
                return False
                
        except Exception:
            return False
    
    def generate_json_documentation(self, output_file: str = None) -> str:
        """Generate JSON documentation for programmatic use"""
        if not output_file:
            output_file = "commands.json"
        
        commands = self.extract_commands_from_code()
        
        # Add metadata
        documentation = {
            'metadata': {
                'generated_at': datetime.now().isoformat(),
                'version': '2.0.0',
                'total_commands': len(commands),
                'total_subcommands': sum(len(cmd['subcommands']) for cmd in commands.values())
            },
            'commands': commands
        }
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        with open(output_path, 'w') as f:
            json.dump(documentation, f, indent=2, default=str)
        
        self.sync_results['files_updated'] += 1
        
        if self.verbose:
            print(f"📄 Generated JSON documentation: {output_path}")
        
        return output_path
    
    def sync_all_documentation(self) -> Dict[str, Any]:
        """Sync all documentation files"""
        print("🔄 Starting documentation synchronization...")
        
        # Generate markdown documentation
        md_file = self.generate_command_documentation()
        
        # Update command counts
        self.update_command_counts(md_file)
        
        # Sync examples
        self.sync_command_examples(md_file)
        
        # Generate JSON documentation
        json_file = self.generate_json_documentation()
        
        # Generate summary
        self.generate_sync_summary()
        
        return self.sync_results
    
    def generate_sync_summary(self, output_file: str = None) -> str:
        """Generate synchronization summary report"""
        if not output_file:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_file = f"docs_sync_summary_{timestamp}.json"
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        with open(output_path, 'w') as f:
            json.dump(self.sync_results, f, indent=2, default=str)
        
        return output_path
    
    def print_summary(self):
        """Print synchronization summary"""
        print("\n" + "=" * 60)
        print("📋 Documentation Sync Summary")
        print("=" * 60)
        print(f"Files Updated: {self.sync_results['files_updated']}")
        print(f"Commands Synced: {self.sync_results['commands_synced']}")
        print(f"Examples Updated: {self.sync_results['examples_updated']}")
        
        if self.sync_results['errors']:
            print(f"\n❌ Errors:")
            for error in self.sync_results['errors']:
                print(f"   - {error}")
        
        if self.sync_results['warnings']:
            print(f"\n⚠️  Warnings:")
            for warning in self.sync_results['warnings']:
                print(f"   - {warning}")


def main():
    """Main entry point for documentation sync"""
    parser = argparse.ArgumentParser(description='TuskTSK CLI Documentation Sync')
    parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose output')
    parser.add_argument('--markdown', action='store_true', help='Generate markdown documentation')
    parser.add_argument('--json', action='store_true', help='Generate JSON documentation')
    parser.add_argument('--update-counts', action='store_true', help='Update command counts')
    parser.add_argument('--sync-examples', action='store_true', help='Sync command examples')
    parser.add_argument('--all', action='store_true', help='Run all sync operations')
    parser.add_argument('--output', '-o', help='Output file for sync results')
    
    args = parser.parse_args()
    
    # Create sync instance
    syncer = DocumentationSync(verbose=args.verbose)
    
    # Run sync operations
    if args.all:
        print("🚀 Running complete documentation synchronization...")
        syncer.sync_all_documentation()
    else:
        if args.markdown:
            syncer.generate_command_documentation()
        
        if args.json:
            syncer.generate_json_documentation()
        
        if args.update_counts:
            syncer.update_command_counts("commands_documentation.md")
        
        if args.sync_examples:
            syncer.sync_command_examples("commands_documentation.md")
    
    # Print summary
    syncer.print_summary()
    
    # Save results
    if args.output:
        output_file = syncer.generate_sync_summary(args.output)
    else:
        output_file = syncer.generate_sync_summary()
    
    if args.verbose:
        print(f"\n💾 Sync results saved to: {output_file}")
    
    # Exit with appropriate code
    total_issues = len(syncer.sync_results['errors']) + len(syncer.sync_results['warnings'])
    sys.exit(0 if total_issues == 0 else 1)


if __name__ == '__main__':
    main() 