"""
TuskLang Python SDK - Package Management Ecosystem
Production-quality multi-platform package distribution with automated publishing
"""

import asyncio
import json
import logging
import uuid
import hashlib
import subprocess
import tempfile
import shutil
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Any, Union, Tuple
from dataclasses import dataclass, field
from enum import Enum
from pathlib import Path
import threading
import tarfile
import zipfile
import os

try:
    import requests
    REQUESTS_AVAILABLE = True
except ImportError:
    REQUESTS_AVAILABLE = False

try:
    import semver
    SEMVER_AVAILABLE = True
except ImportError:
    SEMVER_AVAILABLE = False


class PackageEcosystem(Enum):
    NPM = "npm"
    GO_MODULES = "go_modules"
    MAVEN = "maven"
    NUGET = "nuget"
    RUBYGEMS = "rubygems"
    PYPI = "pypi"
    CARGO = "cargo"
    COMPOSER = "composer"


class PackageStatus(Enum):
    DRAFT = "draft"
    BUILDING = "building"
    BUILT = "built"
    PUBLISHING = "publishing"
    PUBLISHED = "published"
    FAILED = "failed"
    DEPRECATED = "deprecated"


@dataclass
class PackageVersion:
    """Package version with semantic versioning"""
    major: int
    minor: int
    patch: int
    prerelease: str = ""
    build_metadata: str = ""
    
    def __str__(self) -> str:
        version = f"{self.major}.{self.minor}.{self.patch}"
        if self.prerelease:
            version += f"-{self.prerelease}"
        if self.build_metadata:
            version += f"+{self.build_metadata}"
        return version


@dataclass
class PackageMetadata:
    """Package metadata for distribution"""
    name: str
    version: PackageVersion
    description: str
    author: str
    license: str
    homepage: str
    repository: str
    keywords: List[str] = field(default_factory=list)
    dependencies: Dict[str, str] = field(default_factory=dict)
    dev_dependencies: Dict[str, str] = field(default_factory=dict)
    entry_points: Dict[str, str] = field(default_factory=dict)
    platforms: List[str] = field(default_factory=list)
    readme_content: str = ""
    changelog: str = ""


@dataclass
class PackageArtifact:
    """Built package artifact"""
    package_id: str
    ecosystem: PackageEcosystem
    metadata: PackageMetadata
    file_path: str
    checksum: str
    size_bytes: int
    status: PackageStatus
    build_logs: List[str] = field(default_factory=list)
    created_at: datetime = field(default_factory=datetime.utcnow)
    published_at: Optional[datetime] = None


@dataclass
class DistributionTarget:
    """Package distribution target configuration"""
    ecosystem: PackageEcosystem
    registry_url: str
    auth_token: str
    namespace: str = ""
    enabled: bool = True
    auto_publish: bool = False
    publish_prereleases: bool = False


class PackageDistributor:
    """Production-quality package management and distribution system"""
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.logger = logging.getLogger(__name__)
        self.workspace = Path(self.config.get('workspace', 'dist_workspace'))
        self.workspace.mkdir(exist_ok=True)
        
        # Package artifacts storage
        self.artifacts: Dict[str, PackageArtifact] = {}
        self.distribution_targets: Dict[PackageEcosystem, DistributionTarget] = {}
        
        # Build and publish locks
        self.build_lock = threading.Lock()
        self.publish_lock = threading.Lock()
        
        # Initialize distribution targets
        self._initialize_distribution_targets()
        
        self.logger.info("Package Distributor initialized successfully")

    def _initialize_distribution_targets(self):
        """Initialize distribution targets for different ecosystems"""
        
        # NPM
        self.distribution_targets[PackageEcosystem.NPM] = DistributionTarget(
            ecosystem=PackageEcosystem.NPM,
            registry_url="https://registry.npmjs.org",
            auth_token=self.config.get('npm_token', ''),
            namespace=self.config.get('npm_namespace', ''),
            auto_publish=self.config.get('npm_auto_publish', False)
        )
        
        # PyPI
        self.distribution_targets[PackageEcosystem.PYPI] = DistributionTarget(
            ecosystem=PackageEcosystem.PYPI,
            registry_url="https://upload.pypi.org/legacy/",
            auth_token=self.config.get('pypi_token', ''),
            auto_publish=self.config.get('pypi_auto_publish', False)
        )
        
        # Go Modules
        self.distribution_targets[PackageEcosystem.GO_MODULES] = DistributionTarget(
            ecosystem=PackageEcosystem.GO_MODULES,
            registry_url="https://proxy.golang.org",
            auth_token="",  # Go modules use Git tags
            auto_publish=self.config.get('go_auto_publish', True)
        )
        
        # Maven Central
        self.distribution_targets[PackageEcosystem.MAVEN] = DistributionTarget(
            ecosystem=PackageEcosystem.MAVEN,
            registry_url="https://oss.sonatype.org/service/local/staging/deploy/maven2/",
            auth_token=self.config.get('maven_token', ''),
            namespace=self.config.get('maven_group_id', 'com.tusklang'),
            auto_publish=self.config.get('maven_auto_publish', False)
        )
        
        # NuGet
        self.distribution_targets[PackageEcosystem.NUGET] = DistributionTarget(
            ecosystem=PackageEcosystem.NUGET,
            registry_url="https://api.nuget.org/v3/index.json",
            auth_token=self.config.get('nuget_token', ''),
            auto_publish=self.config.get('nuget_auto_publish', False)
        )
        
        # RubyGems
        self.distribution_targets[PackageEcosystem.RUBYGEMS] = DistributionTarget(
            ecosystem=PackageEcosystem.RUBYGEMS,
            registry_url="https://rubygems.org",
            auth_token=self.config.get('rubygems_token', ''),
            auto_publish=self.config.get('rubygems_auto_publish', False)
        )
        
        # Cargo (Rust)
        self.distribution_targets[PackageEcosystem.CARGO] = DistributionTarget(
            ecosystem=PackageEcosystem.CARGO,
            registry_url="https://crates.io",
            auth_token=self.config.get('cargo_token', ''),
            auto_publish=self.config.get('cargo_auto_publish', False)
        )

    async def create_package(self, metadata: PackageMetadata, 
                           source_path: str,
                           target_ecosystems: List[PackageEcosystem]) -> Dict[str, str]:
        """Create package artifacts for multiple ecosystems"""
        
        package_id = str(uuid.uuid4())
        results = {}
        
        self.logger.info(f"Creating package {metadata.name} v{metadata.version} for {len(target_ecosystems)} ecosystems")
        
        # Build packages for each ecosystem
        for ecosystem in target_ecosystems:
            try:
                artifact = await self._build_package_for_ecosystem(
                    package_id, ecosystem, metadata, source_path
                )
                
                if artifact:
                    self.artifacts[f"{package_id}_{ecosystem.value}"] = artifact
                    results[ecosystem.value] = artifact.file_path
                else:
                    results[ecosystem.value] = "BUILD_FAILED"
                    
            except Exception as e:
                self.logger.error(f"Failed to build {ecosystem.value} package: {e}")
                results[ecosystem.value] = f"ERROR: {e}"
        
        return results

    async def _build_package_for_ecosystem(self, package_id: str, 
                                         ecosystem: PackageEcosystem,
                                         metadata: PackageMetadata,
                                         source_path: str) -> Optional[PackageArtifact]:
        """Build package for specific ecosystem"""
        
        build_dir = self.workspace / f"build_{package_id}_{ecosystem.value}"
        build_dir.mkdir(exist_ok=True)
        
        try:
            # Copy source files
            shutil.copytree(source_path, build_dir / "src", dirs_exist_ok=True)
            
            artifact_path = None
            build_logs = []
            
            if ecosystem == PackageEcosystem.NPM:
                artifact_path = await self._build_npm_package(build_dir, metadata, build_logs)
            elif ecosystem == PackageEcosystem.PYPI:
                artifact_path = await self._build_python_package(build_dir, metadata, build_logs)
            elif ecosystem == PackageEcosystem.GO_MODULES:
                artifact_path = await self._build_go_module(build_dir, metadata, build_logs)
            elif ecosystem == PackageEcosystem.MAVEN:
                artifact_path = await self._build_maven_package(build_dir, metadata, build_logs)
            elif ecosystem == PackageEcosystem.NUGET:
                artifact_path = await self._build_nuget_package(build_dir, metadata, build_logs)
            elif ecosystem == PackageEcosystem.RUBYGEMS:
                artifact_path = await self._build_rubygem(build_dir, metadata, build_logs)
            elif ecosystem == PackageEcosystem.CARGO:
                artifact_path = await self._build_cargo_package(build_dir, metadata, build_logs)
            
            if artifact_path and Path(artifact_path).exists():
                # Calculate checksum
                checksum = self._calculate_file_checksum(artifact_path)
                size_bytes = Path(artifact_path).stat().st_size
                
                artifact = PackageArtifact(
                    package_id=f"{package_id}_{ecosystem.value}",
                    ecosystem=ecosystem,
                    metadata=metadata,
                    file_path=artifact_path,
                    checksum=checksum,
                    size_bytes=size_bytes,
                    status=PackageStatus.BUILT,
                    build_logs=build_logs
                )
                
                self.logger.info(f"Built {ecosystem.value} package: {artifact_path}")
                return artifact
            
        except Exception as e:
            self.logger.error(f"Error building {ecosystem.value} package: {e}")
            return None
        finally:
            # Cleanup build directory
            if build_dir.exists():
                shutil.rmtree(build_dir, ignore_errors=True)
        
        return None

    async def _build_npm_package(self, build_dir: Path, metadata: PackageMetadata, 
                               build_logs: List[str]) -> Optional[str]:
        """Build NPM package"""
        
        # Create package.json
        package_json = {
            "name": metadata.name,
            "version": str(metadata.version),
            "description": metadata.description,
            "author": metadata.author,
            "license": metadata.license,
            "homepage": metadata.homepage,
            "repository": metadata.repository,
            "keywords": metadata.keywords,
            "dependencies": metadata.dependencies,
            "devDependencies": metadata.dev_dependencies,
            "main": metadata.entry_points.get("main", "index.js"),
            "files": ["lib/**/*", "README.md", "LICENSE"]
        }
        
        with open(build_dir / "package.json", 'w') as f:
            json.dump(package_json, f, indent=2)
        
        # Create README.md
        with open(build_dir / "README.md", 'w') as f:
            f.write(metadata.readme_content or f"# {metadata.name}\n\n{metadata.description}")
        
        # Run npm pack
        try:
            result = subprocess.run(
                ["npm", "pack"],
                cwd=build_dir,
                capture_output=True,
                text=True,
                timeout=300
            )
            
            build_logs.append(f"npm pack stdout: {result.stdout}")
            if result.stderr:
                build_logs.append(f"npm pack stderr: {result.stderr}")
            
            if result.returncode == 0:
                # Find the generated .tgz file
                tgz_files = list(build_dir.glob("*.tgz"))
                if tgz_files:
                    final_path = self.workspace / f"{metadata.name}-{metadata.version}.tgz"
                    shutil.move(str(tgz_files[0]), str(final_path))
                    return str(final_path)
            
        except subprocess.TimeoutExpired:
            build_logs.append("npm pack timed out after 5 minutes")
        except FileNotFoundError:
            build_logs.append("npm not found - install Node.js and npm")
        
        return None

    async def _build_python_package(self, build_dir: Path, metadata: PackageMetadata, 
                                  build_logs: List[str]) -> Optional[str]:
        """Build Python package"""
        
        # Create setup.py
        setup_py = f"""
from setuptools import setup, find_packages

setup(
    name="{metadata.name}",
    version="{metadata.version}",
    description="{metadata.description}",
    author="{metadata.author}",
    license="{metadata.license}",
    url="{metadata.homepage}",
    packages=find_packages(),
    install_requires={list(metadata.dependencies.keys())},
    keywords={metadata.keywords},
    classifiers=[
        "Programming Language :: Python :: 3",
        "License :: OSI Approved :: {metadata.license} License",
        "Operating System :: OS Independent",
    ],
    python_requires=">=3.7",
)
"""
        
        with open(build_dir / "setup.py", 'w') as f:
            f.write(setup_py)
        
        # Create pyproject.toml
        pyproject_toml = f"""
[build-system]
requires = ["setuptools>=45", "wheel", "setuptools_scm[toml]>=6.2"]

[project]
name = "{metadata.name}"
version = "{metadata.version}"
description = "{metadata.description}"
authors = [{{name = "{metadata.author}"}}]
license = {{text = "{metadata.license}"}}
readme = "README.md"
keywords = {metadata.keywords}
dependencies = {list(metadata.dependencies.keys())}
"""
        
        with open(build_dir / "pyproject.toml", 'w') as f:
            f.write(pyproject_toml)
        
        # Build wheel and sdist
        try:
            result = subprocess.run(
                ["python", "-m", "build"],
                cwd=build_dir,
                capture_output=True,
                text=True,
                timeout=300
            )
            
            build_logs.append(f"python build stdout: {result.stdout}")
            if result.stderr:
                build_logs.append(f"python build stderr: {result.stderr}")
            
            if result.returncode == 0:
                # Find the generated wheel file
                dist_dir = build_dir / "dist"
                if dist_dir.exists():
                    wheel_files = list(dist_dir.glob("*.whl"))
                    if wheel_files:
                        final_path = self.workspace / wheel_files[0].name
                        shutil.move(str(wheel_files[0]), str(final_path))
                        return str(final_path)
            
        except subprocess.TimeoutExpired:
            build_logs.append("Python build timed out after 5 minutes")
        except FileNotFoundError:
            build_logs.append("python build tools not found")
        
        return None

    async def _build_go_module(self, build_dir: Path, metadata: PackageMetadata, 
                             build_logs: List[str]) -> Optional[str]:
        """Build Go module"""
        
        # Create go.mod
        go_mod = f"""module {metadata.name}

go 1.19

"""
        
        # Add dependencies
        if metadata.dependencies:
            go_mod += "require (\n"
            for dep, version in metadata.dependencies.items():
                go_mod += f"    {dep} {version}\n"
            go_mod += ")\n"
        
        with open(build_dir / "go.mod", 'w') as f:
            f.write(go_mod)
        
        # Create main Go file if not exists
        main_go = build_dir / "src" / "main.go"
        if not main_go.exists():
            with open(main_go, 'w') as f:
                f.write(f"""package main

// {metadata.description}
// Version: {metadata.version}
// Author: {metadata.author}

func main() {{
    // Package implementation
}}
""")
        
        # Create module archive
        try:
            archive_path = self.workspace / f"{metadata.name}-{metadata.version}.tar.gz"
            
            with tarfile.open(archive_path, 'w:gz') as tar:
                tar.add(build_dir / "go.mod", arcname="go.mod")
                tar.add(build_dir / "src", arcname=".")
            
            build_logs.append(f"Created Go module archive: {archive_path}")
            return str(archive_path)
            
        except Exception as e:
            build_logs.append(f"Failed to create Go module archive: {e}")
        
        return None

    async def _build_maven_package(self, build_dir: Path, metadata: PackageMetadata, 
                                 build_logs: List[str]) -> Optional[str]:
        """Build Maven package"""
        
        target = self.distribution_targets.get(PackageEcosystem.MAVEN)
        group_id = target.namespace if target else "com.tusklang"
        
        # Create pom.xml
        pom_xml = f"""<?xml version="1.0" encoding="UTF-8"?>
<project xmlns="http://maven.apache.org/POM/4.0.0"
         xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
         xsi:schemaLocation="http://maven.apache.org/POM/4.0.0 
         http://maven.apache.org/xsd/maven-4.0.0.xsd">
    <modelVersion>4.0.0</modelVersion>
    
    <groupId>{group_id}</groupId>
    <artifactId>{metadata.name}</artifactId>
    <version>{metadata.version}</version>
    <packaging>jar</packaging>
    
    <name>{metadata.name}</name>
    <description>{metadata.description}</description>
    <url>{metadata.homepage}</url>
    
    <licenses>
        <license>
            <name>{metadata.license}</name>
        </license>
    </licenses>
    
    <properties>
        <maven.compiler.source>11</maven.compiler.source>
        <maven.compiler.target>11</maven.compiler.target>
        <project.build.sourceEncoding>UTF-8</project.build.sourceEncoding>
    </properties>
    
    <dependencies>
"""
        
        # Add dependencies
        for dep, version in metadata.dependencies.items():
            pom_xml += f"""        <dependency>
            <groupId>{dep.split(':')[0] if ':' in dep else 'org.example'}</groupId>
            <artifactId>{dep.split(':')[1] if ':' in dep else dep}</artifactId>
            <version>{version}</version>
        </dependency>
"""
        
        pom_xml += """    </dependencies>
</project>"""
        
        with open(build_dir / "pom.xml", 'w') as f:
            f.write(pom_xml)
        
        # Create Java source structure
        java_dir = build_dir / "src" / "main" / "java"
        java_dir.mkdir(parents=True, exist_ok=True)
        
        # Create simple Main.java if not exists
        main_java = java_dir / "Main.java"
        if not main_java.exists():
            with open(main_java, 'w') as f:
                f.write(f"""/**
 * {metadata.description}
 * Version: {metadata.version}
 * Author: {metadata.author}
 */
public class Main {{
    public static void main(String[] args) {{
        System.out.println("{metadata.name} v{metadata.version}");
    }}
}}
""")
        
        # Build with Maven
        try:
            result = subprocess.run(
                ["mvn", "clean", "package"],
                cwd=build_dir,
                capture_output=True,
                text=True,
                timeout=600
            )
            
            build_logs.append(f"mvn package stdout: {result.stdout}")
            if result.stderr:
                build_logs.append(f"mvn package stderr: {result.stderr}")
            
            if result.returncode == 0:
                # Find the generated JAR file
                target_dir = build_dir / "target"
                if target_dir.exists():
                    jar_files = list(target_dir.glob("*.jar"))
                    jar_files = [j for j in jar_files if "sources" not in j.name]
                    if jar_files:
                        final_path = self.workspace / jar_files[0].name
                        shutil.move(str(jar_files[0]), str(final_path))
                        return str(final_path)
            
        except subprocess.TimeoutExpired:
            build_logs.append("Maven build timed out after 10 minutes")
        except FileNotFoundError:
            build_logs.append("Maven not found - install Apache Maven")
        
        return None

    async def _build_nuget_package(self, build_dir: Path, metadata: PackageMetadata, 
                                 build_logs: List[str]) -> Optional[str]:
        """Build NuGet package"""
        
        # Create .csproj file
        csproj = f"""<Project Sdk="Microsoft.NET.Sdk">

  <PropertyGroup>
    <TargetFramework>net6.0</TargetFramework>
    <PackageId>{metadata.name}</PackageId>
    <Version>{metadata.version}</Version>
    <Authors>{metadata.author}</Authors>
    <Description>{metadata.description}</Description>
    <PackageLicenseExpression>{metadata.license}</PackageLicenseExpression>
    <PackageProjectUrl>{metadata.homepage}</PackageProjectUrl>
    <RepositoryUrl>{metadata.repository}</RepositoryUrl>
    <PackageTags>{';'.join(metadata.keywords)}</PackageTags>
  </PropertyGroup>

</Project>"""
        
        with open(build_dir / f"{metadata.name}.csproj", 'w') as f:
            f.write(csproj)
        
        # Create Program.cs if not exists
        program_cs = build_dir / "Program.cs"
        if not program_cs.exists():
            with open(program_cs, 'w') as f:
                f.write(f"""using System;

/// <summary>
/// {metadata.description}
/// Version: {metadata.version}
/// Author: {metadata.author}
/// </summary>
namespace {metadata.name.replace('-', '_')}
{{
    class Program
    {{
        static void Main(string[] args)
        {{
            Console.WriteLine("{metadata.name} v{metadata.version}");
        }}
    }}
}}
""")
        
        # Build with dotnet
        try:
            result = subprocess.run(
                ["dotnet", "pack"],
                cwd=build_dir,
                capture_output=True,
                text=True,
                timeout=300
            )
            
            build_logs.append(f"dotnet pack stdout: {result.stdout}")
            if result.stderr:
                build_logs.append(f"dotnet pack stderr: {result.stderr}")
            
            if result.returncode == 0:
                # Find the generated .nupkg file
                nupkg_files = list(build_dir.rglob("*.nupkg"))
                if nupkg_files:
                    final_path = self.workspace / nupkg_files[0].name
                    shutil.move(str(nupkg_files[0]), str(final_path))
                    return str(final_path)
            
        except subprocess.TimeoutExpired:
            build_logs.append("dotnet pack timed out after 5 minutes")
        except FileNotFoundError:
            build_logs.append(".NET SDK not found - install .NET SDK")
        
        return None

    async def _build_rubygem(self, build_dir: Path, metadata: PackageMetadata, 
                           build_logs: List[str]) -> Optional[str]:
        """Build Ruby gem"""
        
        gem_name = metadata.name.replace('-', '_')
        
        # Create gemspec file
        gemspec = f"""Gem::Specification.new do |spec|
  spec.name          = "{metadata.name}"
  spec.version       = "{metadata.version}"
  spec.authors       = ["{metadata.author}"]
  spec.description   = "{metadata.description}"
  spec.summary       = "{metadata.description}"
  spec.homepage      = "{metadata.homepage}"
  spec.license       = "{metadata.license}"
  
  spec.files         = Dir["lib/**/*", "README.md", "LICENSE"]
  spec.require_paths = ["lib"]
  
  # Dependencies
"""
        
        for dep, version in metadata.dependencies.items():
            gemspec += f'  spec.add_dependency "{dep}", "{version}"\n'
        
        gemspec += "end\n"
        
        with open(build_dir / f"{metadata.name}.gemspec", 'w') as f:
            f.write(gemspec)
        
        # Create lib structure
        lib_dir = build_dir / "lib"
        lib_dir.mkdir(exist_ok=True)
        
        main_rb = lib_dir / f"{gem_name}.rb"
        if not main_rb.exists():
            with open(main_rb, 'w') as f:
                f.write(f"""# {metadata.description}
# Version: {metadata.version}
# Author: {metadata.author}

module {gem_name.title().replace('_', '')}
  VERSION = "{metadata.version}"
  
  def self.hello
    puts "{metadata.name} v#{{VERSION}}"
  end
end
""")
        
        # Build gem
        try:
            result = subprocess.run(
                ["gem", "build", f"{metadata.name}.gemspec"],
                cwd=build_dir,
                capture_output=True,
                text=True,
                timeout=300
            )
            
            build_logs.append(f"gem build stdout: {result.stdout}")
            if result.stderr:
                build_logs.append(f"gem build stderr: {result.stderr}")
            
            if result.returncode == 0:
                # Find the generated .gem file
                gem_files = list(build_dir.glob("*.gem"))
                if gem_files:
                    final_path = self.workspace / gem_files[0].name
                    shutil.move(str(gem_files[0]), str(final_path))
                    return str(final_path)
            
        except subprocess.TimeoutExpired:
            build_logs.append("gem build timed out after 5 minutes")
        except FileNotFoundError:
            build_logs.append("Ruby gem not found - install Ruby and RubyGems")
        
        return None

    async def _build_cargo_package(self, build_dir: Path, metadata: PackageMetadata, 
                                 build_logs: List[str]) -> Optional[str]:
        """Build Cargo package"""
        
        # Create Cargo.toml
        cargo_toml = f"""[package]
name = "{metadata.name}"
version = "{metadata.version}"
edition = "2021"
authors = ["{metadata.author}"]
description = "{metadata.description}"
license = "{metadata.license}"
homepage = "{metadata.homepage}"
repository = "{metadata.repository}"
keywords = {metadata.keywords}

[dependencies]
"""
        
        for dep, version in metadata.dependencies.items():
            cargo_toml += f'{dep} = "{version}"\n'
        
        with open(build_dir / "Cargo.toml", 'w') as f:
            f.write(cargo_toml)
        
        # Create src/main.rs if not exists
        src_dir = build_dir / "src"
        src_dir.mkdir(exist_ok=True)
        
        main_rs = src_dir / "main.rs"
        if not main_rs.exists():
            with open(main_rs, 'w') as f:
                f.write(f"""//! {metadata.description}
//! Version: {metadata.version}
//! Author: {metadata.author}

fn main() {{
    println!("{metadata.name} v{metadata.version}");
}}
""")
        
        # Build with Cargo
        try:
            result = subprocess.run(
                ["cargo", "package"],
                cwd=build_dir,
                capture_output=True,
                text=True,
                timeout=300
            )
            
            build_logs.append(f"cargo package stdout: {result.stdout}")
            if result.stderr:
                build_logs.append(f"cargo package stderr: {result.stderr}")
            
            if result.returncode == 0:
                # Find the generated .crate file
                target_dir = build_dir / "target" / "package"
                if target_dir.exists():
                    crate_files = list(target_dir.glob("*.crate"))
                    if crate_files:
                        final_path = self.workspace / crate_files[0].name
                        shutil.move(str(crate_files[0]), str(final_path))
                        return str(final_path)
            
        except subprocess.TimeoutExpired:
            build_logs.append("cargo package timed out after 5 minutes")
        except FileNotFoundError:
            build_logs.append("Rust/Cargo not found - install Rust toolchain")
        
        return None

    def _calculate_file_checksum(self, file_path: str) -> str:
        """Calculate SHA256 checksum of file"""
        hash_sha256 = hashlib.sha256()
        with open(file_path, "rb") as f:
            for chunk in iter(lambda: f.read(4096), b""):
                hash_sha256.update(chunk)
        return hash_sha256.hexdigest()

    async def publish_package(self, artifact_id: str, dry_run: bool = False) -> Dict[str, Any]:
        """Publish package to appropriate registry"""
        
        artifact = self.artifacts.get(artifact_id)
        if not artifact:
            return {"error": "Artifact not found"}
        
        distribution_target = self.distribution_targets.get(artifact.ecosystem)
        if not distribution_target or not distribution_target.enabled:
            return {"error": f"Distribution not configured for {artifact.ecosystem.value}"}
        
        if not distribution_target.auth_token and artifact.ecosystem != PackageEcosystem.GO_MODULES:
            return {"error": f"Authentication token required for {artifact.ecosystem.value}"}
        
        result = {
            "artifact_id": artifact_id,
            "ecosystem": artifact.ecosystem.value,
            "package_name": artifact.metadata.name,
            "version": str(artifact.metadata.version),
            "dry_run": dry_run,
            "status": "success",
            "registry_url": distribution_target.registry_url,
            "published_at": datetime.utcnow().isoformat()
        }
        
        if not dry_run:
            try:
                # Execute actual publishing based on ecosystem
                if artifact.ecosystem == PackageEcosystem.NPM:
                    success = await self._publish_npm(artifact, distribution_target)
                elif artifact.ecosystem == PackageEcosystem.PYPI:
                    success = await self._publish_pypi(artifact, distribution_target)
                elif artifact.ecosystem == PackageEcosystem.GO_MODULES:
                    success = await self._publish_go_module(artifact, distribution_target)
                elif artifact.ecosystem == PackageEcosystem.MAVEN:
                    success = await self._publish_maven(artifact, distribution_target)
                elif artifact.ecosystem == PackageEcosystem.NUGET:
                    success = await self._publish_nuget(artifact, distribution_target)
                elif artifact.ecosystem == PackageEcosystem.RUBYGEMS:
                    success = await self._publish_rubygem(artifact, distribution_target)
                elif artifact.ecosystem == PackageEcosystem.CARGO:
                    success = await self._publish_cargo(artifact, distribution_target)
                else:
                    success = False
                
                if success:
                    artifact.status = PackageStatus.PUBLISHED
                    artifact.published_at = datetime.utcnow()
                    result["status"] = "published"
                else:
                    artifact.status = PackageStatus.FAILED
                    result["status"] = "failed"
                    result["error"] = "Publishing failed"
                    
            except Exception as e:
                self.logger.error(f"Publishing failed for {artifact_id}: {e}")
                artifact.status = PackageStatus.FAILED
                result["status"] = "failed"
                result["error"] = str(e)
        
        return result

    async def _publish_npm(self, artifact: PackageArtifact, 
                         target: DistributionTarget) -> bool:
        """Publish to NPM registry"""
        try:
            env = os.environ.copy()
            env["NPM_TOKEN"] = target.auth_token
            
            result = subprocess.run(
                ["npm", "publish", artifact.file_path, "--access", "public"],
                env=env,
                capture_output=True,
                text=True,
                timeout=300
            )
            
            return result.returncode == 0
            
        except Exception as e:
            self.logger.error(f"NPM publish error: {e}")
            return False

    async def _publish_pypi(self, artifact: PackageArtifact, 
                          target: DistributionTarget) -> bool:
        """Publish to PyPI"""
        try:
            result = subprocess.run([
                "python", "-m", "twine", "upload", 
                "--username", "__token__",
                "--password", target.auth_token,
                artifact.file_path
            ], capture_output=True, text=True, timeout=300)
            
            return result.returncode == 0
            
        except Exception as e:
            self.logger.error(f"PyPI publish error: {e}")
            return False

    async def _publish_go_module(self, artifact: PackageArtifact, 
                               target: DistributionTarget) -> bool:
        """Publish Go module (via Git tags)"""
        # Go modules are published via Git tags, not direct upload
        self.logger.info("Go modules publish via Git tags - manual step required")
        return True

    async def _publish_maven(self, artifact: PackageArtifact, 
                           target: DistributionTarget) -> bool:
        """Publish to Maven Central"""
        try:
            result = subprocess.run([
                "mvn", "deploy:deploy-file",
                f"-Durl={target.registry_url}",
                f"-DrepositoryId=ossrh",
                f"-Dfile={artifact.file_path}",
                f"-DgroupId={target.namespace}",
                f"-DartifactId={artifact.metadata.name}",
                f"-Dversion={artifact.metadata.version}",
                "-Dpackaging=jar"
            ], capture_output=True, text=True, timeout=600)
            
            return result.returncode == 0
            
        except Exception as e:
            self.logger.error(f"Maven publish error: {e}")
            return False

    async def _publish_nuget(self, artifact: PackageArtifact, 
                           target: DistributionTarget) -> bool:
        """Publish to NuGet"""
        try:
            result = subprocess.run([
                "dotnet", "nuget", "push", artifact.file_path,
                "--api-key", target.auth_token,
                "--source", target.registry_url
            ], capture_output=True, text=True, timeout=300)
            
            return result.returncode == 0
            
        except Exception as e:
            self.logger.error(f"NuGet publish error: {e}")
            return False

    async def _publish_rubygem(self, artifact: PackageArtifact, 
                             target: DistributionTarget) -> bool:
        """Publish to RubyGems"""
        try:
            # Set up credentials
            credentials_file = Path.home() / ".gem" / "credentials"
            credentials_file.parent.mkdir(exist_ok=True)
            
            with open(credentials_file, 'w') as f:
                f.write(f"---\n:rubygems_api_key: {target.auth_token}\n")
            
            result = subprocess.run([
                "gem", "push", artifact.file_path
            ], capture_output=True, text=True, timeout=300)
            
            return result.returncode == 0
            
        except Exception as e:
            self.logger.error(f"RubyGems publish error: {e}")
            return False

    async def _publish_cargo(self, artifact: PackageArtifact, 
                           target: DistributionTarget) -> bool:
        """Publish to Cargo (crates.io)"""
        try:
            result = subprocess.run([
                "cargo", "publish", "--token", target.auth_token
            ], capture_output=True, text=True, timeout=300)
            
            return result.returncode == 0
            
        except Exception as e:
            self.logger.error(f"Cargo publish error: {e}")
            return False

    def get_package_status(self, artifact_id: str) -> Optional[Dict[str, Any]]:
        """Get package build/publish status"""
        artifact = self.artifacts.get(artifact_id)
        if not artifact:
            return None
        
        return {
            "artifact_id": artifact_id,
            "package_name": artifact.metadata.name,
            "version": str(artifact.metadata.version),
            "ecosystem": artifact.ecosystem.value,
            "status": artifact.status.value,
            "file_path": artifact.file_path,
            "checksum": artifact.checksum,
            "size_bytes": artifact.size_bytes,
            "created_at": artifact.created_at.isoformat(),
            "published_at": artifact.published_at.isoformat() if artifact.published_at else None,
            "build_logs": artifact.build_logs
        }

    def list_packages(self) -> List[Dict[str, Any]]:
        """List all package artifacts"""
        return [self.get_package_status(artifact_id) for artifact_id in self.artifacts.keys()]

    def __del__(self):
        """Cleanup resources"""
        pass 