#!/usr/bin/env python3
"""
Performance Testing for TuskTSK CLI Commands
============================================
Measures and optimizes command execution time
"""

import sys
import os
import json
import time
import argparse
import subprocess
import statistics
from pathlib import Path
from typing import Dict, List, Any, Optional, Tuple
from datetime import datetime
import psutil
import threading
import concurrent.futures

# Add parent directory to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))


class PerformanceTester:
    """Tests CLI command performance"""
    
    def __init__(self, verbose: bool = False):
        self.verbose = verbose
        self.performance_results = {
            'timestamp': datetime.now().isoformat(),
            'commands_tested': 0,
            'total_execution_time': 0,
            'average_execution_time': 0,
            'slowest_command': None,
            'fastest_command': None,
            'command_details': [],
            'performance_issues': [],
            'recommendations': []
        }
        
    def measure_command_performance(self, command: str, args: List[str] = None, iterations: int = 5) -> Dict[str, Any]:
        """Measure performance of a single command"""
        if args is None:
            args = []
        
        execution_times = []
        memory_usage = []
        cpu_usage = []
        
        print(f"⚡ Testing performance: {' '.join([command] + args)}")
        
        for i in range(iterations):
            if self.verbose:
                print(f"  Run {i+1}/{iterations}...")
            
            # Start monitoring
            process = psutil.Process()
            start_time = time.time()
            start_memory = process.memory_info().rss
            start_cpu = process.cpu_percent()
            
            # Execute command
            try:
                cmd = ['python', '-m', 'tusktsk.cli.main', command] + args
                result = subprocess.run(
                    cmd,
                    capture_output=True,
                    text=True,
                    timeout=30
                )
                
                end_time = time.time()
                end_memory = process.memory_info().rss
                end_cpu = process.cpu_percent()
                
                execution_time = end_time - start_time
                memory_delta = end_memory - start_memory
                cpu_avg = (start_cpu + end_cpu) / 2
                
                execution_times.append(execution_time)
                memory_usage.append(memory_delta)
                cpu_usage.append(cpu_avg)
                
                if self.verbose:
                    print(f"    Execution time: {execution_time:.3f}s")
                    print(f"    Memory delta: {memory_delta / 1024:.1f}KB")
                    print(f"    CPU usage: {cpu_avg:.1f}%")
                
            except subprocess.TimeoutExpired:
                execution_times.append(30.0)  # Timeout
                memory_usage.append(0)
                cpu_usage.append(0)
                if self.verbose:
                    print("    Timeout after 30s")
            except Exception as e:
                if self.verbose:
                    print(f"    Error: {e}")
                break
        
        if not execution_times:
            return {
                'command': command,
                'args': args,
                'status': 'failed',
                'error': 'No successful executions'
            }
        
        # Calculate statistics
        avg_time = statistics.mean(execution_times)
        min_time = min(execution_times)
        max_time = max(execution_times)
        std_dev = statistics.stdev(execution_times) if len(execution_times) > 1 else 0
        
        avg_memory = statistics.mean(memory_usage) if memory_usage else 0
        avg_cpu = statistics.mean(cpu_usage) if cpu_usage else 0
        
        result = {
            'command': command,
            'args': args,
            'status': 'success',
            'iterations': len(execution_times),
            'execution_time': {
                'average': round(avg_time, 3),
                'minimum': round(min_time, 3),
                'maximum': round(max_time, 3),
                'std_deviation': round(std_dev, 3)
            },
            'resource_usage': {
                'memory_delta_kb': round(avg_memory / 1024, 1),
                'cpu_percent': round(avg_cpu, 1)
            },
            'performance_rating': self.rate_performance(avg_time, avg_memory, avg_cpu)
        }
        
        return result
    
    def rate_performance(self, execution_time: float, memory_delta: float, cpu_usage: float) -> str:
        """Rate command performance"""
        if execution_time < 0.1:
            return 'excellent'
        elif execution_time < 0.5:
            return 'good'
        elif execution_time < 2.0:
            return 'acceptable'
        elif execution_time < 10.0:
            return 'slow'
        else:
            return 'very_slow'
    
    def test_all_commands(self) -> Dict[str, Any]:
        """Test performance of all CLI commands"""
        # Define test commands with their arguments
        test_commands = [
            ('db', ['status']),
            ('db', ['init']),
            ('config', ['list']),
            ('cache', ['status']),
            ('test', ['unit']),
            ('serve', ['3000']),
            ('utility', ['format', 'test.tsk']),
            ('binary', ['compile', 'test.tsk']),
            ('ai', ['analyze', 'test input']),
            ('peanuts', ['list']),
            ('css', ['validate', 'test.css']),
            ('license', ['check']),
            ('dependency', ['list'])
        ]
        
        print(f"🚀 Testing performance of {len(test_commands)} commands...")
        
        for command, args in test_commands:
            result = self.measure_command_performance(command, args)
            self.performance_results['command_details'].append(result)
            self.performance_results['commands_tested'] += 1
            
            if result['status'] == 'success':
                self.performance_results['total_execution_time'] += result['execution_time']['average']
        
        # Calculate overall statistics
        if self.performance_results['commands_tested'] > 0:
            self.performance_results['average_execution_time'] = (
                self.performance_results['total_execution_time'] / self.performance_results['commands_tested']
            )
        
        # Find slowest and fastest commands
        successful_commands = [cmd for cmd in self.performance_results['command_details'] if cmd['status'] == 'success']
        
        if successful_commands:
            slowest = max(successful_commands, key=lambda x: x['execution_time']['average'])
            fastest = min(successful_commands, key=lambda x: x['execution_time']['average'])
            
            self.performance_results['slowest_command'] = {
                'command': slowest['command'],
                'args': slowest['args'],
                'time': slowest['execution_time']['average']
            }
            
            self.performance_results['fastest_command'] = {
                'command': fastest['command'],
                'args': fastest['args'],
                'time': fastest['execution_time']['average']
            }
        
        # Generate recommendations
        self.generate_performance_recommendations()
        
        return self.performance_results
    
    def generate_performance_recommendations(self):
        """Generate performance improvement recommendations"""
        recommendations = []
        
        # Check for slow commands
        slow_commands = [
            cmd for cmd in self.performance_results['command_details']
            if cmd['status'] == 'success' and cmd['performance_rating'] in ['slow', 'very_slow']
        ]
        
        if slow_commands:
            recommendations.append({
                'type': 'slow_commands',
                'message': f"Found {len(slow_commands)} slow commands that need optimization",
                'commands': [f"{cmd['command']} {' '.join(cmd['args'])}" for cmd in slow_commands]
            })
        
        # Check for high memory usage
        high_memory_commands = [
            cmd for cmd in self.performance_results['command_details']
            if cmd['status'] == 'success' and cmd['resource_usage']['memory_delta_kb'] > 1024
        ]
        
        if high_memory_commands:
            recommendations.append({
                'type': 'high_memory',
                'message': f"Found {len(high_memory_commands)} commands with high memory usage",
                'commands': [f"{cmd['command']} {' '.join(cmd['args'])}" for cmd in high_memory_commands]
            })
        
        # Check for high CPU usage
        high_cpu_commands = [
            cmd for cmd in self.performance_results['command_details']
            if cmd['status'] == 'success' and cmd['resource_usage']['cpu_percent'] > 50
        ]
        
        if high_cpu_commands:
            recommendations.append({
                'type': 'high_cpu',
                'message': f"Found {len(high_cpu_commands)} commands with high CPU usage",
                'commands': [f"{cmd['command']} {' '.join(cmd['args'])}" for cmd in high_cpu_commands]
            })
        
        # General recommendations
        if self.performance_results['average_execution_time'] > 1.0:
            recommendations.append({
                'type': 'general',
                'message': "Overall average execution time is high. Consider implementing caching and optimization."
            })
        
        self.performance_results['recommendations'] = recommendations
    
    def test_concurrent_execution(self, max_workers: int = 4) -> Dict[str, Any]:
        """Test concurrent command execution"""
        print(f"🔄 Testing concurrent execution with {max_workers} workers...")
        
        test_commands = [
            ('db', ['status']),
            ('config', ['list']),
            ('cache', ['status']),
            ('test', ['unit'])
        ]
        
        concurrent_results = {
            'max_workers': max_workers,
            'total_commands': len(test_commands),
            'execution_times': [],
            'successful_executions': 0,
            'failed_executions': 0
        }
        
        start_time = time.time()
        
        with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as executor:
            # Submit all commands
            future_to_command = {
                executor.submit(self.measure_command_performance, cmd, args): (cmd, args)
                for cmd, args in test_commands
            }
            
            # Collect results
            for future in concurrent.futures.as_completed(future_to_command):
                command, args = future_to_command[future]
                try:
                    result = future.result()
                    if result['status'] == 'success':
                        concurrent_results['successful_executions'] += 1
                        concurrent_results['execution_times'].append(result['execution_time']['average'])
                    else:
                        concurrent_results['failed_executions'] += 1
                except Exception as e:
                    concurrent_results['failed_executions'] += 1
                    if self.verbose:
                        print(f"Error in concurrent execution: {e}")
        
        end_time = time.time()
        concurrent_results['total_time'] = end_time - start_time
        
        if concurrent_results['execution_times']:
            concurrent_results['average_time'] = statistics.mean(concurrent_results['execution_times'])
            concurrent_results['throughput'] = concurrent_results['successful_executions'] / concurrent_results['total_time']
        
        return concurrent_results
    
    def test_memory_usage(self) -> Dict[str, Any]:
        """Test memory usage patterns"""
        print("🧠 Testing memory usage patterns...")
        
        memory_results = {
            'baseline_memory': 0,
            'peak_memory': 0,
            'memory_leaks': [],
            'commands_memory_usage': []
        }
        
        process = psutil.Process()
        
        # Get baseline memory
        memory_results['baseline_memory'] = process.memory_info().rss
        
        # Test memory usage for each command
        test_commands = [
            ('db', ['status']),
            ('config', ['list']),
            ('cache', ['status']),
            ('test', ['unit'])
        ]
        
        for command, args in test_commands:
            # Measure memory before
            memory_before = process.memory_info().rss
            
            # Execute command
            try:
                cmd = ['python', '-m', 'tusktsk.cli.main', command] + args
                subprocess.run(cmd, capture_output=True, text=True, timeout=10)
                
                # Measure memory after
                memory_after = process.memory_info().rss
                memory_delta = memory_after - memory_before
                
                memory_results['commands_memory_usage'].append({
                    'command': command,
                    'args': args,
                    'memory_before_mb': round(memory_before / 1024 / 1024, 2),
                    'memory_after_mb': round(memory_after / 1024 / 1024, 2),
                    'memory_delta_mb': round(memory_delta / 1024 / 1024, 2)
                })
                
                # Check for potential memory leaks
                if memory_delta > 10 * 1024 * 1024:  # 10MB threshold
                    memory_results['memory_leaks'].append({
                        'command': command,
                        'args': args,
                        'memory_delta_mb': round(memory_delta / 1024 / 1024, 2)
                    })
                
                # Update peak memory
                memory_results['peak_memory'] = max(memory_results['peak_memory'], memory_after)
                
            except Exception as e:
                if self.verbose:
                    print(f"Error testing memory for {command}: {e}")
        
        return memory_results
    
    def generate_performance_report(self, output_file: str = None) -> str:
        """Generate comprehensive performance report"""
        if not output_file:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_file = f"performance_report_{timestamp}.json"
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        with open(output_path, 'w') as f:
            json.dump(self.performance_results, f, indent=2, default=str)
        
        return output_path
    
    def print_summary(self):
        """Print performance summary"""
        print("\n" + "=" * 60)
        print("📊 Performance Test Summary")
        print("=" * 60)
        print(f"Commands Tested: {self.performance_results['commands_tested']}")
        print(f"Total Execution Time: {self.performance_results['total_execution_time']:.3f}s")
        print(f"Average Execution Time: {self.performance_results['average_execution_time']:.3f}s")
        
        if self.performance_results['slowest_command']:
            slowest = self.performance_results['slowest_command']
            print(f"Slowest Command: {' '.join([slowest['command']] + slowest['args'])} ({slowest['time']:.3f}s)")
        
        if self.performance_results['fastest_command']:
            fastest = self.performance_results['fastest_command']
            print(f"Fastest Command: {' '.join([fastest['command']] + fastest['args'])} ({fastest['time']:.3f}s)")
        
        if self.performance_results['recommendations']:
            print(f"\n💡 Recommendations:")
            for rec in self.performance_results['recommendations']:
                print(f"  - {rec['message']}")
                if 'commands' in rec:
                    for cmd in rec['commands']:
                        print(f"    * {cmd}")


def main():
    """Main entry point for performance testing"""
    parser = argparse.ArgumentParser(description='TuskTSK CLI Performance Tester')
    parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose output')
    parser.add_argument('--command', help='Test specific command')
    parser.add_argument('--args', nargs='*', help='Command arguments')
    parser.add_argument('--iterations', type=int, default=5, help='Number of test iterations')
    parser.add_argument('--concurrent', action='store_true', help='Test concurrent execution')
    parser.add_argument('--memory', action='store_true', help='Test memory usage')
    parser.add_argument('--output', '-o', help='Output file for performance results')
    parser.add_argument('--all', action='store_true', help='Run all performance tests')
    
    args = parser.parse_args()
    
    # Create performance tester
    tester = PerformanceTester(verbose=args.verbose)
    
    # Run tests
    if args.all:
        print("🚀 Running comprehensive performance tests...")
        tester.test_all_commands()
        
        if args.concurrent:
            concurrent_results = tester.test_concurrent_execution()
            tester.performance_results['concurrent_results'] = concurrent_results
        
        if args.memory:
            memory_results = tester.test_memory_usage()
            tester.performance_results['memory_results'] = memory_results
    else:
        if args.command:
            result = tester.measure_command_performance(args.command, args.args, args.iterations)
            tester.performance_results['command_details'].append(result)
            tester.performance_results['commands_tested'] = 1
        
        if args.concurrent:
            concurrent_results = tester.test_concurrent_execution()
            tester.performance_results['concurrent_results'] = concurrent_results
        
        if args.memory:
            memory_results = tester.test_memory_usage()
            tester.performance_results['memory_results'] = memory_results
    
    # Print summary
    tester.print_summary()
    
    # Save results
    if args.output:
        output_file = tester.generate_performance_report(args.output)
    else:
        output_file = tester.generate_performance_report()
    
    if args.verbose:
        print(f"\n💾 Performance results saved to: {output_file}")
    
    # Exit with appropriate code
    slow_commands = len([cmd for cmd in tester.performance_results['command_details'] 
                        if cmd.get('performance_rating') in ['slow', 'very_slow']])
    sys.exit(0 if slow_commands == 0 else 1)


if __name__ == '__main__':
    main() 