#!/usr/bin/env python3
"""
Code Linting for TuskTSK CLI Commands
=====================================
Lints CLI command implementations for code quality and consistency
"""

import sys
import os
import json
import ast
import argparse
from pathlib import Path
from typing import Dict, List, Any, Optional, Set
from datetime import datetime
import subprocess

# Add parent directory to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))


class CommandLinter:
    """Lints CLI command implementations"""
    
    def __init__(self, verbose: bool = False):
        self.verbose = verbose
        self.lint_results = {
            'timestamp': datetime.now().isoformat(),
            'files_checked': 0,
            'files_with_issues': 0,
            'total_issues': 0,
            'issues_by_type': {},
            'file_details': [],
            'errors': []
        }
        
    def lint_file(self, file_path: str) -> Dict[str, Any]:
        """Lint a single Python file"""
        file_result = {
            'file': file_path,
            'issues': [],
            'score': 100,
            'has_errors': False
        }
        
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Parse AST
            tree = ast.parse(content)
            
            # Run various linting checks
            issues = []
            
            # Check for common issues
            issues.extend(self.check_imports(tree, file_path))
            issues.extend(self.check_function_definitions(tree, file_path))
            issues.extend(self.check_variable_usage(tree, file_path))
            issues.extend(self.check_error_handling(tree, file_path))
            issues.extend(self.check_documentation(tree, file_path))
            issues.extend(self.check_code_style(tree, file_path))
            issues.extend(self.check_security_issues(tree, file_path))
            
            file_result['issues'] = issues
            file_result['total_issues'] = len(issues)
            
            # Calculate score (100 - issues * 2, minimum 0)
            file_result['score'] = max(0, 100 - len(issues) * 2)
            
            if issues:
                file_result['has_errors'] = True
                self.lint_results['files_with_issues'] += 1
            
            self.lint_results['files_checked'] += 1
            self.lint_results['total_issues'] += len(issues)
            
            # Update issue counts by type
            for issue in issues:
                issue_type = issue['type']
                if issue_type not in self.lint_results['issues_by_type']:
                    self.lint_results['issues_by_type'][issue_type] = 0
                self.lint_results['issues_by_type'][issue_type] += 1
            
            self.lint_results['file_details'].append(file_result)
            
        except Exception as e:
            self.lint_results['errors'].append(f"Error linting {file_path}: {e}")
            file_result['has_errors'] = True
            file_result['issues'].append({
                'line': 0,
                'type': 'parse_error',
                'message': f"Failed to parse file: {e}",
                'severity': 'error'
            })
        
        return file_result
    
    def check_imports(self, tree: ast.AST, file_path: str) -> List[Dict[str, Any]]:
        """Check import statements"""
        issues = []
        
        for node in ast.walk(tree):
            if isinstance(node, ast.Import):
                for alias in node.names:
                    # Check for unused imports
                    if alias.name.startswith('_'):
                        issues.append({
                            'line': node.lineno,
                            'type': 'unused_import',
                            'message': f"Unused import: {alias.name}",
                            'severity': 'warning'
                        })
            
            elif isinstance(node, ast.ImportFrom):
                # Check for wildcard imports
                if node.names[0].name == '*':
                    issues.append({
                        'line': node.lineno,
                        'type': 'wildcard_import',
                        'message': "Avoid wildcard imports",
                        'severity': 'warning'
                    })
        
        return issues
    
    def check_function_definitions(self, tree: ast.AST, file_path: str) -> List[Dict[str, Any]]:
        """Check function definitions"""
        issues = []
        
        for node in ast.walk(tree):
            if isinstance(node, ast.FunctionDef):
                # Check for missing docstrings
                if not ast.get_docstring(node):
                    issues.append({
                        'line': node.lineno,
                        'type': 'missing_docstring',
                        'message': f"Function '{node.name}' missing docstring",
                        'severity': 'warning'
                    })
                
                # Check for long functions
                function_lines = self.count_function_lines(node)
                if function_lines > 50:
                    issues.append({
                        'line': node.lineno,
                        'type': 'long_function',
                        'message': f"Function '{node.name}' is {function_lines} lines long (consider breaking it down)",
                        'severity': 'warning'
                    })
                
                # Check for too many arguments
                if len(node.args.args) > 7:
                    issues.append({
                        'line': node.lineno,
                        'type': 'too_many_arguments',
                        'message': f"Function '{node.name}' has {len(node.args.args)} arguments (consider using a config object)",
                        'severity': 'warning'
                    })
        
        return issues
    
    def count_function_lines(self, node: ast.FunctionDef) -> int:
        """Count lines in a function"""
        if not node.body:
            return 0
        
        start_line = node.lineno
        end_line = max(child.lineno for child in ast.walk(node) if hasattr(child, 'lineno'))
        return end_line - start_line + 1
    
    def check_variable_usage(self, tree: ast.AST, file_path: str) -> List[Dict[str, Any]]:
        """Check variable usage"""
        issues = []
        
        for node in ast.walk(tree):
            if isinstance(node, ast.Name):
                # Check for single letter variable names (except in loops)
                if len(node.id) == 1 and node.id not in ['i', 'j', 'k']:
                    parent = self.get_parent_node(tree, node)
                    if not (isinstance(parent, ast.For) and parent.target == node):
                        issues.append({
                            'line': node.lineno,
                            'type': 'single_letter_variable',
                            'message': f"Single letter variable name: {node.id}",
                            'severity': 'warning'
                        })
        
        return issues
    
    def get_parent_node(self, tree: ast.AST, target_node: ast.AST) -> Optional[ast.AST]:
        """Get parent node of target node"""
        for node in ast.walk(tree):
            for child in ast.iter_child_nodes(node):
                if child == target_node:
                    return node
        return None
    
    def check_error_handling(self, tree: ast.AST, file_path: str) -> List[Dict[str, Any]]:
        """Check error handling"""
        issues = []
        
        for node in ast.walk(tree):
            if isinstance(node, ast.Try):
                # Check for bare except clauses
                for handler in node.handlers:
                    if handler.type is None:
                        issues.append({
                            'line': handler.lineno,
                            'type': 'bare_except',
                            'message': "Bare except clause - specify exception type",
                            'severity': 'error'
                        })
        
        return issues
    
    def check_documentation(self, tree: ast.AST, file_path: str) -> List[Dict[str, Any]]:
        """Check documentation"""
        issues = []
        
        # Check module docstring
        module_docstring = ast.get_docstring(tree)
        if not module_docstring:
            issues.append({
                'line': 1,
                'type': 'missing_module_docstring',
                'message': "Module missing docstring",
                'severity': 'warning'
            })
        
        # Check class docstrings
        for node in ast.walk(tree):
            if isinstance(node, ast.ClassDef):
                if not ast.get_docstring(node):
                    issues.append({
                        'line': node.lineno,
                        'type': 'missing_class_docstring',
                        'message': f"Class '{node.name}' missing docstring",
                        'severity': 'warning'
                    })
        
        return issues
    
    def check_code_style(self, tree: ast.AST, file_path: str) -> List[Dict[str, Any]]:
        """Check code style"""
        issues = []
        
        for node in ast.walk(tree):
            if isinstance(node, ast.Compare):
                # Check for comparison with None
                for comparator in node.comparators:
                    if isinstance(comparator, ast.Constant) and comparator.value is None:
                        issues.append({
                            'line': node.lineno,
                            'type': 'none_comparison',
                            'message': "Use 'is None' instead of '== None'",
                            'severity': 'warning'
                        })
        
        return issues
    
    def check_security_issues(self, tree: ast.AST, file_path: str) -> List[Dict[str, Any]]:
        """Check for security issues"""
        issues = []
        
        for node in ast.walk(tree):
            if isinstance(node, ast.Call):
                # Check for dangerous functions
                if isinstance(node.func, ast.Name):
                    dangerous_functions = ['eval', 'exec', 'input']
                    if node.func.id in dangerous_functions:
                        issues.append({
                            'line': node.lineno,
                            'type': 'dangerous_function',
                            'message': f"Dangerous function call: {node.func.id}",
                            'severity': 'error'
                        })
        
        return issues
    
    def lint_cli_commands(self) -> Dict[str, Any]:
        """Lint all CLI command files"""
        cli_dir = os.path.join(os.path.dirname(__file__), '..', '..', 'cli', 'commands')
        
        if not os.path.exists(cli_dir):
            self.lint_results['errors'].append(f"CLI commands directory not found: {cli_dir}")
            return self.lint_results
        
        # Find all Python files in commands directory
        command_files = []
        for root, dirs, files in os.walk(cli_dir):
            for file in files:
                if file.endswith('.py') and not file.startswith('__'):
                    command_files.append(os.path.join(root, file))
        
        print(f"🔍 Linting {len(command_files)} CLI command files...")
        
        for file_path in command_files:
            if self.verbose:
                print(f"  📝 Linting: {os.path.basename(file_path)}")
            
            self.lint_file(file_path)
        
        return self.lint_results
    
    def run_external_linters(self) -> Dict[str, Any]:
        """Run external linters (flake8, pylint, etc.)"""
        external_results = {}
        
        # Try to run flake8
        try:
            result = subprocess.run(
                ['flake8', '--format=json'],
                capture_output=True,
                text=True,
                cwd=os.path.join(os.path.dirname(__file__), '..', '..')
            )
            
            if result.returncode == 0:
                external_results['flake8'] = {
                    'status': 'passed',
                    'issues': 0
                }
            else:
                try:
                    flake8_issues = json.loads(result.stdout)
                    external_results['flake8'] = {
                        'status': 'failed',
                        'issues': len(flake8_issues),
                        'details': flake8_issues
                    }
                except json.JSONDecodeError:
                    external_results['flake8'] = {
                        'status': 'error',
                        'message': 'Failed to parse flake8 output'
                    }
        except FileNotFoundError:
            external_results['flake8'] = {
                'status': 'not_available',
                'message': 'flake8 not installed'
            }
        
        # Try to run pylint
        try:
            result = subprocess.run(
                ['pylint', '--output-format=json'],
                capture_output=True,
                text=True,
                cwd=os.path.join(os.path.dirname(__file__), '..', '..')
            )
            
            if result.returncode == 0:
                external_results['pylint'] = {
                    'status': 'passed',
                    'issues': 0
                }
            else:
                try:
                    pylint_issues = json.loads(result.stdout)
                    external_results['pylint'] = {
                        'status': 'failed',
                        'issues': len(pylint_issues),
                        'details': pylint_issues
                    }
                except json.JSONDecodeError:
                    external_results['pylint'] = {
                        'status': 'error',
                        'message': 'Failed to parse pylint output'
                    }
        except FileNotFoundError:
            external_results['pylint'] = {
                'status': 'not_available',
                'message': 'pylint not installed'
            }
        
        return external_results
    
    def generate_lint_report(self, output_file: str = None) -> str:
        """Generate comprehensive linting report"""
        if not output_file:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_file = f"lint_report_{timestamp}.json"
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        # Add external linter results
        self.lint_results['external_linters'] = self.run_external_linters()
        
        with open(output_path, 'w') as f:
            json.dump(self.lint_results, f, indent=2, default=str)
        
        return output_path
    
    def print_summary(self):
        """Print linting summary"""
        print("\n" + "=" * 60)
        print("📋 Linting Summary")
        print("=" * 60)
        print(f"Files Checked: {self.lint_results['files_checked']}")
        print(f"Files with Issues: {self.lint_results['files_with_issues']}")
        print(f"Total Issues: {self.lint_results['total_issues']}")
        
        if self.lint_results['issues_by_type']:
            print(f"\nIssues by Type:")
            for issue_type, count in self.lint_results['issues_by_type'].items():
                print(f"  - {issue_type}: {count}")
        
        # Calculate overall score
        if self.lint_results['files_checked'] > 0:
            total_score = sum(file['score'] for file in self.lint_results['file_details'])
            average_score = total_score / self.lint_results['files_checked']
            print(f"\nOverall Code Quality Score: {average_score:.1f}/100")
        
        if self.lint_results['errors']:
            print(f"\n❌ Errors:")
            for error in self.lint_results['errors']:
                print(f"   - {error}")


def main():
    """Main entry point for command linting"""
    parser = argparse.ArgumentParser(description='TuskTSK CLI Command Linter')
    parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose output')
    parser.add_argument('--file', help='Lint specific file')
    parser.add_argument('--output', '-o', help='Output file for lint results')
    parser.add_argument('--external', action='store_true', help='Run external linters')
    
    args = parser.parse_args()
    
    # Create linter
    linter = CommandLinter(verbose=args.verbose)
    
    # Run linting
    if args.file:
        print(f"🔍 Linting specific file: {args.file}")
        linter.lint_file(args.file)
    else:
        print("🚀 Running comprehensive CLI command linting...")
        linter.lint_cli_commands()
    
    # Print summary
    linter.print_summary()
    
    # Save results
    if args.output:
        output_file = linter.generate_lint_report(args.output)
    else:
        output_file = linter.generate_lint_report()
    
    if args.verbose:
        print(f"\n💾 Lint results saved to: {output_file}")
    
    # Exit with appropriate code
    sys.exit(0 if linter.lint_results['total_issues'] == 0 else 1)


if __name__ == '__main__':
    main() 