#!/usr/bin/env python3
"""
Test script for TuskTSK AI components
Tests basic import and functionality without requiring all dependencies
"""

import sys
import os
import traceback

def test_imports():
    """Test importing AI components"""
    print("Testing AI component imports...")
    
    try:
        # Test basic imports
        from advanced_features.ai.ai_code_generator import AICodeGenerator
        print("✓ AI Code Generator imported successfully")
        
        from advanced_features.ai.predictive_optimizer import PredictiveOptimizer
        print("✓ Predictive Optimizer imported successfully")
        
        from advanced_features.ai.syntax_intelligence import AISyntaxIntelligence
        print("✓ Syntax Intelligence imported successfully")
        
        from advanced_features.ai.machine_learning_engine import MachineLearningEngine
        print("✓ Machine Learning Engine imported successfully")
        
        from advanced_features.ai.deep_learning_engine import DeepLearningEngine
        print("✓ Deep Learning Engine imported successfully")
        
        from advanced_features.ai.goal_implementation import Goal8Implementation
        print("✓ Goal Implementation imported successfully")
        
        return True
        
    except Exception as e:
        print(f"✗ Import failed: {e}")
        traceback.print_exc()
        return False

def test_basic_functionality():
    """Test basic functionality of AI components"""
    print("\nTesting basic functionality...")
    
    try:
        # Test AI Code Generator
        from advanced_features.ai.ai_code_generator import AICodeGenerator, CodeGenerationRequest, CodeGenerationType
        
        generator = AICodeGenerator()
        request = CodeGenerationRequest(
            description="Create a simple math operator",
            generation_type=CodeGenerationType.OPERATOR
        )
        
        result = generator.generate_code(request)
        print(f"✓ AI Code Generator: Generated code with confidence {result.confidence}")
        
        # Test Predictive Optimizer
        from advanced_features.ai.predictive_optimizer import PredictiveOptimizer
        
        optimizer = PredictiveOptimizer()
        print("✓ Predictive Optimizer: Initialized successfully")
        
        # Test Syntax Intelligence
        from advanced_features.ai.syntax_intelligence import AISyntaxIntelligence, CodeContext
        
        syntax_ai = AISyntaxIntelligence()
        context = CodeContext(current_line="", cursor_position=0)
        suggestions = syntax_ai.get_autocomplete(context)
        print(f"✓ Syntax Intelligence: Generated {len(suggestions)} suggestions")
        
        return True
        
    except Exception as e:
        print(f"✗ Functionality test failed: {e}")
        traceback.print_exc()
        return False

def test_g7_integration():
    """Test g7 component integration"""
    print("\nTesting g7 integration...")
    
    try:
        from advanced_features.g7.error_handler import handle_errors
        from advanced_features.g7.monitoring_framework import monitor_operation
        from advanced_features.g7.performance_engine import optimize_operation
        
        print("✓ g7 components imported successfully")
        
        # Test decorator usage
        @monitor_operation("test_operation")
        @handle_errors(retry=True)
        def test_function():
            return "success"
        
        result = test_function()
        print(f"✓ Decorator test: {result}")
        
        return True
        
    except Exception as e:
        print(f"✗ g7 integration failed: {e}")
        traceback.print_exc()
        return False

def main():
    """Main test function"""
    print("TuskTSK AI Components Test")
    print("=" * 40)
    
    # Change to the correct directory
    os.chdir(os.path.dirname(os.path.abspath(__file__)))
    
    # Run tests
    import_success = test_imports()
    g7_success = test_g7_integration()
    
    # Only test functionality if imports work
    if import_success and g7_success:
        func_success = test_basic_functionality()
    else:
        func_success = False
    
    # Summary
    print("\n" + "=" * 40)
    print("Test Summary:")
    print(f"Imports: {'✓ PASS' if import_success else '✗ FAIL'}")
    print(f"g7 Integration: {'✓ PASS' if g7_success else '✗ FAIL'}")
    print(f"Functionality: {'✓ PASS' if func_success else '✗ FAIL'}")
    
    if import_success and g7_success and func_success:
        print("\n🎉 All tests passed! AI components are working correctly.")
        return 0
    else:
        print("\n❌ Some tests failed. Check the output above for details.")
        return 1

if __name__ == "__main__":
    sys.exit(main()) 