"""
TuskLang Binary Format Implementation for Python

This module provides a complete implementation of the TuskLang binary format (.pnt)
that is compatible with the Go reference implementation.
"""

import struct
import time
import zlib
from typing import Dict, List, Any, Optional, Union, Tuple
from dataclasses import dataclass
from enum import IntEnum
import numpy as np


class TypeIdentifier(IntEnum):
    """Type identifiers for binary format values."""
    NULL = 0x00
    BOOL = 0x01
    INT8 = 0x02
    UINT8 = 0x03
    INT16 = 0x04
    UINT16 = 0x05
    INT32 = 0x06
    UINT32 = 0x07
    INT64 = 0x08
    UINT64 = 0x09
    FLOAT32 = 0x0A
    FLOAT64 = 0x0B
    STRING = 0x0C
    BYTES = 0x0D
    ARRAY = 0x0E
    OBJECT = 0x0F
    REFERENCE = 0x10
    TIMESTAMP = 0x11
    DURATION = 0x12
    DECIMAL = 0x13


class FeatureFlags:
    """Feature flags for binary format files."""
    COMPRESSED = 1 << 0
    ENCRYPTED = 1 << 1
    INDEXED = 1 << 2
    VALIDATED = 1 << 3
    STREAMING = 1 << 4


# Magic bytes
MAGIC_HEADER = b"PNT\0"
MAGIC_FOOTER = b"\0TNP"

# Version constants
VERSION_MAJOR = 1
VERSION_MINOR = 0
VERSION_PATCH = 0


@dataclass
class Header:
    """File header structure."""
    magic: bytes = MAGIC_HEADER
    major: int = VERSION_MAJOR
    minor: int = VERSION_MINOR
    patch: int = VERSION_PATCH
    flags: int = 0
    data_offset: int = 64
    data_size: int = 0
    index_offset: int = 0
    index_size: int = 0
    checksum: int = 0
    reserved: bytes = b'\x00' * 40


@dataclass
class Footer:
    """File footer structure."""
    magic: bytes = MAGIC_FOOTER
    file_size: int = 0
    checksum: int = 0
    reserved: bytes = b'\x00' * 4


@dataclass
class IndexEntry:
    """Index table entry."""
    key_length: int = 0
    key: str = ""
    value_offset: int = 0
    value_type: int = 0
    value_size: int = 0


@dataclass
class Value:
    """Configuration value."""
    type_id: int
    data: Any
    size: int = 0
    offset: int = 0


@dataclass
class Object:
    """Configuration object."""
    entries: Dict[str, Value]
    size: int = 0
    offset: int = 0


@dataclass
class Array:
    """Configuration array."""
    elements: List[Value]
    size: int = 0
    offset: int = 0


class BinaryFormatError(Exception):
    """Exception raised for binary format errors."""
    
    def __init__(self, message: str, offset: int = 0, error_type: str = "unknown"):
        self.message = message
        self.offset = offset
        self.error_type = error_type
        super().__init__(f"binary format error at offset {offset} ({error_type}): {message}")


def encode_length(length: int) -> bytes:
    """Encode length using compact encoding."""
    if length <= 127:
        return struct.pack('<B', length)
    elif length <= 16383:
        return struct.pack('<H', length | 0x8000)
    else:
        return b'\xFF' + struct.pack('<I', length)


def decode_length(data: bytes, offset: int = 0) -> Tuple[int, int]:
    """Decode length from compact encoding."""
    if offset >= len(data):
        raise BinaryFormatError("unexpected end of data", offset, "length_decode")
    
    first_byte = data[offset]
    if first_byte <= 127:
        return first_byte, offset + 1
    elif first_byte <= 0xFF:
        if offset + 1 >= len(data):
            raise BinaryFormatError("unexpected end of data", offset, "length_decode")
        second_byte = data[offset + 1]
        length = ((first_byte & 0x7F) << 8) | second_byte
        return length, offset + 2
    else:
        if offset + 3 >= len(data):
            raise BinaryFormatError("unexpected end of data", offset, "length_decode")
        length = struct.unpack('<I', data[offset + 1:offset + 4] + b'\x00')[0]
        return length, offset + 4


def calculate_crc32(data: bytes) -> int:
    """Calculate CRC32 checksum."""
    return zlib.crc32(data) & 0xFFFFFFFF


def value_to_bytes(value: Value) -> bytes:
    """Convert a value to its binary representation."""
    if value.type_id == TypeIdentifier.NULL:
        return b''
    
    elif value.type_id == TypeIdentifier.BOOL:
        return struct.pack('<B', 1 if value.data else 0)
    
    elif value.type_id == TypeIdentifier.INT8:
        return struct.pack('<b', value.data)
    
    elif value.type_id == TypeIdentifier.UINT8:
        return struct.pack('<B', value.data)
    
    elif value.type_id == TypeIdentifier.INT16:
        return struct.pack('<h', value.data)
    
    elif value.type_id == TypeIdentifier.UINT16:
        return struct.pack('<H', value.data)
    
    elif value.type_id == TypeIdentifier.INT32:
        return struct.pack('<i', value.data)
    
    elif value.type_id == TypeIdentifier.UINT32:
        return struct.pack('<I', value.data)
    
    elif value.type_id == TypeIdentifier.INT64:
        return struct.pack('<q', value.data)
    
    elif value.type_id == TypeIdentifier.UINT64:
        return struct.pack('<Q', value.data)
    
    elif value.type_id == TypeIdentifier.FLOAT32:
        return struct.pack('<f', value.data)
    
    elif value.type_id == TypeIdentifier.FLOAT64:
        return struct.pack('<d', value.data)
    
    elif value.type_id == TypeIdentifier.STRING:
        str_bytes = value.data.encode('utf-8')
        return encode_length(len(str_bytes)) + str_bytes
    
    elif value.type_id == TypeIdentifier.BYTES:
        return encode_length(len(value.data)) + value.data
    
    elif value.type_id == TypeIdentifier.TIMESTAMP:
        if isinstance(value.data, (int, float)):
            timestamp = int(value.data)
        else:
            timestamp = int(value.data.timestamp())
        return struct.pack('<Q', timestamp)
    
    elif value.type_id == TypeIdentifier.DURATION:
        if isinstance(value.data, (int, float)):
            nanoseconds = int(value.data)
        else:
            nanoseconds = int(value.data.total_seconds() * 1e9)
        return struct.pack('<Q', nanoseconds)
    
    elif value.type_id == TypeIdentifier.REFERENCE:
        return struct.pack('<Q', value.data)
    
    else:
        raise BinaryFormatError(f"unsupported type identifier: {value.type_id}", 0, "unsupported_type")


class BinaryReader:
    """Reader for binary format files."""
    
    def __init__(self, filename: str):
        """Initialize reader with file path."""
        self.filename = filename
        self.file = None
        self.offset = 0
        self.buffer = bytearray(4096)  # 4KB buffer
    
    def __enter__(self):
        """Context manager entry."""
        self.file = open(self.filename, 'rb')
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        """Context manager exit."""
        if self.file:
            self.file.close()
    
    def read_header(self) -> Header:
        """Read and validate file header."""
        if not self.file:
            raise BinaryFormatError("file not open", 0, "file_not_open")
        
        # Read header bytes
        header_data = self.file.read(64)
        if len(header_data) != 64:
            raise BinaryFormatError("incomplete header", self.offset, "header_read")
        
        # Parse header
        header = Header()
        header.magic = header_data[0:4]
        header.major = header_data[4]
        header.minor = header_data[5]
        header.patch = header_data[6]
        header.flags = header_data[7]
        header.data_offset = struct.unpack('<Q', header_data[8:16])[0]
        header.data_size = struct.unpack('<Q', header_data[16:24])[0]
        header.index_offset = struct.unpack('<Q', header_data[24:32])[0]
        header.index_size = struct.unpack('<Q', header_data[32:40])[0]
        header.checksum = struct.unpack('<I', header_data[40:44])[0]
        header.reserved = header_data[44:64]
        
        # Validate magic bytes
        if header.magic != MAGIC_HEADER:
            raise BinaryFormatError(
                f"invalid magic bytes: expected {MAGIC_HEADER}, got {header.magic}",
                self.offset, "magic_validation"
            )
        
        # Validate version
        if header.major != VERSION_MAJOR:
            raise BinaryFormatError(
                f"unsupported major version: {header.major}",
                self.offset, "version_validation"
            )
        
        # Validate checksum
        checksum_data = header_data[0:40] + header_data[44:64]
        calculated_checksum = calculate_crc32(checksum_data)
        
        if header.checksum != calculated_checksum:
            raise BinaryFormatError(
                f"header checksum mismatch: expected {calculated_checksum:08x}, got {header.checksum:08x}",
                self.offset, "checksum_validation"
            )
        
        self.offset = 64
        return header
    
    def read_value(self) -> Value:
        """Read a single value from current position."""
        if not self.file:
            raise BinaryFormatError("file not open", self.offset, "file_not_open")
        
        # Read type identifier
        type_data = self.file.read(1)
        if len(type_data) != 1:
            raise BinaryFormatError("failed to read type identifier", self.offset, "type_read")
        
        type_id = type_data[0]
        self.offset += 1
        
        value = Value(type_id=type_id, offset=self.offset)
        
        # Read value data based on type
        if type_id == TypeIdentifier.NULL:
            value.data = None
            value.size = 0
        
        elif type_id == TypeIdentifier.BOOL:
            bool_data = self.file.read(1)
            if len(bool_data) != 1:
                raise BinaryFormatError("failed to read bool value", self.offset, "bool_read")
            value.data = bool(bool_data[0])
            value.size = 1
            self.offset += 1
        
        elif type_id == TypeIdentifier.INT8:
            int_data = self.file.read(1)
            if len(int_data) != 1:
                raise BinaryFormatError("failed to read int8 value", self.offset, "int8_read")
            value.data = struct.unpack('<b', int_data)[0]
            value.size = 1
            self.offset += 1
        
        elif type_id == TypeIdentifier.UINT8:
            uint_data = self.file.read(1)
            if len(uint_data) != 1:
                raise BinaryFormatError("failed to read uint8 value", self.offset, "uint8_read")
            value.data = struct.unpack('<B', uint_data)[0]
            value.size = 1
            self.offset += 1
        
        elif type_id == TypeIdentifier.INT16:
            int_data = self.file.read(2)
            if len(int_data) != 2:
                raise BinaryFormatError("failed to read int16 value", self.offset, "int16_read")
            value.data = struct.unpack('<h', int_data)[0]
            value.size = 2
            self.offset += 2
        
        elif type_id == TypeIdentifier.UINT16:
            uint_data = self.file.read(2)
            if len(uint_data) != 2:
                raise BinaryFormatError("failed to read uint16 value", self.offset, "uint16_read")
            value.data = struct.unpack('<H', uint_data)[0]
            value.size = 2
            self.offset += 2
        
        elif type_id == TypeIdentifier.INT32:
            int_data = self.file.read(4)
            if len(int_data) != 4:
                raise BinaryFormatError("failed to read int32 value", self.offset, "int32_read")
            value.data = struct.unpack('<i', int_data)[0]
            value.size = 4
            self.offset += 4
        
        elif type_id == TypeIdentifier.UINT32:
            uint_data = self.file.read(4)
            if len(uint_data) != 4:
                raise BinaryFormatError("failed to read uint32 value", self.offset, "uint32_read")
            value.data = struct.unpack('<I', uint_data)[0]
            value.size = 4
            self.offset += 4
        
        elif type_id == TypeIdentifier.INT64:
            int_data = self.file.read(8)
            if len(int_data) != 8:
                raise BinaryFormatError("failed to read int64 value", self.offset, "int64_read")
            value.data = struct.unpack('<q', int_data)[0]
            value.size = 8
            self.offset += 8
        
        elif type_id == TypeIdentifier.UINT64:
            uint_data = self.file.read(8)
            if len(uint_data) != 8:
                raise BinaryFormatError("failed to read uint64 value", self.offset, "uint64_read")
            value.data = struct.unpack('<Q', uint_data)[0]
            value.size = 8
            self.offset += 8
        
        elif type_id == TypeIdentifier.FLOAT32:
            float_data = self.file.read(4)
            if len(float_data) != 4:
                raise BinaryFormatError("failed to read float32 value", self.offset, "float32_read")
            value.data = struct.unpack('<f', float_data)[0]
            value.size = 4
            self.offset += 4
        
        elif type_id == TypeIdentifier.FLOAT64:
            float_data = self.file.read(8)
            if len(float_data) != 8:
                raise BinaryFormatError("failed to read float64 value", self.offset, "float64_read")
            value.data = struct.unpack('<d', float_data)[0]
            value.size = 8
            self.offset += 8
        
        elif type_id == TypeIdentifier.STRING:
            value.data = self.read_string()
            value.size = len(value.data.encode('utf-8'))
        
        elif type_id == TypeIdentifier.BYTES:
            value.data = self.read_bytes()
            value.size = len(value.data)
        
        elif type_id == TypeIdentifier.TIMESTAMP:
            timestamp_data = self.file.read(8)
            if len(timestamp_data) != 8:
                raise BinaryFormatError("failed to read timestamp value", self.offset, "timestamp_read")
            unix_time = struct.unpack('<Q', timestamp_data)[0]
            value.data = time.time()  # Convert to time object
            value.size = 8
            self.offset += 8
        
        elif type_id == TypeIdentifier.DURATION:
            duration_data = self.file.read(8)
            if len(duration_data) != 8:
                raise BinaryFormatError("failed to read duration value", self.offset, "duration_read")
            nanoseconds = struct.unpack('<Q', duration_data)[0]
            value.data = nanoseconds / 1e9  # Convert to seconds
            value.size = 8
            self.offset += 8
        
        elif type_id == TypeIdentifier.REFERENCE:
            ref_data = self.file.read(8)
            if len(ref_data) != 8:
                raise BinaryFormatError("failed to read reference offset", self.offset, "reference_read")
            value.data = struct.unpack('<Q', ref_data)[0]
            value.size = 8
            self.offset += 8
        
        elif type_id == TypeIdentifier.ARRAY:
            value.data = self.read_array()
            value.size = value.data.size
        
        elif type_id == TypeIdentifier.OBJECT:
            value.data = self.read_object()
            value.size = value.data.size
        
        else:
            raise BinaryFormatError(f"unknown type identifier: {type_id}", self.offset, "unknown_type")
        
        return value
    
    def read_string(self) -> str:
        """Read a string value with length prefix."""
        # Read length
        length_data = self.file.read(1)
        if len(length_data) != 1:
            raise BinaryFormatError("failed to read string length", self.offset, "string_length")
        
        first_byte = length_data[0]
        if first_byte <= 127:
            length = first_byte
            self.offset += 1
        elif first_byte <= 0xFF:
            second_byte = self.file.read(1)
            if len(second_byte) != 1:
                raise BinaryFormatError("failed to read string length", self.offset, "string_length")
            length = ((first_byte & 0x7F) << 8) | second_byte[0]
            self.offset += 2
        else:
            length_bytes = self.file.read(3)
            if len(length_bytes) != 3:
                raise BinaryFormatError("failed to read string length", self.offset, "string_length")
            length = struct.unpack('<I', length_bytes + b'\x00')[0]
            self.offset += 4
        
        if length > 0:
            str_data = self.file.read(length)
            if len(str_data) != length:
                raise BinaryFormatError("failed to read string data", self.offset, "string_data")
            self.offset += length
            return str_data.decode('utf-8')
        
        return ""
    
    def read_bytes(self) -> bytes:
        """Read a byte array with length prefix."""
        # Read length
        length_data = self.file.read(1)
        if len(length_data) != 1:
            raise BinaryFormatError("failed to read bytes length", self.offset, "bytes_length")
        
        first_byte = length_data[0]
        if first_byte <= 127:
            length = first_byte
            self.offset += 1
        elif first_byte <= 0xFF:
            second_byte = self.file.read(1)
            if len(second_byte) != 1:
                raise BinaryFormatError("failed to read bytes length", self.offset, "bytes_length")
            length = ((first_byte & 0x7F) << 8) | second_byte[0]
            self.offset += 2
        else:
            length_bytes = self.file.read(3)
            if len(length_bytes) != 3:
                raise BinaryFormatError("failed to read bytes length", self.offset, "bytes_length")
            length = struct.unpack('<I', length_bytes + b'\x00')[0]
            self.offset += 4
        
        if length > 0:
            bytes_data = self.file.read(length)
            if len(bytes_data) != length:
                raise BinaryFormatError("failed to read bytes data", self.offset, "bytes_data")
            self.offset += length
            return bytes_data
        
        return b""
    
    def read_array(self) -> Array:
        """Read an array value."""
        # Read length
        length_data = self.file.read(1)
        if len(length_data) != 1:
            raise BinaryFormatError("failed to read array length", self.offset, "array_length")
        
        first_byte = length_data[0]
        if first_byte <= 127:
            length = first_byte
            self.offset += 1
        elif first_byte <= 0xFF:
            second_byte = self.file.read(1)
            if len(second_byte) != 1:
                raise BinaryFormatError("failed to read array length", self.offset, "array_length")
            length = ((first_byte & 0x7F) << 8) | second_byte[0]
            self.offset += 2
        else:
            length_bytes = self.file.read(3)
            if len(length_bytes) != 3:
                raise BinaryFormatError("failed to read array length", self.offset, "array_length")
            length = struct.unpack('<I', length_bytes + b'\x00')[0]
            self.offset += 4
        
        array = Array(elements=[], offset=self.offset)
        
        # Read array elements
        for i in range(length):
            element = self.read_value()
            array.elements.append(element)
        
        # Calculate total size
        array.size = self.offset - array.offset
        return array
    
    def read_object(self) -> Object:
        """Read an object value."""
        # Read length
        length_data = self.file.read(1)
        if len(length_data) != 1:
            raise BinaryFormatError("failed to read object length", self.offset, "object_length")
        
        first_byte = length_data[0]
        if first_byte <= 127:
            length = first_byte
            self.offset += 1
        elif first_byte <= 0xFF:
            second_byte = self.file.read(1)
            if len(second_byte) != 1:
                raise BinaryFormatError("failed to read object length", self.offset, "object_length")
            length = ((first_byte & 0x7F) << 8) | second_byte[0]
            self.offset += 2
        else:
            length_bytes = self.file.read(3)
            if len(length_bytes) != 3:
                raise BinaryFormatError("failed to read object length", self.offset, "object_length")
            length = struct.unpack('<I', length_bytes + b'\x00')[0]
            self.offset += 4
        
        object_data = Object(entries={}, offset=self.offset)
        
        # Read object entries
        for i in range(length):
            # Read key
            key = self.read_string()
            
            # Read value
            value = self.read_value()
            
            object_data.entries[key] = value
        
        # Calculate total size
        object_data.size = self.offset - object_data.offset
        return object_data
    
    def read_footer(self) -> Footer:
        """Read and validate file footer."""
        if not self.file:
            raise BinaryFormatError("file not open", self.offset, "file_not_open")
        
        footer_data = self.file.read(16)
        if len(footer_data) != 16:
            raise BinaryFormatError("incomplete footer", self.offset, "footer_read")
        
        footer = Footer()
        footer.magic = footer_data[0:4]
        footer.file_size = struct.unpack('<I', footer_data[4:8])[0]
        footer.checksum = struct.unpack('<I', footer_data[8:12])[0]
        footer.reserved = footer_data[12:16]
        
        # Validate magic bytes
        if footer.magic != MAGIC_FOOTER:
            raise BinaryFormatError(
                f"invalid footer magic bytes: expected {MAGIC_FOOTER}, got {footer.magic}",
                self.offset, "footer_magic"
            )
        
        return footer


class BinaryWriter:
    """Writer for binary format files."""
    
    def __init__(self, filename: str):
        """Initialize writer with file path."""
        self.filename = filename
        self.file = None
        self.offset = 0
        self.buffer = bytearray(4096)  # 4KB buffer
    
    def __enter__(self):
        """Context manager entry."""
        self.file = open(self.filename, 'wb')
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        """Context manager exit."""
        if self.file:
            self.file.close()
    
    def write_header(self, header: Header) -> None:
        """Write file header."""
        if not self.file:
            raise BinaryFormatError("file not open", self.offset, "file_not_open")
        
        # Set magic bytes
        header.magic = MAGIC_HEADER
        
        # Set version
        header.major = VERSION_MAJOR
        header.minor = VERSION_MINOR
        header.patch = VERSION_PATCH
        
        # Set data offset
        header.data_offset = 64
        
        # Calculate checksum (we'll update this after writing)
        header.checksum = 0
        
        # Write header
        header_data = (
            header.magic +
            struct.pack('<BBBB', header.major, header.minor, header.patch, header.flags) +
            struct.pack('<QQQQ', header.data_offset, header.data_size, header.index_offset, header.index_size) +
            struct.pack('<I', header.checksum) +
            b'\x00' * 4 +  # Padding
            header.reserved
        )
        
        self.file.write(header_data)
        self.offset = 64
    
    def write_value(self, value: Value) -> None:
        """Write a single value."""
        if not self.file:
            raise BinaryFormatError("file not open", self.offset, "file_not_open")
        
        # Write type identifier
        self.file.write(struct.pack('<B', value.type_id))
        self.offset += 1
        
        # Write value data based on type
        if value.type_id == TypeIdentifier.NULL:
            value.size = 0
        
        elif value.type_id == TypeIdentifier.BOOL:
            bool_val = 1 if value.data else 0
            self.file.write(struct.pack('<B', bool_val))
            value.size = 1
            self.offset += 1
        
        elif value.type_id == TypeIdentifier.INT8:
            self.file.write(struct.pack('<b', value.data))
            value.size = 1
            self.offset += 1
        
        elif value.type_id == TypeIdentifier.UINT8:
            self.file.write(struct.pack('<B', value.data))
            value.size = 1
            self.offset += 1
        
        elif value.type_id == TypeIdentifier.INT16:
            self.file.write(struct.pack('<h', value.data))
            value.size = 2
            self.offset += 2
        
        elif value.type_id == TypeIdentifier.UINT16:
            self.file.write(struct.pack('<H', value.data))
            value.size = 2
            self.offset += 2
        
        elif value.type_id == TypeIdentifier.INT32:
            self.file.write(struct.pack('<i', value.data))
            value.size = 4
            self.offset += 4
        
        elif value.type_id == TypeIdentifier.UINT32:
            self.file.write(struct.pack('<I', value.data))
            value.size = 4
            self.offset += 4
        
        elif value.type_id == TypeIdentifier.INT64:
            self.file.write(struct.pack('<q', value.data))
            value.size = 8
            self.offset += 8
        
        elif value.type_id == TypeIdentifier.UINT64:
            self.file.write(struct.pack('<Q', value.data))
            value.size = 8
            self.offset += 8
        
        elif value.type_id == TypeIdentifier.FLOAT32:
            self.file.write(struct.pack('<f', value.data))
            value.size = 4
            self.offset += 4
        
        elif value.type_id == TypeIdentifier.FLOAT64:
            self.file.write(struct.pack('<d', value.data))
            value.size = 8
            self.offset += 8
        
        elif value.type_id == TypeIdentifier.STRING:
            self.write_string(value.data)
            value.size = len(value.data.encode('utf-8'))
        
        elif value.type_id == TypeIdentifier.BYTES:
            self.write_bytes(value.data)
            value.size = len(value.data)
        
        elif value.type_id == TypeIdentifier.TIMESTAMP:
            if isinstance(value.data, (int, float)):
                timestamp = int(value.data)
            else:
                timestamp = int(value.data.timestamp())
            self.file.write(struct.pack('<Q', timestamp))
            value.size = 8
            self.offset += 8
        
        elif value.type_id == TypeIdentifier.DURATION:
            if isinstance(value.data, (int, float)):
                nanoseconds = int(value.data)
            else:
                nanoseconds = int(value.data.total_seconds() * 1e9)
            self.file.write(struct.pack('<Q', nanoseconds))
            value.size = 8
            self.offset += 8
        
        elif value.type_id == TypeIdentifier.REFERENCE:
            self.file.write(struct.pack('<Q', value.data))
            value.size = 8
            self.offset += 8
        
        elif value.type_id == TypeIdentifier.ARRAY:
            self.write_array(value.data)
            value.size = value.data.size
        
        elif value.type_id == TypeIdentifier.OBJECT:
            self.write_object(value.data)
            value.size = value.data.size
        
        else:
            raise BinaryFormatError(f"unknown type identifier: {value.type_id}", self.offset, "unknown_type")
    
    def write_string(self, string_data: str) -> None:
        """Write a string with length prefix."""
        str_bytes = string_data.encode('utf-8')
        length_bytes = encode_length(len(str_bytes))
        
        self.file.write(length_bytes)
        self.offset += len(length_bytes)
        
        if len(str_bytes) > 0:
            self.file.write(str_bytes)
            self.offset += len(str_bytes)
    
    def write_bytes(self, bytes_data: bytes) -> None:
        """Write a byte array with length prefix."""
        length_bytes = encode_length(len(bytes_data))
        
        self.file.write(length_bytes)
        self.offset += len(length_bytes)
        
        if len(bytes_data) > 0:
            self.file.write(bytes_data)
            self.offset += len(bytes_data)
    
    def write_array(self, array: Array) -> None:
        """Write an array value."""
        length_bytes = encode_length(len(array.elements))
        
        self.file.write(length_bytes)
        self.offset += len(length_bytes)
        
        array.offset = self.offset
        
        # Write array elements
        for element in array.elements:
            self.write_value(element)
        
        # Calculate total size
        array.size = self.offset - array.offset
    
    def write_object(self, object_data: Object) -> None:
        """Write an object value."""
        length_bytes = encode_length(len(object_data.entries))
        
        self.file.write(length_bytes)
        self.offset += len(length_bytes)
        
        object_data.offset = self.offset
        
        # Write object entries
        for key, value in object_data.entries.items():
            # Write key
            self.write_string(key)
            
            # Write value
            self.write_value(value)
        
        # Calculate total size
        object_data.size = self.offset - object_data.offset
    
    def write_footer(self, footer: Footer) -> None:
        """Write file footer."""
        if not self.file:
            raise BinaryFormatError("file not open", self.offset, "file_not_open")
        
        # Set magic bytes
        footer.magic = MAGIC_FOOTER
        
        # Set file size
        footer.file_size = self.offset + 16  # +16 for footer itself
        
        # Calculate checksum of entire file
        self.file.flush()
        
        # Read entire file for checksum calculation
        self.file.seek(0)
        file_data = self.file.read(self.offset)
        
        footer.checksum = calculate_crc32(file_data)
        
        # Write footer
        footer_data = (
            footer.magic +
            struct.pack('<II', footer.file_size, footer.checksum) +
            footer.reserved
        )
        
        self.file.write(footer_data)
        self.offset += 16
    
    def flush(self) -> None:
        """Flush any buffered data to the underlying file."""
        if self.file:
            self.file.flush()


# Convenience functions for easy usage
def read_pnt_file(filename: str) -> Dict[str, Any]:
    """Read a .pnt file and return the configuration as a dictionary."""
    with BinaryReader(filename) as reader:
        header = reader.read_header()
        object_data = reader.read_object()
        footer = reader.read_footer()
        
        # Convert to dictionary
        result = {}
        for key, value in object_data.entries.items():
            result[key] = value.data
        
        return result


def write_pnt_file(filename: str, config: Dict[str, Any]) -> None:
    """Write a configuration dictionary to a .pnt file."""
    with BinaryWriter(filename) as writer:
        # Convert dictionary to object
        entries = {}
        for key, value in config.items():
            if value is None:
                type_id = TypeIdentifier.NULL
            elif isinstance(value, bool):
                type_id = TypeIdentifier.BOOL
            elif isinstance(value, int):
                if -128 <= value <= 127:
                    type_id = TypeIdentifier.INT8
                elif 0 <= value <= 255:
                    type_id = TypeIdentifier.UINT8
                elif -32768 <= value <= 32767:
                    type_id = TypeIdentifier.INT16
                elif 0 <= value <= 65535:
                    type_id = TypeIdentifier.UINT16
                elif -2147483648 <= value <= 2147483647:
                    type_id = TypeIdentifier.INT32
                elif 0 <= value <= 4294967295:
                    type_id = TypeIdentifier.UINT32
                else:
                    type_id = TypeIdentifier.INT64
            elif isinstance(value, float):
                type_id = TypeIdentifier.FLOAT64
            elif isinstance(value, str):
                type_id = TypeIdentifier.STRING
            elif isinstance(value, bytes):
                type_id = TypeIdentifier.BYTES
            else:
                raise ValueError(f"unsupported type for key '{key}': {type(value)}")
            
            entries[key] = Value(type_id=type_id, data=value)
        
        object_data = Object(entries=entries)
        
        # Write file
        header = Header()
        writer.write_header(header)
        writer.write_object(object_data)
        footer = Footer()
        writer.write_footer(footer) 