"""
TuskLang Python SDK - Task Manager (g11.2)
Production task scheduling system with cron-like scheduling, priority queues and retries
"""

import asyncio
import heapq
import json
import logging
import pickle
import threading
import time
import uuid
from collections import defaultdict, deque
from dataclasses import dataclass, field, asdict
from datetime import datetime, timedelta
from enum import Enum
from typing import Dict, List, Optional, Set, Any, Callable, Union, Tuple
import re
import concurrent.futures


class TaskPriority(Enum):
    CRITICAL = 1
    HIGH = 2
    NORMAL = 3
    LOW = 4
    BACKGROUND = 5


class TaskStatus(Enum):
    SCHEDULED = "scheduled"
    QUEUED = "queued"
    RUNNING = "running"
    COMPLETED = "completed"
    FAILED = "failed"
    CANCELLED = "cancelled"
    PAUSED = "paused"
    RETRYING = "retrying"
    TIMEOUT = "timeout"


class ScheduleType(Enum):
    ONCE = "once"
    RECURRING = "recurring"
    CRON = "cron"
    INTERVAL = "interval"
    EVENT_BASED = "event_based"


@dataclass
class TaskResult:
    """Task execution result"""
    task_id: str
    execution_id: str
    status: TaskStatus
    start_time: datetime
    end_time: datetime
    duration_seconds: float
    output: Any = None
    error: Optional[str] = None
    attempt_count: int = 1
    resource_usage: Dict[str, float] = field(default_factory=dict)
    logs: List[str] = field(default_factory=list)


@dataclass
class Schedule:
    """Task schedule definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    schedule_type: ScheduleType = ScheduleType.ONCE
    
    # One-time scheduling
    run_at: Optional[datetime] = None
    
    # Recurring scheduling
    cron_expression: str = ""  # "0 9 * * MON-FRI" 
    interval_seconds: int = 0
    
    # Event-based
    event_pattern: str = ""
    
    # Limits
    max_runs: Optional[int] = None
    end_date: Optional[datetime] = None
    
    # Metadata
    created_at: datetime = field(default_factory=datetime.now)
    is_active: bool = True


@dataclass 
class TaskDefinition:
    """Task definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    
    # Execution details
    function: Optional[Callable] = None
    function_name: str = ""  # For serialization
    args: tuple = field(default_factory=tuple)
    kwargs: Dict[str, Any] = field(default_factory=dict)
    
    # Scheduling
    schedule: Optional[Schedule] = None
    priority: TaskPriority = TaskPriority.NORMAL
    
    # Retry settings
    max_retries: int = 3
    retry_delay: int = 60  # seconds
    retry_backoff: float = 2.0  # Exponential backoff multiplier
    max_retry_delay: int = 3600  # Maximum retry delay
    
    # Execution constraints
    timeout: int = 3600  # seconds
    max_memory_mb: int = 1024
    max_cpu_percent: float = 100.0
    
    # Dependencies
    depends_on: List[str] = field(default_factory=list)  # Task IDs
    
    # Metadata
    created_at: datetime = field(default_factory=datetime.now)
    created_by: str = "system"
    tags: List[str] = field(default_factory=list)
    is_active: bool = True
    
    # Execution history limits
    max_executions_to_keep: int = 100


@dataclass
class TaskExecution:
    """Task execution instance"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    task_id: str = ""
    
    status: TaskStatus = TaskStatus.QUEUED
    priority: TaskPriority = TaskPriority.NORMAL
    scheduled_time: datetime = field(default_factory=datetime.now)
    start_time: Optional[datetime] = None
    end_time: Optional[datetime] = None
    
    # Execution context
    attempt_count: int = 0
    next_retry_time: Optional[datetime] = None
    
    # Results
    result: Optional[TaskResult] = None
    
    # Runtime info
    worker_id: Optional[str] = None
    memory_usage_mb: float = 0.0
    cpu_usage_percent: float = 0.0


class CronParser:
    """Cron expression parser"""
    
    def __init__(self):
        self.months = {
            'JAN': 1, 'FEB': 2, 'MAR': 3, 'APR': 4, 'MAY': 5, 'JUN': 6,
            'JUL': 7, 'AUG': 8, 'SEP': 9, 'OCT': 10, 'NOV': 11, 'DEC': 12
        }
        self.days = {
            'SUN': 0, 'MON': 1, 'TUE': 2, 'WED': 3, 
            'THU': 4, 'FRI': 5, 'SAT': 6
        }
    
    def parse(self, cron_expression: str) -> Dict[str, Set[int]]:
        """Parse cron expression into field sets"""
        # Format: minute hour day month dayofweek
        # Examples:
        # "0 9 * * *" - Every day at 9:00 AM
        # "0 9 * * MON-FRI" - Weekdays at 9:00 AM
        # "*/15 * * * *" - Every 15 minutes
        
        parts = cron_expression.strip().split()
        if len(parts) != 5:
            raise ValueError(f"Invalid cron expression: {cron_expression}")
        
        minute_str, hour_str, day_str, month_str, dow_str = parts
        
        return {
            'minute': self._parse_field(minute_str, 0, 59),
            'hour': self._parse_field(hour_str, 0, 23),
            'day': self._parse_field(day_str, 1, 31),
            'month': self._parse_field(month_str, 1, 12, self.months),
            'dow': self._parse_field(dow_str, 0, 6, self.days)  # 0=Sunday
        }
    
    def _parse_field(self, field: str, min_val: int, max_val: int, 
                    name_mapping: Optional[Dict[str, int]] = None) -> Set[int]:
        """Parse individual cron field"""
        if field == '*':
            return set(range(min_val, max_val + 1))
        
        values = set()
        
        # Handle comma-separated values
        for part in field.split(','):
            part = part.strip()
            
            # Handle ranges (e.g., "1-5", "MON-FRI")
            if '-' in part:
                start_str, end_str = part.split('-', 1)
                start = self._parse_value(start_str, name_mapping)
                end = self._parse_value(end_str, name_mapping)
                values.update(range(start, end + 1))
            
            # Handle step values (e.g., "*/15", "1-5/2")
            elif '/' in part:
                range_part, step_str = part.split('/', 1)
                step = int(step_str)
                
                if range_part == '*':
                    range_values = set(range(min_val, max_val + 1))
                elif '-' in range_part:
                    start_str, end_str = range_part.split('-', 1)
                    start = self._parse_value(start_str, name_mapping)
                    end = self._parse_value(end_str, name_mapping)
                    range_values = set(range(start, end + 1))
                else:
                    start = self._parse_value(range_part, name_mapping)
                    range_values = set(range(start, max_val + 1))
                
                values.update(v for v in range_values if (v - min_val) % step == 0)
            
            # Handle single values
            else:
                values.add(self._parse_value(part, name_mapping))
        
        return values
    
    def _parse_value(self, value_str: str, name_mapping: Optional[Dict[str, int]] = None) -> int:
        """Parse single value with optional name mapping"""
        if name_mapping and value_str.upper() in name_mapping:
            return name_mapping[value_str.upper()]
        return int(value_str)
    
    def next_run_time(self, cron_expression: str, from_time: datetime) -> datetime:
        """Calculate next run time for cron expression"""
        parsed = self.parse(cron_expression)
        
        # Start from the next minute
        current = from_time.replace(second=0, microsecond=0) + timedelta(minutes=1)
        
        # Find next matching time
        for _ in range(366 * 24 * 60):  # Max iterations to prevent infinite loop
            if (current.minute in parsed['minute'] and
                current.hour in parsed['hour'] and
                current.day in parsed['day'] and
                current.month in parsed['month'] and
                current.weekday() + 1 % 7 in parsed['dow']):  # Convert weekday
                return current
            
            current += timedelta(minutes=1)
        
        raise ValueError(f"Could not find next run time for: {cron_expression}")


class TaskQueue:
    """Priority task queue"""
    
    def __init__(self):
        self._heap: List[Tuple[int, datetime, str, TaskExecution]] = []
        self._tasks: Dict[str, TaskExecution] = {}
        self._lock = threading.RLock()
        self._condition = threading.Condition(self._lock)
    
    def put(self, task_execution: TaskExecution):
        """Add task to queue"""
        with self._condition:
            # Priority queue uses min-heap, so we negate priority for max-heap behavior
            priority_value = -task_execution.priority.value
            
            heapq.heappush(
                self._heap,
                (priority_value, task_execution.scheduled_time, task_execution.id, task_execution)
            )
            self._tasks[task_execution.id] = task_execution
            self._condition.notify_all()
    
    def get(self, timeout: Optional[float] = None) -> Optional[TaskExecution]:
        """Get highest priority task that's ready to run"""
        with self._condition:
            while True:
                # Find ready tasks
                now = datetime.now()
                ready_tasks = []
                
                # Check all tasks in heap
                temp_heap = []
                while self._heap:
                    priority, scheduled_time, task_id, task_exec = heapq.heappop(self._heap)
                    
                    if scheduled_time <= now and task_exec.status == TaskStatus.QUEUED:
                        ready_tasks.append((priority, scheduled_time, task_id, task_exec))
                    else:
                        temp_heap.append((priority, scheduled_time, task_id, task_exec))
                
                # Restore heap
                self._heap = temp_heap
                heapq.heapify(self._heap)
                
                if ready_tasks:
                    # Return highest priority ready task
                    priority, scheduled_time, task_id, task_exec = min(ready_tasks)
                    del self._tasks[task_id]
                    return task_exec
                
                # No ready tasks - wait
                if not self._condition.wait(timeout):
                    return None
    
    def remove(self, task_id: str) -> bool:
        """Remove task from queue"""
        with self._condition:
            if task_id in self._tasks:
                del self._tasks[task_id]
                # Rebuild heap without the removed task
                self._heap = [
                    (p, t, tid, task) for p, t, tid, task in self._heap
                    if tid != task_id
                ]
                heapq.heapify(self._heap)
                return True
            return False
    
    def size(self) -> int:
        """Get queue size"""
        with self._lock:
            return len(self._tasks)
    
    def get_tasks(self) -> List[TaskExecution]:
        """Get all queued tasks"""
        with self._lock:
            return list(self._tasks.values())


class TaskWorker:
    """Task execution worker"""
    
    def __init__(self, worker_id: str, task_manager: 'TaskManager'):
        self.worker_id = worker_id
        self.task_manager = task_manager
        self.is_running = False
        self.current_task: Optional[TaskExecution] = None
        self.logger = logging.getLogger(__name__)
        self.executor = concurrent.futures.ThreadPoolExecutor(max_workers=1)
    
    async def start(self):
        """Start worker loop"""
        self.is_running = True
        self.logger.info(f"Worker {self.worker_id} started")
        
        while self.is_running:
            try:
                # Get next task from queue
                task_execution = self.task_manager.task_queue.get(timeout=1.0)
                
                if task_execution:
                    await self._execute_task(task_execution)
                
            except Exception as e:
                self.logger.error(f"Worker {self.worker_id} error: {e}")
                await asyncio.sleep(1)
        
        self.logger.info(f"Worker {self.worker_id} stopped")
    
    def stop(self):
        """Stop worker"""
        self.is_running = False
        self.executor.shutdown(wait=True)
    
    async def _execute_task(self, task_execution: TaskExecution):
        """Execute single task"""
        self.current_task = task_execution
        task_execution.worker_id = self.worker_id
        task_execution.status = TaskStatus.RUNNING
        task_execution.start_time = datetime.now()
        
        task_def = self.task_manager.tasks.get(task_execution.task_id)
        if not task_def:
            await self._complete_task(task_execution, TaskStatus.FAILED, "Task definition not found")
            return
        
        self.logger.info(f"Worker {self.worker_id} executing task {task_execution.task_id}")
        
        try:
            # Execute task function
            if task_def.function:
                result = await self._run_task_function(task_def, task_execution)
                await self._complete_task(task_execution, TaskStatus.COMPLETED, None, result)
            else:
                await self._complete_task(task_execution, TaskStatus.FAILED, "No function defined")
                
        except Exception as e:
            error_msg = str(e)
            self.logger.error(f"Task {task_execution.task_id} failed: {error_msg}")
            
            # Check if should retry
            if task_execution.attempt_count < task_def.max_retries:
                await self._schedule_retry(task_execution, task_def, error_msg)
            else:
                await self._complete_task(task_execution, TaskStatus.FAILED, error_msg)
        
        finally:
            self.current_task = None
    
    async def _run_task_function(self, task_def: TaskDefinition, task_execution: TaskExecution) -> Any:
        """Run task function with timeout"""
        loop = asyncio.get_event_loop()
        
        # Run in thread pool with timeout
        future = loop.run_in_executor(
            self.executor,
            task_def.function,
            *task_def.args,
            **task_def.kwargs
        )
        
        try:
            result = await asyncio.wait_for(future, timeout=task_def.timeout)
            return result
        except asyncio.TimeoutError:
            task_execution.status = TaskStatus.TIMEOUT
            raise TimeoutError(f"Task timed out after {task_def.timeout} seconds")
    
    async def _schedule_retry(self, task_execution: TaskExecution, task_def: TaskDefinition, error_msg: str):
        """Schedule task retry"""
        task_execution.attempt_count += 1
        task_execution.status = TaskStatus.RETRYING
        
        # Calculate retry delay with exponential backoff
        base_delay = task_def.retry_delay
        backoff_delay = base_delay * (task_def.retry_backoff ** (task_execution.attempt_count - 1))
        retry_delay = min(backoff_delay, task_def.max_retry_delay)
        
        task_execution.next_retry_time = datetime.now() + timedelta(seconds=retry_delay)
        task_execution.scheduled_time = task_execution.next_retry_time
        task_execution.status = TaskStatus.QUEUED
        
        # Add back to queue
        self.task_manager.task_queue.put(task_execution)
        
        self.logger.info(f"Scheduled retry for task {task_execution.task_id} in {retry_delay} seconds (attempt {task_execution.attempt_count})")
    
    async def _complete_task(self, task_execution: TaskExecution, status: TaskStatus, 
                           error: Optional[str] = None, result: Any = None):
        """Complete task execution"""
        task_execution.status = status
        task_execution.end_time = datetime.now()
        
        # Create task result
        if task_execution.start_time:
            duration = (task_execution.end_time - task_execution.start_time).total_seconds()
        else:
            duration = 0.0
        
        task_result = TaskResult(
            task_id=task_execution.task_id,
            execution_id=task_execution.id,
            status=status,
            start_time=task_execution.start_time or datetime.now(),
            end_time=task_execution.end_time,
            duration_seconds=duration,
            output=result,
            error=error,
            attempt_count=task_execution.attempt_count
        )
        
        task_execution.result = task_result
        
        # Store execution result
        await self.task_manager._store_execution_result(task_execution)
        
        # Schedule next run if recurring
        await self.task_manager._schedule_next_run(task_execution.task_id)
        
        self.logger.info(f"Task {task_execution.task_id} completed with status: {status.value}")


class TaskManager:
    """Main task management system"""
    
    def __init__(self, num_workers: int = 4):
        self.tasks: Dict[str, TaskDefinition] = {}
        self.task_queue = TaskQueue()
        self.execution_history: Dict[str, deque] = defaultdict(lambda: deque(maxlen=1000))
        self.cron_parser = CronParser()
        self.logger = logging.getLogger(__name__)
        
        # Workers
        self.num_workers = num_workers
        self.workers: List[TaskWorker] = []
        self.is_running = False
        
        # Scheduler
        self.scheduler_task: Optional[asyncio.Task] = None
        
        # Metrics
        self.metrics = {
            'tasks_created': 0,
            'executions_total': 0,
            'executions_completed': 0,
            'executions_failed': 0,
            'current_queue_size': 0
        }
    
    async def start(self):
        """Start task manager"""
        if self.is_running:
            return
        
        self.is_running = True
        
        # Start workers
        self.workers = [TaskWorker(f"worker-{i}", self) for i in range(self.num_workers)]
        worker_tasks = [asyncio.create_task(worker.start()) for worker in self.workers]
        
        # Start scheduler
        self.scheduler_task = asyncio.create_task(self._scheduler_loop())
        
        self.logger.info(f"Task manager started with {self.num_workers} workers")
        
        # Wait for all tasks
        await asyncio.gather(*worker_tasks, self.scheduler_task, return_exceptions=True)
    
    async def stop(self):
        """Stop task manager"""
        if not self.is_running:
            return
        
        self.is_running = False
        
        # Stop workers
        for worker in self.workers:
            worker.stop()
        
        # Stop scheduler
        if self.scheduler_task:
            self.scheduler_task.cancel()
        
        self.logger.info("Task manager stopped")
    
    def create_task(self, task_def: TaskDefinition) -> str:
        """Create new task"""
        self.tasks[task_def.id] = task_def
        self.metrics['tasks_created'] += 1
        
        # Schedule initial run if needed
        if task_def.schedule:
            asyncio.create_task(self._schedule_task(task_def))
        
        self.logger.info(f"Created task: {task_def.name} ({task_def.id})")
        return task_def.id
    
    def create_simple_task(self, name: str, function: Callable, 
                          schedule_type: ScheduleType = ScheduleType.ONCE,
                          cron_expression: str = "",
                          run_at: Optional[datetime] = None,
                          interval_seconds: int = 0,
                          priority: TaskPriority = TaskPriority.NORMAL,
                          max_retries: int = 3,
                          *args, **kwargs) -> str:
        """Create simple task with minimal configuration"""
        schedule = None
        if schedule_type != ScheduleType.ONCE or cron_expression or run_at or interval_seconds:
            schedule = Schedule(
                schedule_type=schedule_type,
                cron_expression=cron_expression,
                run_at=run_at,
                interval_seconds=interval_seconds
            )
        
        task_def = TaskDefinition(
            name=name,
            function=function,
            args=args,
            kwargs=kwargs,
            schedule=schedule,
            priority=priority,
            max_retries=max_retries
        )
        
        return self.create_task(task_def)
    
    def schedule_task_now(self, task_id: str, priority: Optional[TaskPriority] = None) -> str:
        """Schedule task for immediate execution"""
        task_def = self.tasks.get(task_id)
        if not task_def:
            raise ValueError(f"Task not found: {task_id}")
        
        execution = TaskExecution(
            task_id=task_id,
            priority=priority or task_def.priority,
            scheduled_time=datetime.now()
        )
        
        self.task_queue.put(execution)
        self.metrics['executions_total'] += 1
        
        self.logger.info(f"Scheduled immediate execution for task: {task_id}")
        return execution.id
    
    async def _scheduler_loop(self):
        """Main scheduler loop"""
        self.logger.info("Scheduler started")
        
        while self.is_running:
            try:
                await self._check_scheduled_tasks()
                await asyncio.sleep(30)  # Check every 30 seconds
                
                # Update metrics
                self.metrics['current_queue_size'] = self.task_queue.size()
                
            except Exception as e:
                self.logger.error(f"Scheduler error: {e}")
                await asyncio.sleep(60)
        
        self.logger.info("Scheduler stopped")
    
    async def _check_scheduled_tasks(self):
        """Check for tasks that need to be scheduled"""
        now = datetime.now()
        
        for task_def in self.tasks.values():
            if not task_def.is_active or not task_def.schedule:
                continue
            
            should_schedule, next_run = self._should_schedule_task(task_def, now)
            
            if should_schedule:
                execution = TaskExecution(
                    task_id=task_def.id,
                    priority=task_def.priority,
                    scheduled_time=next_run
                )
                
                self.task_queue.put(execution)
                self.metrics['executions_total'] += 1
                
                self.logger.info(f"Scheduled task: {task_def.name} for {next_run}")
    
    def _should_schedule_task(self, task_def: TaskDefinition, now: datetime) -> Tuple[bool, datetime]:
        """Check if task should be scheduled"""
        schedule = task_def.schedule
        if not schedule or not schedule.is_active:
            return False, now
        
        # Check if we've reached max runs
        if schedule.max_runs:
            execution_count = len(self.execution_history[task_def.id])
            if execution_count >= schedule.max_runs:
                return False, now
        
        # Check end date
        if schedule.end_date and now > schedule.end_date:
            return False, now
        
        if schedule.schedule_type == ScheduleType.ONCE:
            if schedule.run_at and schedule.run_at <= now:
                # Check if already executed
                executions = self.execution_history[task_def.id]
                if not executions:
                    return True, schedule.run_at
        
        elif schedule.schedule_type == ScheduleType.CRON:
            if schedule.cron_expression:
                try:
                    next_run = self.cron_parser.next_run_time(schedule.cron_expression, now)
                    return True, next_run
                except Exception as e:
                    self.logger.error(f"Invalid cron expression for task {task_def.id}: {e}")
        
        elif schedule.schedule_type == ScheduleType.INTERVAL:
            if schedule.interval_seconds > 0:
                # Check last execution
                executions = self.execution_history[task_def.id]
                if not executions:
                    return True, now  # First run
                
                last_execution = max(executions, key=lambda x: x.start_time or datetime.min)
                if last_execution.end_time:
                    next_run = last_execution.end_time + timedelta(seconds=schedule.interval_seconds)
                    if next_run <= now:
                        return True, next_run
        
        return False, now
    
    async def _schedule_task(self, task_def: TaskDefinition):
        """Schedule task based on its schedule definition"""
        if not task_def.schedule:
            return
        
        schedule = task_def.schedule
        now = datetime.now()
        
        if schedule.schedule_type == ScheduleType.ONCE and schedule.run_at:
            execution = TaskExecution(
                task_id=task_def.id,
                priority=task_def.priority,
                scheduled_time=schedule.run_at
            )
            self.task_queue.put(execution)
            self.metrics['executions_total'] += 1
    
    async def _schedule_next_run(self, task_id: str):
        """Schedule next run for recurring tasks"""
        task_def = self.tasks.get(task_id)
        if not task_def or not task_def.schedule:
            return
        
        schedule = task_def.schedule
        if schedule.schedule_type in [ScheduleType.RECURRING, ScheduleType.CRON, ScheduleType.INTERVAL]:
            # This will be handled by the scheduler loop
            pass
    
    async def _store_execution_result(self, task_execution: TaskExecution):
        """Store task execution result"""
        task_def = self.tasks.get(task_execution.task_id)
        if not task_def:
            return
        
        # Add to execution history
        history = self.execution_history[task_execution.task_id]
        history.append(task_execution)
        
        # Limit history size
        if len(history) > task_def.max_executions_to_keep:
            history.popleft()
        
        # Update metrics
        if task_execution.status == TaskStatus.COMPLETED:
            self.metrics['executions_completed'] += 1
        elif task_execution.status == TaskStatus.FAILED:
            self.metrics['executions_failed'] += 1
    
    def get_task(self, task_id: str) -> Optional[TaskDefinition]:
        """Get task definition"""
        return self.tasks.get(task_id)
    
    def get_task_executions(self, task_id: str, limit: int = 50) -> List[TaskExecution]:
        """Get task execution history"""
        executions = list(self.execution_history[task_id])
        executions.sort(key=lambda x: x.start_time or datetime.min, reverse=True)
        return executions[:limit]
    
    def cancel_task(self, task_id: str) -> bool:
        """Cancel queued task executions"""
        task_def = self.tasks.get(task_id)
        if task_def:
            task_def.is_active = False
            
            # Remove from queue
            removed = self.task_queue.remove(task_id)
            self.logger.info(f"Cancelled task: {task_id}")
            return removed
        return False
    
    def pause_task(self, task_id: str) -> bool:
        """Pause task scheduling"""
        task_def = self.tasks.get(task_id)
        if task_def:
            task_def.is_active = False
            return True
        return False
    
    def resume_task(self, task_id: str) -> bool:
        """Resume task scheduling"""
        task_def = self.tasks.get(task_id)
        if task_def:
            task_def.is_active = True
            return True
        return False
    
    def get_queue_status(self) -> Dict[str, Any]:
        """Get queue status"""
        queued_tasks = self.task_queue.get_tasks()
        
        status_counts = defaultdict(int)
        priority_counts = defaultdict(int)
        
        for task_exec in queued_tasks:
            status_counts[task_exec.status.value] += 1
            priority_counts[task_exec.priority.value] += 1
        
        return {
            'total_queued': len(queued_tasks),
            'status_breakdown': dict(status_counts),
            'priority_breakdown': dict(priority_counts),
            'workers_active': len([w for w in self.workers if w.current_task])
        }
    
    def get_metrics(self) -> Dict[str, Any]:
        """Get task manager metrics"""
        return {
            **self.metrics,
            'active_tasks': len([t for t in self.tasks.values() if t.is_active]),
            'total_tasks': len(self.tasks),
            'workers_count': len(self.workers),
            'queue_status': self.get_queue_status()
        }


if __name__ == "__main__":
    # Example task functions
    async def example_task(message: str, delay: int = 1):
        """Example task function"""
        await asyncio.sleep(delay)
        print(f"Task executed: {message}")
        return {"message": message, "timestamp": datetime.now().isoformat()}
    
    def sync_task(value: int):
        """Example synchronous task"""
        time.sleep(1)
        result = value * 2
        print(f"Sync task result: {result}")
        return result
    
    async def main():
        # Create task manager
        task_manager = TaskManager(num_workers=2)
        
        # Create tasks
        
        # One-time task
        task1_id = task_manager.create_simple_task(
            name="One-time Task",
            function=lambda: example_task("Hello World!"),
            schedule_type=ScheduleType.ONCE,
            run_at=datetime.now() + timedelta(seconds=2)
        )
        
        # Recurring task with interval
        task2_id = task_manager.create_simple_task(
            name="Interval Task",
            function=lambda: sync_task(42),
            schedule_type=ScheduleType.INTERVAL,
            interval_seconds=5,
            priority=TaskPriority.HIGH
        )
        
        # Cron-based task
        task3_id = task_manager.create_simple_task(
            name="Cron Task",
            function=lambda: example_task("Cron execution", 0),
            schedule_type=ScheduleType.CRON,
            cron_expression="*/2 * * * *",  # Every 2 minutes
            max_retries=1
        )
        
        print(f"Created tasks: {task1_id}, {task2_id}, {task3_id}")
        
        # Schedule immediate execution
        exec_id = task_manager.schedule_task_now(task1_id, TaskPriority.CRITICAL)
        print(f"Scheduled immediate execution: {exec_id}")
        
        # Start task manager (run for a short time for demo)
        async def run_manager():
            await asyncio.sleep(10)  # Run for 10 seconds
            await task_manager.stop()
        
        await asyncio.gather(
            task_manager.start(),
            run_manager()
        )
        
        # Get metrics
        metrics = task_manager.get_metrics()
        print(f"Final metrics: {json.dumps(metrics, indent=2)}")
        
        # Get execution history
        executions = task_manager.get_task_executions(task1_id)
        print(f"Task {task1_id} executions: {len(executions)}")
        
        print("\ng11.2: Task Manager with Scheduling - COMPLETED ✅")
    
    asyncio.run(main()) 