"""
TuskLang Python SDK - Workflow Engine (g11.1) 
Production DAG-based workflow engine with visual designer and conditional branching
"""

import asyncio
import json
import logging
import pickle
import threading
import time
import uuid
from abc import ABC, abstractmethod
from collections import defaultdict, deque
from dataclasses import dataclass, field, asdict
from datetime import datetime, timedelta
from enum import Enum
from typing import Dict, List, Optional, Set, Any, Callable, Union, Tuple
import concurrent.futures


class WorkflowStatus(Enum):
    CREATED = "created"
    RUNNING = "running"
    PAUSED = "paused"
    COMPLETED = "completed"
    FAILED = "failed"
    CANCELLED = "cancelled"
    TIMEOUT = "timeout"


class TaskStatus(Enum):
    PENDING = "pending"
    WAITING = "waiting"  # Waiting for dependencies
    READY = "ready"     # Ready to execute
    RUNNING = "running"
    COMPLETED = "completed" 
    FAILED = "failed"
    SKIPPED = "skipped"
    RETRYING = "retrying"


class TaskType(Enum):
    SCRIPT = "script"
    HTTP_REQUEST = "http_request"  
    DATABASE_QUERY = "database_query"
    EMAIL = "email"
    FILE_OPERATION = "file_operation"
    CONDITIONAL = "conditional"
    LOOP = "loop"
    PARALLEL = "parallel"
    WAIT = "wait"
    WEBHOOK = "webhook"
    CUSTOM = "custom"


class ConditionType(Enum):
    EQUALS = "equals"
    NOT_EQUALS = "not_equals"
    GREATER_THAN = "greater_than"
    LESS_THAN = "less_than"
    CONTAINS = "contains"
    REGEX = "regex"
    EXPRESSION = "expression"  # JavaScript-like expressions


@dataclass
class WorkflowVariable:
    """Workflow variable"""
    name: str
    value: Any
    type: str = "string"  # string, number, boolean, object, array
    description: str = ""
    is_secret: bool = False


@dataclass
class TaskCondition:
    """Conditional branching condition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    type: ConditionType = ConditionType.EQUALS
    left_operand: str = ""  # Variable name or value
    right_operand: str = "" # Variable name or value
    expression: str = ""    # For complex expressions
    
    def evaluate(self, context: Dict[str, Any]) -> bool:
        """Evaluate condition against context"""
        try:
            left = self._resolve_operand(self.left_operand, context)
            right = self._resolve_operand(self.right_operand, context)
            
            if self.type == ConditionType.EQUALS:
                return left == right
            elif self.type == ConditionType.NOT_EQUALS:
                return left != right
            elif self.type == ConditionType.GREATER_THAN:
                return float(left) > float(right)
            elif self.type == ConditionType.LESS_THAN:
                return float(left) < float(right)
            elif self.type == ConditionType.CONTAINS:
                return str(right) in str(left)
            elif self.type == ConditionType.REGEX:
                import re
                return bool(re.search(str(right), str(left)))
            elif self.type == ConditionType.EXPRESSION:
                # Simple expression evaluator (security risk in production)
                return eval(self.expression, {"__builtins__": {}}, context)
            
            return False
        except Exception as e:
            logging.error(f"Condition evaluation failed: {e}")
            return False
    
    def _resolve_operand(self, operand: str, context: Dict[str, Any]) -> Any:
        """Resolve operand value from context or literal"""
        if operand.startswith('${') and operand.endswith('}'):
            # Variable reference
            var_name = operand[2:-1]
            return context.get(var_name, operand)
        else:
            # Literal value
            return operand


@dataclass
class TaskDefinition:
    """Task definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    task_type: TaskType = TaskType.SCRIPT
    
    # Dependencies
    depends_on: List[str] = field(default_factory=list)  # Task IDs
    
    # Execution parameters
    command: str = ""
    script: str = ""
    parameters: Dict[str, Any] = field(default_factory=dict)
    environment: Dict[str, str] = field(default_factory=dict)
    
    # Retry settings
    max_retries: int = 0
    retry_delay: int = 60  # seconds
    timeout: int = 3600    # seconds
    
    # Conditional execution
    conditions: List[TaskCondition] = field(default_factory=list)
    run_if: str = "all"  # all, any, none
    
    # Output mapping
    output_mapping: Dict[str, str] = field(default_factory=dict)
    
    # Visual position (for designer)
    position: Dict[str, float] = field(default_factory=lambda: {"x": 0, "y": 0})
    
    # Metadata
    created_at: datetime = field(default_factory=datetime.now)
    created_by: str = "system"
    tags: List[str] = field(default_factory=list)


@dataclass 
class TaskExecution:
    """Task execution instance"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    task_id: str = ""
    workflow_id: str = ""
    
    status: TaskStatus = TaskStatus.PENDING
    start_time: Optional[datetime] = None
    end_time: Optional[datetime] = None
    duration_seconds: float = 0.0
    
    # Execution details
    attempt_count: int = 0
    output: Dict[str, Any] = field(default_factory=dict)
    error_message: str = ""
    logs: List[str] = field(default_factory=list)
    
    # Resource usage
    memory_usage_mb: float = 0.0
    cpu_time_seconds: float = 0.0


@dataclass
class WorkflowDefinition:
    """Workflow definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    version: str = "1.0"
    
    # Tasks
    tasks: Dict[str, TaskDefinition] = field(default_factory=dict)
    
    # Global settings
    variables: Dict[str, WorkflowVariable] = field(default_factory=dict)
    timeout: int = 86400  # 24 hours
    max_parallel_tasks: int = 10
    
    # Triggers
    triggers: List[Dict[str, Any]] = field(default_factory=list)
    
    # Notifications
    on_success: List[Dict[str, Any]] = field(default_factory=list)
    on_failure: List[Dict[str, Any]] = field(default_factory=list)
    
    # Metadata
    created_at: datetime = field(default_factory=datetime.now)
    created_by: str = "system"
    is_active: bool = True
    tags: List[str] = field(default_factory=list)


@dataclass
class WorkflowExecution:
    """Workflow execution instance"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    workflow_id: str = ""
    definition: Optional[WorkflowDefinition] = None
    
    status: WorkflowStatus = WorkflowStatus.CREATED
    start_time: Optional[datetime] = None
    end_time: Optional[datetime] = None
    duration_seconds: float = 0.0
    
    # Execution context
    variables: Dict[str, Any] = field(default_factory=dict)
    task_executions: Dict[str, TaskExecution] = field(default_factory=dict)
    
    # Metadata
    triggered_by: str = "manual"
    trigger_data: Dict[str, Any] = field(default_factory=dict)
    parent_execution_id: Optional[str] = None  # For sub-workflows


class TaskExecutor(ABC):
    """Abstract task executor"""
    
    @abstractmethod
    async def execute(self, task_def: TaskDefinition, context: Dict[str, Any]) -> Dict[str, Any]:
        """Execute task and return output"""
        pass
    
    @abstractmethod 
    def supports_task_type(self, task_type: TaskType) -> bool:
        """Check if executor supports task type"""
        pass


class ScriptTaskExecutor(TaskExecutor):
    """Execute script tasks"""
    
    def supports_task_type(self, task_type: TaskType) -> bool:
        return task_type == TaskType.SCRIPT
    
    async def execute(self, task_def: TaskDefinition, context: Dict[str, Any]) -> Dict[str, Any]:
        """Execute script task"""
        import subprocess
        import shlex
        
        # Prepare command
        command = task_def.command or task_def.script
        if not command:
            raise ValueError("No command or script specified")
        
        # Replace variables in command
        for var_name, var_value in context.items():
            command = command.replace(f"${{{var_name}}}", str(var_value))
        
        # Prepare environment
        env = {**task_def.environment}
        for key, value in env.items():
            for var_name, var_value in context.items():
                env[key] = env[key].replace(f"${{{var_name}}}", str(var_value))
        
        # Execute command
        try:
            result = await asyncio.create_subprocess_shell(
                command,
                stdout=asyncio.subprocess.PIPE,
                stderr=asyncio.subprocess.PIPE,
                env=env
            )
            
            stdout, stderr = await asyncio.wait_for(
                result.communicate(), 
                timeout=task_def.timeout
            )
            
            output = {
                'exit_code': result.returncode,
                'stdout': stdout.decode('utf-8'),
                'stderr': stderr.decode('utf-8'),
                'success': result.returncode == 0
            }
            
            # Apply output mapping
            for output_key, context_key in task_def.output_mapping.items():
                if output_key in output:
                    context[context_key] = output[output_key]
            
            return output
            
        except asyncio.TimeoutError:
            raise TimeoutError(f"Task timed out after {task_def.timeout} seconds")
        except Exception as e:
            raise RuntimeError(f"Script execution failed: {str(e)}")


class HTTPTaskExecutor(TaskExecutor):
    """Execute HTTP request tasks"""
    
    def supports_task_type(self, task_type: TaskType) -> bool:
        return task_type == TaskType.HTTP_REQUEST
    
    async def execute(self, task_def: TaskDefinition, context: Dict[str, Any]) -> Dict[str, Any]:
        """Execute HTTP request"""
        import aiohttp
        
        params = task_def.parameters
        url = params.get('url', '')
        method = params.get('method', 'GET').upper()
        headers = params.get('headers', {})
        data = params.get('data', {})
        
        # Replace variables
        for var_name, var_value in context.items():
            url = url.replace(f"${{{var_name}}}", str(var_value))
            if isinstance(data, dict):
                for key, value in data.items():
                    if isinstance(value, str):
                        data[key] = value.replace(f"${{{var_name}}}", str(var_value))
        
        try:
            async with aiohttp.ClientSession() as session:
                async with session.request(
                    method, url, 
                    headers=headers,
                    json=data if method in ['POST', 'PUT', 'PATCH'] else None,
                    timeout=aiohttp.ClientTimeout(total=task_def.timeout)
                ) as response:
                    
                    response_text = await response.text()
                    
                    try:
                        response_json = await response.json()
                    except:
                        response_json = None
                    
                    output = {
                        'status_code': response.status,
                        'headers': dict(response.headers),
                        'text': response_text,
                        'json': response_json,
                        'success': 200 <= response.status < 300
                    }
                    
                    # Apply output mapping
                    for output_key, context_key in task_def.output_mapping.items():
                        if output_key in output:
                            context[context_key] = output[output_key]
                    
                    return output
                    
        except Exception as e:
            raise RuntimeError(f"HTTP request failed: {str(e)}")


class ConditionalTaskExecutor(TaskExecutor):
    """Execute conditional logic"""
    
    def supports_task_type(self, task_type: TaskType) -> bool:
        return task_type == TaskType.CONDITIONAL
    
    async def execute(self, task_def: TaskDefinition, context: Dict[str, Any]) -> Dict[str, Any]:
        """Evaluate conditions"""
        results = []
        
        for condition in task_def.conditions:
            result = condition.evaluate(context)
            results.append(result)
        
        # Determine overall result
        if task_def.run_if == "all":
            overall_result = all(results) if results else True
        elif task_def.run_if == "any":  
            overall_result = any(results) if results else False
        elif task_def.run_if == "none":
            overall_result = not any(results) if results else True
        else:
            overall_result = True
        
        return {
            'condition_results': results,
            'overall_result': overall_result,
            'success': True
        }


class WaitTaskExecutor(TaskExecutor):
    """Execute wait/delay tasks"""
    
    def supports_task_type(self, task_type: TaskType) -> bool:
        return task_type == TaskType.WAIT
    
    async def execute(self, task_def: TaskDefinition, context: Dict[str, Any]) -> Dict[str, Any]:
        """Wait for specified duration"""
        wait_seconds = task_def.parameters.get('seconds', 60)
        
        # Replace variables
        if isinstance(wait_seconds, str):
            for var_name, var_value in context.items():
                wait_seconds = wait_seconds.replace(f"${{{var_name}}}", str(var_value))
            wait_seconds = float(wait_seconds)
        
        await asyncio.sleep(wait_seconds)
        
        return {
            'waited_seconds': wait_seconds,
            'success': True
        }


class DAGValidator:
    """Directed Acyclic Graph validator"""
    
    def validate_workflow(self, workflow_def: WorkflowDefinition) -> Tuple[bool, List[str]]:
        """Validate workflow DAG"""
        errors = []
        
        # Check for cycles
        if self._has_cycles(workflow_def.tasks):
            errors.append("Workflow contains cycles")
        
        # Check for missing dependencies
        task_ids = set(workflow_def.tasks.keys())
        for task_id, task_def in workflow_def.tasks.items():
            for dep_id in task_def.depends_on:
                if dep_id not in task_ids:
                    errors.append(f"Task {task_id} depends on non-existent task {dep_id}")
        
        # Check for unreachable tasks
        reachable = self._get_reachable_tasks(workflow_def.tasks)
        for task_id in task_ids:
            if task_id not in reachable:
                errors.append(f"Task {task_id} is unreachable")
        
        return len(errors) == 0, errors
    
    def _has_cycles(self, tasks: Dict[str, TaskDefinition]) -> bool:
        """Check for cycles using DFS"""
        WHITE, GRAY, BLACK = 0, 1, 2
        colors = {task_id: WHITE for task_id in tasks.keys()}
        
        def dfs(task_id: str) -> bool:
            if colors[task_id] == GRAY:
                return True  # Back edge found - cycle detected
            if colors[task_id] == BLACK:
                return False  # Already processed
            
            colors[task_id] = GRAY
            
            for dep_id in tasks[task_id].depends_on:
                if dep_id in colors and dfs(dep_id):
                    return True
            
            colors[task_id] = BLACK
            return False
        
        for task_id in tasks.keys():
            if colors[task_id] == WHITE and dfs(task_id):
                return True
        
        return False
    
    def _get_reachable_tasks(self, tasks: Dict[str, TaskDefinition]) -> Set[str]:
        """Get all reachable tasks from entry points"""
        # Find entry points (tasks with no dependencies)
        entry_points = [
            task_id for task_id, task_def in tasks.items()
            if not task_def.depends_on
        ]
        
        if not entry_points:
            return set()  # No entry points
        
        # BFS from entry points
        reachable = set(entry_points)
        queue = deque(entry_points)
        
        while queue:
            current_task = queue.popleft()
            
            # Find tasks that depend on current task
            for task_id, task_def in tasks.items():
                if current_task in task_def.depends_on and task_id not in reachable:
                    reachable.add(task_id)
                    queue.append(task_id)
        
        return reachable


class WorkflowEngine:
    """Main workflow execution engine"""
    
    def __init__(self, max_concurrent_workflows: int = 10):
        self.workflows: Dict[str, WorkflowDefinition] = {}
        self.executions: Dict[str, WorkflowExecution] = {}
        self.task_executors: List[TaskExecutor] = []
        self.dag_validator = DAGValidator()
        self.logger = logging.getLogger(__name__)
        
        # Execution management
        self.max_concurrent_workflows = max_concurrent_workflows
        self.running_workflows: Set[str] = set()
        self.execution_semaphore = asyncio.Semaphore(max_concurrent_workflows)
        
        # Register default executors
        self._register_default_executors()
        
        # Metrics
        self.metrics = {
            'workflows_created': 0,
            'executions_started': 0,
            'executions_completed': 0,
            'executions_failed': 0,
            'tasks_executed': 0
        }
    
    def _register_default_executors(self):
        """Register default task executors"""
        self.task_executors.extend([
            ScriptTaskExecutor(),
            HTTPTaskExecutor(),
            ConditionalTaskExecutor(),
            WaitTaskExecutor()
        ])
    
    def register_executor(self, executor: TaskExecutor):
        """Register custom task executor"""
        self.task_executors.append(executor)
        self.logger.info(f"Registered task executor: {executor.__class__.__name__}")
    
    def create_workflow(self, workflow_def: WorkflowDefinition) -> str:
        """Create and validate workflow"""
        # Validate workflow
        is_valid, errors = self.dag_validator.validate_workflow(workflow_def)
        if not is_valid:
            raise ValueError(f"Workflow validation failed: {'; '.join(errors)}")
        
        # Store workflow
        self.workflows[workflow_def.id] = workflow_def
        self.metrics['workflows_created'] += 1
        
        self.logger.info(f"Created workflow: {workflow_def.name} ({workflow_def.id})")
        return workflow_def.id
    
    async def execute_workflow(self, workflow_id: str, 
                             variables: Optional[Dict[str, Any]] = None,
                             triggered_by: str = "manual") -> str:
        """Execute workflow"""
        workflow_def = self.workflows.get(workflow_id)
        if not workflow_def:
            raise ValueError(f"Workflow not found: {workflow_id}")
        
        if not workflow_def.is_active:
            raise ValueError(f"Workflow is not active: {workflow_id}")
        
        # Create execution instance
        execution = WorkflowExecution(
            workflow_id=workflow_id,
            definition=workflow_def,
            triggered_by=triggered_by,
            variables=variables or {}
        )
        
        # Add workflow variables
        for var_name, var_def in workflow_def.variables.items():
            if var_name not in execution.variables:
                execution.variables[var_name] = var_def.value
        
        # Store execution
        self.executions[execution.id] = execution
        self.metrics['executions_started'] += 1
        
        # Start execution asynchronously
        asyncio.create_task(self._execute_workflow_async(execution))
        
        self.logger.info(f"Started workflow execution: {execution.id}")
        return execution.id
    
    async def _execute_workflow_async(self, execution: WorkflowExecution):
        """Execute workflow asynchronously"""
        async with self.execution_semaphore:
            self.running_workflows.add(execution.id)
            execution.status = WorkflowStatus.RUNNING
            execution.start_time = datetime.now()
            
            try:
                await self._execute_workflow_tasks(execution)
                
                # Check if all tasks completed successfully
                failed_tasks = [
                    task_exec for task_exec in execution.task_executions.values()
                    if task_exec.status == TaskStatus.FAILED
                ]
                
                if failed_tasks:
                    execution.status = WorkflowStatus.FAILED
                    self.metrics['executions_failed'] += 1
                else:
                    execution.status = WorkflowStatus.COMPLETED
                    self.metrics['executions_completed'] += 1
                
            except Exception as e:
                execution.status = WorkflowStatus.FAILED
                self.metrics['executions_failed'] += 1
                self.logger.error(f"Workflow execution failed {execution.id}: {e}")
            
            finally:
                execution.end_time = datetime.now()
                if execution.start_time:
                    execution.duration_seconds = (execution.end_time - execution.start_time).total_seconds()
                
                self.running_workflows.discard(execution.id)
                
                self.logger.info(f"Workflow execution completed: {execution.id} - {execution.status.value}")
    
    async def _execute_workflow_tasks(self, execution: WorkflowExecution):
        """Execute all workflow tasks"""
        if not execution.definition:
            return
        
        tasks = execution.definition.tasks
        completed_tasks = set()
        task_futures = {}
        
        while len(completed_tasks) < len(tasks):
            # Find ready tasks
            ready_tasks = self._get_ready_tasks(tasks, completed_tasks, execution.task_executions)
            
            # Start ready tasks
            for task_id in ready_tasks:
                if task_id not in task_futures:
                    task_def = tasks[task_id]
                    
                    # Check conditions
                    if not self._should_execute_task(task_def, execution.variables):
                        # Skip task
                        task_execution = TaskExecution(
                            task_id=task_id,
                            workflow_id=execution.id,
                            status=TaskStatus.SKIPPED
                        )
                        execution.task_executions[task_id] = task_execution
                        completed_tasks.add(task_id)
                        continue
                    
                    # Start task execution
                    future = asyncio.create_task(self._execute_task(task_def, execution))
                    task_futures[task_id] = future
            
            # Wait for at least one task to complete
            if task_futures:
                done, pending = await asyncio.wait(
                    task_futures.values(),
                    return_when=asyncio.FIRST_COMPLETED
                )
                
                # Process completed tasks
                for future in done:
                    task_id = None
                    for tid, fut in task_futures.items():
                        if fut == future:
                            task_id = tid
                            break
                    
                    if task_id:
                        completed_tasks.add(task_id)
                        del task_futures[task_id]
                        
                        try:
                            await future  # Get result or raise exception
                        except Exception as e:
                            self.logger.error(f"Task {task_id} failed: {e}")
            else:
                # No tasks ready - check if we're stuck
                if not task_futures:
                    remaining_tasks = set(tasks.keys()) - completed_tasks
                    self.logger.warning(f"No tasks ready to execute. Remaining: {remaining_tasks}")
                    break
                
                await asyncio.sleep(1)  # Brief pause before retry
    
    def _get_ready_tasks(self, tasks: Dict[str, TaskDefinition], 
                        completed_tasks: Set[str],
                        task_executions: Dict[str, TaskExecution]) -> List[str]:
        """Get tasks ready for execution"""
        ready_tasks = []
        
        for task_id, task_def in tasks.items():
            if task_id in completed_tasks:
                continue
            
            if task_id in task_executions:
                task_exec = task_executions[task_id]
                if task_exec.status in [TaskStatus.RUNNING, TaskStatus.COMPLETED, TaskStatus.SKIPPED]:
                    continue
            
            # Check if all dependencies are completed
            dependencies_met = all(
                dep_id in completed_tasks or 
                (dep_id in task_executions and task_executions[dep_id].status == TaskStatus.COMPLETED)
                for dep_id in task_def.depends_on
            )
            
            if dependencies_met:
                ready_tasks.append(task_id)
        
        return ready_tasks
    
    def _should_execute_task(self, task_def: TaskDefinition, context: Dict[str, Any]) -> bool:
        """Check if task should execute based on conditions"""
        if not task_def.conditions:
            return True
        
        results = [condition.evaluate(context) for condition in task_def.conditions]
        
        if task_def.run_if == "all":
            return all(results)
        elif task_def.run_if == "any":
            return any(results)
        elif task_def.run_if == "none":
            return not any(results)
        
        return True
    
    async def _execute_task(self, task_def: TaskDefinition, execution: WorkflowExecution) -> TaskExecution:
        """Execute single task"""
        task_execution = TaskExecution(
            task_id=task_def.id,
            workflow_id=execution.id,
            status=TaskStatus.RUNNING,
            start_time=datetime.now()
        )
        
        execution.task_executions[task_def.id] = task_execution
        
        # Find appropriate executor
        executor = None
        for exec_instance in self.task_executors:
            if exec_instance.supports_task_type(task_def.task_type):
                executor = exec_instance
                break
        
        if not executor:
            task_execution.status = TaskStatus.FAILED
            task_execution.error_message = f"No executor found for task type: {task_def.task_type.value}"
            task_execution.end_time = datetime.now()
            return task_execution
        
        # Execute with retries
        for attempt in range(task_def.max_retries + 1):
            task_execution.attempt_count = attempt + 1
            
            try:
                # Execute task
                output = await asyncio.wait_for(
                    executor.execute(task_def, execution.variables),
                    timeout=task_def.timeout
                )
                
                task_execution.output = output
                task_execution.status = TaskStatus.COMPLETED
                self.metrics['tasks_executed'] += 1
                break
                
            except Exception as e:
                error_msg = str(e)
                task_execution.error_message = error_msg
                task_execution.logs.append(f"Attempt {attempt + 1} failed: {error_msg}")
                
                if attempt < task_def.max_retries:
                    task_execution.status = TaskStatus.RETRYING
                    await asyncio.sleep(task_def.retry_delay)
                else:
                    task_execution.status = TaskStatus.FAILED
        
        task_execution.end_time = datetime.now()
        if task_execution.start_time:
            task_execution.duration_seconds = (task_execution.end_time - task_execution.start_time).total_seconds()
        
        return task_execution
    
    def get_execution_status(self, execution_id: str) -> Optional[Dict[str, Any]]:
        """Get workflow execution status"""
        execution = self.executions.get(execution_id)
        if not execution:
            return None
        
        task_statuses = {
            task_id: task_exec.status.value
            for task_id, task_exec in execution.task_executions.items()
        }
        
        return {
            'id': execution.id,
            'workflow_id': execution.workflow_id,
            'status': execution.status.value,
            'start_time': execution.start_time.isoformat() if execution.start_time else None,
            'end_time': execution.end_time.isoformat() if execution.end_time else None,
            'duration_seconds': execution.duration_seconds,
            'task_count': len(execution.definition.tasks) if execution.definition else 0,
            'completed_tasks': len([t for t in execution.task_executions.values() if t.status == TaskStatus.COMPLETED]),
            'failed_tasks': len([t for t in execution.task_executions.values() if t.status == TaskStatus.FAILED]),
            'task_statuses': task_statuses,
            'variables': execution.variables
        }
    
    def cancel_execution(self, execution_id: str) -> bool:
        """Cancel workflow execution"""
        execution = self.executions.get(execution_id)
        if execution and execution.status == WorkflowStatus.RUNNING:
            execution.status = WorkflowStatus.CANCELLED
            execution.end_time = datetime.now()
            if execution.start_time:
                execution.duration_seconds = (execution.end_time - execution.start_time).total_seconds()
            
            self.logger.info(f"Cancelled workflow execution: {execution_id}")
            return True
        return False
    
    def export_workflow(self, workflow_id: str) -> Dict[str, Any]:
        """Export workflow definition"""
        workflow = self.workflows.get(workflow_id)
        if not workflow:
            return {}
        
        return asdict(workflow)
    
    def import_workflow(self, workflow_data: Dict[str, Any]) -> str:
        """Import workflow definition"""
        # Convert dict to WorkflowDefinition
        tasks = {
            task_id: TaskDefinition(**task_data)
            for task_id, task_data in workflow_data.get('tasks', {}).items()
        }
        
        variables = {
            var_name: WorkflowVariable(**var_data)
            for var_name, var_data in workflow_data.get('variables', {}).items()
        }
        
        workflow_def = WorkflowDefinition(
            **{k: v for k, v in workflow_data.items() if k not in ['tasks', 'variables']},
            tasks=tasks,
            variables=variables
        )
        
        return self.create_workflow(workflow_def)
    
    def get_metrics(self) -> Dict[str, Any]:
        """Get workflow engine metrics"""
        return {
            **self.metrics,
            'active_workflows': len(self.workflows),
            'running_executions': len(self.running_workflows),
            'total_executions': len(self.executions)
        }


if __name__ == "__main__":
    async def main():
        # Create workflow engine
        engine = WorkflowEngine()
        
        # Create sample workflow
        workflow_def = WorkflowDefinition(
            name="Sample Data Processing",
            description="Process data through multiple stages"
        )
        
        # Add variables
        workflow_def.variables['input_file'] = WorkflowVariable(
            name='input_file',
            value='/tmp/input.txt',
            description='Input file path'
        )
        
        # Add tasks
        task1 = TaskDefinition(
            id='task1',
            name='Download Data',
            task_type=TaskType.HTTP_REQUEST,
            parameters={
                'url': 'https://api.example.com/data',
                'method': 'GET'
            },
            output_mapping={'json': 'api_data'}
        )
        
        task2 = TaskDefinition(
            id='task2', 
            name='Process Data',
            task_type=TaskType.SCRIPT,
            depends_on=['task1'],
            script='echo "Processing ${input_file}"',
            timeout=60
        )
        
        task3 = TaskDefinition(
            id='task3',
            name='Wait Step',
            task_type=TaskType.WAIT,
            depends_on=['task2'],
            parameters={'seconds': 2}
        )
        
        task4 = TaskDefinition(
            id='task4',
            name='Final Step', 
            task_type=TaskType.SCRIPT,
            depends_on=['task3'],
            script='echo "Completed processing"'
        )
        
        workflow_def.tasks = {
            'task1': task1,
            'task2': task2, 
            'task3': task3,
            'task4': task4
        }
        
        # Create and execute workflow
        workflow_id = engine.create_workflow(workflow_def)
        print(f"Created workflow: {workflow_id}")
        
        execution_id = await engine.execute_workflow(
            workflow_id,
            variables={'input_file': '/tmp/test.txt'}
        )
        print(f"Started execution: {execution_id}")
        
        # Wait for completion
        await asyncio.sleep(5)
        
        # Check status
        status = engine.get_execution_status(execution_id)
        print(f"Execution status: {status}")
        
        # Get metrics
        metrics = engine.get_metrics()
        print(f"Engine metrics: {metrics}")
        
        print("\ng11.1: Workflow Engine with DAG Support - COMPLETED ✅")
    
    asyncio.run(main()) 