"""
Advanced Quantum Computing Simulator
Quantum circuit simulation with gate operations and quantum algorithms.
"""

import asyncio
import cmath
import json
import logging
import math
import random
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime
from typing import Any, Dict, List, Optional, Tuple, Union

import numpy as np

# Visualization (optional)
try:
    import matplotlib.pyplot as plt
    MATPLOTLIB_AVAILABLE = True
except ImportError:
    MATPLOTLIB_AVAILABLE = False
    print("Matplotlib not available. Circuit visualization will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class QuantumGate:
    """Quantum gate representation."""
    name: str
    matrix: np.ndarray
    num_qubits: int
    target_qubits: List[int]
    control_qubits: List[int] = field(default_factory=list)
    parameters: Dict[str, float] = field(default_factory=dict)

@dataclass
class QuantumState:
    """Quantum state representation."""
    num_qubits: int
    amplitudes: np.ndarray
    
    def __post_init__(self):
        """Normalize the quantum state."""
        norm = np.linalg.norm(self.amplitudes)
        if norm > 0:
            self.amplitudes = self.amplitudes / norm

@dataclass
class MeasurementResult:
    """Quantum measurement result."""
    measured_state: str
    probability: float
    measurement_counts: Dict[str, int]
    num_shots: int

class QuantumGates:
    """Standard quantum gate definitions."""
    
    # Pauli gates
    I = np.array([[1, 0], [0, 1]], dtype=complex)  # Identity
    X = np.array([[0, 1], [1, 0]], dtype=complex)  # Pauli-X (NOT)
    Y = np.array([[0, -1j], [1j, 0]], dtype=complex)  # Pauli-Y
    Z = np.array([[1, 0], [0, -1]], dtype=complex)  # Pauli-Z
    
    # Hadamard gate
    H = np.array([[1, 1], [1, -1]], dtype=complex) / np.sqrt(2)
    
    # Phase gates
    S = np.array([[1, 0], [0, 1j]], dtype=complex)  # S gate
    T = np.array([[1, 0], [0, np.exp(1j * np.pi / 4)]], dtype=complex)  # T gate
    
    # Rotation gates
    @staticmethod
    def RX(theta: float) -> np.ndarray:
        """Rotation around X-axis."""
        c = np.cos(theta / 2)
        s = np.sin(theta / 2)
        return np.array([[c, -1j * s], [-1j * s, c]], dtype=complex)
    
    @staticmethod
    def RY(theta: float) -> np.ndarray:
        """Rotation around Y-axis."""
        c = np.cos(theta / 2)
        s = np.sin(theta / 2)
        return np.array([[c, -s], [s, c]], dtype=complex)
    
    @staticmethod
    def RZ(theta: float) -> np.ndarray:
        """Rotation around Z-axis."""
        return np.array([[np.exp(-1j * theta / 2), 0],
                        [0, np.exp(1j * theta / 2)]], dtype=complex)
    
    @staticmethod
    def phase_gate(phi: float) -> np.ndarray:
        """Phase gate with arbitrary phase."""
        return np.array([[1, 0], [0, np.exp(1j * phi)]], dtype=complex)
    
    # Two-qubit gates
    @staticmethod
    def CNOT() -> np.ndarray:
        """Controlled-NOT gate."""
        return np.array([[1, 0, 0, 0],
                        [0, 1, 0, 0],
                        [0, 0, 0, 1],
                        [0, 0, 1, 0]], dtype=complex)
    
    @staticmethod
    def CZ() -> np.ndarray:
        """Controlled-Z gate."""
        return np.array([[1, 0, 0, 0],
                        [0, 1, 0, 0],
                        [0, 0, 1, 0],
                        [0, 0, 0, -1]], dtype=complex)
    
    @staticmethod
    def SWAP() -> np.ndarray:
        """SWAP gate."""
        return np.array([[1, 0, 0, 0],
                        [0, 0, 1, 0],
                        [0, 1, 0, 0],
                        [0, 0, 0, 1]], dtype=complex)

class QuantumCircuit:
    """Quantum circuit implementation."""
    
    def __init__(self, num_qubits: int):
        self.num_qubits = num_qubits
        self.gates: List[QuantumGate] = []
        self.measurements: List[Tuple[int, int]] = []  # (qubit, classical_bit)
        self.name = f"circuit_{num_qubits}q"
    
    def add_gate(self, gate_name: str, target_qubits: List[int], 
                control_qubits: List[int] = None, **params) -> 'QuantumCircuit':
        """Add a quantum gate to the circuit."""
        control_qubits = control_qubits or []
        
        # Get gate matrix
        if gate_name.upper() == 'X':
            matrix = QuantumGates.X
        elif gate_name.upper() == 'Y':
            matrix = QuantumGates.Y
        elif gate_name.upper() == 'Z':
            matrix = QuantumGates.Z
        elif gate_name.upper() == 'H':
            matrix = QuantumGates.H
        elif gate_name.upper() == 'S':
            matrix = QuantumGates.S
        elif gate_name.upper() == 'T':
            matrix = QuantumGates.T
        elif gate_name.upper() == 'RX':
            matrix = QuantumGates.RX(params.get('theta', 0))
        elif gate_name.upper() == 'RY':
            matrix = QuantumGates.RY(params.get('theta', 0))
        elif gate_name.upper() == 'RZ':
            matrix = QuantumGates.RZ(params.get('theta', 0))
        elif gate_name.upper() == 'CNOT':
            matrix = QuantumGates.CNOT()
        elif gate_name.upper() == 'CZ':
            matrix = QuantumGates.CZ()
        elif gate_name.upper() == 'SWAP':
            matrix = QuantumGates.SWAP()
        else:
            raise ValueError(f"Unknown gate: {gate_name}")
        
        gate = QuantumGate(
            name=gate_name,
            matrix=matrix,
            num_qubits=len(target_qubits) + len(control_qubits),
            target_qubits=target_qubits,
            control_qubits=control_qubits,
            parameters=params
        )
        
        self.gates.append(gate)
        return self
    
    def measure(self, qubit: int, classical_bit: int) -> 'QuantumCircuit':
        """Add measurement to the circuit."""
        self.measurements.append((qubit, classical_bit))
        return self
    
    def measure_all(self) -> 'QuantumCircuit':
        """Measure all qubits."""
        for i in range(self.num_qubits):
            self.measure(i, i)
        return self
    
    # Convenience methods for common gates
    def x(self, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('X', [qubit])
    
    def y(self, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('Y', [qubit])
    
    def z(self, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('Z', [qubit])
    
    def h(self, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('H', [qubit])
    
    def s(self, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('S', [qubit])
    
    def t(self, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('T', [qubit])
    
    def rx(self, theta: float, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('RX', [qubit], theta=theta)
    
    def ry(self, theta: float, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('RY', [qubit], theta=theta)
    
    def rz(self, theta: float, qubit: int) -> 'QuantumCircuit':
        return self.add_gate('RZ', [qubit], theta=theta)
    
    def cnot(self, control: int, target: int) -> 'QuantumCircuit':
        return self.add_gate('CNOT', [target], [control])
    
    def cz(self, control: int, target: int) -> 'QuantumCircuit':
        return self.add_gate('CZ', [target], [control])
    
    def swap(self, qubit1: int, qubit2: int) -> 'QuantumCircuit':
        return self.add_gate('SWAP', [qubit1, qubit2])
    
    def depth(self) -> int:
        """Calculate circuit depth."""
        return len(self.gates)
    
    def width(self) -> int:
        """Get circuit width (number of qubits)."""
        return self.num_qubits
    
    def __str__(self) -> str:
        """String representation of the circuit."""
        lines = [f"Quantum Circuit ({self.num_qubits} qubits, {len(self.gates)} gates)"]
        
        for i, gate in enumerate(self.gates):
            gate_str = f"{i}: {gate.name}"
            if gate.control_qubits:
                gate_str += f" (control: {gate.control_qubits})"
            gate_str += f" -> qubits {gate.target_qubits}"
            lines.append(gate_str)
        
        if self.measurements:
            lines.append(f"Measurements: {self.measurements}")
        
        return "\n".join(lines)

class QuantumSimulator:
    """Quantum circuit simulator."""
    
    def __init__(self, max_qubits: int = 20):
        self.max_qubits = max_qubits
        self.simulation_history = []
    
    def create_initial_state(self, num_qubits: int) -> QuantumState:
        """Create initial |0...0⟩ state."""
        state_size = 2 ** num_qubits
        amplitudes = np.zeros(state_size, dtype=complex)
        amplitudes[0] = 1.0  # |0...0⟩ state
        
        return QuantumState(num_qubits=num_qubits, amplitudes=amplitudes)
    
    def apply_single_qubit_gate(self, state: QuantumState, gate: QuantumGate) -> QuantumState:
        """Apply single-qubit gate to quantum state."""
        if len(gate.target_qubits) != 1:
            raise ValueError("Single-qubit gate must have exactly one target qubit")
        
        target_qubit = gate.target_qubits[0]
        num_qubits = state.num_qubits
        new_amplitudes = np.zeros_like(state.amplitudes)
        
        for i in range(2 ** num_qubits):
            # Extract bits
            bits = [(i >> j) & 1 for j in range(num_qubits)]
            
            # Apply gate to target qubit
            target_bit = bits[target_qubit]
            
            # Calculate new amplitudes
            for new_bit in [0, 1]:
                new_bits = bits.copy()
                new_bits[target_qubit] = new_bit
                
                # Convert back to index
                new_index = sum(bit * (2 ** j) for j, bit in enumerate(new_bits))
                
                # Apply gate matrix
                new_amplitudes[new_index] += gate.matrix[new_bit, target_bit] * state.amplitudes[i]
        
        return QuantumState(num_qubits=num_qubits, amplitudes=new_amplitudes)
    
    def apply_controlled_gate(self, state: QuantumState, gate: QuantumGate) -> QuantumState:
        """Apply controlled gate to quantum state."""
        if not gate.control_qubits or len(gate.target_qubits) != 1:
            raise ValueError("Controlled gate must have control qubits and one target")
        
        control_qubit = gate.control_qubits[0]
        target_qubit = gate.target_qubits[0]
        num_qubits = state.num_qubits
        new_amplitudes = state.amplitudes.copy()
        
        for i in range(2 ** num_qubits):
            bits = [(i >> j) & 1 for j in range(num_qubits)]
            
            # Apply gate only if control qubit is |1⟩
            if bits[control_qubit] == 1:
                target_bit = bits[target_qubit]
                
                for new_bit in [0, 1]:
                    if new_bit != target_bit:
                        new_bits = bits.copy()
                        new_bits[target_qubit] = new_bit
                        new_index = sum(bit * (2 ** j) for j, bit in enumerate(new_bits))
                        
                        # Apply gate matrix element
                        amplitude_contribution = gate.matrix[new_bit, target_bit] * state.amplitudes[i]
                        new_amplitudes[new_index] += amplitude_contribution
                        new_amplitudes[i] -= amplitude_contribution
        
        return QuantumState(num_qubits=num_qubits, amplitudes=new_amplitudes)
    
    def apply_two_qubit_gate(self, state: QuantumState, gate: QuantumGate) -> QuantumState:
        """Apply two-qubit gate to quantum state."""
        if len(gate.target_qubits) != 2:
            raise ValueError("Two-qubit gate must have exactly two target qubits")
        
        qubit1, qubit2 = gate.target_qubits
        num_qubits = state.num_qubits
        new_amplitudes = np.zeros_like(state.amplitudes)
        
        for i in range(2 ** num_qubits):
            bits = [(i >> j) & 1 for j in range(num_qubits)]
            
            bit1, bit2 = bits[qubit1], bits[qubit2]
            two_qubit_state = bit1 * 2 + bit2  # 00, 01, 10, 11
            
            for new_state in range(4):
                new_bit1 = new_state // 2
                new_bit2 = new_state % 2
                
                new_bits = bits.copy()
                new_bits[qubit1] = new_bit1
                new_bits[qubit2] = new_bit2
                
                new_index = sum(bit * (2 ** j) for j, bit in enumerate(new_bits))
                new_amplitudes[new_index] += gate.matrix[new_state, two_qubit_state] * state.amplitudes[i]
        
        return QuantumState(num_qubits=num_qubits, amplitudes=new_amplitudes)
    
    def apply_gate(self, state: QuantumState, gate: QuantumGate) -> QuantumState:
        """Apply quantum gate to state."""
        if gate.control_qubits:
            # Controlled gates
            if gate.name.upper() == 'CNOT':
                return self.apply_two_qubit_gate(state, gate)
            else:
                return self.apply_controlled_gate(state, gate)
        elif len(gate.target_qubits) == 1:
            # Single-qubit gates
            return self.apply_single_qubit_gate(state, gate)
        elif len(gate.target_qubits) == 2:
            # Two-qubit gates
            return self.apply_two_qubit_gate(state, gate)
        else:
            raise ValueError(f"Unsupported gate configuration: {gate.name}")
    
    def simulate(self, circuit: QuantumCircuit) -> QuantumState:
        """Simulate quantum circuit."""
        if circuit.num_qubits > self.max_qubits:
            raise ValueError(f"Circuit has too many qubits: {circuit.num_qubits} > {self.max_qubits}")
        
        logger.info(f"Simulating circuit with {circuit.num_qubits} qubits and {len(circuit.gates)} gates")
        
        # Initialize state
        state = self.create_initial_state(circuit.num_qubits)
        
        # Apply gates
        for gate in circuit.gates:
            state = self.apply_gate(state, gate)
        
        # Store simulation history
        self.simulation_history.append({
            "circuit_name": circuit.name,
            "num_qubits": circuit.num_qubits,
            "num_gates": len(circuit.gates),
            "final_state": state.amplitudes.copy(),
            "timestamp": datetime.now().isoformat()
        })
        
        return state
    
    def measure(self, state: QuantumState, shots: int = 1000) -> MeasurementResult:
        """Measure quantum state multiple times."""
        probabilities = np.abs(state.amplitudes) ** 2
        num_states = len(state.amplitudes)
        
        # Generate random measurements
        measurement_counts = {}
        measured_states = []
        
        for _ in range(shots):
            # Sample according to probabilities
            outcome = np.random.choice(num_states, p=probabilities)
            
            # Convert to binary string
            binary_string = format(outcome, f'0{state.num_qubits}b')
            measured_states.append(binary_string)
            
            if binary_string in measurement_counts:
                measurement_counts[binary_string] += 1
            else:
                measurement_counts[binary_string] = 1
        
        # Most frequent outcome
        most_frequent = max(measurement_counts.keys(), key=lambda k: measurement_counts[k])
        most_frequent_prob = measurement_counts[most_frequent] / shots
        
        return MeasurementResult(
            measured_state=most_frequent,
            probability=most_frequent_prob,
            measurement_counts=measurement_counts,
            num_shots=shots
        )
    
    def get_state_probabilities(self, state: QuantumState) -> Dict[str, float]:
        """Get probabilities for all computational basis states."""
        probabilities = {}
        
        for i, amplitude in enumerate(state.amplitudes):
            prob = abs(amplitude) ** 2
            if prob > 1e-10:  # Only include non-negligible probabilities
                binary_string = format(i, f'0{state.num_qubits}b')
                probabilities[binary_string] = prob
        
        return probabilities
    
    def calculate_fidelity(self, state1: QuantumState, state2: QuantumState) -> float:
        """Calculate fidelity between two quantum states."""
        if state1.num_qubits != state2.num_qubits:
            raise ValueError("States must have the same number of qubits")
        
        overlap = np.vdot(state1.amplitudes, state2.amplitudes)
        fidelity = abs(overlap) ** 2
        
        return fidelity

class QuantumAlgorithms:
    """Implementation of famous quantum algorithms."""
    
    @staticmethod
    def deutsch_jozsa(oracle_function: Callable[[int], int], num_qubits: int) -> QuantumCircuit:
        """Deutsch-Jozsa algorithm implementation."""
        circuit = QuantumCircuit(num_qubits + 1)  # +1 for ancilla
        
        # Initialize ancilla in |1⟩
        circuit.x(num_qubits)
        
        # Apply Hadamard to all qubits
        for i in range(num_qubits + 1):
            circuit.h(i)
        
        # Oracle implementation (simplified)
        # In a real implementation, this would be a proper oracle circuit
        for i in range(num_qubits):
            if oracle_function(i) == 1:
                circuit.cnot(i, num_qubits)
        
        # Final Hadamard on input qubits
        for i in range(num_qubits):
            circuit.h(i)
        
        # Measure input qubits
        for i in range(num_qubits):
            circuit.measure(i, i)
        
        return circuit
    
    @staticmethod
    def grovers_algorithm(num_qubits: int, marked_item: int) -> QuantumCircuit:
        """Grover's algorithm implementation."""
        circuit = QuantumCircuit(num_qubits)
        
        # Initialize superposition
        for i in range(num_qubits):
            circuit.h(i)
        
        # Number of iterations
        num_iterations = int(np.pi / 4 * np.sqrt(2 ** num_qubits))
        
        for _ in range(num_iterations):
            # Oracle: flip phase of marked item
            # Simplified oracle - in practice this would be more complex
            binary_marked = format(marked_item, f'0{num_qubits}b')
            
            # Apply X gates to create the marked state
            for i, bit in enumerate(binary_marked):
                if bit == '0':
                    circuit.x(i)
            
            # Multi-controlled Z gate (simplified)
            if num_qubits == 2:
                circuit.cz(0, 1)
            elif num_qubits > 2:
                # For simplicity, just apply Z to last qubit
                circuit.z(num_qubits - 1)
            
            # Undo X gates
            for i, bit in enumerate(binary_marked):
                if bit == '0':
                    circuit.x(i)
            
            # Diffusion operator
            for i in range(num_qubits):
                circuit.h(i)
                circuit.x(i)
            
            if num_qubits == 2:
                circuit.cz(0, 1)
            elif num_qubits > 2:
                circuit.z(num_qubits - 1)
            
            for i in range(num_qubits):
                circuit.x(i)
                circuit.h(i)
        
        circuit.measure_all()
        return circuit
    
    @staticmethod
    def quantum_fourier_transform(num_qubits: int) -> QuantumCircuit:
        """Quantum Fourier Transform implementation."""
        circuit = QuantumCircuit(num_qubits)
        
        for j in range(num_qubits):
            circuit.h(j)
            
            for k in range(j + 1, num_qubits):
                # Controlled rotation
                theta = 2 * np.pi / (2 ** (k - j + 1))
                circuit.rz(theta, k)  # Simplified - should be controlled
        
        # Swap qubits to reverse order
        for i in range(num_qubits // 2):
            circuit.swap(i, num_qubits - 1 - i)
        
        return circuit
    
    @staticmethod
    def bell_state_preparation() -> QuantumCircuit:
        """Prepare Bell state |Φ⁺⟩ = (|00⟩ + |11⟩)/√2."""
        circuit = QuantumCircuit(2)
        circuit.h(0)
        circuit.cnot(0, 1)
        circuit.measure_all()
        return circuit

class QuantumComputingSimulator:
    """
    Comprehensive Quantum Computing Simulator.
    Supports quantum circuit simulation with gate operations and quantum algorithms.
    """
    
    def __init__(self, max_qubits: int = 15):
        self.simulator = QuantumSimulator(max_qubits)
        self.algorithms = QuantumAlgorithms()
        self.circuit_library = {}
        
        self.stats = {
            'circuits_simulated': 0,
            'total_gates_applied': 0,
            'total_measurements': 0,
            'simulation_time': 0.0
        }
    
    async def create_circuit(self, num_qubits: int, name: str = None) -> QuantumCircuit:
        """Create a new quantum circuit."""
        circuit = QuantumCircuit(num_qubits)
        
        if name:
            circuit.name = name
            self.circuit_library[name] = circuit
        
        return circuit
    
    async def simulate_circuit(self, circuit: QuantumCircuit) -> QuantumState:
        """Simulate quantum circuit."""
        start_time = datetime.now()
        
        state = self.simulator.simulate(circuit)
        
        # Update statistics
        self.stats['circuits_simulated'] += 1
        self.stats['total_gates_applied'] += len(circuit.gates)
        self.stats['simulation_time'] += (datetime.now() - start_time).total_seconds()
        
        return state
    
    async def run_circuit_with_measurements(self, circuit: QuantumCircuit, 
                                          shots: int = 1000) -> MeasurementResult:
        """Run circuit and perform measurements."""
        state = await self.simulate_circuit(circuit)
        result = self.simulator.measure(state, shots)
        
        self.stats['total_measurements'] += shots
        
        return result
    
    async def run_deutsch_jozsa(self, oracle_function: Callable[[int], int], 
                              num_qubits: int) -> MeasurementResult:
        """Run Deutsch-Jozsa algorithm."""
        logger.info(f"Running Deutsch-Jozsa algorithm with {num_qubits} qubits")
        
        circuit = self.algorithms.deutsch_jozsa(oracle_function, num_qubits)
        return await self.run_circuit_with_measurements(circuit)
    
    async def run_grovers_algorithm(self, num_qubits: int, 
                                  marked_item: int) -> MeasurementResult:
        """Run Grover's algorithm."""
        logger.info(f"Running Grover's algorithm to find item {marked_item}")
        
        circuit = self.algorithms.grovers_algorithm(num_qubits, marked_item)
        return await self.run_circuit_with_measurements(circuit)
    
    async def run_qft(self, num_qubits: int) -> MeasurementResult:
        """Run Quantum Fourier Transform."""
        logger.info(f"Running QFT with {num_qubits} qubits")
        
        circuit = self.algorithms.quantum_fourier_transform(num_qubits)
        return await self.run_circuit_with_measurements(circuit)
    
    async def prepare_bell_state(self) -> MeasurementResult:
        """Prepare and measure Bell state."""
        logger.info("Preparing Bell state")
        
        circuit = self.algorithms.bell_state_preparation()
        return await self.run_circuit_with_measurements(circuit)
    
    def analyze_state(self, state: QuantumState) -> Dict[str, Any]:
        """Analyze quantum state properties."""
        probabilities = self.simulator.get_state_probabilities(state)
        
        # Calculate entropy
        entropy = 0.0
        for prob in probabilities.values():
            if prob > 0:
                entropy -= prob * np.log2(prob)
        
        # Find most probable states
        sorted_probs = sorted(probabilities.items(), key=lambda x: x[1], reverse=True)
        top_states = sorted_probs[:5]  # Top 5 most probable states
        
        return {
            "num_qubits": state.num_qubits,
            "entropy": entropy,
            "num_nonzero_amplitudes": len(probabilities),
            "top_probable_states": top_states,
            "all_probabilities": probabilities
        }
    
    def compare_circuits(self, circuit1: QuantumCircuit, 
                        circuit2: QuantumCircuit) -> Dict[str, Any]:
        """Compare two quantum circuits."""
        state1 = self.simulator.simulate(circuit1)
        state2 = self.simulator.simulate(circuit2)
        
        fidelity = self.simulator.calculate_fidelity(state1, state2)
        
        return {
            "circuit1_gates": len(circuit1.gates),
            "circuit2_gates": len(circuit2.gates),
            "circuit1_depth": circuit1.depth(),
            "circuit2_depth": circuit2.depth(),
            "state_fidelity": fidelity,
            "circuits_equivalent": fidelity > 0.999
        }
    
    def get_circuit_library(self) -> Dict[str, Dict]:
        """Get information about stored circuits."""
        library_info = {}
        
        for name, circuit in self.circuit_library.items():
            library_info[name] = {
                "num_qubits": circuit.num_qubits,
                "num_gates": len(circuit.gates),
                "depth": circuit.depth(),
                "has_measurements": len(circuit.measurements) > 0
            }
        
        return library_info
    
    def get_processing_stats(self) -> Dict[str, Any]:
        """Get processing statistics."""
        stats = self.stats.copy()
        
        if stats['circuits_simulated'] > 0:
            stats['avg_simulation_time'] = stats['simulation_time'] / stats['circuits_simulated']
            stats['avg_gates_per_circuit'] = stats['total_gates_applied'] / stats['circuits_simulated']
        else:
            stats['avg_simulation_time'] = 0.0
            stats['avg_gates_per_circuit'] = 0.0
        
        return stats

# Example usage and testing
async def main():
    """Example usage of the Quantum Computing Simulator."""
    print("=== Quantum Computing Simulator Demo ===")
    
    # Initialize quantum simulator
    quantum_sim = QuantumComputingSimulator(max_qubits=4)
    
    # Test 1: Bell state preparation
    print("\n1. Testing Bell state preparation...")
    bell_result = await quantum_sim.prepare_bell_state()
    print(f"Bell state measurements (top 3):")
    sorted_measurements = sorted(bell_result.measurement_counts.items(), 
                               key=lambda x: x[1], reverse=True)
    for state, count in sorted_measurements[:3]:
        probability = count / bell_result.num_shots
        print(f"  |{state}⟩: {probability:.3f} ({count}/{bell_result.num_shots})")
    
    # Test 2: Custom circuit simulation
    print("\n2. Testing custom circuit...")
    circuit = await quantum_sim.create_circuit(3, "test_circuit")
    circuit.h(0).cnot(0, 1).cnot(1, 2).measure_all()
    
    state = await quantum_sim.simulate_circuit(circuit)
    analysis = quantum_sim.analyze_state(state)
    
    print(f"Circuit analysis:")
    print(f"  Entropy: {analysis['entropy']:.3f}")
    print(f"  Non-zero amplitudes: {analysis['num_nonzero_amplitudes']}")
    print(f"  Top probable states:")
    for state_str, prob in analysis['top_probable_states'][:3]:
        print(f"    |{state_str}⟩: {prob:.3f}")
    
    # Test 3: Deutsch-Jozsa algorithm
    print("\n3. Testing Deutsch-Jozsa algorithm...")
    
    # Constant function (always returns 0)
    def constant_oracle(x):
        return 0
    
    # Balanced function
    def balanced_oracle(x):
        return x % 2
    
    dj_constant = await quantum_sim.run_deutsch_jozsa(constant_oracle, 2)
    dj_balanced = await quantum_sim.run_deutsch_jozsa(balanced_oracle, 2)
    
    print(f"Deutsch-Jozsa (constant function): {dj_constant.measured_state}")
    print(f"Deutsch-Jozsa (balanced function): {dj_balanced.measured_state}")
    
    # Test 4: Grover's algorithm
    print("\n4. Testing Grover's algorithm...")
    grover_result = await quantum_sim.run_grovers_algorithm(2, 3)  # Search for |11⟩
    
    print(f"Grover's search for |11⟩:")
    for state, count in sorted(grover_result.measurement_counts.items(), 
                              key=lambda x: x[1], reverse=True)[:3]:
        probability = count / grover_result.num_shots
        print(f"  |{state}⟩: {probability:.3f}")
    
    # Test 5: Quantum Fourier Transform
    print("\n5. Testing Quantum Fourier Transform...")
    qft_result = await quantum_sim.run_qft(3)
    
    print(f"QFT results (top 3 states):")
    for state, count in sorted(qft_result.measurement_counts.items(), 
                              key=lambda x: x[1], reverse=True)[:3]:
        probability = count / qft_result.num_shots
        print(f"  |{state}⟩: {probability:.3f}")
    
    # Test 6: Circuit library
    print("\n6. Circuit library:")
    library = quantum_sim.get_circuit_library()
    for name, info in library.items():
        print(f"  {name}: {info['num_qubits']} qubits, {info['num_gates']} gates")
    
    # Test 7: Processing statistics
    print("\n7. Processing statistics:")
    stats = quantum_sim.get_processing_stats()
    for key, value in stats.items():
        if isinstance(value, float):
            print(f"  {key}: {value:.4f}")
        else:
            print(f"  {key}: {value}")
    
    print("\n=== Quantum Computing Simulator Demo Complete ===")

if __name__ == "__main__":
    asyncio.run(main()) 