"""
Advanced Message Queue Systems
NATS, AMQP (RabbitMQ), and Apache Kafka support for TuskLang.
"""

import asyncio
import json
import logging
import time
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime
from typing import Any, Callable, Dict, List, Optional, Union

# NATS Support
try:
    import nats
    from nats.aio.client import Client as NATS
    NATS_AVAILABLE = True
except ImportError:
    NATS_AVAILABLE = False
    print("NATS library not available. @nats operator will be limited.")

# AMQP/RabbitMQ Support
try:
    import aio_pika
    from aio_pika import connect_robust, ExchangeType, Message
    AMQP_AVAILABLE = True
except ImportError:
    AMQP_AVAILABLE = False
    print("aio_pika library not available. @amqp operator will be limited.")

# Kafka Support
try:
    from kafka import KafkaProducer, KafkaConsumer
    from kafka.admin import KafkaAdminClient, NewTopic
    import aiokafka
    KAFKA_AVAILABLE = True
except ImportError:
    KAFKA_AVAILABLE = False
    print("kafka-python library not available. @kafka operator will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class MessageEnvelope:
    """Universal message envelope."""
    topic: str
    payload: Any
    headers: Dict[str, str] = field(default_factory=dict)
    timestamp: datetime = field(default_factory=datetime.now)
    message_id: str = ""
    correlation_id: Optional[str] = None

@dataclass
class QueueConfig:
    """Queue configuration."""
    name: str
    durable: bool = True
    auto_delete: bool = False
    exclusive: bool = False
    arguments: Dict[str, Any] = field(default_factory=dict)

@dataclass
class ExchangeConfig:
    """Exchange configuration."""
    name: str
    type: str = "direct"
    durable: bool = True
    auto_delete: bool = False
    arguments: Dict[str, Any] = field(default_factory=dict)

class NATSOperator:
    """@nats operator implementation."""
    
    def __init__(self):
        self.client = None
        self.subscriptions = {}
        self.connection_config = {}
    
    async def connect(self, servers: List[str] = None, **kwargs) -> bool:
        """Connect to NATS server."""
        if not NATS_AVAILABLE:
            logger.error("NATS library not available")
            return False
        
        try:
            servers = servers or ["nats://localhost:4222"]
            self.client = NATS()
            
            await self.client.connect(servers=servers, **kwargs)
            logger.info(f"Connected to NATS servers: {servers}")
            return True
        except Exception as e:
            logger.error(f"Error connecting to NATS: {str(e)}")
            return False
    
    async def publish(self, subject: str, payload: Any, reply_to: str = None) -> bool:
        """Publish message to NATS subject."""
        if not self.client:
            logger.error("Not connected to NATS")
            return False
        
        try:
            if isinstance(payload, dict):
                data = json.dumps(payload).encode()
            elif isinstance(payload, str):
                data = payload.encode()
            else:
                data = str(payload).encode()
            
            await self.client.publish(subject, data, reply=reply_to)
            logger.debug(f"Published to NATS subject: {subject}")
            return True
        except Exception as e:
            logger.error(f"Error publishing to NATS: {str(e)}")
            return False
    
    async def subscribe(self, subject: str, handler: Callable, queue: str = None) -> str:
        """Subscribe to NATS subject."""
        if not self.client:
            logger.error("Not connected to NATS")
            return ""
        
        try:
            async def message_handler(msg):
                try:
                    # Try to decode as JSON, fallback to string
                    try:
                        payload = json.loads(msg.data.decode())
                    except json.JSONDecodeError:
                        payload = msg.data.decode()
                    
                    envelope = MessageEnvelope(
                        topic=msg.subject,
                        payload=payload,
                        timestamp=datetime.now(),
                        message_id=str(id(msg))
                    )
                    
                    await handler(envelope)
                    
                    # Send reply if requested
                    if msg.reply:
                        response = {"status": "processed", "timestamp": datetime.now().isoformat()}
                        await self.client.publish(msg.reply, json.dumps(response).encode())
                        
                except Exception as e:
                    logger.error(f"Error handling NATS message: {str(e)}")
            
            if queue:
                subscription = await self.client.subscribe(subject, queue=queue, cb=message_handler)
            else:
                subscription = await self.client.subscribe(subject, cb=message_handler)
            
            sub_id = str(id(subscription))
            self.subscriptions[sub_id] = subscription
            
            logger.info(f"Subscribed to NATS subject: {subject} (queue: {queue})")
            return sub_id
        except Exception as e:
            logger.error(f"Error subscribing to NATS: {str(e)}")
            return ""
    
    async def request(self, subject: str, payload: Any, timeout: int = 5) -> Optional[MessageEnvelope]:
        """Send request and wait for response."""
        if not self.client:
            logger.error("Not connected to NATS")
            return None
        
        try:
            if isinstance(payload, dict):
                data = json.dumps(payload).encode()
            elif isinstance(payload, str):
                data = payload.encode()
            else:
                data = str(payload).encode()
            
            response = await self.client.request(subject, data, timeout=timeout)
            
            try:
                payload = json.loads(response.data.decode())
            except json.JSONDecodeError:
                payload = response.data.decode()
            
            return MessageEnvelope(
                topic=response.subject,
                payload=payload,
                timestamp=datetime.now(),
                message_id=str(id(response))
            )
        except Exception as e:
            logger.error(f"Error making NATS request: {str(e)}")
            return None
    
    async def disconnect(self) -> bool:
        """Disconnect from NATS."""
        try:
            if self.client:
                await self.client.drain()
                await self.client.close()
                self.client = None
                logger.info("Disconnected from NATS")
            return True
        except Exception as e:
            logger.error(f"Error disconnecting from NATS: {str(e)}")
            return False

class AMQPOperator:
    """@amqp operator implementation."""
    
    def __init__(self):
        self.connection = None
        self.channel = None
        self.exchanges = {}
        self.queues = {}
        self.consumers = {}
    
    async def connect(self, url: str = "amqp://guest:guest@localhost:5672/") -> bool:
        """Connect to AMQP broker (RabbitMQ)."""
        if not AMQP_AVAILABLE:
            logger.error("aio_pika library not available")
            return False
        
        try:
            self.connection = await connect_robust(url)
            self.channel = await self.connection.channel()
            
            logger.info("Connected to AMQP broker")
            return True
        except Exception as e:
            logger.error(f"Error connecting to AMQP: {str(e)}")
            return False
    
    async def declare_exchange(self, config: ExchangeConfig) -> bool:
        """Declare AMQP exchange."""
        if not self.channel:
            logger.error("Not connected to AMQP")
            return False
        
        try:
            exchange_type = getattr(ExchangeType, config.type.upper(), ExchangeType.DIRECT)
            
            exchange = await self.channel.declare_exchange(
                config.name,
                exchange_type,
                durable=config.durable,
                auto_delete=config.auto_delete,
                arguments=config.arguments
            )
            
            self.exchanges[config.name] = exchange
            logger.info(f"Declared AMQP exchange: {config.name}")
            return True
        except Exception as e:
            logger.error(f"Error declaring exchange: {str(e)}")
            return False
    
    async def declare_queue(self, config: QueueConfig) -> bool:
        """Declare AMQP queue."""
        if not self.channel:
            logger.error("Not connected to AMQP")
            return False
        
        try:
            queue = await self.channel.declare_queue(
                config.name,
                durable=config.durable,
                auto_delete=config.auto_delete,
                exclusive=config.exclusive,
                arguments=config.arguments
            )
            
            self.queues[config.name] = queue
            logger.info(f"Declared AMQP queue: {config.name}")
            return True
        except Exception as e:
            logger.error(f"Error declaring queue: {str(e)}")
            return False
    
    async def bind_queue(self, queue_name: str, exchange_name: str, routing_key: str = "") -> bool:
        """Bind queue to exchange."""
        if queue_name not in self.queues or exchange_name not in self.exchanges:
            logger.error(f"Queue {queue_name} or exchange {exchange_name} not found")
            return False
        
        try:
            queue = self.queues[queue_name]
            exchange = self.exchanges[exchange_name]
            
            await queue.bind(exchange, routing_key)
            logger.info(f"Bound queue {queue_name} to exchange {exchange_name}")
            return True
        except Exception as e:
            logger.error(f"Error binding queue: {str(e)}")
            return False
    
    async def publish(self, exchange_name: str, routing_key: str, payload: Any, 
                     headers: Dict[str, str] = None) -> bool:
        """Publish message to AMQP exchange."""
        if exchange_name not in self.exchanges:
            logger.error(f"Exchange {exchange_name} not found")
            return False
        
        try:
            if isinstance(payload, dict):
                body = json.dumps(payload).encode()
                content_type = "application/json"
            else:
                body = str(payload).encode()
                content_type = "text/plain"
            
            message = Message(
                body,
                content_type=content_type,
                headers=headers or {},
                timestamp=datetime.now()
            )
            
            exchange = self.exchanges[exchange_name]
            await exchange.publish(message, routing_key)
            
            logger.debug(f"Published to AMQP exchange: {exchange_name}")
            return True
        except Exception as e:
            logger.error(f"Error publishing to AMQP: {str(e)}")
            return False
    
    async def consume(self, queue_name: str, handler: Callable) -> str:
        """Start consuming messages from queue."""
        if queue_name not in self.queues:
            logger.error(f"Queue {queue_name} not found")
            return ""
        
        try:
            async def message_handler(message):
                try:
                    async with message.process():
                        # Try to decode as JSON
                        try:
                            payload = json.loads(message.body.decode())
                        except json.JSONDecodeError:
                            payload = message.body.decode()
                        
                        envelope = MessageEnvelope(
                            topic=queue_name,
                            payload=payload,
                            headers=dict(message.headers) if message.headers else {},
                            timestamp=message.timestamp or datetime.now(),
                            message_id=message.message_id or str(id(message))
                        )
                        
                        await handler(envelope)
                except Exception as e:
                    logger.error(f"Error handling AMQP message: {str(e)}")
            
            queue = self.queues[queue_name]
            consumer_tag = await queue.consume(message_handler)
            
            self.consumers[consumer_tag] = queue
            logger.info(f"Started consuming from AMQP queue: {queue_name}")
            return consumer_tag
        except Exception as e:
            logger.error(f"Error starting consumer: {str(e)}")
            return ""
    
    async def disconnect(self) -> bool:
        """Disconnect from AMQP broker."""
        try:
            if self.connection:
                await self.connection.close()
                self.connection = None
                self.channel = None
                logger.info("Disconnected from AMQP")
            return True
        except Exception as e:
            logger.error(f"Error disconnecting from AMQP: {str(e)}")
            return False

class KafkaOperator:
    """@kafka operator implementation."""
    
    def __init__(self):
        self.producer = None
        self.consumers = {}
        self.admin_client = None
        self.bootstrap_servers = []
    
    def connect_producer(self, bootstrap_servers: List[str] = None, **kwargs) -> bool:
        """Connect Kafka producer."""
        if not KAFKA_AVAILABLE:
            logger.error("kafka-python library not available")
            return False
        
        try:
            self.bootstrap_servers = bootstrap_servers or ['localhost:9092']
            
            self.producer = KafkaProducer(
                bootstrap_servers=self.bootstrap_servers,
                value_serializer=lambda x: json.dumps(x).encode('utf-8') if isinstance(x, dict) else str(x).encode('utf-8'),
                **kwargs
            )
            
            logger.info("Connected Kafka producer")
            return True
        except Exception as e:
            logger.error(f"Error connecting Kafka producer: {str(e)}")
            return False
    
    def connect_admin(self, bootstrap_servers: List[str] = None) -> bool:
        """Connect Kafka admin client."""
        if not KAFKA_AVAILABLE:
            return False
        
        try:
            bootstrap_servers = bootstrap_servers or self.bootstrap_servers or ['localhost:9092']
            
            self.admin_client = KafkaAdminClient(bootstrap_servers=bootstrap_servers)
            logger.info("Connected Kafka admin client")
            return True
        except Exception as e:
            logger.error(f"Error connecting Kafka admin: {str(e)}")
            return False
    
    def create_topic(self, topic_name: str, num_partitions: int = 1, 
                    replication_factor: int = 1) -> bool:
        """Create Kafka topic."""
        if not self.admin_client:
            logger.error("Kafka admin client not connected")
            return False
        
        try:
            topic = NewTopic(
                name=topic_name,
                num_partitions=num_partitions,
                replication_factor=replication_factor
            )
            
            self.admin_client.create_topics([topic])
            logger.info(f"Created Kafka topic: {topic_name}")
            return True
        except Exception as e:
            logger.error(f"Error creating Kafka topic: {str(e)}")
            return False
    
    def publish(self, topic: str, payload: Any, key: str = None, 
               headers: Dict[str, bytes] = None) -> bool:
        """Publish message to Kafka topic."""
        if not self.producer:
            logger.error("Kafka producer not connected")
            return False
        
        try:
            future = self.producer.send(
                topic,
                value=payload,
                key=key.encode('utf-8') if key else None,
                headers=list(headers.items()) if headers else None
            )
            
            # Wait for send to complete
            future.get(timeout=10)
            
            logger.debug(f"Published to Kafka topic: {topic}")
            return True
        except Exception as e:
            logger.error(f"Error publishing to Kafka: {str(e)}")
            return False
    
    def create_consumer(self, topics: List[str], group_id: str, 
                       handler: Callable, **kwargs) -> str:
        """Create Kafka consumer."""
        if not KAFKA_AVAILABLE:
            logger.error("Kafka library not available")
            return ""
        
        try:
            consumer = KafkaConsumer(
                *topics,
                bootstrap_servers=self.bootstrap_servers,
                group_id=group_id,
                value_deserializer=lambda m: json.loads(m.decode('utf-8')) if m else None,
                **kwargs
            )
            
            consumer_id = f"{group_id}_{id(consumer)}"
            self.consumers[consumer_id] = {
                'consumer': consumer,
                'handler': handler,
                'topics': topics,
                'active': True
            }
            
            # Start consumer in background
            asyncio.create_task(self._consume_messages(consumer_id))
            
            logger.info(f"Created Kafka consumer: {group_id}")
            return consumer_id
        except Exception as e:
            logger.error(f"Error creating Kafka consumer: {str(e)}")
            return ""
    
    async def _consume_messages(self, consumer_id: str):
        """Background task to consume messages."""
        if consumer_id not in self.consumers:
            return
        
        consumer_info = self.consumers[consumer_id]
        consumer = consumer_info['consumer']
        handler = consumer_info['handler']
        
        try:
            while consumer_info['active']:
                messages = consumer.poll(timeout_ms=1000)
                
                for topic_partition, message_list in messages.items():
                    for message in message_list:
                        try:
                            envelope = MessageEnvelope(
                                topic=message.topic,
                                payload=message.value,
                                headers={k: v.decode('utf-8') if isinstance(v, bytes) else str(v) 
                                        for k, v in (message.headers or [])},
                                timestamp=datetime.fromtimestamp(message.timestamp / 1000) if message.timestamp else datetime.now(),
                                message_id=f"{message.topic}_{message.partition}_{message.offset}"
                            )
                            
                            await handler(envelope)
                        except Exception as e:
                            logger.error(f"Error handling Kafka message: {str(e)}")
                
                await asyncio.sleep(0.1)  # Prevent tight loop
        except Exception as e:
            logger.error(f"Error in Kafka consumer loop: {str(e)}")
        finally:
            consumer.close()
    
    def stop_consumer(self, consumer_id: str) -> bool:
        """Stop Kafka consumer."""
        if consumer_id in self.consumers:
            self.consumers[consumer_id]['active'] = False
            logger.info(f"Stopped Kafka consumer: {consumer_id}")
            return True
        return False
    
    def disconnect(self) -> bool:
        """Disconnect Kafka clients."""
        try:
            # Stop all consumers
            for consumer_id in list(self.consumers.keys()):
                self.stop_consumer(consumer_id)
            
            # Close producer
            if self.producer:
                self.producer.flush()
                self.producer.close()
                self.producer = None
            
            # Close admin client
            if self.admin_client:
                self.admin_client.close()
                self.admin_client = None
            
            logger.info("Disconnected Kafka clients")
            return True
        except Exception as e:
            logger.error(f"Error disconnecting Kafka: {str(e)}")
            return False

class MessageQueueSystems:
    """
    Advanced Message Queue Systems for TuskLang.
    Implements @nats, @amqp, and @kafka operators.
    """
    
    def __init__(self):
        self.nats = NATSOperator()
        self.amqp = AMQPOperator()
        self.kafka = KafkaOperator()
        
        self.stats = {
            'nats_messages': 0,
            'amqp_messages': 0,
            'kafka_messages': 0,
            'total_subscriptions': 0,
            'active_connections': 0
        }
    
    # NATS operator methods
    async def nats_connect(self, servers: List[str] = None, **kwargs) -> bool:
        """Connect to NATS (@nats operator - connect)."""
        success = await self.nats.connect(servers, **kwargs)
        if success:
            self.stats['active_connections'] += 1
        return success
    
    async def nats_publish(self, subject: str, payload: Any, reply_to: str = None) -> bool:
        """Publish to NATS (@nats operator - publish)."""
        success = await self.nats.publish(subject, payload, reply_to)
        if success:
            self.stats['nats_messages'] += 1
        return success
    
    async def nats_subscribe(self, subject: str, handler: Callable, queue: str = None) -> str:
        """Subscribe to NATS (@nats operator - subscribe)."""
        sub_id = await self.nats.subscribe(subject, handler, queue)
        if sub_id:
            self.stats['total_subscriptions'] += 1
        return sub_id
    
    async def nats_request(self, subject: str, payload: Any, timeout: int = 5) -> Optional[MessageEnvelope]:
        """NATS request-response (@nats operator - request)."""
        return await self.nats.request(subject, payload, timeout)
    
    # AMQP operator methods
    async def amqp_connect(self, url: str = "amqp://guest:guest@localhost:5672/") -> bool:
        """Connect to AMQP (@amqp operator - connect)."""
        success = await self.amqp.connect(url)
        if success:
            self.stats['active_connections'] += 1
        return success
    
    async def amqp_declare_exchange(self, name: str, exchange_type: str = "direct", 
                                  durable: bool = True) -> bool:
        """Declare AMQP exchange (@amqp operator - exchange)."""
        config = ExchangeConfig(name=name, type=exchange_type, durable=durable)
        return await self.amqp.declare_exchange(config)
    
    async def amqp_declare_queue(self, name: str, durable: bool = True) -> bool:
        """Declare AMQP queue (@amqp operator - queue)."""
        config = QueueConfig(name=name, durable=durable)
        return await self.amqp.declare_queue(config)
    
    async def amqp_bind_queue(self, queue: str, exchange: str, routing_key: str = "") -> bool:
        """Bind AMQP queue to exchange (@amqp operator - bind)."""
        return await self.amqp.bind_queue(queue, exchange, routing_key)
    
    async def amqp_publish(self, exchange: str, routing_key: str, payload: Any,
                         headers: Dict[str, str] = None) -> bool:
        """Publish to AMQP (@amqp operator - publish)."""
        success = await self.amqp.publish(exchange, routing_key, payload, headers)
        if success:
            self.stats['amqp_messages'] += 1
        return success
    
    async def amqp_consume(self, queue: str, handler: Callable) -> str:
        """Consume from AMQP queue (@amqp operator - consume)."""
        consumer_id = await self.amqp.consume(queue, handler)
        if consumer_id:
            self.stats['total_subscriptions'] += 1
        return consumer_id
    
    # Kafka operator methods
    def kafka_connect_producer(self, servers: List[str] = None, **kwargs) -> bool:
        """Connect Kafka producer (@kafka operator - producer)."""
        success = self.kafka.connect_producer(servers, **kwargs)
        if success:
            self.stats['active_connections'] += 1
        return success
    
    def kafka_create_topic(self, topic: str, partitions: int = 1, replication: int = 1) -> bool:
        """Create Kafka topic (@kafka operator - topic)."""
        if not self.kafka.admin_client:
            self.kafka.connect_admin()
        return self.kafka.create_topic(topic, partitions, replication)
    
    def kafka_publish(self, topic: str, payload: Any, key: str = None, 
                     headers: Dict[str, bytes] = None) -> bool:
        """Publish to Kafka (@kafka operator - publish)."""
        success = self.kafka.publish(topic, payload, key, headers)
        if success:
            self.stats['kafka_messages'] += 1
        return success
    
    def kafka_consume(self, topics: List[str], group_id: str, handler: Callable, **kwargs) -> str:
        """Consume from Kafka (@kafka operator - consume)."""
        consumer_id = self.kafka.create_consumer(topics, group_id, handler, **kwargs)
        if consumer_id:
            self.stats['total_subscriptions'] += 1
        return consumer_id
    
    def kafka_stop_consumer(self, consumer_id: str) -> bool:
        """Stop Kafka consumer."""
        return self.kafka.stop_consumer(consumer_id)
    
    # Utility methods
    def get_stats(self) -> Dict[str, Any]:
        """Get message queue statistics."""
        return self.stats.copy()
    
    def get_connection_status(self) -> Dict[str, bool]:
        """Get connection status for all message queues."""
        return {
            'nats_connected': self.nats.client is not None,
            'amqp_connected': self.amqp.connection is not None,
            'kafka_producer_connected': self.kafka.producer is not None,
            'kafka_admin_connected': self.kafka.admin_client is not None
        }
    
    async def disconnect_all(self) -> bool:
        """Disconnect from all message queue systems."""
        results = []
        
        results.append(await self.nats.disconnect())
        results.append(await self.amqp.disconnect())
        results.append(self.kafka.disconnect())
        
        if all(results):
            self.stats['active_connections'] = 0
            logger.info("Disconnected from all message queue systems")
            return True
        return False

# Example usage and testing
async def main():
    """Example usage of Message Queue Systems."""
    print("=== Message Queue Systems Demo ===")
    
    # Initialize the system
    mq_systems = MessageQueueSystems()
    
    # Test 1: NATS
    print("\n1. Testing NATS (@nats operator)...")
    
    if NATS_AVAILABLE:
        try:
            # Connect
            connected = await mq_systems.nats_connect(["nats://demo.nats.io:4222"])
            if connected:
                # Define message handler
                async def nats_handler(envelope: MessageEnvelope):
                    print(f"NATS received: {envelope.payload} on {envelope.topic}")
                
                # Subscribe
                sub_id = await mq_systems.nats_subscribe("test.subject", nats_handler)
                
                # Publish
                if sub_id:
                    await asyncio.sleep(0.1)  # Give subscription time to register
                    await mq_systems.nats_publish("test.subject", {"message": "Hello NATS!"})
                
                await asyncio.sleep(0.5)  # Wait for message delivery
        except Exception as e:
            print(f"NATS test failed: {str(e)}")
    else:
        print("NATS library not available - skipping test")
    
    # Test 2: AMQP/RabbitMQ
    print("\n2. Testing AMQP/RabbitMQ (@amqp operator)...")
    print("AMQP test requires RabbitMQ server - skipping in demo")
    
    # Test 3: Kafka
    print("\n3. Testing Apache Kafka (@kafka operator)...")
    
    if KAFKA_AVAILABLE:
        try:
            # Connect producer
            producer_connected = mq_systems.kafka_connect_producer(["localhost:9092"])
            if producer_connected:
                # Create topic
                mq_systems.kafka_create_topic("test-topic", partitions=1)
                
                # Define message handler
                async def kafka_handler(envelope: MessageEnvelope):
                    print(f"Kafka received: {envelope.payload} on {envelope.topic}")
                
                # Create consumer
                consumer_id = mq_systems.kafka_consume(
                    ["test-topic"], 
                    "test-group", 
                    kafka_handler
                )
                
                # Publish message
                if consumer_id:
                    await asyncio.sleep(1)  # Give consumer time to connect
                    mq_systems.kafka_publish("test-topic", {"message": "Hello Kafka!"})
                    
                    await asyncio.sleep(2)  # Wait for message delivery
                    mq_systems.kafka_stop_consumer(consumer_id)
        except Exception as e:
            print(f"Kafka test failed: {str(e)}")
    else:
        print("Kafka library not available - skipping test")
    
    # Test 4: Statistics
    print("\n4. Message queue statistics:")
    stats = mq_systems.get_stats()
    for key, value in stats.items():
        print(f"  {key}: {value}")
    
    # Test 5: Connection status
    print("\n5. Connection status:")
    status = mq_systems.get_connection_status()
    for system, connected in status.items():
        print(f"  {system}: {'✓' if connected else '✗'}")
    
    # Test 6: Available libraries
    print("\n6. Available message queue support:")
    print(f"  NATS: {NATS_AVAILABLE}")
    print(f"  AMQP (RabbitMQ): {AMQP_AVAILABLE}")
    print(f"  Apache Kafka: {KAFKA_AVAILABLE}")
    
    # Cleanup
    await mq_systems.disconnect_all()
    
    print("\n=== Message Queue Systems Demo Complete ===")

if __name__ == "__main__":
    asyncio.run(main()) 