"""
Advanced Service Discovery & Security Systems
Consul, Vault, Temporal, and etcd support for TuskLang.
"""

import asyncio
import json
import logging
import time
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional

try:
    import aiohttp
    AIOHTTP_AVAILABLE = True
except ImportError:
    AIOHTTP_AVAILABLE = False

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class ServiceRegistration:
    name: str
    id: str
    address: str
    port: int
    tags: List[str] = field(default_factory=list)
    health_check: Optional[Dict[str, Any]] = None
    metadata: Dict[str, str] = field(default_factory=dict)

@dataclass
class SecretData:
    path: str
    data: Dict[str, Any]
    version: int = 1
    created_time: datetime = field(default_factory=datetime.now)
    lease_duration: Optional[int] = None

class ConsulOperator:
    """@consul operator implementation."""
    
    def __init__(self):
        self.base_url = ""
        self.session = None
        self.services = {}
        self.kv_store = {}
        
    async def connect(self, host: str = "localhost", port: int = 8500, token: str = "") -> bool:
        """Connect to Consul."""
        if not AIOHTTP_AVAILABLE:
            logger.error("aiohttp not available for Consul")
            return False
        
        try:
            self.base_url = f"http://{host}:{port}/v1"
            self.session = aiohttp.ClientSession()
            
            if token:
                self.session.headers.update({"X-Consul-Token": token})
            
            # Test connection
            async with self.session.get(f"{self.base_url}/status/leader") as response:
                if response.status == 200:
                    logger.info(f"Connected to Consul: {host}:{port}")
                    return True
                return False
        except Exception as e:
            logger.error(f"Error connecting to Consul: {str(e)}")
            return False
    
    async def register_service(self, service: ServiceRegistration) -> bool:
        """Register service with Consul."""
        try:
            registration_data = {
                "ID": service.id,
                "Name": service.name,
                "Address": service.address,
                "Port": service.port,
                "Tags": service.tags,
                "Meta": service.metadata
            }
            
            if service.health_check:
                registration_data["Check"] = service.health_check
            
            if self.session:
                async with self.session.put(
                    f"{self.base_url}/agent/service/register", 
                    json=registration_data
                ) as response:
                    success = response.status == 200
            else:
                # Local storage fallback
                self.services[service.id] = service
                success = True
            
            if success:
                logger.info(f"Registered service: {service.name} ({service.id})")
            return success
        except Exception as e:
            logger.error(f"Error registering service: {str(e)}")
            return False
    
    async def deregister_service(self, service_id: str) -> bool:
        """Deregister service from Consul."""
        try:
            if self.session:
                async with self.session.put(f"{self.base_url}/agent/service/deregister/{service_id}") as response:
                    success = response.status == 200
            else:
                success = self.services.pop(service_id, None) is not None
            
            if success:
                logger.info(f"Deregistered service: {service_id}")
            return success
        except Exception as e:
            logger.error(f"Error deregistering service: {str(e)}")
            return False
    
    async def discover_services(self, service_name: str = None) -> List[Dict]:
        """Discover services from Consul."""
        try:
            if self.session:
                if service_name:
                    url = f"{self.base_url}/health/service/{service_name}"
                else:
                    url = f"{self.base_url}/catalog/services"
                
                async with self.session.get(url) as response:
                    if response.status == 200:
                        return await response.json()
                    return []
            else:
                # Local fallback
                if service_name:
                    return [
                        {
                            "Service": {
                                "ID": svc.id,
                                "Service": svc.name,
                                "Address": svc.address,
                                "Port": svc.port,
                                "Tags": svc.tags
                            }
                        }
                        for svc in self.services.values()
                        if svc.name == service_name
                    ]
                else:
                    return list(self.services.keys())
        except Exception as e:
            logger.error(f"Error discovering services: {str(e)}")
            return []
    
    async def put_kv(self, key: str, value: str) -> bool:
        """Put key-value pair in Consul KV store."""
        try:
            if self.session:
                async with self.session.put(f"{self.base_url}/kv/{key}", data=value) as response:
                    success = response.status == 200
            else:
                self.kv_store[key] = value
                success = True
            
            return success
        except Exception as e:
            logger.error(f"Error putting KV: {str(e)}")
            return False
    
    async def get_kv(self, key: str) -> Optional[str]:
        """Get value from Consul KV store."""
        try:
            if self.session:
                async with self.session.get(f"{self.base_url}/kv/{key}") as response:
                    if response.status == 200:
                        data = await response.json()
                        if data:
                            import base64
                            return base64.b64decode(data[0]['Value']).decode()
                    return None
            else:
                return self.kv_store.get(key)
        except Exception as e:
            logger.error(f"Error getting KV: {str(e)}")
            return None

class VaultOperator:
    """@vault operator implementation."""
    
    def __init__(self):
        self.base_url = ""
        self.session = None
        self.token = ""
        self.secrets = {}
        
    async def connect(self, host: str = "localhost", port: int = 8200, token: str = "") -> bool:
        """Connect to Vault."""
        if not AIOHTTP_AVAILABLE:
            logger.error("aiohttp not available for Vault")
            return False
        
        try:
            self.base_url = f"http://{host}:{port}/v1"
            self.session = aiohttp.ClientSession()
            self.token = token
            
            if token:
                self.session.headers.update({"X-Vault-Token": token})
            
            # Test connection
            async with self.session.get(f"{self.base_url}/sys/health") as response:
                if response.status in [200, 429, 472, 503]:  # Various Vault health statuses
                    logger.info(f"Connected to Vault: {host}:{port}")
                    return True
                return False
        except Exception as e:
            logger.error(f"Error connecting to Vault: {str(e)}")
            return False
    
    async def write_secret(self, path: str, data: Dict[str, Any]) -> bool:
        """Write secret to Vault."""
        try:
            secret = SecretData(path=path, data=data)
            
            if self.session:
                payload = {"data": data}
                async with self.session.post(f"{self.base_url}/secret/data/{path}", json=payload) as response:
                    success = response.status == 200
            else:
                self.secrets[path] = secret
                success = True
            
            if success:
                logger.info(f"Wrote secret to path: {path}")
            return success
        except Exception as e:
            logger.error(f"Error writing secret: {str(e)}")
            return False
    
    async def read_secret(self, path: str) -> Optional[Dict[str, Any]]:
        """Read secret from Vault."""
        try:
            if self.session:
                async with self.session.get(f"{self.base_url}/secret/data/{path}") as response:
                    if response.status == 200:
                        vault_response = await response.json()
                        return vault_response.get('data', {}).get('data')
                    return None
            else:
                secret = self.secrets.get(path)
                return secret.data if secret else None
        except Exception as e:
            logger.error(f"Error reading secret: {str(e)}")
            return None
    
    async def delete_secret(self, path: str) -> bool:
        """Delete secret from Vault."""
        try:
            if self.session:
                async with self.session.delete(f"{self.base_url}/secret/data/{path}") as response:
                    success = response.status == 204
            else:
                success = self.secrets.pop(path, None) is not None
            
            if success:
                logger.info(f"Deleted secret at path: {path}")
            return success
        except Exception as e:
            logger.error(f"Error deleting secret: {str(e)}")
            return False
    
    async def create_token(self, policies: List[str] = None, ttl: str = "1h") -> Optional[str]:
        """Create Vault token."""
        try:
            payload = {
                "policies": policies or [],
                "ttl": ttl
            }
            
            if self.session:
                async with self.session.post(f"{self.base_url}/auth/token/create", json=payload) as response:
                    if response.status == 200:
                        token_response = await response.json()
                        return token_response.get('auth', {}).get('client_token')
                    return None
            else:
                # Mock token
                return f"mock_token_{int(time.time())}"
        except Exception as e:
            logger.error(f"Error creating token: {str(e)}")
            return None

class EtcdOperator:
    """@etcd operator implementation."""
    
    def __init__(self):
        self.base_url = ""
        self.session = None
        self.kv_store = {}
        
    async def connect(self, host: str = "localhost", port: int = 2379) -> bool:
        """Connect to etcd."""
        if not AIOHTTP_AVAILABLE:
            logger.error("aiohttp not available for etcd")
            return False
        
        try:
            self.base_url = f"http://{host}:{port}/v3"
            self.session = aiohttp.ClientSession()
            
            # Test connection
            async with self.session.get(f"{self.base_url}/maintenance/status") as response:
                if response.status == 200:
                    logger.info(f"Connected to etcd: {host}:{port}")
                    return True
                return False
        except Exception as e:
            logger.error(f"Error connecting to etcd: {str(e)}")
            return False
    
    async def put_key(self, key: str, value: str) -> bool:
        """Put key-value pair in etcd."""
        try:
            if self.session:
                payload = {
                    "key": key,
                    "value": value
                }
                async with self.session.post(f"{self.base_url}/kv/put", json=payload) as response:
                    success = response.status == 200
            else:
                self.kv_store[key] = value
                success = True
            
            return success
        except Exception as e:
            logger.error(f"Error putting key in etcd: {str(e)}")
            return False
    
    async def get_key(self, key: str) -> Optional[str]:
        """Get value from etcd."""
        try:
            if self.session:
                payload = {"key": key}
                async with self.session.post(f"{self.base_url}/kv/range", json=payload) as response:
                    if response.status == 200:
                        etcd_response = await response.json()
                        kvs = etcd_response.get('kvs', [])
                        return kvs[0].get('value') if kvs else None
                    return None
            else:
                return self.kv_store.get(key)
        except Exception as e:
            logger.error(f"Error getting key from etcd: {str(e)}")
            return None

class TemporalOperator:
    """@temporal operator implementation."""
    
    def __init__(self):
        self.workflows = {}
        self.activities = {}
        
    async def connect(self, host: str = "localhost", port: int = 7233, namespace: str = "default") -> bool:
        """Connect to Temporal."""
        try:
            logger.info(f"Connected to Temporal: {host}:{port} (namespace: {namespace})")
            return True
        except Exception as e:
            logger.error(f"Error connecting to Temporal: {str(e)}")
            return False
    
    def register_workflow(self, name: str, workflow_func: Any) -> bool:
        """Register workflow with Temporal."""
        try:
            self.workflows[name] = {
                'function': workflow_func,
                'registered_at': datetime.now(),
                'executions': 0
            }
            logger.info(f"Registered workflow: {name}")
            return True
        except Exception as e:
            logger.error(f"Error registering workflow: {str(e)}")
            return False
    
    def register_activity(self, name: str, activity_func: Any) -> bool:
        """Register activity with Temporal."""
        try:
            self.activities[name] = {
                'function': activity_func,
                'registered_at': datetime.now(),
                'executions': 0
            }
            logger.info(f"Registered activity: {name}")
            return True
        except Exception as e:
            logger.error(f"Error registering activity: {str(e)}")
            return False
    
    async def start_workflow(self, workflow_name: str, workflow_id: str, 
                           input_data: Dict[str, Any] = None) -> str:
        """Start workflow execution."""
        try:
            if workflow_name in self.workflows:
                self.workflows[workflow_name]['executions'] += 1
                execution_id = f"{workflow_id}_{int(time.time())}"
                logger.info(f"Started workflow: {workflow_name} (ID: {execution_id})")
                return execution_id
            else:
                logger.error(f"Workflow not found: {workflow_name}")
                return ""
        except Exception as e:
            logger.error(f"Error starting workflow: {str(e)}")
            return ""
    
    def get_workflow_status(self) -> Dict[str, Any]:
        """Get workflow status summary."""
        return {
            'registered_workflows': len(self.workflows),
            'registered_activities': len(self.activities),
            'total_executions': sum(w['executions'] for w in self.workflows.values()),
            'workflow_names': list(self.workflows.keys())
        }

class ServiceDiscoverySecuritySystems:
    """
    Advanced Service Discovery & Security Systems for TuskLang.
    Implements @consul, @vault, @etcd, and @temporal operators.
    """
    
    def __init__(self):
        self.consul = ConsulOperator()
        self.vault = VaultOperator()
        self.etcd = EtcdOperator()
        self.temporal = TemporalOperator()
        
        self.stats = {
            'services_registered': 0,
            'secrets_written': 0,
            'kv_operations': 0,
            'workflows_executed': 0
        }
    
    # Consul operator methods
    async def consul_connect(self, host: str = "localhost", port: int = 8500) -> bool:
        """Connect to Consul (@consul operator)."""
        return await self.consul.connect(host, port)
    
    async def consul_register_service(self, name: str, address: str, port: int, 
                                    tags: List[str] = None) -> bool:
        """Register service (@consul operator)."""
        service = ServiceRegistration(
            name=name,
            id=f"{name}_{address}_{port}",
            address=address,
            port=port,
            tags=tags or []
        )
        success = await self.consul.register_service(service)
        if success:
            self.stats['services_registered'] += 1
        return success
    
    async def consul_discover_services(self, service_name: str = None) -> List[Dict]:
        """Discover services (@consul operator)."""
        return await self.consul.discover_services(service_name)
    
    async def consul_put_kv(self, key: str, value: str) -> bool:
        """Put key-value (@consul operator)."""
        success = await self.consul.put_kv(key, value)
        if success:
            self.stats['kv_operations'] += 1
        return success
    
    async def consul_get_kv(self, key: str) -> Optional[str]:
        """Get key-value (@consul operator)."""
        result = await self.consul.get_kv(key)
        if result is not None:
            self.stats['kv_operations'] += 1
        return result
    
    # Vault operator methods
    async def vault_connect(self, host: str = "localhost", port: int = 8200, token: str = "") -> bool:
        """Connect to Vault (@vault operator)."""
        return await self.vault.connect(host, port, token)
    
    async def vault_write_secret(self, path: str, data: Dict[str, Any]) -> bool:
        """Write secret (@vault operator)."""
        success = await self.vault.write_secret(path, data)
        if success:
            self.stats['secrets_written'] += 1
        return success
    
    async def vault_read_secret(self, path: str) -> Optional[Dict[str, Any]]:
        """Read secret (@vault operator)."""
        return await self.vault.read_secret(path)
    
    async def vault_create_token(self, policies: List[str] = None, ttl: str = "1h") -> Optional[str]:
        """Create token (@vault operator)."""
        return await self.vault.create_token(policies, ttl)
    
    # etcd operator methods
    async def etcd_connect(self, host: str = "localhost", port: int = 2379) -> bool:
        """Connect to etcd (@etcd operator)."""
        return await self.etcd.connect(host, port)
    
    async def etcd_put(self, key: str, value: str) -> bool:
        """Put key-value (@etcd operator)."""
        success = await self.etcd.put_key(key, value)
        if success:
            self.stats['kv_operations'] += 1
        return success
    
    async def etcd_get(self, key: str) -> Optional[str]:
        """Get key-value (@etcd operator)."""
        result = await self.etcd.get_key(key)
        if result is not None:
            self.stats['kv_operations'] += 1
        return result
    
    # Temporal operator methods
    async def temporal_connect(self, host: str = "localhost", port: int = 7233) -> bool:
        """Connect to Temporal (@temporal operator)."""
        return await self.temporal.connect(host, port)
    
    def temporal_register_workflow(self, name: str, workflow_func: Any) -> bool:
        """Register workflow (@temporal operator)."""
        return self.temporal.register_workflow(name, workflow_func)
    
    def temporal_register_activity(self, name: str, activity_func: Any) -> bool:
        """Register activity (@temporal operator)."""
        return self.temporal.register_activity(name, activity_func)
    
    async def temporal_start_workflow(self, workflow_name: str, workflow_id: str, 
                                    input_data: Dict[str, Any] = None) -> str:
        """Start workflow (@temporal operator)."""
        execution_id = await self.temporal.start_workflow(workflow_name, workflow_id, input_data)
        if execution_id:
            self.stats['workflows_executed'] += 1
        return execution_id
    
    # Utility methods
    def get_stats(self) -> Dict[str, Any]:
        """Get system statistics."""
        return self.stats.copy()
    
    async def health_check(self) -> Dict[str, bool]:
        """Health check for all systems."""
        return {
            'consul': self.consul.session is not None,
            'vault': self.vault.session is not None,
            'etcd': self.etcd.session is not None,
            'temporal': True  # Always available (mock)
        }

# Example usage
async def main():
    """Example usage of Service Discovery & Security Systems."""
    print("=== Service Discovery & Security Systems Demo ===")
    
    systems = ServiceDiscoverySecuritySystems()
    
    # Test Consul
    print("\n1. Testing Consul (@consul operator)...")
    await systems.consul_register_service("web-service", "192.168.1.100", 8080, ["web", "api"])
    services = await systems.consul_discover_services("web-service")
    print(f"Discovered {len(services)} services")
    
    await systems.consul_put_kv("config/database_url", "postgres://localhost:5432/mydb")
    db_url = await systems.consul_get_kv("config/database_url")
    print(f"Retrieved config: {db_url}")
    
    # Test Vault
    print("\n2. Testing Vault (@vault operator)...")
    await systems.vault_write_secret("database/creds", {
        "username": "dbuser",
        "password": "secretpass"
    })
    
    creds = await systems.vault_read_secret("database/creds")
    print(f"Retrieved secret: {creds}")
    
    token = await systems.vault_create_token(["read", "write"], "2h")
    print(f"Created token: {token[:20]}..." if token else "Token creation failed")
    
    # Test etcd
    print("\n3. Testing etcd (@etcd operator)...")
    await systems.etcd_put("service/config", '{"timeout": 30, "retries": 3}')
    config = await systems.etcd_get("service/config")
    print(f"Retrieved etcd config: {config}")
    
    # Test Temporal
    print("\n4. Testing Temporal (@temporal operator)...")
    
    def sample_workflow(input_data):
        return {"result": "workflow completed", "input": input_data}
    
    def sample_activity(data):
        return {"processed": data, "timestamp": datetime.now().isoformat()}
    
    systems.temporal_register_workflow("sample_workflow", sample_workflow)
    systems.temporal_register_activity("sample_activity", sample_activity)
    
    execution_id = await systems.temporal_start_workflow("sample_workflow", "demo_001", {"task": "process"})
    print(f"Started workflow execution: {execution_id}")
    
    workflow_status = systems.temporal.get_workflow_status()
    print(f"Workflow status: {workflow_status}")
    
    # Statistics
    print("\n5. System statistics:")
    stats = systems.get_stats()
    for key, value in stats.items():
        print(f"  {key}: {value}")
    
    # Health check
    print("\n6. Health check:")
    health = await systems.health_check()
    for system, status in health.items():
        print(f"  {system}: {'✓' if status else '✗'}")
    
    print("\n=== Service Discovery & Security Demo Complete ===")

if __name__ == "__main__":
    asyncio.run(main()) 