"""
Advanced Operators Integration for TuskLang Python SDK
Integrates all advanced operators including ETCD, Elasticsearch, Zipkin, Grafana, 
Istio, Temporal, AMQP, SQLite Enhanced, and Custom Framework with the existing 
TuskLang operator system.
"""

import asyncio
import json
import re
from typing import Any, Dict, List, Optional
from datetime import datetime

# Import existing communication protocols
import sys
import os
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'communications'))
from advanced_communication_protocols import AdvancedCommunicationProtocols

# Import database integration systems
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'databases'))
try:
    from database_integration_systems import DatabaseIntegrationSystems
    DB_INTEGRATION_AVAILABLE = True
except ImportError:
    DB_INTEGRATION_AVAILABLE = False

# Import message queue systems
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'queues'))
try:
    from message_queue_systems import MessageQueueSystems
    QUEUE_INTEGRATION_AVAILABLE = True
except ImportError:
    QUEUE_INTEGRATION_AVAILABLE = False

# Import monitoring systems  
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'monitoring'))
try:
    from monitoring_observability_systems import MonitoringObservabilitySystems
    MONITORING_INTEGRATION_AVAILABLE = True
except ImportError:
    MONITORING_INTEGRATION_AVAILABLE = False

# Import new advanced operators
# ETCD Operator
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'infrastructure'))
try:
    from etcd_operator import EtcdOperator, EtcdConfig
    ETCD_AVAILABLE = True
except ImportError:
    ETCD_AVAILABLE = False

# Elasticsearch Operator
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'search'))
try:
    from elasticsearch_operator import ElasticsearchOperator, ElasticsearchConfig
    ELASTICSEARCH_AVAILABLE = True
except ImportError:
    ELASTICSEARCH_AVAILABLE = False

# Zipkin Operator
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'observability'))
try:
    from zipkin_operator import ZipkinOperator, ZipkinConfig
    ZIPKIN_AVAILABLE = True
except ImportError:
    ZIPKIN_AVAILABLE = False

# Grafana Operator
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'visualization'))
try:
    from grafana_operator import GrafanaOperator, GrafanaConfig
    GRAFANA_AVAILABLE = True
except ImportError:
    GRAFANA_AVAILABLE = False

# Istio Operator
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'servicemesh'))
try:
    from istio_operator import IstioOperator, IstioConfig
    ISTIO_AVAILABLE = True
except ImportError:
    ISTIO_AVAILABLE = False

# Temporal Operator
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'workflow'))
try:
    from temporal_operator import TemporalOperator, TemporalConfig
    TEMPORAL_AVAILABLE = True
except ImportError:
    TEMPORAL_AVAILABLE = False

# AMQP Operator
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..', 'operators', 'messaging'))
try:
    from amqp_operator import AMQPOperator, AMQPConfig
    AMQP_AVAILABLE = True
except ImportError:
    AMQP_AVAILABLE = False

# SQLite Enhanced Operator
try:
    from sqlite_enhanced_operator import SQLiteEnhancedOperator, SQLiteConfig
    SQLITE_ENHANCED_AVAILABLE = True
except ImportError:
    SQLITE_ENHANCED_AVAILABLE = False

# Custom Operator Framework
sys.path.append(os.path.join(os.path.dirname(__file__), '..', 'framework'))
try:
    from custom_operator_framework import CustomOperatorFramework, BaseOperator, OperatorType
    CUSTOM_FRAMEWORK_AVAILABLE = True
except ImportError:
    CUSTOM_FRAMEWORK_AVAILABLE = False


class AdvancedOperatorIntegration:
    """
    Integration layer for advanced operators in TuskLang.
    Provides seamless integration of all advanced operators including infrastructure,
    search, observability, visualization, service mesh, workflow, messaging,
    enhanced database, and custom framework capabilities.
    """
    
    def __init__(self):
        # Initialize existing communication protocols
        self.protocols = AdvancedCommunicationProtocols()
        
        # Initialize existing database integration if available
        if DB_INTEGRATION_AVAILABLE:
            self.database_systems = DatabaseIntegrationSystems()
        else:
            self.database_systems = None
        
        # Initialize existing message queue systems if available
        if QUEUE_INTEGRATION_AVAILABLE:
            self.queue_systems = MessageQueueSystems()
        else:
            self.queue_systems = None
        
        # Initialize existing monitoring systems if available
        if MONITORING_INTEGRATION_AVAILABLE:
            self.monitoring_systems = MonitoringObservabilitySystems()
        else:
            self.monitoring_systems = None
        
        # Initialize new advanced operators
        # Infrastructure
        if ETCD_AVAILABLE:
            self.etcd_operator = EtcdOperator()
        else:
            self.etcd_operator = None
        
        # Search
        if ELASTICSEARCH_AVAILABLE:
            self.elasticsearch_operator = ElasticsearchOperator()
        else:
            self.elasticsearch_operator = None
        
        # Observability
        if ZIPKIN_AVAILABLE:
            self.zipkin_operator = ZipkinOperator()
        else:
            self.zipkin_operator = None
        
        # Visualization
        if GRAFANA_AVAILABLE:
            self.grafana_operator = GrafanaOperator()
        else:
            self.grafana_operator = None
        
        # Service Mesh
        if ISTIO_AVAILABLE:
            self.istio_operator = IstioOperator()
        else:
            self.istio_operator = None
        
        # Workflow
        if TEMPORAL_AVAILABLE:
            self.temporal_operator = TemporalOperator()
        else:
            self.temporal_operator = None
        
        # Messaging
        if AMQP_AVAILABLE:
            self.amqp_operator = AMQPOperator()
        else:
            self.amqp_operator = None
        
        # Enhanced Database
        if SQLITE_ENHANCED_AVAILABLE:
            self.sqlite_enhanced_operator = SQLiteEnhancedOperator()
        else:
            self.sqlite_enhanced_operator = None
        
        # Custom Framework
        if CUSTOM_FRAMEWORK_AVAILABLE:
            self.custom_framework = CustomOperatorFramework()
        else:
            self.custom_framework = None
        
        # Enhanced operator execution statistics
        self.operator_stats = {
            'graphql_queries': 0,
            'grpc_calls': 0,
            'websocket_messages': 0,
            'sse_events': 0,
            'db_queries': 0,
            'queue_messages': 0,
            'monitoring_events': 0,
            'etcd_operations': 0,
            'elasticsearch_operations': 0,
            'zipkin_traces': 0,
            'grafana_operations': 0,
            'istio_operations': 0,
            'temporal_workflows': 0,
            'amqp_messages': 0,
            'sqlite_enhanced_operations': 0,
            'custom_operator_executions': 0
        }
    
    # Existing operator methods (keeping for compatibility)
    async def execute_graphql(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @graphql operator"""
        try:
            # Parse @graphql("query", {variables})
            match = re.match(r'@graphql\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @graphql syntax"}
            
            query = match.group(1)
            variables = {}
            if match.group(2):
                variables = json.loads(match.group(2))
            
            result = await self.protocols.execute_graphql(query, variables)
            self.operator_stats['graphql_queries'] += 1
            
            return result
            
        except Exception as e:
            return {"error": str(e)}
    
    # New ETCD operator methods (@etcd)
    async def execute_etcd(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @etcd operator"""
        if not self.etcd_operator:
            return {"error": "ETCD operator not available"}
        
        try:
            # Parse @etcd("operation", {params})
            match = re.match(r'@etcd\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @etcd syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate ETCD operation
            if operation == "get":
                result = await self.etcd_operator.get(params.get("key", ""))
                return {"value": result.value if result else None}
                
            elif operation == "put":
                result = await self.etcd_operator.put(
                    params.get("key", ""),
                    params.get("value", ""),
                    lease_id=params.get("lease_id")
                )
                return {"success": result}
                
            elif operation == "delete":
                result = await self.etcd_operator.delete(params.get("key", ""))
                return {"success": result}
                
            elif operation == "watch":
                # This would need special handling for async operations
                return {"error": "Watch operation requires callback setup"}
                
            else:
                return {"error": f"Unknown ETCD operation: {operation}"}
            
            self.operator_stats['etcd_operations'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New Elasticsearch operator methods (@elasticsearch)
    async def execute_elasticsearch(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @elasticsearch operator"""
        if not self.elasticsearch_operator:
            return {"error": "Elasticsearch operator not available"}
        
        try:
            # Parse @elasticsearch("operation", {params})
            match = re.match(r'@elasticsearch\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @elasticsearch syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate Elasticsearch operation
            if operation == "search":
                from elasticsearch_operator import ElasticsearchQuery
                query = ElasticsearchQuery(
                    query=params.get("query", {}),
                    size=params.get("size", 10),
                    from_=params.get("from", 0)
                )
                result = await self.elasticsearch_operator.search(query, params.get("indices"))
                return result
                
            elif operation == "index":
                from elasticsearch_operator import ElasticsearchDocument
                document = ElasticsearchDocument(
                    index=params.get("index", ""),
                    id=params.get("id"),
                    source=params.get("document", {})
                )
                result = await self.elasticsearch_operator.index_document(document)
                return result
                
            elif operation == "delete":
                result = await self.elasticsearch_operator.delete_document(
                    params.get("index", ""),
                    params.get("id", "")
                )
                return result
                
            else:
                return {"error": f"Unknown Elasticsearch operation: {operation}"}
            
            self.operator_stats['elasticsearch_operations'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New Zipkin operator methods (@zipkin)
    async def execute_zipkin(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @zipkin operator"""
        if not self.zipkin_operator:
            return {"error": "Zipkin operator not available"}
        
        try:
            # Parse @zipkin("operation", {params})
            match = re.match(r'@zipkin\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @zipkin syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate Zipkin operation
            if operation == "trace":
                # Create trace context
                with self.zipkin_operator.trace(
                    params.get("operation_name", "tsk-operation"),
                    tags=params.get("tags", {}),
                    kind=params.get("kind")
                ) as span:
                    # Add annotations if provided
                    if params.get("annotations"):
                        for annotation in params["annotations"]:
                            self.zipkin_operator.add_annotation(annotation)
                    
                    return {"trace_id": span.trace_id if span else None}
                    
            elif operation == "add_tag":
                self.zipkin_operator.add_tag(params.get("key", ""), params.get("value", ""))
                return {"success": True}
                
            elif operation == "add_annotation":
                self.zipkin_operator.add_annotation(params.get("value", ""))
                return {"success": True}
                
            else:
                return {"error": f"Unknown Zipkin operation: {operation}"}
            
            self.operator_stats['zipkin_traces'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New Grafana operator methods (@grafana)
    async def execute_grafana(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @grafana operator"""
        if not self.grafana_operator:
            return {"error": "Grafana operator not available"}
        
        try:
            # Parse @grafana("operation", {params})
            match = re.match(r'@grafana\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @grafana syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate Grafana operation
            if operation == "create_dashboard":
                from grafana_operator import GrafanaDashboard
                dashboard = GrafanaDashboard(
                    title=params.get("title", ""),
                    panels=params.get("panels", []),
                    tags=params.get("tags", [])
                )
                result = await self.grafana_operator.create_dashboard(dashboard)
                return result
                
            elif operation == "query":
                result = await self.grafana_operator.query_datasource(
                    params.get("datasource", ""),
                    params.get("query", {})
                )
                return result
                
            elif operation == "search_dashboards":
                result = await self.grafana_operator.search_dashboards(
                    params.get("query", ""),
                    params.get("tags", [])
                )
                return result
                
            else:
                return {"error": f"Unknown Grafana operation: {operation}"}
            
            self.operator_stats['grafana_operations'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New Istio operator methods (@istio)
    async def execute_istio(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @istio operator"""
        if not self.istio_operator:
            return {"error": "Istio operator not available"}
        
        try:
            # Parse @istio("operation", {params})
            match = re.match(r'@istio\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @istio syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate Istio operation
            if operation == "create_virtual_service":
                from istio_operator import VirtualService
                vs = VirtualService(
                    name=params.get("name", ""),
                    namespace=params.get("namespace", "default"),
                    hosts=params.get("hosts", []),
                    http_routes=params.get("http_routes", [])
                )
                result = await self.istio_operator.create_virtual_service(vs)
                return {"success": result}
                
            elif operation == "configure_canary":
                result = await self.istio_operator.configure_canary_deployment(
                    params.get("service_name", ""),
                    params.get("namespace", "default"),
                    params.get("stable_version", "v1"),
                    params.get("canary_version", "v2"),
                    params.get("canary_weight", 10)
                )
                return {"success": result}
                
            else:
                return {"error": f"Unknown Istio operation: {operation}"}
            
            self.operator_stats['istio_operations'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New Temporal operator methods (@temporal)
    async def execute_temporal(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @temporal operator"""
        if not self.temporal_operator:
            return {"error": "Temporal operator not available"}
        
        try:
            # Parse @temporal("operation", {params})
            match = re.match(r'@temporal\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @temporal syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate Temporal operation
            if operation == "start_workflow":
                workflow_id = await self.temporal_operator.start_workflow(
                    params.get("workflow_name", ""),
                    *params.get("args", [])
                )
                return {"workflow_id": workflow_id}
                
            elif operation == "get_result":
                result = await self.temporal_operator.get_workflow_result(
                    params.get("workflow_id", ""),
                    params.get("timeout")
                )
                return {"result": result}
                
            elif operation == "signal":
                result = await self.temporal_operator.signal_workflow(
                    params.get("workflow_id", ""),
                    params.get("signal_name", ""),
                    *params.get("args", [])
                )
                return {"success": result}
                
            else:
                return {"error": f"Unknown Temporal operation: {operation}"}
            
            self.operator_stats['temporal_workflows'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New AMQP operator methods (@amqp)
    async def execute_amqp(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @amqp operator"""
        if not self.amqp_operator:
            return {"error": "AMQP operator not available"}
        
        try:
            # Parse @amqp("operation", {params})
            match = re.match(r'@amqp\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @amqp syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate AMQP operation
            if operation == "publish":
                from amqp_operator import AMQPMessage
                message = AMQPMessage(
                    body=params.get("body", ""),
                    routing_key=params.get("routing_key", ""),
                    exchange=params.get("exchange", "")
                )
                result = await self.amqp_operator.publish_message(message)
                return {"success": result}
                
            elif operation == "declare_queue":
                from amqp_operator import AMQPQueue
                queue = AMQPQueue(
                    name=params.get("name", ""),
                    durable=params.get("durable", True)
                )
                result = await self.amqp_operator.declare_queue(queue)
                return {"success": result}
                
            else:
                return {"error": f"Unknown AMQP operation: {operation}"}
            
            self.operator_stats['amqp_messages'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New SQLite Enhanced operator methods (@sqlite_enhanced)
    async def execute_sqlite_enhanced(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @sqlite_enhanced operator"""
        if not self.sqlite_enhanced_operator:
            return {"error": "SQLite Enhanced operator not available"}
        
        try:
            # Parse @sqlite_enhanced("operation", {params})
            match = re.match(r'@sqlite_enhanced\("([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @sqlite_enhanced syntax"}
            
            operation = match.group(1)
            params = {}
            if match.group(2):
                params = json.loads(match.group(2))
            
            # Route to appropriate SQLite Enhanced operation
            if operation == "query":
                result = await self.sqlite_enhanced_operator.execute_query(
                    params.get("sql", ""),
                    params.get("params"),
                    analyze=params.get("analyze", False)
                )
                return {
                    "data": result.data,
                    "execution_time": result.execution_time,
                    "query_plan": result.query_plan
                }
                
            elif operation == "fts_search":
                result = await self.sqlite_enhanced_operator.fts_search(
                    params.get("table", ""),
                    params.get("query", ""),
                    params.get("limit", 100)
                )
                return {"results": result}
                
            elif operation == "create_backup":
                backup_path = await self.sqlite_enhanced_operator.create_backup(
                    params.get("backup_name")
                )
                return {"backup_path": backup_path}
                
            else:
                return {"error": f"Unknown SQLite Enhanced operation: {operation}"}
            
            self.operator_stats['sqlite_enhanced_operations'] += 1
            
        except Exception as e:
            return {"error": str(e)}
    
    # New Custom Framework operator methods (@custom)
    async def execute_custom(self, expression: str, context: Dict[str, Any]) -> Any:
        """Execute @custom operator"""
        if not self.custom_framework:
            return {"error": "Custom Framework not available"}
        
        try:
            # Parse @custom("operator_id.operation", {params})
            match = re.match(r'@custom\("([^.]+)\.([^"]+)"\s*(?:,\s*({.+}))?\)', expression)
            if not match:
                return {"error": "Invalid @custom syntax (use operator_id.operation)"}
            
            operator_id = match.group(1)
            operation = match.group(2)
            params = {}
            if match.group(3):
                params = json.loads(match.group(3))
            
            # Execute custom operator
            result = await self.custom_framework.execute_operator(
                operator_id, 
                operation, 
                *params.get("args", []),
                **params.get("kwargs", {})
            )
            
            self.operator_stats['custom_operator_executions'] += 1
            return {"result": result}
            
        except Exception as e:
            return {"error": str(e)}
    
    # Master operator execution method
    async def execute_operator(self, expression: str, context: Optional[Dict[str, Any]] = None) -> Any:
        """
        Master method to execute any advanced operator.
        Routes the expression to the appropriate operator based on the operator prefix.
        """
        if context is None:
            context = {}
        
        try:
            # Extract operator type from expression
            operator_match = re.match(r'@(\w+)', expression)
            if not operator_match:
                return {"error": "Invalid operator syntax"}
            
            operator_type = operator_match.group(1)
            
            # Route to appropriate operator
            if operator_type == "graphql":
                return await self.execute_graphql(expression, context)
            elif operator_type == "etcd":
                return await self.execute_etcd(expression, context)
            elif operator_type == "elasticsearch":
                return await self.execute_elasticsearch(expression, context)
            elif operator_type == "zipkin":
                return await self.execute_zipkin(expression, context)
            elif operator_type == "grafana":
                return await self.execute_grafana(expression, context)
            elif operator_type == "istio":
                return await self.execute_istio(expression, context)
            elif operator_type == "temporal":
                return await self.execute_temporal(expression, context)
            elif operator_type == "amqp":
                return await self.execute_amqp(expression, context)
            elif operator_type == "sqlite_enhanced":
                return await self.execute_sqlite_enhanced(expression, context)
            elif operator_type == "custom":
                return await self.execute_custom(expression, context)
            else:
                return {"error": f"Unknown operator type: {operator_type}"}
                
        except Exception as e:
            return {"error": f"Operator execution failed: {str(e)}"}
    
    def get_available_operators(self) -> List[str]:
        """Get list of all available operators."""
        operators = []
        
        # Existing operators
        operators.extend(["graphql", "grpc", "websocket", "sse"])
        
        # New advanced operators
        if ETCD_AVAILABLE:
            operators.append("etcd")
        if ELASTICSEARCH_AVAILABLE:
            operators.append("elasticsearch")
        if ZIPKIN_AVAILABLE:
            operators.append("zipkin")
        if GRAFANA_AVAILABLE:
            operators.append("grafana")
        if ISTIO_AVAILABLE:
            operators.append("istio")
        if TEMPORAL_AVAILABLE:
            operators.append("temporal")
        if AMQP_AVAILABLE:
            operators.append("amqp")
        if SQLITE_ENHANCED_AVAILABLE:
            operators.append("sqlite_enhanced")
        if CUSTOM_FRAMEWORK_AVAILABLE:
            operators.append("custom")
        
        return sorted(operators)
    
    def get_operator_statistics(self) -> Dict[str, Any]:
        """Get comprehensive statistics for all operators."""
        stats = {
            'total_operations': sum(self.operator_stats.values()),
            'operator_stats': self.operator_stats.copy(),
            'available_operators': self.get_available_operators(),
            'operator_availability': {
                'etcd': ETCD_AVAILABLE,
                'elasticsearch': ELASTICSEARCH_AVAILABLE,
                'zipkin': ZIPKIN_AVAILABLE,
                'grafana': GRAFANA_AVAILABLE,
                'istio': ISTIO_AVAILABLE,
                'temporal': TEMPORAL_AVAILABLE,
                'amqp': AMQP_AVAILABLE,
                'sqlite_enhanced': SQLITE_ENHANCED_AVAILABLE,
                'custom_framework': CUSTOM_FRAMEWORK_AVAILABLE
            }
        }
        
        # Add individual operator statistics
        individual_stats = {}
        
        if self.etcd_operator:
            individual_stats['etcd'] = self.etcd_operator.get_statistics()
        if self.elasticsearch_operator:
            individual_stats['elasticsearch'] = self.elasticsearch_operator.get_statistics()
        if self.zipkin_operator:
            individual_stats['zipkin'] = self.zipkin_operator.get_statistics()
        if self.grafana_operator:
            individual_stats['grafana'] = self.grafana_operator.get_statistics()
        if self.istio_operator:
            individual_stats['istio'] = self.istio_operator.get_statistics()
        if self.temporal_operator:
            individual_stats['temporal'] = self.temporal_operator.get_statistics()
        if self.amqp_operator:
            individual_stats['amqp'] = self.amqp_operator.get_statistics()
        if self.sqlite_enhanced_operator:
            individual_stats['sqlite_enhanced'] = self.sqlite_enhanced_operator.get_statistics()
        if self.custom_framework:
            individual_stats['custom_framework'] = self.custom_framework.get_framework_statistics()
        
        stats['individual_operator_stats'] = individual_stats
        
        return stats
    
    async def close_all_operators(self):
        """Close all operators and cleanup resources."""
        cleanup_tasks = []
        
        # Close new advanced operators
        if self.etcd_operator:
            cleanup_tasks.append(self.etcd_operator.close())
        if self.elasticsearch_operator:
            cleanup_tasks.append(self.elasticsearch_operator.close())
        if self.zipkin_operator:
            cleanup_tasks.append(self.zipkin_operator.close())
        if self.grafana_operator:
            cleanup_tasks.append(self.grafana_operator.close())
        if self.istio_operator:
            cleanup_tasks.append(self.istio_operator.close())
        if self.temporal_operator:
            cleanup_tasks.append(self.temporal_operator.close())
        if self.amqp_operator:
            cleanup_tasks.append(self.amqp_operator.close())
        if self.sqlite_enhanced_operator:
            cleanup_tasks.append(self.sqlite_enhanced_operator.close())
        if self.custom_framework:
            cleanup_tasks.append(self.custom_framework.shutdown())
        
        # Execute all cleanup tasks
        if cleanup_tasks:
            await asyncio.gather(*cleanup_tasks, return_exceptions=True) 