"""
TuskLang Context Management
Provides context for operator execution and configuration.
"""

import asyncio
import json
import logging
from dataclasses import dataclass, field
from datetime import datetime
from typing import Any, Dict, List, Optional, Union

logger = logging.getLogger(__name__)

@dataclass
class TuskContext:
    """TuskLang execution context."""
    
    # Basic context information
    request_id: Optional[str] = None
    user_id: Optional[str] = None
    session_id: Optional[str] = None
    timestamp: datetime = field(default_factory=datetime.now)
    
    # Configuration and settings
    config: Dict[str, Any] = field(default_factory=dict)
    environment: str = "development"
    
    # Operator-specific data
    operator_data: Dict[str, Any] = field(default_factory=dict)
    
    # Metrics and monitoring
    metrics: Dict[str, Any] = field(default_factory=dict)
    start_time: Optional[datetime] = None
    
    # Error handling
    errors: List[Dict[str, Any]] = field(default_factory=list)
    warnings: List[str] = field(default_factory=list)
    
    def __post_init__(self):
        """Initialize context after creation."""
        if self.start_time is None:
            self.start_time = datetime.now()
    
    def add_metric(self, name: str, value: Any, tags: Optional[Dict[str, str]] = None):
        """Add a metric to the context."""
        metric = {
            'name': name,
            'value': value,
            'timestamp': datetime.now().isoformat(),
            'tags': tags or {}
        }
        self.metrics[name] = metric
    
    def add_error(self, error: str, details: Optional[Dict[str, Any]] = None):
        """Add an error to the context."""
        error_info = {
            'message': error,
            'timestamp': datetime.now().isoformat(),
            'details': details or {}
        }
        self.errors.append(error_info)
        logger.error(f"Context error: {error}")
    
    def add_warning(self, warning: str):
        """Add a warning to the context."""
        self.warnings.append(warning)
        logger.warning(f"Context warning: {warning}")
    
    def get_operator_data(self, operator: str, key: str, default: Any = None) -> Any:
        """Get operator-specific data."""
        return self.operator_data.get(operator, {}).get(key, default)
    
    def set_operator_data(self, operator: str, key: str, value: Any):
        """Set operator-specific data."""
        if operator not in self.operator_data:
            self.operator_data[operator] = {}
        self.operator_data[operator][key] = value
    
    def get_execution_time(self) -> float:
        """Get execution time in seconds."""
        if self.start_time:
            return (datetime.now() - self.start_time).total_seconds()
        return 0.0
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert context to dictionary."""
        return {
            'request_id': self.request_id,
            'user_id': self.user_id,
            'session_id': self.session_id,
            'timestamp': self.timestamp.isoformat(),
            'environment': self.environment,
            'config': self.config,
            'operator_data': self.operator_data,
            'metrics': self.metrics,
            'errors': self.errors,
            'warnings': self.warnings,
            'execution_time': self.get_execution_time()
        }
    
    def from_dict(self, data: Dict[str, Any]):
        """Load context from dictionary."""
        self.request_id = data.get('request_id')
        self.user_id = data.get('user_id')
        self.session_id = data.get('session_id')
        self.environment = data.get('environment', 'development')
        self.config = data.get('config', {})
        self.operator_data = data.get('operator_data', {})
        self.metrics = data.get('metrics', {})
        self.errors = data.get('errors', [])
        self.warnings = data.get('warnings', [])
    
    def clear(self):
        """Clear context data."""
        self.operator_data.clear()
        self.metrics.clear()
        self.errors.clear()
        self.warnings.clear()
        self.start_time = datetime.now()
    
    def is_healthy(self) -> bool:
        """Check if context is healthy (no critical errors)."""
        return len(self.errors) == 0
    
    def get_summary(self) -> Dict[str, Any]:
        """Get context summary."""
        return {
            'request_id': self.request_id,
            'execution_time': self.get_execution_time(),
            'error_count': len(self.errors),
            'warning_count': len(self.warnings),
            'metric_count': len(self.metrics),
            'is_healthy': self.is_healthy()
        } 