"""
Custom Operator Framework - Extensible Operator System
Production-ready framework for developing, registering, and managing custom TuskLang operators
with plugin architecture, lifecycle management, and comprehensive tooling.
"""

import asyncio
import importlib
import inspect
import json
import logging
import os
import sys
from abc import ABC, abstractmethod
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, Type, Union
import yaml
import ast
import re
import threading
from enum import Enum

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class OperatorLifecycle(Enum):
    """Operator lifecycle states."""
    REGISTERED = "registered"
    INITIALIZED = "initialized"
    ACTIVE = "active"
    SUSPENDED = "suspended"
    ERROR = "error"
    UNREGISTERED = "unregistered"

class OperatorType(Enum):
    """Operator types."""
    DATABASE = "database"
    MESSAGING = "messaging"
    NETWORK = "network"
    COMPUTATION = "computation"
    INTEGRATION = "integration"
    UTILITY = "utility"
    CUSTOM = "custom"

@dataclass
class OperatorMetadata:
    """Operator metadata structure."""
    name: str
    version: str
    description: str
    author: str
    type: OperatorType
    dependencies: List[str] = field(default_factory=list)
    parameters: Dict[str, Any] = field(default_factory=dict)
    tags: List[str] = field(default_factory=list)
    documentation_url: Optional[str] = None
    source_url: Optional[str] = None
    license: str = "MIT"
    min_python_version: str = "3.8"
    max_instances: int = -1  # -1 for unlimited

@dataclass
class OperatorMethod:
    """Operator method definition."""
    name: str
    function: Callable
    parameters: Dict[str, Any] = field(default_factory=dict)
    return_type: Optional[Type] = None
    description: str = ""
    examples: List[str] = field(default_factory=list)
    async_method: bool = False

@dataclass
class OperatorInstance:
    """Running operator instance."""
    operator_id: str
    operator_class: Type
    instance: Any
    metadata: OperatorMetadata
    lifecycle_state: OperatorLifecycle = OperatorLifecycle.REGISTERED
    created_at: datetime = field(default_factory=datetime.now)
    last_used: datetime = field(default_factory=datetime.now)
    usage_count: int = 0
    error_count: int = 0
    last_error: Optional[str] = None

class BaseOperator(ABC):
    """Base class for all custom operators."""
    
    def __init__(self):
        self.operator_id: Optional[str] = None
        self.metadata: Optional[OperatorMetadata] = None
        self.config: Dict[str, Any] = {}
        self.stats: Dict[str, Any] = {
            'operations_performed': 0,
            'errors_occurred': 0,
            'last_operation_time': None
        }
        self._initialized = False
    
    @abstractmethod
    async def initialize(self, config: Dict[str, Any]) -> bool:
        """Initialize the operator with configuration."""
        pass
    
    @abstractmethod
    async def execute(self, operation: str, *args, **kwargs) -> Any:
        """Execute operator operation."""
        pass
    
    async def cleanup(self) -> bool:
        """Cleanup resources."""
        return True
    
    def get_metadata(self) -> OperatorMetadata:
        """Get operator metadata."""
        if self.metadata:
            return self.metadata
        
        # Default metadata from class
        return OperatorMetadata(
            name=self.__class__.__name__,
            version="1.0.0",
            description="Custom operator",
            author="Unknown",
            type=OperatorType.CUSTOM
        )
    
    def get_available_operations(self) -> List[str]:
        """Get list of available operations."""
        operations = []
        for method_name in dir(self):
            if not method_name.startswith('_') and callable(getattr(self, method_name)):
                if method_name not in ['initialize', 'execute', 'cleanup', 'get_metadata', 'get_available_operations']:
                    operations.append(method_name)
        return operations
    
    def validate_config(self, config: Dict[str, Any]) -> bool:
        """Validate configuration."""
        return True
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get operator statistics."""
        return self.stats.copy()

class OperatorValidator:
    """Validates operator implementations and configurations."""
    
    def __init__(self):
        self.validation_rules = {
            'class_structure': self._validate_class_structure,
            'metadata': self._validate_metadata,
            'methods': self._validate_methods,
            'documentation': self._validate_documentation,
            'dependencies': self._validate_dependencies
        }
    
    def validate_operator(self, operator_class: Type[BaseOperator]) -> Dict[str, Any]:
        """Comprehensive operator validation."""
        results = {
            'valid': True,
            'warnings': [],
            'errors': [],
            'details': {}
        }
        
        for rule_name, validator in self.validation_rules.items():
            try:
                rule_result = validator(operator_class)
                results['details'][rule_name] = rule_result
                
                if rule_result.get('errors'):
                    results['errors'].extend(rule_result['errors'])
                    results['valid'] = False
                
                if rule_result.get('warnings'):
                    results['warnings'].extend(rule_result['warnings'])
                    
            except Exception as e:
                results['errors'].append(f"Validation rule {rule_name} failed: {str(e)}")
                results['valid'] = False
        
        return results
    
    def _validate_class_structure(self, operator_class: Type[BaseOperator]) -> Dict[str, Any]:
        """Validate class structure."""
        result = {'errors': [], 'warnings': []}
        
        # Check inheritance
        if not issubclass(operator_class, BaseOperator):
            result['errors'].append("Operator must inherit from BaseOperator")
        
        # Check required methods
        required_methods = ['initialize', 'execute']
        for method in required_methods:
            if not hasattr(operator_class, method):
                result['errors'].append(f"Missing required method: {method}")
        
        # Check method signatures
        if hasattr(operator_class, 'initialize'):
            init_sig = inspect.signature(operator_class.initialize)
            if 'config' not in init_sig.parameters:
                result['errors'].append("initialize method must accept 'config' parameter")
        
        return result
    
    def _validate_metadata(self, operator_class: Type[BaseOperator]) -> Dict[str, Any]:
        """Validate operator metadata."""
        result = {'errors': [], 'warnings': []}
        
        instance = operator_class()
        metadata = instance.get_metadata()
        
        # Required fields
        required_fields = ['name', 'version', 'description', 'author']
        for field in required_fields:
            if not getattr(metadata, field) or getattr(metadata, field) == "":
                result['warnings'].append(f"Metadata field '{field}' is empty or missing")
        
        # Version format
        version_pattern = r'^\d+\.\d+\.\d+$'
        if not re.match(version_pattern, metadata.version):
            result['warnings'].append("Version should follow semantic versioning (e.g., '1.0.0')")
        
        return result
    
    def _validate_methods(self, operator_class: Type[BaseOperator]) -> Dict[str, Any]:
        """Validate operator methods."""
        result = {'errors': [], 'warnings': []}
        
        instance = operator_class()
        methods = instance.get_available_operations()
        
        if not methods:
            result['warnings'].append("No custom operations defined")
        
        # Check method implementations
        for method_name in methods:
            method = getattr(instance, method_name)
            if not inspect.isdefined(method):
                result['warnings'].append(f"Method '{method_name}' has no implementation")
        
        return result
    
    def _validate_documentation(self, operator_class: Type[BaseOperator]) -> Dict[str, Any]:
        """Validate documentation."""
        result = {'errors': [], 'warnings': []}
        
        if not operator_class.__doc__:
            result['warnings'].append("Class missing docstring")
        
        # Check method docstrings
        for name, method in inspect.getmembers(operator_class, predicate=inspect.isfunction):
            if not name.startswith('_') and not method.__doc__:
                result['warnings'].append(f"Method '{name}' missing docstring")
        
        return result
    
    def _validate_dependencies(self, operator_class: Type[BaseOperator]) -> Dict[str, Any]:
        """Validate dependencies."""
        result = {'errors': [], 'warnings': []}
        
        instance = operator_class()
        metadata = instance.get_metadata()
        
        # Check if dependencies are available
        for dependency in metadata.dependencies:
            try:
                importlib.import_module(dependency)
            except ImportError:
                result['warnings'].append(f"Optional dependency '{dependency}' not available")
        
        return result

class OperatorRegistry:
    """Central registry for all operators."""
    
    def __init__(self):
        self.operators: Dict[str, OperatorInstance] = {}
        self.operator_types: Dict[OperatorType, List[str]] = {}
        self.plugin_paths: List[Path] = []
        self._lock = threading.Lock()
        
        # Initialize type mapping
        for op_type in OperatorType:
            self.operator_types[op_type] = []
    
    def register_operator(self, operator_class: Type[BaseOperator], 
                         operator_id: Optional[str] = None) -> str:
        """Register operator in the registry."""
        with self._lock:
            # Generate ID if not provided
            if operator_id is None:
                operator_id = f"{operator_class.__name__}_{len(self.operators)}"
            
            # Check if already registered
            if operator_id in self.operators:
                raise ValueError(f"Operator {operator_id} already registered")
            
            # Create instance
            instance = operator_class()
            metadata = instance.get_metadata()
            
            # Create operator instance record
            operator_instance = OperatorInstance(
                operator_id=operator_id,
                operator_class=operator_class,
                instance=instance,
                metadata=metadata
            )
            
            # Store in registry
            self.operators[operator_id] = operator_instance
            self.operator_types[metadata.type].append(operator_id)
            
            logger.info(f"Registered operator: {operator_id}")
            return operator_id
    
    def unregister_operator(self, operator_id: str) -> bool:
        """Unregister operator."""
        with self._lock:
            if operator_id not in self.operators:
                return False
            
            operator_instance = self.operators[operator_id]
            
            # Cleanup operator
            if operator_instance.instance._initialized:
                asyncio.create_task(operator_instance.instance.cleanup())
            
            # Remove from type mapping
            op_type = operator_instance.metadata.type
            if operator_id in self.operator_types[op_type]:
                self.operator_types[op_type].remove(operator_id)
            
            # Remove from registry
            del self.operators[operator_id]
            
            logger.info(f"Unregistered operator: {operator_id}")
            return True
    
    def get_operator(self, operator_id: str) -> Optional[OperatorInstance]:
        """Get operator instance."""
        return self.operators.get(operator_id)
    
    def list_operators(self, op_type: Optional[OperatorType] = None) -> List[OperatorInstance]:
        """List registered operators."""
        if op_type is None:
            return list(self.operators.values())
        
        operator_ids = self.operator_types.get(op_type, [])
        return [self.operators[op_id] for op_id in operator_ids if op_id in self.operators]
    
    def search_operators(self, query: str) -> List[OperatorInstance]:
        """Search operators by name, description, or tags."""
        results = []
        query_lower = query.lower()
        
        for operator_instance in self.operators.values():
            metadata = operator_instance.metadata
            
            # Search in name, description, and tags
            searchable_text = f"{metadata.name} {metadata.description} {' '.join(metadata.tags)}".lower()
            
            if query_lower in searchable_text:
                results.append(operator_instance)
        
        return results
    
    def add_plugin_path(self, path: Union[str, Path]):
        """Add path to search for operator plugins."""
        plugin_path = Path(path)
        if plugin_path.exists() and plugin_path not in self.plugin_paths:
            self.plugin_paths.append(plugin_path)
            sys.path.insert(0, str(plugin_path))
    
    def discover_operators(self) -> List[str]:
        """Discover and load operators from plugin paths."""
        discovered = []
        
        for plugin_path in self.plugin_paths:
            for py_file in plugin_path.glob("**/*.py"):
                try:
                    # Load module
                    spec = importlib.util.spec_from_file_location(
                        py_file.stem, py_file
                    )
                    module = importlib.util.module_from_spec(spec)
                    spec.loader.exec_module(module)
                    
                    # Find operator classes
                    for name, obj in inspect.getmembers(module, inspect.isclass):
                        if (issubclass(obj, BaseOperator) and 
                            obj != BaseOperator and 
                            name not in [op.metadata.name for op in self.operators.values()]):
                            
                            operator_id = self.register_operator(obj)
                            discovered.append(operator_id)
                            
                except Exception as e:
                    logger.warning(f"Error loading operator from {py_file}: {str(e)}")
        
        return discovered

class OperatorDocumentationGenerator:
    """Generates documentation for operators."""
    
    def __init__(self, registry: OperatorRegistry):
        self.registry = registry
    
    def generate_operator_docs(self, operator_id: str, format: str = "markdown") -> str:
        """Generate documentation for specific operator."""
        operator_instance = self.registry.get_operator(operator_id)
        if not operator_instance:
            raise ValueError(f"Operator {operator_id} not found")
        
        if format.lower() == "markdown":
            return self._generate_markdown_docs(operator_instance)
        elif format.lower() == "json":
            return self._generate_json_docs(operator_instance)
        else:
            raise ValueError(f"Unsupported format: {format}")
    
    def _generate_markdown_docs(self, operator_instance: OperatorInstance) -> str:
        """Generate Markdown documentation."""
        metadata = operator_instance.metadata
        instance = operator_instance.instance
        
        doc = f"# {metadata.name}\n\n"
        doc += f"**Version:** {metadata.version}\n"
        doc += f"**Author:** {metadata.author}\n"
        doc += f"**Type:** {metadata.type.value}\n"
        doc += f"**License:** {metadata.license}\n\n"
        
        doc += f"## Description\n\n{metadata.description}\n\n"
        
        # Dependencies
        if metadata.dependencies:
            doc += "## Dependencies\n\n"
            for dep in metadata.dependencies:
                doc += f"- {dep}\n"
            doc += "\n"
        
        # Operations
        operations = instance.get_available_operations()
        if operations:
            doc += "## Available Operations\n\n"
            for op in operations:
                method = getattr(instance, op)
                doc += f"### {op}\n\n"
                if method.__doc__:
                    doc += f"{method.__doc__}\n\n"
                
                # Method signature
                sig = inspect.signature(method)
                doc += f"**Signature:** `{op}{sig}`\n\n"
        
        # Configuration
        if metadata.parameters:
            doc += "## Configuration Parameters\n\n"
            for param, details in metadata.parameters.items():
                doc += f"- **{param}**: {details}\n"
            doc += "\n"
        
        # Tags
        if metadata.tags:
            doc += f"**Tags:** {', '.join(metadata.tags)}\n\n"
        
        return doc
    
    def _generate_json_docs(self, operator_instance: OperatorInstance) -> str:
        """Generate JSON documentation."""
        metadata = operator_instance.metadata
        instance = operator_instance.instance
        
        docs = {
            "name": metadata.name,
            "version": metadata.version,
            "description": metadata.description,
            "author": metadata.author,
            "type": metadata.type.value,
            "license": metadata.license,
            "dependencies": metadata.dependencies,
            "parameters": metadata.parameters,
            "tags": metadata.tags,
            "operations": []
        }
        
        # Add operations
        operations = instance.get_available_operations()
        for op in operations:
            method = getattr(instance, op)
            sig = inspect.signature(method)
            
            op_doc = {
                "name": op,
                "signature": str(sig),
                "description": method.__doc__ or "",
                "parameters": {
                    param.name: {
                        "type": param.annotation.__name__ if param.annotation != inspect.Parameter.empty else "Any",
                        "default": param.default if param.default != inspect.Parameter.empty else None
                    }
                    for param in sig.parameters.values()
                }
            }
            docs["operations"].append(op_doc)
        
        return json.dumps(docs, indent=2)
    
    def generate_registry_docs(self, format: str = "markdown") -> str:
        """Generate documentation for entire registry."""
        if format.lower() == "markdown":
            return self._generate_registry_markdown()
        elif format.lower() == "json":
            return self._generate_registry_json()
        else:
            raise ValueError(f"Unsupported format: {format}")
    
    def _generate_registry_markdown(self) -> str:
        """Generate Markdown documentation for registry."""
        doc = "# TuskLang Operator Registry\n\n"
        doc += f"**Total Operators:** {len(self.registry.operators)}\n\n"
        
        # Group by type
        for op_type in OperatorType:
            operators = self.registry.list_operators(op_type)
            if operators:
                doc += f"## {op_type.value.title()} Operators\n\n"
                
                for operator_instance in operators:
                    metadata = operator_instance.metadata
                    doc += f"### {metadata.name} ({metadata.version})\n"
                    doc += f"{metadata.description}\n\n"
                    doc += f"**Author:** {metadata.author}  \n"
                    if metadata.tags:
                        doc += f"**Tags:** {', '.join(metadata.tags)}  \n"
                    doc += "\n"
        
        return doc
    
    def _generate_registry_json(self) -> str:
        """Generate JSON documentation for registry."""
        registry_data = {
            "total_operators": len(self.registry.operators),
            "types": {},
            "operators": []
        }
        
        # Count by type
        for op_type in OperatorType:
            count = len(self.registry.operator_types[op_type])
            if count > 0:
                registry_data["types"][op_type.value] = count
        
        # Operator summaries
        for operator_instance in self.registry.operators.values():
            metadata = operator_instance.metadata
            registry_data["operators"].append({
                "id": operator_instance.operator_id,
                "name": metadata.name,
                "version": metadata.version,
                "type": metadata.type.value,
                "description": metadata.description,
                "author": metadata.author,
                "tags": metadata.tags
            })
        
        return json.dumps(registry_data, indent=2)

class CustomOperatorFramework:
    """Main framework class for custom operator management."""
    
    def __init__(self):
        self.registry = OperatorRegistry()
        self.validator = OperatorValidator()
        self.doc_generator = OperatorDocumentationGenerator(self.registry)
        self.operator_stats = {
            'total_registered': 0,
            'total_executions': 0,
            'total_errors': 0,
            'active_operators': 0
        }
        self._executor = ThreadPoolExecutor(max_workers=10)
    
    # Core Framework Methods
    def register_operator(self, operator_class: Type[BaseOperator], 
                         operator_id: Optional[str] = None,
                         validate: bool = True) -> str:
        """Register operator with optional validation."""
        if validate:
            validation_result = self.validator.validate_operator(operator_class)
            
            if not validation_result['valid']:
                raise ValueError(f"Operator validation failed: {validation_result['errors']}")
            
            if validation_result['warnings']:
                for warning in validation_result['warnings']:
                    logger.warning(f"Operator validation warning: {warning}")
        
        operator_id = self.registry.register_operator(operator_class, operator_id)
        self.operator_stats['total_registered'] += 1
        
        return operator_id
    
    async def initialize_operator(self, operator_id: str, config: Dict[str, Any]) -> bool:
        """Initialize operator with configuration."""
        operator_instance = self.registry.get_operator(operator_id)
        if not operator_instance:
            raise ValueError(f"Operator {operator_id} not found")
        
        try:
            # Validate config
            if not operator_instance.instance.validate_config(config):
                raise ValueError("Configuration validation failed")
            
            # Initialize operator
            success = await operator_instance.instance.initialize(config)
            
            if success:
                operator_instance.instance._initialized = True
                operator_instance.lifecycle_state = OperatorLifecycle.INITIALIZED
                self.operator_stats['active_operators'] += 1
            else:
                operator_instance.lifecycle_state = OperatorLifecycle.ERROR
                operator_instance.last_error = "Initialization failed"
            
            return success
            
        except Exception as e:
            operator_instance.lifecycle_state = OperatorLifecycle.ERROR
            operator_instance.last_error = str(e)
            operator_instance.error_count += 1
            logger.error(f"Error initializing operator {operator_id}: {str(e)}")
            raise
    
    async def execute_operator(self, operator_id: str, operation: str, 
                              *args, **kwargs) -> Any:
        """Execute operator operation."""
        operator_instance = self.registry.get_operator(operator_id)
        if not operator_instance:
            raise ValueError(f"Operator {operator_id} not found")
        
        if not operator_instance.instance._initialized:
            raise RuntimeError(f"Operator {operator_id} not initialized")
        
        try:
            operator_instance.lifecycle_state = OperatorLifecycle.ACTIVE
            operator_instance.last_used = datetime.now()
            operator_instance.usage_count += 1
            
            # Execute operation
            result = await operator_instance.instance.execute(operation, *args, **kwargs)
            
            # Update stats
            operator_instance.instance.stats['operations_performed'] += 1
            operator_instance.instance.stats['last_operation_time'] = datetime.now()
            self.operator_stats['total_executions'] += 1
            
            return result
            
        except Exception as e:
            operator_instance.lifecycle_state = OperatorLifecycle.ERROR
            operator_instance.last_error = str(e)
            operator_instance.error_count += 1
            
            # Update error stats
            operator_instance.instance.stats['errors_occurred'] += 1
            self.operator_stats['total_errors'] += 1
            
            logger.error(f"Error executing {operation} on {operator_id}: {str(e)}")
            raise
    
    # Development Tools
    def create_operator_template(self, operator_name: str, operator_type: OperatorType,
                               output_path: Optional[Path] = None) -> str:
        """Create operator template for development."""
        template = f'''"""
{operator_name} - Custom TuskLang Operator
Generated by Custom Operator Framework
"""

from aa_python.core.framework.custom_operator_framework import BaseOperator, OperatorMetadata, OperatorType
from typing import Any, Dict


class {operator_name}(BaseOperator):
    """
    {operator_name} operator implementation.
    
    TODO: Add detailed description of what this operator does.
    """
    
    def get_metadata(self) -> OperatorMetadata:
        """Get operator metadata."""
        return OperatorMetadata(
            name="{operator_name}",
            version="1.0.0",
            description="Custom {operator_name.lower()} operator",
            author="Your Name",
            type=OperatorType.{operator_type.name},
            dependencies=[],  # Add required Python packages here
            parameters={{
                # Define configuration parameters here
                # "host": "Connection host (default: localhost)",
                # "port": "Connection port (default: 8080)"
            }},
            tags=["{operator_type.value}"]
        )
    
    async def initialize(self, config: Dict[str, Any]) -> bool:
        """Initialize the operator with configuration."""
        self.config = config
        
        # TODO: Initialize your operator here
        # Example:
        # self.client = YourClient(
        #     host=config.get('host', 'localhost'),
        #     port=config.get('port', 8080)
        # )
        
        return True
    
    async def execute(self, operation: str, *args, **kwargs) -> Any:
        """Execute operator operation."""
        # Route operations to appropriate methods
        if operation == "example_operation":
            return await self.example_operation(*args, **kwargs)
        else:
            raise ValueError(f"Unknown operation: {{operation}}")
    
    # TODO: Implement your custom operations here
    async def example_operation(self, data: Any) -> Dict[str, Any]:
        """
        Example operation that processes data.
        
        Args:
            data: Input data to process
            
        Returns:
            Dict containing processed results
        """
        # TODO: Implement your operation logic
        return {{"result": f"Processed: {{data}}"}}
    
    async def cleanup(self) -> bool:
        """Cleanup resources."""
        # TODO: Close connections, cleanup resources
        return True
    
    def validate_config(self, config: Dict[str, Any]) -> bool:
        """Validate configuration parameters."""
        # TODO: Add configuration validation logic
        return True


# Auto-registration (optional)
# if __name__ == "__main__":
#     from aa_python.core.framework.custom_operator_framework import CustomOperatorFramework
#     framework = CustomOperatorFramework()
#     framework.register_operator({operator_name})
'''
        
        if output_path:
            output_file = output_path / f"{operator_name.lower()}_operator.py"
            output_file.write_text(template)
            return str(output_file)
        
        return template
    
    def validate_operator_class(self, operator_class: Type[BaseOperator]) -> Dict[str, Any]:
        """Validate operator class."""
        return self.validator.validate_operator(operator_class)
    
    def generate_docs(self, operator_id: Optional[str] = None, 
                     format: str = "markdown") -> str:
        """Generate documentation."""
        if operator_id:
            return self.doc_generator.generate_operator_docs(operator_id, format)
        else:
            return self.doc_generator.generate_registry_docs(format)
    
    # Registry Management
    def list_operators(self, op_type: Optional[OperatorType] = None) -> List[Dict[str, Any]]:
        """List operators with summary information."""
        operators = self.registry.list_operators(op_type)
        
        return [
            {
                'id': op.operator_id,
                'name': op.metadata.name,
                'version': op.metadata.version,
                'type': op.metadata.type.value,
                'description': op.metadata.description,
                'state': op.lifecycle_state.value,
                'usage_count': op.usage_count,
                'error_count': op.error_count,
                'last_used': op.last_used.isoformat() if op.last_used else None
            }
            for op in operators
        ]
    
    def get_operator_info(self, operator_id: str) -> Optional[Dict[str, Any]]:
        """Get detailed operator information."""
        operator_instance = self.registry.get_operator(operator_id)
        if not operator_instance:
            return None
        
        metadata = operator_instance.metadata
        
        return {
            'id': operator_instance.operator_id,
            'metadata': {
                'name': metadata.name,
                'version': metadata.version,
                'description': metadata.description,
                'author': metadata.author,
                'type': metadata.type.value,
                'dependencies': metadata.dependencies,
                'parameters': metadata.parameters,
                'tags': metadata.tags,
                'license': metadata.license
            },
            'lifecycle': {
                'state': operator_instance.lifecycle_state.value,
                'created_at': operator_instance.created_at.isoformat(),
                'last_used': operator_instance.last_used.isoformat() if operator_instance.last_used else None,
                'usage_count': operator_instance.usage_count,
                'error_count': operator_instance.error_count,
                'last_error': operator_instance.last_error
            },
            'statistics': operator_instance.instance.get_statistics(),
            'operations': operator_instance.instance.get_available_operations()
        }
    
    def discover_plugins(self, plugin_paths: List[Union[str, Path]]) -> List[str]:
        """Discover and load operator plugins."""
        discovered = []
        
        for path in plugin_paths:
            self.registry.add_plugin_path(path)
        
        discovered = self.registry.discover_operators()
        self.operator_stats['total_registered'] += len(discovered)
        
        return discovered
    
    def get_framework_statistics(self) -> Dict[str, Any]:
        """Get comprehensive framework statistics."""
        return {
            'framework_stats': self.operator_stats.copy(),
            'registry_stats': {
                'total_operators': len(self.registry.operators),
                'operators_by_type': {
                    op_type.value: len(operators)
                    for op_type, operators in self.registry.operator_types.items()
                    if operators
                }
            },
            'lifecycle_distribution': {
                state.value: sum(1 for op in self.registry.operators.values() 
                               if op.lifecycle_state == state)
                for state in OperatorLifecycle
            }
        }
    
    async def shutdown(self):
        """Shutdown framework and cleanup."""
        # Cleanup all operators
        for operator_id in list(self.registry.operators.keys()):
            operator_instance = self.registry.get_operator(operator_id)
            if operator_instance and operator_instance.instance._initialized:
                try:
                    await operator_instance.instance.cleanup()
                except Exception as e:
                    logger.warning(f"Error cleaning up {operator_id}: {str(e)}")
            
            self.registry.unregister_operator(operator_id)
        
        # Shutdown executor
        self._executor.shutdown(wait=True)
        
        logger.info("Custom Operator Framework shutdown complete")

# Export the framework components
__all__ = [
    'CustomOperatorFramework', 'BaseOperator', 'OperatorMetadata', 'OperatorType',
    'OperatorLifecycle', 'OperatorMethod', 'OperatorInstance', 'OperatorValidator',
    'OperatorRegistry', 'OperatorDocumentationGenerator'
] 