"""
Advanced Database Integration Systems
MongoDB, PostgreSQL, MySQL, and Redis support for TuskLang.
"""

import asyncio
import json
import logging
import time
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime
from typing import Any, Dict, List, Optional, Union

# MongoDB Support
try:
    import motor.motor_asyncio
    from pymongo import MongoClient
    from pymongo.errors import PyMongoError
    MONGODB_AVAILABLE = True
except ImportError:
    MONGODB_AVAILABLE = False
    print("pymongo/motor not available. @mongodb operator will be limited.")

# PostgreSQL Support
try:
    import asyncpg
    import psycopg2
    POSTGRESQL_AVAILABLE = True
except ImportError:
    POSTGRESQL_AVAILABLE = False
    print("asyncpg/psycopg2 not available. @postgresql operator will be limited.")

# MySQL Support
try:
    import aiomysql
    import pymysql
    MYSQL_AVAILABLE = True
except ImportError:
    MYSQL_AVAILABLE = False
    print("aiomysql/pymysql not available. @mysql operator will be limited.")

# Redis Support
try:
    import aioredis
    import redis
    REDIS_AVAILABLE = True
except ImportError:
    REDIS_AVAILABLE = False
    print("aioredis/redis not available. @redis operator will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class DatabaseConfig:
    """Database connection configuration."""
    host: str = "localhost"
    port: int = 0
    database: str = ""
    username: str = ""
    password: str = ""
    ssl: bool = False
    options: Dict[str, Any] = field(default_factory=dict)

@dataclass
class QueryResult:
    """Database query result."""
    success: bool
    data: Any = None
    affected_rows: int = 0
    execution_time: float = 0.0
    error: Optional[str] = None

class MongoDBOperator:
    """@mongodb operator implementation."""
    
    def __init__(self):
        self.client = None
        self.database = None
        self.collections = {}
    
    async def connect(self, config: DatabaseConfig) -> bool:
        """Connect to MongoDB."""
        if not MONGODB_AVAILABLE:
            logger.error("MongoDB library not available")
            return False
        
        try:
            # Build connection string
            if config.username and config.password:
                auth_str = f"{config.username}:{config.password}@"
            else:
                auth_str = ""
            
            port = config.port or 27017
            connection_string = f"mongodb://{auth_str}{config.host}:{port}/{config.database}"
            
            self.client = motor.motor_asyncio.AsyncIOMotorClient(connection_string)
            
            # Test connection
            await self.client.admin.command('ping')
            
            self.database = self.client[config.database]
            logger.info(f"Connected to MongoDB: {config.host}:{port}")
            return True
        except Exception as e:
            logger.error(f"Error connecting to MongoDB: {str(e)}")
            return False
    
    async def insert_one(self, collection: str, document: Dict[str, Any]) -> QueryResult:
        """Insert single document."""
        if not self.database:
            return QueryResult(success=False, error="Not connected to MongoDB")
        
        try:
            start_time = time.time()
            
            coll = self.database[collection]
            result = await coll.insert_one(document)
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data={"inserted_id": str(result.inserted_id)},
                affected_rows=1,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def insert_many(self, collection: str, documents: List[Dict[str, Any]]) -> QueryResult:
        """Insert multiple documents."""
        if not self.database:
            return QueryResult(success=False, error="Not connected to MongoDB")
        
        try:
            start_time = time.time()
            
            coll = self.database[collection]
            result = await coll.insert_many(documents)
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data={"inserted_ids": [str(id_) for id_ in result.inserted_ids]},
                affected_rows=len(result.inserted_ids),
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def find_one(self, collection: str, filter_dict: Dict[str, Any] = None) -> QueryResult:
        """Find single document."""
        if not self.database:
            return QueryResult(success=False, error="Not connected to MongoDB")
        
        try:
            start_time = time.time()
            
            coll = self.database[collection]
            document = await coll.find_one(filter_dict or {})
            
            execution_time = time.time() - start_time
            
            if document:
                # Convert ObjectId to string
                if '_id' in document:
                    document['_id'] = str(document['_id'])
            
            return QueryResult(
                success=True,
                data=document,
                affected_rows=1 if document else 0,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def find_many(self, collection: str, filter_dict: Dict[str, Any] = None,
                       limit: int = None) -> QueryResult:
        """Find multiple documents."""
        if not self.database:
            return QueryResult(success=False, error="Not connected to MongoDB")
        
        try:
            start_time = time.time()
            
            coll = self.database[collection]
            cursor = coll.find(filter_dict or {})
            
            if limit:
                cursor = cursor.limit(limit)
            
            documents = []
            async for doc in cursor:
                if '_id' in doc:
                    doc['_id'] = str(doc['_id'])
                documents.append(doc)
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data=documents,
                affected_rows=len(documents),
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def update_one(self, collection: str, filter_dict: Dict[str, Any],
                        update_dict: Dict[str, Any]) -> QueryResult:
        """Update single document."""
        if not self.database:
            return QueryResult(success=False, error="Not connected to MongoDB")
        
        try:
            start_time = time.time()
            
            coll = self.database[collection]
            result = await coll.update_one(filter_dict, {"$set": update_dict})
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data={"matched": result.matched_count, "modified": result.modified_count},
                affected_rows=result.modified_count,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def delete_one(self, collection: str, filter_dict: Dict[str, Any]) -> QueryResult:
        """Delete single document."""
        if not self.database:
            return QueryResult(success=False, error="Not connected to MongoDB")
        
        try:
            start_time = time.time()
            
            coll = self.database[collection]
            result = await coll.delete_one(filter_dict)
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data={"deleted_count": result.deleted_count},
                affected_rows=result.deleted_count,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def create_index(self, collection: str, keys: Dict[str, int]) -> QueryResult:
        """Create index on collection."""
        if not self.database:
            return QueryResult(success=False, error="Not connected to MongoDB")
        
        try:
            start_time = time.time()
            
            coll = self.database[collection]
            index_name = await coll.create_index(list(keys.items()))
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data={"index_name": index_name},
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))

class PostgreSQLOperator:
    """@postgresql operator implementation."""
    
    def __init__(self):
        self.pool = None
        
    async def connect(self, config: DatabaseConfig) -> bool:
        """Connect to PostgreSQL."""
        if not POSTGRESQL_AVAILABLE:
            logger.error("PostgreSQL library not available")
            return False
        
        try:
            port = config.port or 5432
            
            connection_string = f"postgresql://{config.username}:{config.password}@{config.host}:{port}/{config.database}"
            
            self.pool = await asyncpg.create_pool(
                connection_string,
                min_size=1,
                max_size=10,
                **config.options
            )
            
            logger.info(f"Connected to PostgreSQL: {config.host}:{port}")
            return True
        except Exception as e:
            logger.error(f"Error connecting to PostgreSQL: {str(e)}")
            return False
    
    async def execute_query(self, query: str, params: List[Any] = None) -> QueryResult:
        """Execute SQL query."""
        if not self.pool:
            return QueryResult(success=False, error="Not connected to PostgreSQL")
        
        try:
            start_time = time.time()
            
            async with self.pool.acquire() as connection:
                result = await connection.execute(query, *(params or []))
                
                execution_time = time.time() - start_time
                
                # Extract affected rows from result string
                affected_rows = 0
                if isinstance(result, str):
                    parts = result.split()
                    if len(parts) >= 2 and parts[-1].isdigit():
                        affected_rows = int(parts[-1])
                
                return QueryResult(
                    success=True,
                    data=result,
                    affected_rows=affected_rows,
                    execution_time=execution_time
                )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def fetch_one(self, query: str, params: List[Any] = None) -> QueryResult:
        """Fetch single row."""
        if not self.pool:
            return QueryResult(success=False, error="Not connected to PostgreSQL")
        
        try:
            start_time = time.time()
            
            async with self.pool.acquire() as connection:
                row = await connection.fetchrow(query, *(params or []))
                
                execution_time = time.time() - start_time
                
                data = dict(row) if row else None
                
                return QueryResult(
                    success=True,
                    data=data,
                    affected_rows=1 if data else 0,
                    execution_time=execution_time
                )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def fetch_all(self, query: str, params: List[Any] = None) -> QueryResult:
        """Fetch all rows."""
        if not self.pool:
            return QueryResult(success=False, error="Not connected to PostgreSQL")
        
        try:
            start_time = time.time()
            
            async with self.pool.acquire() as connection:
                rows = await connection.fetch(query, *(params or []))
                
                execution_time = time.time() - start_time
                
                data = [dict(row) for row in rows]
                
                return QueryResult(
                    success=True,
                    data=data,
                    affected_rows=len(data),
                    execution_time=execution_time
                )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def execute_transaction(self, queries: List[Tuple[str, List[Any]]]) -> QueryResult:
        """Execute multiple queries in transaction."""
        if not self.pool:
            return QueryResult(success=False, error="Not connected to PostgreSQL")
        
        try:
            start_time = time.time()
            total_affected = 0
            
            async with self.pool.acquire() as connection:
                async with connection.transaction():
                    for query, params in queries:
                        result = await connection.execute(query, *(params or []))
                        
                        # Extract affected rows
                        if isinstance(result, str):
                            parts = result.split()
                            if len(parts) >= 2 and parts[-1].isdigit():
                                total_affected += int(parts[-1])
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data=f"Transaction completed: {len(queries)} queries",
                affected_rows=total_affected,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))

class MySQLOperator:
    """@mysql operator implementation."""
    
    def __init__(self):
        self.pool = None
    
    async def connect(self, config: DatabaseConfig) -> bool:
        """Connect to MySQL."""
        if not MYSQL_AVAILABLE:
            logger.error("MySQL library not available")
            return False
        
        try:
            port = config.port or 3306
            
            self.pool = await aiomysql.create_pool(
                host=config.host,
                port=port,
                user=config.username,
                password=config.password,
                db=config.database,
                minsize=1,
                maxsize=10,
                **config.options
            )
            
            logger.info(f"Connected to MySQL: {config.host}:{port}")
            return True
        except Exception as e:
            logger.error(f"Error connecting to MySQL: {str(e)}")
            return False
    
    async def execute_query(self, query: str, params: List[Any] = None) -> QueryResult:
        """Execute SQL query."""
        if not self.pool:
            return QueryResult(success=False, error="Not connected to MySQL")
        
        try:
            start_time = time.time()
            
            async with self.pool.acquire() as connection:
                async with connection.cursor() as cursor:
                    affected = await cursor.execute(query, params)
                    await connection.commit()
                    
                    execution_time = time.time() - start_time
                    
                    return QueryResult(
                        success=True,
                        data=f"Query executed successfully",
                        affected_rows=affected,
                        execution_time=execution_time
                    )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def fetch_one(self, query: str, params: List[Any] = None) -> QueryResult:
        """Fetch single row."""
        if not self.pool:
            return QueryResult(success=False, error="Not connected to MySQL")
        
        try:
            start_time = time.time()
            
            async with self.pool.acquire() as connection:
                async with connection.cursor(aiomysql.DictCursor) as cursor:
                    await cursor.execute(query, params)
                    row = await cursor.fetchone()
                    
                    execution_time = time.time() - start_time
                    
                    return QueryResult(
                        success=True,
                        data=row,
                        affected_rows=1 if row else 0,
                        execution_time=execution_time
                    )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def fetch_all(self, query: str, params: List[Any] = None) -> QueryResult:
        """Fetch all rows."""
        if not self.pool:
            return QueryResult(success=False, error="Not connected to MySQL")
        
        try:
            start_time = time.time()
            
            async with self.pool.acquire() as connection:
                async with connection.cursor(aiomysql.DictCursor) as cursor:
                    await cursor.execute(query, params)
                    rows = await cursor.fetchall()
                    
                    execution_time = time.time() - start_time
                    
                    return QueryResult(
                        success=True,
                        data=rows,
                        affected_rows=len(rows),
                        execution_time=execution_time
                    )
        except Exception as e:
            return QueryResult(success=False, error=str(e))

class RedisOperator:
    """@redis operator implementation."""
    
    def __init__(self):
        self.client = None
    
    async def connect(self, config: DatabaseConfig) -> bool:
        """Connect to Redis."""
        if not REDIS_AVAILABLE:
            logger.error("Redis library not available")
            return False
        
        try:
            port = config.port or 6379
            
            self.client = await aioredis.create_redis_pool(
                f'redis://{config.host}:{port}',
                password=config.password if config.password else None,
                db=int(config.database) if config.database else 0,
                **config.options
            )
            
            # Test connection
            await self.client.ping()
            
            logger.info(f"Connected to Redis: {config.host}:{port}")
            return True
        except Exception as e:
            logger.error(f"Error connecting to Redis: {str(e)}")
            return False
    
    async def set_key(self, key: str, value: Any, expire: int = None) -> QueryResult:
        """Set key-value pair."""
        if not self.client:
            return QueryResult(success=False, error="Not connected to Redis")
        
        try:
            start_time = time.time()
            
            # Serialize complex values as JSON
            if isinstance(value, (dict, list)):
                value = json.dumps(value)
            
            result = await self.client.set(key, value, expire=expire)
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data=result,
                affected_rows=1,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def get_key(self, key: str) -> QueryResult:
        """Get value by key."""
        if not self.client:
            return QueryResult(success=False, error="Not connected to Redis")
        
        try:
            start_time = time.time()
            
            value = await self.client.get(key)
            
            if value:
                value = value.decode('utf-8')
                
                # Try to deserialize JSON
                try:
                    value = json.loads(value)
                except json.JSONDecodeError:
                    pass  # Keep as string
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data=value,
                affected_rows=1 if value is not None else 0,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def delete_key(self, key: str) -> QueryResult:
        """Delete key."""
        if not self.client:
            return QueryResult(success=False, error="Not connected to Redis")
        
        try:
            start_time = time.time()
            
            deleted = await self.client.delete(key)
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data={"deleted": deleted},
                affected_rows=deleted,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def hash_set(self, key: str, field: str, value: Any) -> QueryResult:
        """Set hash field."""
        if not self.client:
            return QueryResult(success=False, error="Not connected to Redis")
        
        try:
            start_time = time.time()
            
            if isinstance(value, (dict, list)):
                value = json.dumps(value)
            
            result = await self.client.hset(key, field, value)
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data=result,
                affected_rows=1,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))
    
    async def hash_get(self, key: str, field: str = None) -> QueryResult:
        """Get hash field(s)."""
        if not self.client:
            return QueryResult(success=False, error="Not connected to Redis")
        
        try:
            start_time = time.time()
            
            if field:
                value = await self.client.hget(key, field)
                if value:
                    value = value.decode('utf-8')
                    try:
                        value = json.loads(value)
                    except json.JSONDecodeError:
                        pass
            else:
                hash_data = await self.client.hgetall(key)
                value = {}
                for k, v in hash_data.items():
                    k_str = k.decode('utf-8')
                    v_str = v.decode('utf-8')
                    try:
                        v_str = json.loads(v_str)
                    except json.JSONDecodeError:
                        pass
                    value[k_str] = v_str
            
            execution_time = time.time() - start_time
            
            return QueryResult(
                success=True,
                data=value,
                affected_rows=1 if value else 0,
                execution_time=execution_time
            )
        except Exception as e:
            return QueryResult(success=False, error=str(e))

class DatabaseIntegrationSystems:
    """
    Advanced Database Integration Systems for TuskLang.
    Implements @mongodb, @postgresql, @mysql, and @redis operators.
    """
    
    def __init__(self):
        self.mongodb = MongoDBOperator()
        self.postgresql = PostgreSQLOperator()
        self.mysql = MySQLOperator()
        self.redis = RedisOperator()
        
        self.stats = {
            'mongodb_queries': 0,
            'postgresql_queries': 0,
            'mysql_queries': 0,
            'redis_operations': 0,
            'total_execution_time': 0.0,
            'active_connections': 0
        }
    
    # MongoDB operator methods
    async def mongodb_connect(self, host: str = "localhost", port: int = 27017, 
                            database: str = "test", username: str = "", 
                            password: str = "") -> bool:
        """Connect to MongoDB (@mongodb operator - connect)."""
        config = DatabaseConfig(
            host=host, port=port, database=database, 
            username=username, password=password
        )
        success = await self.mongodb.connect(config)
        if success:
            self.stats['active_connections'] += 1
        return success
    
    async def mongodb_insert(self, collection: str, document: Dict[str, Any]) -> QueryResult:
        """Insert document (@mongodb operator - insert)."""
        result = await self.mongodb.insert_one(collection, document)
        if result.success:
            self.stats['mongodb_queries'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    async def mongodb_find(self, collection: str, filter_dict: Dict[str, Any] = None) -> QueryResult:
        """Find documents (@mongodb operator - find)."""
        result = await self.mongodb.find_many(collection, filter_dict)
        if result.success:
            self.stats['mongodb_queries'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    # PostgreSQL operator methods
    async def postgresql_connect(self, host: str = "localhost", port: int = 5432,
                               database: str = "postgres", username: str = "postgres",
                               password: str = "") -> bool:
        """Connect to PostgreSQL (@postgresql operator - connect)."""
        config = DatabaseConfig(
            host=host, port=port, database=database,
            username=username, password=password
        )
        success = await self.postgresql.connect(config)
        if success:
            self.stats['active_connections'] += 1
        return success
    
    async def postgresql_query(self, query: str, params: List[Any] = None) -> QueryResult:
        """Execute PostgreSQL query (@postgresql operator - query)."""
        result = await self.postgresql.execute_query(query, params)
        if result.success:
            self.stats['postgresql_queries'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    async def postgresql_select(self, query: str, params: List[Any] = None) -> QueryResult:
        """Execute PostgreSQL SELECT (@postgresql operator - select)."""
        result = await self.postgresql.fetch_all(query, params)
        if result.success:
            self.stats['postgresql_queries'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    # MySQL operator methods
    async def mysql_connect(self, host: str = "localhost", port: int = 3306,
                          database: str = "mysql", username: str = "root",
                          password: str = "") -> bool:
        """Connect to MySQL (@mysql operator - connect)."""
        config = DatabaseConfig(
            host=host, port=port, database=database,
            username=username, password=password
        )
        success = await self.mysql.connect(config)
        if success:
            self.stats['active_connections'] += 1
        return success
    
    async def mysql_query(self, query: str, params: List[Any] = None) -> QueryResult:
        """Execute MySQL query (@mysql operator - query)."""
        result = await self.mysql.execute_query(query, params)
        if result.success:
            self.stats['mysql_queries'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    async def mysql_select(self, query: str, params: List[Any] = None) -> QueryResult:
        """Execute MySQL SELECT (@mysql operator - select)."""
        result = await self.mysql.fetch_all(query, params)
        if result.success:
            self.stats['mysql_queries'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    # Redis operator methods
    async def redis_connect(self, host: str = "localhost", port: int = 6379,
                          database: str = "0", password: str = "") -> bool:
        """Connect to Redis (@redis operator - connect)."""
        config = DatabaseConfig(
            host=host, port=port, database=database, password=password
        )
        success = await self.redis.connect(config)
        if success:
            self.stats['active_connections'] += 1
        return success
    
    async def redis_set(self, key: str, value: Any, expire: int = None) -> QueryResult:
        """Set Redis key (@redis operator - set)."""
        result = await self.redis.set_key(key, value, expire)
        if result.success:
            self.stats['redis_operations'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    async def redis_get(self, key: str) -> QueryResult:
        """Get Redis key (@redis operator - get)."""
        result = await self.redis.get_key(key)
        if result.success:
            self.stats['redis_operations'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    async def redis_hset(self, key: str, field: str, value: Any) -> QueryResult:
        """Set Redis hash field (@redis operator - hset)."""
        result = await self.redis.hash_set(key, field, value)
        if result.success:
            self.stats['redis_operations'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    async def redis_hget(self, key: str, field: str = None) -> QueryResult:
        """Get Redis hash field (@redis operator - hget)."""
        result = await self.redis.hash_get(key, field)
        if result.success:
            self.stats['redis_operations'] += 1
            self.stats['total_execution_time'] += result.execution_time
        return result
    
    # Utility methods
    def get_stats(self) -> Dict[str, Any]:
        """Get database statistics."""
        stats = self.stats.copy()
        total_queries = sum([
            stats['mongodb_queries'],
            stats['postgresql_queries'], 
            stats['mysql_queries'],
            stats['redis_operations']
        ])
        
        if total_queries > 0:
            stats['average_execution_time'] = stats['total_execution_time'] / total_queries
        else:
            stats['average_execution_time'] = 0.0
        
        return stats
    
    def get_connection_status(self) -> Dict[str, bool]:
        """Get connection status for all databases."""
        return {
            'mongodb_connected': self.mongodb.database is not None,
            'postgresql_connected': self.postgresql.pool is not None,
            'mysql_connected': self.mysql.pool is not None,
            'redis_connected': self.redis.client is not None
        }

# Example usage and testing
async def main():
    """Example usage of Database Integration Systems."""
    print("=== Database Integration Systems Demo ===")
    
    # Initialize the system
    db_systems = DatabaseIntegrationSystems()
    
    # Test 1: MongoDB
    print("\n1. Testing MongoDB (@mongodb operator)...")
    
    if MONGODB_AVAILABLE:
        print("MongoDB test requires running MongoDB server - skipping connection")
        print("MongoDB operators available: connect, insert, find, update, delete")
    else:
        print("MongoDB library not available")
    
    # Test 2: PostgreSQL
    print("\n2. Testing PostgreSQL (@postgresql operator)...")
    
    if POSTGRESQL_AVAILABLE:
        print("PostgreSQL test requires running PostgreSQL server - skipping connection")
        print("PostgreSQL operators available: connect, query, select, transaction")
    else:
        print("PostgreSQL library not available")
    
    # Test 3: MySQL
    print("\n3. Testing MySQL (@mysql operator)...")
    
    if MYSQL_AVAILABLE:
        print("MySQL test requires running MySQL server - skipping connection")
        print("MySQL operators available: connect, query, select")
    else:
        print("MySQL library not available")
    
    # Test 4: Redis
    print("\n4. Testing Redis (@redis operator)...")
    
    if REDIS_AVAILABLE:
        try:
            # Try to connect to local Redis
            connected = await db_systems.redis_connect()
            if connected:
                print("✓ Connected to Redis")
                
                # Test basic operations
                set_result = await db_systems.redis_set("test_key", {"message": "Hello Redis!"})
                print(f"Set operation: {set_result.success}")
                
                get_result = await db_systems.redis_get("test_key")
                print(f"Get operation: {get_result.data}")
                
                # Test hash operations
                hset_result = await db_systems.redis_hset("test_hash", "field1", "value1")
                print(f"Hash set operation: {hset_result.success}")
                
                hget_result = await db_systems.redis_hget("test_hash", "field1")
                print(f"Hash get operation: {hget_result.data}")
            else:
                print("Redis connection failed - server may not be running")
        except Exception as e:
            print(f"Redis test failed: {str(e)}")
    else:
        print("Redis library not available")
    
    # Test 5: Statistics
    print("\n5. Database statistics:")
    stats = db_systems.get_stats()
    for key, value in stats.items():
        print(f"  {key}: {value}")
    
    # Test 6: Connection status
    print("\n6. Connection status:")
    status = db_systems.get_connection_status()
    for db_name, connected in status.items():
        print(f"  {db_name}: {'✓' if connected else '✗'}")
    
    # Test 7: Available libraries
    print("\n7. Available database support:")
    print(f"  MongoDB: {MONGODB_AVAILABLE}")
    print(f"  PostgreSQL: {POSTGRESQL_AVAILABLE}")
    print(f"  MySQL: {MYSQL_AVAILABLE}")
    print(f"  Redis: {REDIS_AVAILABLE}")
    
    print("\n=== Database Integration Systems Demo Complete ===")

if __name__ == "__main__":
    asyncio.run(main()) 