"""
Enterprise Security & Access Control Systems
RBAC, OAuth2/SAML, MFA, and Audit Logging for TuskLang.
"""

import asyncio
import hashlib
import hmac
import json
import logging
import secrets
import time
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from enum import Enum
from typing import Any, Dict, List, Optional

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class PermissionType(Enum):
    READ = "read"
    WRITE = "write"
    DELETE = "delete"
    EXECUTE = "execute"
    ADMIN = "admin"

@dataclass
class Permission:
    resource: str
    action: PermissionType
    conditions: Dict[str, Any] = field(default_factory=dict)

@dataclass
class Role:
    name: str
    permissions: List[Permission] = field(default_factory=list)
    description: str = ""
    created_at: datetime = field(default_factory=datetime.now)

@dataclass
class User:
    id: str
    username: str
    email: str
    roles: List[str] = field(default_factory=list)
    attributes: Dict[str, Any] = field(default_factory=dict)
    created_at: datetime = field(default_factory=datetime.now)
    last_login: Optional[datetime] = None
    mfa_enabled: bool = False
    mfa_secret: Optional[str] = None

@dataclass
class AuditLog:
    user_id: str
    action: str
    resource: str
    result: str  # SUCCESS, FAILURE, DENIED
    timestamp: datetime = field(default_factory=datetime.now)
    ip_address: Optional[str] = None
    user_agent: Optional[str] = None
    details: Dict[str, Any] = field(default_factory=dict)

class RBACSystem:
    """Role-Based Access Control implementation."""
    
    def __init__(self):
        self.users = {}
        self.roles = {}
        self.permissions = {}
        self.audit_logs = []
        
    def create_role(self, name: str, description: str = "") -> Role:
        """Create a new role."""
        role = Role(name=name, description=description)
        self.roles[name] = role
        logger.info(f"Created role: {name}")
        return role
    
    def add_permission_to_role(self, role_name: str, permission: Permission) -> bool:
        """Add permission to role."""
        if role_name in self.roles:
            self.roles[role_name].permissions.append(permission)
            logger.info(f"Added permission {permission.action.value} on {permission.resource} to role {role_name}")
            return True
        return False
    
    def create_user(self, user_id: str, username: str, email: str) -> User:
        """Create a new user."""
        user = User(id=user_id, username=username, email=email)
        self.users[user_id] = user
        logger.info(f"Created user: {username} ({user_id})")
        return user
    
    def assign_role_to_user(self, user_id: str, role_name: str) -> bool:
        """Assign role to user."""
        if user_id in self.users and role_name in self.roles:
            if role_name not in self.users[user_id].roles:
                self.users[user_id].roles.append(role_name)
                logger.info(f"Assigned role {role_name} to user {user_id}")
                return True
        return False
    
    def check_permission(self, user_id: str, resource: str, action: PermissionType, 
                        context: Dict[str, Any] = None) -> bool:
        """Check if user has permission for action on resource."""
        if user_id not in self.users:
            return False
        
        user = self.users[user_id]
        context = context or {}
        
        # Check all user's roles
        for role_name in user.roles:
            if role_name in self.roles:
                role = self.roles[role_name]
                
                for permission in role.permissions:
                    if (permission.resource == resource or permission.resource == "*") and \
                       (permission.action == action or permission.action == PermissionType.ADMIN):
                        
                        # Check conditions
                        if self._evaluate_conditions(permission.conditions, context, user):
                            return True
        
        return False
    
    def _evaluate_conditions(self, conditions: Dict[str, Any], context: Dict[str, Any], 
                           user: User) -> bool:
        """Evaluate permission conditions."""
        for key, expected_value in conditions.items():
            if key == "time_range":
                # Time-based access control
                now = datetime.now().time()
                start = datetime.strptime(expected_value["start"], "%H:%M").time()
                end = datetime.strptime(expected_value["end"], "%H:%M").time()
                if not (start <= now <= end):
                    return False
            
            elif key == "ip_whitelist":
                client_ip = context.get("ip_address")
                if client_ip and client_ip not in expected_value:
                    return False
            
            elif key == "user_attribute":
                attr_name, attr_value = expected_value["name"], expected_value["value"]
                if user.attributes.get(attr_name) != attr_value:
                    return False
        
        return True
    
    def get_user_permissions(self, user_id: str) -> List[Permission]:
        """Get all permissions for a user."""
        if user_id not in self.users:
            return []
        
        user = self.users[user_id]
        permissions = []
        
        for role_name in user.roles:
            if role_name in self.roles:
                permissions.extend(self.roles[role_name].permissions)
        
        return permissions

class MFASystem:
    """Multi-Factor Authentication system."""
    
    def __init__(self):
        self.user_secrets = {}
        self.backup_codes = {}
        
    def enable_mfa(self, user_id: str) -> str:
        """Enable MFA for user and return secret."""
        secret = secrets.token_urlsafe(32)
        self.user_secrets[user_id] = secret
        
        # Generate backup codes
        backup_codes = [secrets.token_hex(4).upper() for _ in range(10)]
        self.backup_codes[user_id] = backup_codes
        
        logger.info(f"Enabled MFA for user: {user_id}")
        return secret
    
    def verify_totp(self, user_id: str, token: str) -> bool:
        """Verify TOTP token."""
        if user_id not in self.user_secrets:
            return False
        
        # Simplified TOTP verification (in production, use proper TOTP library)
        secret = self.user_secrets[user_id]
        current_time = int(time.time() // 30)  # 30-second time step
        
        # Allow some time drift
        for time_step in [current_time - 1, current_time, current_time + 1]:
            expected_token = self._generate_totp(secret, time_step)
            if token == expected_token:
                return True
        
        return False
    
    def verify_backup_code(self, user_id: str, code: str) -> bool:
        """Verify backup code."""
        if user_id not in self.backup_codes:
            return False
        
        if code.upper() in self.backup_codes[user_id]:
            # Remove used backup code
            self.backup_codes[user_id].remove(code.upper())
            logger.info(f"Used backup code for user: {user_id}")
            return True
        
        return False
    
    def _generate_totp(self, secret: str, time_step: int) -> str:
        """Generate TOTP token (simplified implementation)."""
        # This is a simplified version - use proper TOTP library in production
        key = secret.encode()
        time_bytes = time_step.to_bytes(8, byteorder='big')
        
        # HMAC-SHA1
        hmac_digest = hmac.new(key, time_bytes, hashlib.sha1).digest()
        
        # Dynamic truncation
        offset = hmac_digest[19] & 0xf
        code = (int.from_bytes(hmac_digest[offset:offset + 4], byteorder='big') & 0x7fffffff) % 1000000
        
        return f"{code:06d}"
    
    def get_backup_codes(self, user_id: str) -> List[str]:
        """Get remaining backup codes."""
        return self.backup_codes.get(user_id, []).copy()

class OAuth2System:
    """OAuth2/SAML authentication system."""
    
    def __init__(self):
        self.clients = {}
        self.tokens = {}
        self.authorization_codes = {}
        
    def register_client(self, client_id: str, client_secret: str, 
                       redirect_uris: List[str]) -> bool:
        """Register OAuth2 client."""
        self.clients[client_id] = {
            'client_secret': client_secret,
            'redirect_uris': redirect_uris,
            'created_at': datetime.now()
        }
        logger.info(f"Registered OAuth2 client: {client_id}")
        return True
    
    def generate_authorization_code(self, client_id: str, user_id: str, 
                                  scope: List[str]) -> str:
        """Generate authorization code."""
        if client_id not in self.clients:
            return ""
        
        auth_code = secrets.token_urlsafe(32)
        self.authorization_codes[auth_code] = {
            'client_id': client_id,
            'user_id': user_id,
            'scope': scope,
            'expires_at': datetime.now() + timedelta(minutes=10),
            'used': False
        }
        
        return auth_code
    
    def exchange_code_for_token(self, client_id: str, client_secret: str, 
                               auth_code: str) -> Optional[Dict[str, str]]:
        """Exchange authorization code for access token."""
        # Verify client
        if client_id not in self.clients or \
           self.clients[client_id]['client_secret'] != client_secret:
            return None
        
        # Verify authorization code
        if auth_code not in self.authorization_codes:
            return None
        
        code_data = self.authorization_codes[auth_code]
        if code_data['used'] or datetime.now() > code_data['expires_at']:
            return None
        
        # Mark code as used
        code_data['used'] = True
        
        # Generate tokens
        access_token = secrets.token_urlsafe(32)
        refresh_token = secrets.token_urlsafe(32)
        
        self.tokens[access_token] = {
            'user_id': code_data['user_id'],
            'client_id': client_id,
            'scope': code_data['scope'],
            'expires_at': datetime.now() + timedelta(hours=1),
            'refresh_token': refresh_token
        }
        
        return {
            'access_token': access_token,
            'token_type': 'Bearer',
            'expires_in': 3600,
            'refresh_token': refresh_token
        }
    
    def verify_token(self, access_token: str) -> Optional[Dict[str, Any]]:
        """Verify access token."""
        if access_token in self.tokens:
            token_data = self.tokens[access_token]
            if datetime.now() < token_data['expires_at']:
                return token_data
        return None
    
    def refresh_token(self, refresh_token: str) -> Optional[Dict[str, str]]:
        """Refresh access token."""
        # Find token by refresh token
        for access_token, token_data in self.tokens.items():
            if token_data.get('refresh_token') == refresh_token:
                # Generate new access token
                new_access_token = secrets.token_urlsafe(32)
                new_refresh_token = secrets.token_urlsafe(32)
                
                # Copy token data with new tokens
                self.tokens[new_access_token] = {
                    **token_data,
                    'expires_at': datetime.now() + timedelta(hours=1),
                    'refresh_token': new_refresh_token
                }
                
                # Remove old token
                del self.tokens[access_token]
                
                return {
                    'access_token': new_access_token,
                    'token_type': 'Bearer',
                    'expires_in': 3600,
                    'refresh_token': new_refresh_token
                }
        
        return None

class AuditSystem:
    """Audit logging and compliance system."""
    
    def __init__(self):
        self.logs = []
        self.log_retention_days = 365
        
    def log_access(self, user_id: str, action: str, resource: str, 
                  result: str, context: Dict[str, Any] = None) -> None:
        """Log access attempt."""
        audit_log = AuditLog(
            user_id=user_id,
            action=action,
            resource=resource,
            result=result,
            ip_address=context.get('ip_address') if context else None,
            user_agent=context.get('user_agent') if context else None,
            details=context or {}
        )
        
        self.logs.append(audit_log)
        logger.info(f"Audit log: {user_id} {action} {resource} -> {result}")
    
    def get_audit_logs(self, user_id: str = None, resource: str = None, 
                      start_date: datetime = None, end_date: datetime = None,
                      limit: int = 100) -> List[AuditLog]:
        """Query audit logs."""
        filtered_logs = self.logs
        
        # Filter by user
        if user_id:
            filtered_logs = [log for log in filtered_logs if log.user_id == user_id]
        
        # Filter by resource
        if resource:
            filtered_logs = [log for log in filtered_logs if log.resource == resource]
        
        # Filter by date range
        if start_date:
            filtered_logs = [log for log in filtered_logs if log.timestamp >= start_date]
        
        if end_date:
            filtered_logs = [log for log in filtered_logs if log.timestamp <= end_date]
        
        # Sort by timestamp (newest first) and limit
        filtered_logs.sort(key=lambda x: x.timestamp, reverse=True)
        return filtered_logs[:limit]
    
    def generate_compliance_report(self, report_type: str = "access_summary") -> Dict[str, Any]:
        """Generate compliance report."""
        if report_type == "access_summary":
            return self._generate_access_summary()
        elif report_type == "failed_attempts":
            return self._generate_failed_attempts_report()
        elif report_type == "user_activity":
            return self._generate_user_activity_report()
        else:
            return {"error": f"Unknown report type: {report_type}"}
    
    def _generate_access_summary(self) -> Dict[str, Any]:
        """Generate access summary report."""
        total_logs = len(self.logs)
        successful_access = len([log for log in self.logs if log.result == "SUCCESS"])
        failed_access = len([log for log in self.logs if log.result == "FAILURE"])
        denied_access = len([log for log in self.logs if log.result == "DENIED"])
        
        return {
            "report_type": "access_summary",
            "generated_at": datetime.now().isoformat(),
            "total_events": total_logs,
            "successful_access": successful_access,
            "failed_access": failed_access,
            "denied_access": denied_access,
            "success_rate": (successful_access / total_logs * 100) if total_logs > 0 else 0
        }
    
    def _generate_failed_attempts_report(self) -> Dict[str, Any]:
        """Generate failed attempts report."""
        failed_logs = [log for log in self.logs if log.result in ["FAILURE", "DENIED"]]
        
        # Group by user
        user_failures = {}
        for log in failed_logs:
            if log.user_id not in user_failures:
                user_failures[log.user_id] = 0
            user_failures[log.user_id] += 1
        
        # Get top failed users
        top_failed_users = sorted(user_failures.items(), key=lambda x: x[1], reverse=True)[:10]
        
        return {
            "report_type": "failed_attempts",
            "generated_at": datetime.now().isoformat(),
            "total_failed_attempts": len(failed_logs),
            "unique_users_with_failures": len(user_failures),
            "top_failed_users": [{"user_id": user, "failure_count": count} 
                               for user, count in top_failed_users]
        }
    
    def _generate_user_activity_report(self) -> Dict[str, Any]:
        """Generate user activity report."""
        # Group by user
        user_activity = {}
        for log in self.logs:
            if log.user_id not in user_activity:
                user_activity[log.user_id] = {
                    'total_actions': 0,
                    'successful_actions': 0,
                    'last_activity': log.timestamp
                }
            
            activity = user_activity[log.user_id]
            activity['total_actions'] += 1
            
            if log.result == "SUCCESS":
                activity['successful_actions'] += 1
            
            if log.timestamp > activity['last_activity']:
                activity['last_activity'] = log.timestamp
        
        return {
            "report_type": "user_activity", 
            "generated_at": datetime.now().isoformat(),
            "total_users": len(user_activity),
            "user_summary": [
                {
                    "user_id": user_id,
                    "total_actions": data['total_actions'],
                    "successful_actions": data['successful_actions'],
                    "last_activity": data['last_activity'].isoformat()
                }
                for user_id, data in list(user_activity.items())[:20]  # Top 20 users
            ]
        }
    
    def cleanup_old_logs(self) -> int:
        """Remove old audit logs based on retention policy."""
        cutoff_date = datetime.now() - timedelta(days=self.log_retention_days)
        old_logs_count = len([log for log in self.logs if log.timestamp < cutoff_date])
        
        self.logs = [log for log in self.logs if log.timestamp >= cutoff_date]
        
        if old_logs_count > 0:
            logger.info(f"Cleaned up {old_logs_count} old audit logs")
        
        return old_logs_count

class EnterpriseSecuritySystems:
    """
    Enterprise Security & Access Control Systems for TuskLang.
    Implements RBAC, MFA, OAuth2, and Audit Logging.
    """
    
    def __init__(self):
        self.rbac = RBACSystem()
        self.mfa = MFASystem()
        self.oauth2 = OAuth2System()
        self.audit = AuditSystem()
        
        self.stats = {
            'total_users': 0,
            'total_roles': 0,
            'mfa_enabled_users': 0,
            'oauth2_clients': 0,
            'audit_logs_count': 0
        }
        
        # Initialize default roles
        self._setup_default_roles()
    
    def _setup_default_roles(self):
        """Setup default system roles."""
        # Admin role
        admin_role = self.rbac.create_role("admin", "System administrator with full access")
        self.rbac.add_permission_to_role("admin", Permission("*", PermissionType.ADMIN))
        
        # User role
        user_role = self.rbac.create_role("user", "Standard user with basic permissions")
        self.rbac.add_permission_to_role("user", Permission("profile", PermissionType.READ))
        self.rbac.add_permission_to_role("user", Permission("profile", PermissionType.WRITE))
        
        # Reader role
        reader_role = self.rbac.create_role("reader", "Read-only access")
        self.rbac.add_permission_to_role("reader", Permission("*", PermissionType.READ))
        
        self.stats['total_roles'] = len(self.rbac.roles)
    
    # RBAC methods
    def create_user(self, user_id: str, username: str, email: str, 
                   roles: List[str] = None) -> User:
        """Create user with roles."""
        user = self.rbac.create_user(user_id, username, email)
        
        # Assign roles
        for role in (roles or ["user"]):
            self.rbac.assign_role_to_user(user_id, role)
        
        self.stats['total_users'] = len(self.rbac.users)
        
        # Audit log
        self.audit.log_access(user_id, "CREATE_USER", "user_management", "SUCCESS")
        
        return user
    
    def check_access(self, user_id: str, resource: str, action: str, 
                    context: Dict[str, Any] = None) -> bool:
        """Check user access to resource."""
        try:
            action_enum = PermissionType(action.lower())
        except ValueError:
            action_enum = PermissionType.READ
        
        has_access = self.rbac.check_permission(user_id, resource, action_enum, context)
        
        # Audit log
        result = "SUCCESS" if has_access else "DENIED"
        self.audit.log_access(user_id, f"{action.upper()}_{resource}", resource, result, context)
        
        return has_access
    
    def assign_role(self, user_id: str, role_name: str, admin_user_id: str) -> bool:
        """Assign role to user (admin operation)."""
        # Check if admin has permission
        if not self.check_access(admin_user_id, "user_management", "admin"):
            return False
        
        success = self.rbac.assign_role_to_user(user_id, role_name)
        
        # Audit log
        result = "SUCCESS" if success else "FAILURE"
        self.audit.log_access(admin_user_id, f"ASSIGN_ROLE_{role_name}", f"user:{user_id}", result)
        
        return success
    
    # MFA methods
    def enable_mfa_for_user(self, user_id: str) -> Dict[str, Any]:
        """Enable MFA for user."""
        if user_id not in self.rbac.users:
            return {"success": False, "error": "User not found"}
        
        secret = self.mfa.enable_mfa(user_id)
        backup_codes = self.mfa.get_backup_codes(user_id)
        
        # Update user record
        self.rbac.users[user_id].mfa_enabled = True
        self.rbac.users[user_id].mfa_secret = secret
        
        self.stats['mfa_enabled_users'] = len([u for u in self.rbac.users.values() if u.mfa_enabled])
        
        # Audit log
        self.audit.log_access(user_id, "ENABLE_MFA", "mfa_settings", "SUCCESS")
        
        return {
            "success": True,
            "secret": secret,
            "backup_codes": backup_codes,
            "qr_code_url": f"otpauth://totp/TuskLang:{user_id}?secret={secret}&issuer=TuskLang"
        }
    
    def verify_mfa_token(self, user_id: str, token: str) -> bool:
        """Verify MFA token."""
        if user_id not in self.rbac.users or not self.rbac.users[user_id].mfa_enabled:
            return False
        
        # Try TOTP first, then backup code
        is_valid = self.mfa.verify_totp(user_id, token) or self.mfa.verify_backup_code(user_id, token)
        
        # Audit log
        result = "SUCCESS" if is_valid else "FAILURE"
        self.audit.log_access(user_id, "VERIFY_MFA", "mfa_verification", result)
        
        return is_valid
    
    # OAuth2 methods
    def register_oauth2_client(self, client_id: str, client_secret: str, 
                             redirect_uris: List[str]) -> bool:
        """Register OAuth2 client."""
        success = self.oauth2.register_client(client_id, client_secret, redirect_uris)
        
        if success:
            self.stats['oauth2_clients'] = len(self.oauth2.clients)
        
        return success
    
    def oauth2_authorize(self, client_id: str, user_id: str, scope: List[str]) -> str:
        """Generate OAuth2 authorization code."""
        auth_code = self.oauth2.generate_authorization_code(client_id, user_id, scope)
        
        # Audit log
        result = "SUCCESS" if auth_code else "FAILURE"
        self.audit.log_access(user_id, "OAUTH2_AUTHORIZE", f"client:{client_id}", result)
        
        return auth_code
    
    def oauth2_token(self, client_id: str, client_secret: str, auth_code: str) -> Optional[Dict[str, str]]:
        """Exchange OAuth2 code for token."""
        tokens = self.oauth2.exchange_code_for_token(client_id, client_secret, auth_code)
        
        # Audit log
        result = "SUCCESS" if tokens else "FAILURE"
        self.audit.log_access("system", "OAUTH2_TOKEN", f"client:{client_id}", result)
        
        return tokens
    
    def verify_oauth2_token(self, access_token: str) -> Optional[Dict[str, Any]]:
        """Verify OAuth2 access token."""
        return self.oauth2.verify_token(access_token)
    
    # Audit methods
    def get_audit_logs(self, **kwargs) -> List[AuditLog]:
        """Get audit logs with filtering."""
        logs = self.audit.get_audit_logs(**kwargs)
        self.stats['audit_logs_count'] = len(self.audit.logs)
        return logs
    
    def generate_compliance_report(self, report_type: str) -> Dict[str, Any]:
        """Generate compliance report."""
        return self.audit.generate_compliance_report(report_type)
    
    # Utility methods
    def get_stats(self) -> Dict[str, Any]:
        """Get security system statistics."""
        self.stats.update({
            'total_users': len(self.rbac.users),
            'total_roles': len(self.rbac.roles),
            'mfa_enabled_users': len([u for u in self.rbac.users.values() if u.mfa_enabled]),
            'oauth2_clients': len(self.oauth2.clients),
            'audit_logs_count': len(self.audit.logs)
        })
        return self.stats.copy()
    
    def security_health_check(self) -> Dict[str, Any]:
        """Perform security health check."""
        total_users = len(self.rbac.users)
        mfa_users = len([u for u in self.rbac.users.values() if u.mfa_enabled])
        mfa_adoption = (mfa_users / total_users * 100) if total_users > 0 else 0
        
        # Check for recent failed attempts
        recent_failures = len([log for log in self.audit.logs 
                             if log.result in ["FAILURE", "DENIED"] and 
                             log.timestamp > datetime.now() - timedelta(hours=24)])
        
        return {
            "overall_status": "healthy",
            "mfa_adoption_rate": mfa_adoption,
            "recent_failed_attempts": recent_failures,
            "active_oauth2_tokens": len(self.oauth2.tokens),
            "recommendations": self._generate_security_recommendations(mfa_adoption, recent_failures)
        }
    
    def _generate_security_recommendations(self, mfa_adoption: float, failed_attempts: int) -> List[str]:
        """Generate security recommendations."""
        recommendations = []
        
        if mfa_adoption < 50:
            recommendations.append("Consider mandating MFA for all users - current adoption is low")
        
        if failed_attempts > 100:
            recommendations.append("High number of failed attempts detected - consider implementing rate limiting")
        
        if len(self.oauth2.clients) > 0 and not any("admin" in client_id for client_id in self.oauth2.clients):
            recommendations.append("Consider implementing client registration approval process")
        
        if not recommendations:
            recommendations.append("Security posture looks good - maintain current practices")
        
        return recommendations

# Example usage
async def main():
    """Example usage of Enterprise Security Systems."""
    print("=== Enterprise Security Systems Demo ===")
    
    security = EnterpriseSecuritySystems()
    
    # Test 1: User management and RBAC
    print("\n1. Testing RBAC system...")
    
    # Create users
    admin_user = security.create_user("admin_001", "admin", "admin@company.com", ["admin"])
    regular_user = security.create_user("user_001", "john_doe", "john@company.com", ["user"])
    
    # Test permissions
    can_admin = security.check_access("admin_001", "user_management", "admin")
    can_user_admin = security.check_access("user_001", "user_management", "admin")
    can_user_read = security.check_access("user_001", "profile", "read")
    
    print(f"Admin can manage users: {can_admin}")
    print(f"User can manage users: {can_user_admin}")
    print(f"User can read profile: {can_user_read}")
    
    # Test 2: MFA
    print("\n2. Testing MFA system...")
    
    mfa_result = security.enable_mfa_for_user("user_001")
    print(f"MFA enabled: {mfa_result['success']}")
    if mfa_result['success']:
        print(f"Backup codes generated: {len(mfa_result['backup_codes'])}")
        
        # Test backup code
        backup_code = mfa_result['backup_codes'][0]
        is_valid = security.verify_mfa_token("user_001", backup_code)
        print(f"Backup code verification: {is_valid}")
    
    # Test 3: OAuth2
    print("\n3. Testing OAuth2 system...")
    
    # Register client
    client_registered = security.register_oauth2_client(
        "web_app_123",
        "client_secret_456",
        ["https://app.company.com/callback"]
    )
    print(f"OAuth2 client registered: {client_registered}")
    
    if client_registered:
        # Authorization flow
        auth_code = security.oauth2_authorize("web_app_123", "user_001", ["read", "write"])
        print(f"Authorization code generated: {bool(auth_code)}")
        
        if auth_code:
            tokens = security.oauth2_token("web_app_123", "client_secret_456", auth_code)
            print(f"Access token received: {bool(tokens)}")
            
            if tokens:
                token_data = security.verify_oauth2_token(tokens['access_token'])
                print(f"Token verification: {bool(token_data)}")
    
    # Test 4: Audit logs
    print("\n4. Testing audit system...")
    
    # Generate some activity
    security.check_access("user_001", "sensitive_data", "read")
    security.check_access("user_001", "admin_panel", "write")
    
    # Get audit logs
    user_logs = security.get_audit_logs(user_id="user_001", limit=5)
    print(f"User audit logs: {len(user_logs)}")
    
    # Generate compliance report
    access_report = security.generate_compliance_report("access_summary")
    print(f"Total access events: {access_report['total_events']}")
    print(f"Success rate: {access_report['success_rate']:.1f}%")
    
    # Test 5: Security health check
    print("\n5. Security health check:")
    health = security.security_health_check()
    print(f"Overall status: {health['overall_status']}")
    print(f"MFA adoption: {health['mfa_adoption_rate']:.1f}%")
    print(f"Recommendations: {len(health['recommendations'])}")
    for rec in health['recommendations']:
        print(f"  - {rec}")
    
    # Test 6: Statistics
    print("\n6. System statistics:")
    stats = security.get_stats()
    for key, value in stats.items():
        print(f"  {key}: {value}")
    
    print("\n=== Enterprise Security Demo Complete ===")

if __name__ == "__main__":
    asyncio.run(main()) 