"""
TuskLang Python SDK - Event Bus System (g9.2)
Scalable event distribution with pub/sub patterns, filtering and routing
"""

import asyncio
import json
import logging
import threading
import time
import uuid
from abc import ABC, abstractmethod
from collections import defaultdict
from dataclasses import dataclass, asdict, field
from datetime import datetime
from enum import Enum
from typing import Dict, List, Optional, Any, Callable, Union, Set
import weakref
import fnmatch
import re

try:
    import redis
    REDIS_AVAILABLE = True
except ImportError:
    REDIS_AVAILABLE = False


class EventType(Enum):
    USER_ACTION = "user_action"
    SYSTEM_EVENT = "system_event"
    DATA_CHANGE = "data_change"
    BUSINESS_EVENT = "business_event"
    ERROR_EVENT = "error_event"
    MONITORING = "monitoring"
    NOTIFICATION = "notification"


@dataclass
class Event:
    """Event structure for pub/sub system"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    type: EventType = EventType.SYSTEM_EVENT
    topic: str = ""
    source: str = ""
    data: Dict[str, Any] = field(default_factory=dict)
    metadata: Dict[str, Any] = field(default_factory=dict)
    timestamp: datetime = field(default_factory=datetime.now)
    version: str = "1.0"
    correlation_id: Optional[str] = None
    causation_id: Optional[str] = None
    
    def to_dict(self) -> Dict[str, Any]:
        data = asdict(self)
        data['type'] = self.type.value
        data['timestamp'] = self.timestamp.isoformat()
        return data
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'Event':
        data = data.copy()
        data['type'] = EventType(data['type'])
        data['timestamp'] = datetime.fromisoformat(data['timestamp'])
        return cls(**data)
    
    def matches_pattern(self, pattern: str) -> bool:
        """Check if event matches topic pattern"""
        return fnmatch.fnmatch(self.topic, pattern)


@dataclass
class EventFilter:
    """Event filtering criteria"""
    topic_patterns: List[str] = field(default_factory=list)
    event_types: List[EventType] = field(default_factory=list)
    sources: List[str] = field(default_factory=list)
    data_filters: Dict[str, Any] = field(default_factory=dict)
    metadata_filters: Dict[str, Any] = field(default_factory=dict)
    
    def matches(self, event: Event) -> bool:
        """Check if event matches all filter criteria"""
        # Topic patterns
        if self.topic_patterns:
            if not any(event.matches_pattern(pattern) for pattern in self.topic_patterns):
                return False
        
        # Event types
        if self.event_types:
            if event.type not in self.event_types:
                return False
        
        # Sources
        if self.sources:
            if event.source not in self.sources:
                return False
        
        # Data filters
        for key, value in self.data_filters.items():
            if key not in event.data or event.data[key] != value:
                return False
        
        # Metadata filters
        for key, value in self.metadata_filters.items():
            if key not in event.metadata or event.metadata[key] != value:
                return False
        
        return True


class EventHandler:
    """Event handler with metadata"""
    
    def __init__(self, handler_func: Callable[[Event], Any], 
                 event_filter: Optional[EventFilter] = None,
                 priority: int = 0, async_handler: bool = True):
        self.handler_func = handler_func
        self.event_filter = event_filter or EventFilter()
        self.priority = priority
        self.async_handler = async_handler
        self.id = str(uuid.uuid4())
        self.created_at = datetime.now()
        self.execution_count = 0
        self.error_count = 0
        self.total_execution_time = 0.0
        self.lock = threading.Lock()
    
    async def handle_event(self, event: Event) -> bool:
        """Handle event if it matches filter"""
        if not self.event_filter.matches(event):
            return False
        
        start_time = time.time()
        
        try:
            with self.lock:
                self.execution_count += 1
            
            if self.async_handler and asyncio.iscoroutinefunction(self.handler_func):
                await self.handler_func(event)
            else:
                self.handler_func(event)
            
            execution_time = time.time() - start_time
            
            with self.lock:
                self.total_execution_time += execution_time
            
            return True
            
        except Exception as e:
            execution_time = time.time() - start_time
            
            with self.lock:
                self.error_count += 1
                self.total_execution_time += execution_time
            
            logging.error(f"Event handler {self.id} failed: {e}")
            return False
    
    def get_stats(self) -> Dict[str, Any]:
        """Get handler statistics"""
        with self.lock:
            avg_time = self.total_execution_time / max(1, self.execution_count)
            return {
                'id': self.id,
                'executions': self.execution_count,
                'errors': self.error_count,
                'average_time': avg_time,
                'total_time': self.total_execution_time,
                'error_rate': self.error_count / max(1, self.execution_count)
            }


class EventBus(ABC):
    """Abstract event bus interface"""
    
    @abstractmethod
    async def publish(self, event: Event) -> bool:
        """Publish event to bus"""
        pass
    
    @abstractmethod
    def subscribe(self, handler: EventHandler) -> str:
        """Subscribe event handler"""
        pass
    
    @abstractmethod
    def unsubscribe(self, subscription_id: str) -> bool:
        """Unsubscribe event handler"""
        pass
    
    @abstractmethod
    async def get_stats(self) -> Dict[str, Any]:
        """Get event bus statistics"""
        pass


class InMemoryEventBus(EventBus):
    """In-memory event bus implementation"""
    
    def __init__(self, max_history: int = 1000):
        self.handlers: Dict[str, EventHandler] = {}
        self.event_history: List[Event] = []
        self.max_history = max_history
        self.lock = threading.RLock()
        self.logger = logging.getLogger(__name__)
        
        # Statistics
        self.published_events = 0
        self.failed_publications = 0
        self.total_handlers_executed = 0
        self.start_time = datetime.now()
    
    async def publish(self, event: Event) -> bool:
        """Publish event to all matching handlers"""
        try:
            with self.lock:
                self.published_events += 1
                
                # Add to history
                self.event_history.append(event)
                if len(self.event_history) > self.max_history:
                    self.event_history.pop(0)
                
                # Get matching handlers sorted by priority
                matching_handlers = [
                    handler for handler in self.handlers.values()
                    if handler.event_filter.matches(event)
                ]
                matching_handlers.sort(key=lambda h: h.priority, reverse=True)
            
            # Execute handlers
            tasks = []
            for handler in matching_handlers:
                if handler.async_handler:
                    tasks.append(handler.handle_event(event))
                else:
                    # Run sync handlers in thread pool
                    tasks.append(asyncio.get_event_loop().run_in_executor(
                        None, lambda: asyncio.run(handler.handle_event(event))
                    ))
            
            if tasks:
                results = await asyncio.gather(*tasks, return_exceptions=True)
                successful = sum(1 for r in results if r is True)
                
                with self.lock:
                    self.total_handlers_executed += len(results)
                
                self.logger.info(f"Event {event.id} published to {successful}/{len(results)} handlers")
            
            return True
            
        except Exception as e:
            with self.lock:
                self.failed_publications += 1
            
            self.logger.error(f"Failed to publish event {event.id}: {e}")
            return False
    
    def subscribe(self, handler: EventHandler) -> str:
        """Subscribe event handler"""
        with self.lock:
            self.handlers[handler.id] = handler
        
        self.logger.info(f"Handler {handler.id} subscribed")
        return handler.id
    
    def unsubscribe(self, subscription_id: str) -> bool:
        """Unsubscribe event handler"""
        with self.lock:
            if subscription_id in self.handlers:
                del self.handlers[subscription_id]
                self.logger.info(f"Handler {subscription_id} unsubscribed")
                return True
            return False
    
    async def get_stats(self) -> Dict[str, Any]:
        """Get event bus statistics"""
        with self.lock:
            uptime = (datetime.now() - self.start_time).total_seconds()
            events_per_second = self.published_events / max(1, uptime)
            
            handler_stats = [handler.get_stats() for handler in self.handlers.values()]
            
            return {
                'published_events': self.published_events,
                'failed_publications': self.failed_publications,
                'total_handlers_executed': self.total_handlers_executed,
                'active_handlers': len(self.handlers),
                'events_per_second': events_per_second,
                'uptime_seconds': uptime,
                'handler_stats': handler_stats,
                'event_history_size': len(self.event_history)
            }
    
    def get_event_history(self, limit: int = 100) -> List[Event]:
        """Get recent event history"""
        with self.lock:
            return self.event_history[-limit:] if limit else self.event_history.copy()


class RedisEventBus(EventBus):
    """Redis-based distributed event bus"""
    
    def __init__(self, redis_url: str = "redis://localhost:6379", 
                 channel_prefix: str = "events"):
        if not REDIS_AVAILABLE:
            raise ImportError("Redis not available")
        
        self.redis = redis.from_url(redis_url, decode_responses=True)
        self.channel_prefix = channel_prefix
        self.handlers: Dict[str, EventHandler] = {}
        self.subscriptions: Dict[str, str] = {}  # handler_id -> channel
        self.pubsub = self.redis.pubsub()
        self.logger = logging.getLogger(__name__)
        self.is_listening = False
        
        # Statistics
        self.published_events = 0
        self.received_events = 0
        
        # Start listening task
        self._listen_task = None
    
    async def publish(self, event: Event) -> bool:
        """Publish event to Redis"""
        try:
            channel = f"{self.channel_prefix}:{event.topic}"
            message = json.dumps(event.to_dict())
            
            self.redis.publish(channel, message)
            self.published_events += 1
            
            self.logger.info(f"Event {event.id} published to channel {channel}")
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to publish event {event.id}: {e}")
            return False
    
    def subscribe(self, handler: EventHandler) -> str:
        """Subscribe event handler to Redis channels"""
        # Subscribe to channels based on handler's topic patterns
        for pattern in handler.event_filter.topic_patterns:
            channel = f"{self.channel_prefix}:{pattern}"
            self.pubsub.psubscribe(channel)
            self.subscriptions[handler.id] = channel
        
        self.handlers[handler.id] = handler
        
        # Start listening if not already started
        if not self.is_listening:
            self._listen_task = asyncio.create_task(self._listen_for_events())
            self.is_listening = True
        
        self.logger.info(f"Handler {handler.id} subscribed to Redis channels")
        return handler.id
    
    def unsubscribe(self, subscription_id: str) -> bool:
        """Unsubscribe event handler"""
        if subscription_id in self.handlers:
            if subscription_id in self.subscriptions:
                channel = self.subscriptions[subscription_id]
                self.pubsub.punsubscribe(channel)
                del self.subscriptions[subscription_id]
            
            del self.handlers[subscription_id]
            self.logger.info(f"Handler {subscription_id} unsubscribed")
            return True
        
        return False
    
    async def _listen_for_events(self):
        """Listen for events from Redis"""
        self.logger.info("Started listening for Redis events")
        
        try:
            while self.is_listening:
                message = self.pubsub.get_message(timeout=1.0)
                
                if message and message['type'] == 'pmessage':
                    try:
                        event_data = json.loads(message['data'])
                        event = Event.from_dict(event_data)
                        
                        self.received_events += 1
                        
                        # Execute matching handlers
                        tasks = []
                        for handler in self.handlers.values():
                            if handler.event_filter.matches(event):
                                tasks.append(handler.handle_event(event))
                        
                        if tasks:
                            await asyncio.gather(*tasks, return_exceptions=True)
                        
                    except Exception as e:
                        self.logger.error(f"Error processing Redis event: {e}")
                
                await asyncio.sleep(0.01)  # Small delay
                
        except Exception as e:
            self.logger.error(f"Redis listener error: {e}")
        finally:
            self.logger.info("Redis event listener stopped")
    
    async def get_stats(self) -> Dict[str, Any]:
        """Get Redis event bus statistics"""
        return {
            'published_events': self.published_events,
            'received_events': self.received_events,
            'active_handlers': len(self.handlers),
            'active_subscriptions': len(self.subscriptions),
            'handler_stats': [handler.get_stats() for handler in self.handlers.values()]
        }
    
    def close(self):
        """Close Redis connections"""
        self.is_listening = False
        if self._listen_task:
            self._listen_task.cancel()
        self.pubsub.close()


class EventRouter:
    """Advanced event routing with rules"""
    
    def __init__(self):
        self.routing_rules: List[Dict[str, Any]] = []
        self.logger = logging.getLogger(__name__)
    
    def add_routing_rule(self, source_pattern: str, target_topic: str, 
                        conditions: Optional[Dict[str, Any]] = None,
                        transformations: Optional[List[Callable[[Event], Event]]] = None):
        """Add event routing rule"""
        rule = {
            'source_pattern': source_pattern,
            'target_topic': target_topic,
            'conditions': conditions or {},
            'transformations': transformations or []
        }
        self.routing_rules.append(rule)
        self.logger.info(f"Added routing rule: {source_pattern} -> {target_topic}")
    
    def route_event(self, event: Event) -> List[Event]:
        """Route event based on rules"""
        routed_events = [event]  # Always include original
        
        for rule in self.routing_rules:
            if event.matches_pattern(rule['source_pattern']):
                # Check conditions
                conditions_met = True
                for key, value in rule['conditions'].items():
                    if key in event.data and event.data[key] != value:
                        conditions_met = False
                        break
                
                if conditions_met:
                    # Create routed event
                    routed_event = Event(
                        type=event.type,
                        topic=rule['target_topic'],
                        source=event.source,
                        data=event.data.copy(),
                        metadata=event.metadata.copy(),
                        correlation_id=event.correlation_id,
                        causation_id=event.id  # Original event caused this one
                    )
                    
                    # Apply transformations
                    for transform in rule['transformations']:
                        routed_event = transform(routed_event)
                    
                    routed_events.append(routed_event)
        
        return routed_events


class EventBusManager:
    """High-level event bus manager with routing"""
    
    def __init__(self, event_bus: EventBus):
        self.event_bus = event_bus
        self.router = EventRouter()
        self.middleware: List[Callable[[Event], Event]] = []
        self.logger = logging.getLogger(__name__)
    
    def add_middleware(self, middleware_func: Callable[[Event], Event]):
        """Add event processing middleware"""
        self.middleware.append(middleware_func)
    
    async def publish(self, event: Event) -> bool:
        """Publish event with routing and middleware"""
        try:
            # Apply middleware
            processed_event = event
            for middleware in self.middleware:
                processed_event = middleware(processed_event)
            
            # Route event
            routed_events = self.router.route_event(processed_event)
            
            # Publish all routed events
            results = []
            for routed_event in routed_events:
                result = await self.event_bus.publish(routed_event)
                results.append(result)
            
            return all(results)
            
        except Exception as e:
            self.logger.error(f"Failed to publish event {event.id}: {e}")
            return False
    
    def subscribe(self, topic_patterns: List[str], handler_func: Callable[[Event], Any],
                 event_types: Optional[List[EventType]] = None,
                 **filter_kwargs) -> str:
        """Subscribe with simplified interface"""
        event_filter = EventFilter(
            topic_patterns=topic_patterns,
            event_types=event_types or [],
            **filter_kwargs
        )
        
        handler = EventHandler(handler_func, event_filter)
        return self.event_bus.subscribe(handler)
    
    def unsubscribe(self, subscription_id: str) -> bool:
        """Unsubscribe handler"""
        return self.event_bus.unsubscribe(subscription_id)
    
    def add_route(self, source_pattern: str, target_topic: str, **kwargs):
        """Add routing rule"""
        return self.router.add_routing_rule(source_pattern, target_topic, **kwargs)
    
    async def get_stats(self) -> Dict[str, Any]:
        """Get comprehensive statistics"""
        bus_stats = await self.event_bus.get_stats()
        
        return {
            **bus_stats,
            'middleware_count': len(self.middleware),
            'routing_rules': len(self.router.routing_rules)
        }


# Factory functions
def create_memory_event_bus(max_history: int = 1000) -> InMemoryEventBus:
    """Create in-memory event bus"""
    return InMemoryEventBus(max_history)


def create_redis_event_bus(redis_url: str = "redis://localhost:6379", 
                          channel_prefix: str = "events") -> RedisEventBus:
    """Create Redis event bus"""
    return RedisEventBus(redis_url, channel_prefix)


def create_event_manager(event_bus: EventBus) -> EventBusManager:
    """Create event bus manager"""
    return EventBusManager(event_bus)


if __name__ == "__main__":
    async def user_action_handler(event: Event):
        """Example user action handler"""
        print(f"User action: {event.data}")
    
    async def system_event_handler(event: Event):
        """Example system event handler"""
        print(f"System event: {event.data}")
    
    async def main():
        # Create event bus and manager
        event_bus = create_memory_event_bus()
        manager = create_event_manager(event_bus)
        
        # Subscribe handlers
        user_sub = manager.subscribe(
            ["user.*"], 
            user_action_handler, 
            [EventType.USER_ACTION]
        )
        
        system_sub = manager.subscribe(
            ["system.*"], 
            system_event_handler, 
            [EventType.SYSTEM_EVENT]
        )
        
        # Add routing rule
        manager.add_route("user.login", "notification.welcome")
        
        # Publish events
        user_event = Event(
            type=EventType.USER_ACTION,
            topic="user.login",
            source="auth_service",
            data={"user_id": "123", "username": "john_doe"}
        )
        
        system_event = Event(
            type=EventType.SYSTEM_EVENT,
            topic="system.startup",
            source="main_service",
            data={"version": "1.0.0", "startup_time": datetime.now().isoformat()}
        )
        
        await manager.publish(user_event)
        await manager.publish(system_event)
        
        # Get statistics
        stats = await manager.get_stats()
        print(f"Event bus stats: {json.dumps(stats, indent=2, default=str)}")
        
        print("g9.2: Event Bus System - COMPLETED ✅")
    
    asyncio.run(main()) 