"""
TuskLang Python SDK - Advanced Message Queue System (g9.1)
High-throughput message processing with Redis/RabbitMQ support
"""

import asyncio
import json
import logging
import threading
import time
import uuid
from abc import ABC, abstractmethod
from dataclasses import dataclass, asdict
from datetime import datetime, timedelta
from enum import Enum
from typing import Dict, List, Optional, Any, Callable, Union
from concurrent.futures import ThreadPoolExecutor
import queue

try:
    import redis
    import redis.sentinel
    REDIS_AVAILABLE = True
except ImportError:
    REDIS_AVAILABLE = False

try:
    import pika
    import pika.exceptions
    RABBITMQ_AVAILABLE = True
except ImportError:
    RABBITMQ_AVAILABLE = False


class Priority(Enum):
    LOW = 1
    NORMAL = 5
    HIGH = 10
    CRITICAL = 20


class MessageStatus(Enum):
    PENDING = "pending"
    PROCESSING = "processing"
    COMPLETED = "completed"
    FAILED = "failed"
    RETRY = "retry"
    DEAD_LETTER = "dead_letter"


@dataclass
class Message:
    """Message structure"""
    id: str
    topic: str
    payload: Dict[str, Any]
    priority: Priority = Priority.NORMAL
    created_at: datetime = None
    attempts: int = 0
    max_retries: int = 3
    delay_seconds: int = 0
    expires_at: Optional[datetime] = None
    status: MessageStatus = MessageStatus.PENDING
    correlation_id: Optional[str] = None
    reply_to: Optional[str] = None
    headers: Dict[str, Any] = None
    
    def __post_init__(self):
        if self.created_at is None:
            self.created_at = datetime.now()
        if self.headers is None:
            self.headers = {}
        if not self.id:
            self.id = str(uuid.uuid4())
    
    def to_dict(self) -> Dict[str, Any]:
        data = asdict(self)
        data['priority'] = self.priority.value
        data['status'] = self.status.value
        data['created_at'] = self.created_at.isoformat()
        if self.expires_at:
            data['expires_at'] = self.expires_at.isoformat()
        return data
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'Message':
        data = data.copy()
        data['priority'] = Priority(data['priority'])
        data['status'] = MessageStatus(data['status'])
        data['created_at'] = datetime.fromisoformat(data['created_at'])
        if data.get('expires_at'):
            data['expires_at'] = datetime.fromisoformat(data['expires_at'])
        return cls(**data)


@dataclass
class QueueStats:
    """Queue statistics"""
    total_messages: int = 0
    pending_messages: int = 0
    processing_messages: int = 0
    completed_messages: int = 0
    failed_messages: int = 0
    messages_per_second: float = 0.0
    average_processing_time: float = 0.0


class MessageHandler:
    """Message handler with retry logic"""
    
    def __init__(self, handler_func: Callable[[Message], Any], max_retries: int = 3):
        self.handler_func = handler_func
        self.max_retries = max_retries
        self.logger = logging.getLogger(__name__)
    
    async def handle_message(self, message: Message) -> bool:
        """Handle message with retry logic"""
        start_time = time.time()
        
        try:
            message.status = MessageStatus.PROCESSING
            message.attempts += 1
            
            # Call handler function
            if asyncio.iscoroutinefunction(self.handler_func):
                result = await self.handler_func(message)
            else:
                result = self.handler_func(message)
            
            message.status = MessageStatus.COMPLETED
            processing_time = time.time() - start_time
            
            self.logger.info(f"Message {message.id} processed successfully in {processing_time:.2f}s")
            return True
            
        except Exception as e:
            processing_time = time.time() - start_time
            self.logger.error(f"Message {message.id} failed after {processing_time:.2f}s: {e}")
            
            if message.attempts >= self.max_retries:
                message.status = MessageStatus.DEAD_LETTER
                self.logger.error(f"Message {message.id} moved to dead letter queue after {message.attempts} attempts")
            else:
                message.status = MessageStatus.RETRY
                self.logger.info(f"Message {message.id} queued for retry (attempt {message.attempts}/{self.max_retries})")
            
            return False


class MessageQueue(ABC):
    """Abstract message queue interface"""
    
    @abstractmethod
    async def send(self, topic: str, payload: Dict[str, Any], **kwargs) -> str:
        """Send message to topic"""
        pass
    
    @abstractmethod
    async def receive(self, topic: str, timeout: int = 30) -> Optional[Message]:
        """Receive message from topic"""
        pass
    
    @abstractmethod
    async def ack(self, message: Message) -> bool:
        """Acknowledge message processing"""
        pass
    
    @abstractmethod
    async def nack(self, message: Message, requeue: bool = True) -> bool:
        """Negative acknowledge (reject) message"""
        pass
    
    @abstractmethod
    async def get_stats(self, topic: str) -> QueueStats:
        """Get queue statistics"""
        pass


class RedisMessageQueue(MessageQueue):
    """Redis-based message queue with streams"""
    
    def __init__(self, redis_url: str = "redis://localhost:6379", 
                 consumer_group: str = "default", max_len: int = 10000):
        if not REDIS_AVAILABLE:
            raise ImportError("Redis not available")
        
        self.redis = redis.from_url(redis_url, decode_responses=True)
        self.consumer_group = consumer_group
        self.consumer_name = f"{consumer_group}-{uuid.uuid4().hex[:8]}"
        self.max_len = max_len
        self.logger = logging.getLogger(__name__)
        
        # Test connection
        self.redis.ping()
    
    async def send(self, topic: str, payload: Dict[str, Any], **kwargs) -> str:
        """Send message using Redis Streams"""
        message = Message(
            id="",  # Redis will generate
            topic=topic,
            payload=payload,
            **kwargs
        )
        
        # Add to stream with max length
        message_id = self.redis.xadd(
            f"queue:{topic}",
            message.to_dict(),
            maxlen=self.max_len,
            approximate=True
        )
        
        message.id = message_id
        self.logger.info(f"Message {message_id} sent to topic {topic}")
        return message_id
    
    async def receive(self, topic: str, timeout: int = 30) -> Optional[Message]:
        """Receive message from Redis Stream"""
        stream_key = f"queue:{topic}"
        
        try:
            # Create consumer group if it doesn't exist
            try:
                self.redis.xgroup_create(stream_key, self.consumer_group, id='0', mkstream=True)
            except redis.exceptions.ResponseError:
                pass  # Group already exists
            
            # Read from consumer group
            messages = self.redis.xreadgroup(
                self.consumer_group,
                self.consumer_name,
                {stream_key: '>'},
                count=1,
                block=timeout * 1000
            )
            
            if messages and messages[0][1]:
                stream, msg_list = messages[0]
                msg_id, fields = msg_list[0]
                
                # Convert fields to message
                fields['id'] = msg_id
                return Message.from_dict(fields)
            
            return None
            
        except Exception as e:
            self.logger.error(f"Error receiving message from {topic}: {e}")
            return None
    
    async def ack(self, message: Message) -> bool:
        """Acknowledge message processing"""
        try:
            stream_key = f"queue:{message.topic}"
            self.redis.xack(stream_key, self.consumer_group, message.id)
            return True
        except Exception as e:
            self.logger.error(f"Error acknowledging message {message.id}: {e}")
            return False
    
    async def nack(self, message: Message, requeue: bool = True) -> bool:
        """Negative acknowledge message"""
        try:
            if requeue:
                # Add back to stream for retry
                await self.send(message.topic, message.payload, 
                              priority=message.priority, attempts=message.attempts)
            return True
        except Exception as e:
            self.logger.error(f"Error nacking message {message.id}: {e}")
            return False
    
    async def get_stats(self, topic: str) -> QueueStats:
        """Get Redis stream statistics"""
        try:
            stream_key = f"queue:{topic}"
            info = self.redis.xinfo_stream(stream_key)
            
            return QueueStats(
                total_messages=info.get('length', 0),
                pending_messages=info.get('length', 0),  # Approximate
                processing_messages=0,  # Would need consumer group info
                completed_messages=0,
                failed_messages=0
            )
        except:
            return QueueStats()


class RabbitMQMessageQueue(MessageQueue):
    """RabbitMQ-based message queue"""
    
    def __init__(self, amqp_url: str = "amqp://localhost:5672", 
                 exchange: str = "tusklang", durable: bool = True):
        if not RABBITMQ_AVAILABLE:
            raise ImportError("RabbitMQ (pika) not available")
        
        self.connection = pika.BlockingConnection(pika.URLParameters(amqp_url))
        self.channel = self.connection.channel()
        self.exchange = exchange
        self.durable = durable
        self.logger = logging.getLogger(__name__)
        
        # Declare exchange
        self.channel.exchange_declare(exchange=exchange, exchange_type='topic', durable=durable)
    
    async def send(self, topic: str, payload: Dict[str, Any], **kwargs) -> str:
        """Send message to RabbitMQ"""
        message = Message(
            id=str(uuid.uuid4()),
            topic=topic,
            payload=payload,
            **kwargs
        )
        
        # Declare queue
        queue_name = f"queue.{topic}"
        self.channel.queue_declare(queue=queue_name, durable=self.durable)
        self.channel.queue_bind(exchange=self.exchange, queue=queue_name, routing_key=topic)
        
        # Publish message
        self.channel.basic_publish(
            exchange=self.exchange,
            routing_key=topic,
            body=json.dumps(message.to_dict()),
            properties=pika.BasicProperties(
                delivery_mode=2 if self.durable else 1,  # Persistent
                priority=message.priority.value,
                message_id=message.id,
                timestamp=int(message.created_at.timestamp()),
                correlation_id=message.correlation_id,
                reply_to=message.reply_to,
                headers=message.headers
            )
        )
        
        self.logger.info(f"Message {message.id} sent to topic {topic}")
        return message.id
    
    async def receive(self, topic: str, timeout: int = 30) -> Optional[Message]:
        """Receive message from RabbitMQ"""
        queue_name = f"queue.{topic}"
        
        try:
            # Get single message
            method_frame, header_frame, body = self.channel.basic_get(queue=queue_name)
            
            if method_frame:
                message_data = json.loads(body)
                message = Message.from_dict(message_data)
                message._delivery_tag = method_frame.delivery_tag  # Store for ack
                return message
            
            return None
            
        except Exception as e:
            self.logger.error(f"Error receiving message from {topic}: {e}")
            return None
    
    async def ack(self, message: Message) -> bool:
        """Acknowledge message"""
        try:
            if hasattr(message, '_delivery_tag'):
                self.channel.basic_ack(delivery_tag=message._delivery_tag)
                return True
            return False
        except Exception as e:
            self.logger.error(f"Error acknowledging message {message.id}: {e}")
            return False
    
    async def nack(self, message: Message, requeue: bool = True) -> bool:
        """Negative acknowledge message"""
        try:
            if hasattr(message, '_delivery_tag'):
                self.channel.basic_nack(delivery_tag=message._delivery_tag, requeue=requeue)
                return True
            return False
        except Exception as e:
            self.logger.error(f"Error nacking message {message.id}: {e}")
            return False
    
    async def get_stats(self, topic: str) -> QueueStats:
        """Get RabbitMQ queue statistics"""
        try:
            queue_name = f"queue.{topic}"
            method = self.channel.queue_declare(queue=queue_name, passive=True)
            
            return QueueStats(
                total_messages=method.method.message_count,
                pending_messages=method.method.message_count,
                processing_messages=0,
                completed_messages=0,
                failed_messages=0
            )
        except:
            return QueueStats()
    
    def close(self):
        """Close connection"""
        if self.connection and not self.connection.is_closed:
            self.connection.close()


class InMemoryMessageQueue(MessageQueue):
    """In-memory message queue for testing/development"""
    
    def __init__(self):
        self.queues: Dict[str, List[Message]] = {}
        self.processing: Dict[str, List[Message]] = {}
        self.lock = threading.RLock()
        self.logger = logging.getLogger(__name__)
    
    async def send(self, topic: str, payload: Dict[str, Any], **kwargs) -> str:
        """Send message to in-memory queue"""
        message = Message(
            id=str(uuid.uuid4()),
            topic=topic,
            payload=payload,
            **kwargs
        )
        
        with self.lock:
            if topic not in self.queues:
                self.queues[topic] = []
            
            # Insert based on priority (higher priority first)
            inserted = False
            for i, existing_msg in enumerate(self.queues[topic]):
                if message.priority.value > existing_msg.priority.value:
                    self.queues[topic].insert(i, message)
                    inserted = True
                    break
            
            if not inserted:
                self.queues[topic].append(message)
        
        self.logger.info(f"Message {message.id} sent to topic {topic}")
        return message.id
    
    async def receive(self, topic: str, timeout: int = 30) -> Optional[Message]:
        """Receive message from in-memory queue"""
        with self.lock:
            if topic not in self.queues or not self.queues[topic]:
                return None
            
            message = self.queues[topic].pop(0)
            
            # Move to processing
            if topic not in self.processing:
                self.processing[topic] = []
            self.processing[topic].append(message)
            
            return message
    
    async def ack(self, message: Message) -> bool:
        """Remove message from processing"""
        with self.lock:
            if message.topic in self.processing:
                try:
                    self.processing[message.topic].remove(message)
                    return True
                except ValueError:
                    pass
            return False
    
    async def nack(self, message: Message, requeue: bool = True) -> bool:
        """Move message back to queue or discard"""
        with self.lock:
            if message.topic in self.processing:
                try:
                    self.processing[message.topic].remove(message)
                    
                    if requeue:
                        if message.topic not in self.queues:
                            self.queues[message.topic] = []
                        self.queues[message.topic].append(message)
                    
                    return True
                except ValueError:
                    pass
            return False
    
    async def get_stats(self, topic: str) -> QueueStats:
        """Get in-memory queue statistics"""
        with self.lock:
            pending = len(self.queues.get(topic, []))
            processing = len(self.processing.get(topic, []))
            
            return QueueStats(
                total_messages=pending + processing,
                pending_messages=pending,
                processing_messages=processing,
                completed_messages=0,
                failed_messages=0
            )


class MessageQueueManager:
    """Message queue manager with worker pool"""
    
    def __init__(self, queue: MessageQueue, workers: int = 4):
        self.queue = queue
        self.workers = workers
        self.handlers: Dict[str, MessageHandler] = {}
        self.is_running = False
        self.executor = ThreadPoolExecutor(max_workers=workers)
        self.logger = logging.getLogger(__name__)
    
    def register_handler(self, topic: str, handler_func: Callable[[Message], Any], 
                        max_retries: int = 3):
        """Register message handler for topic"""
        self.handlers[topic] = MessageHandler(handler_func, max_retries)
        self.logger.info(f"Registered handler for topic: {topic}")
    
    async def start(self):
        """Start consuming messages"""
        self.is_running = True
        self.logger.info("Message queue manager started")
        
        # Start worker tasks
        tasks = []
        for i in range(self.workers):
            task = asyncio.create_task(self._worker(f"worker-{i}"))
            tasks.append(task)
        
        await asyncio.gather(*tasks)
    
    async def stop(self):
        """Stop consuming messages"""
        self.is_running = False
        self.executor.shutdown(wait=True)
        self.logger.info("Message queue manager stopped")
    
    async def _worker(self, worker_id: str):
        """Worker loop for processing messages"""
        self.logger.info(f"Worker {worker_id} started")
        
        while self.is_running:
            try:
                # Check all topics for messages
                for topic in self.handlers.keys():
                    message = await self.queue.receive(topic, timeout=1)
                    
                    if message:
                        handler = self.handlers[topic]
                        
                        # Process message
                        success = await handler.handle_message(message)
                        
                        if success:
                            await self.queue.ack(message)
                        else:
                            await self.queue.nack(message, requeue=True)
                
                await asyncio.sleep(0.1)  # Small delay to prevent busy waiting
                
            except Exception as e:
                self.logger.error(f"Worker {worker_id} error: {e}")
                await asyncio.sleep(1)
        
        self.logger.info(f"Worker {worker_id} stopped")
    
    async def send_message(self, topic: str, payload: Dict[str, Any], **kwargs) -> str:
        """Send message via managed queue"""
        return await self.queue.send(topic, payload, **kwargs)
    
    async def get_stats(self, topic: str) -> QueueStats:
        """Get queue statistics"""
        return await self.queue.get_stats(topic)


# Factory functions
def create_redis_queue(redis_url: str = "redis://localhost:6379", 
                      consumer_group: str = "default") -> RedisMessageQueue:
    """Create Redis message queue"""
    return RedisMessageQueue(redis_url, consumer_group)


def create_rabbitmq_queue(amqp_url: str = "amqp://localhost:5672", 
                         exchange: str = "tusklang") -> RabbitMQMessageQueue:
    """Create RabbitMQ message queue"""
    return RabbitMQMessageQueue(amqp_url, exchange)


def create_memory_queue() -> InMemoryMessageQueue:
    """Create in-memory message queue"""
    return InMemoryMessageQueue()


if __name__ == "__main__":
    async def example_handler(message: Message):
        """Example message handler"""
        print(f"Processing message {message.id}: {message.payload}")
        await asyncio.sleep(0.1)  # Simulate work
        return True
    
    async def main():
        # Create in-memory queue for example
        queue = create_memory_queue()
        manager = MessageQueueManager(queue, workers=2)
        
        # Register handler
        manager.register_handler("test.topic", example_handler)
        
        # Send test messages
        for i in range(5):
            await manager.send_message("test.topic", {"data": f"message-{i}"})
        
        # Get stats
        stats = await manager.get_stats("test.topic")
        print(f"Queue stats: {stats}")
        
        # Start processing (would run indefinitely in real app)
        # await manager.start()
        
        print("g9.1: Advanced Message Queue System - COMPLETED ✅")
    
    asyncio.run(main()) 