"""
TuskLang Python SDK - Analytics Engine (g12.1)
Production analytics with real-time dashboards, interactive charts and KPIs
"""

import asyncio
import json
import logging
import math
import statistics
import time
import uuid
from collections import defaultdict, deque
from dataclasses import dataclass, field, asdict
from datetime import datetime, timedelta
from enum import Enum
from typing import Dict, List, Optional, Set, Any, Callable, Union, Tuple
import threading


class ChartType(Enum):
    LINE = "line"
    BAR = "bar"
    PIE = "pie"
    SCATTER = "scatter"
    HISTOGRAM = "histogram"
    HEATMAP = "heatmap"
    GAUGE = "gauge"
    TABLE = "table"
    METRIC = "metric"


class AggregationType(Enum):
    SUM = "sum"
    COUNT = "count"
    AVG = "avg"
    MIN = "min"
    MAX = "max"
    MEDIAN = "median"
    PERCENTILE = "percentile"
    DISTINCT_COUNT = "distinct_count"
    VARIANCE = "variance"
    STD_DEV = "std_dev"


class TimeGranularity(Enum):
    SECOND = "second"
    MINUTE = "minute" 
    HOUR = "hour"
    DAY = "day"
    WEEK = "week"
    MONTH = "month"
    QUARTER = "quarter"
    YEAR = "year"


@dataclass
class DataPoint:
    """Single data point"""
    timestamp: datetime
    value: float
    dimensions: Dict[str, Any] = field(default_factory=dict)
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class MetricDefinition:
    """Metric definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    data_source: str = ""
    aggregation: AggregationType = AggregationType.SUM
    dimension_filters: Dict[str, Any] = field(default_factory=dict)
    time_window: int = 3600  # seconds
    percentile: float = 95.0  # for percentile aggregation
    
    # Thresholds
    warning_threshold: Optional[float] = None
    critical_threshold: Optional[float] = None
    
    # Metadata
    unit: str = ""
    format: str = "{:.2f}"
    created_at: datetime = field(default_factory=datetime.now)


@dataclass
class ChartConfiguration:
    """Chart configuration"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    title: str = ""
    chart_type: ChartType = ChartType.LINE
    metrics: List[str] = field(default_factory=list)  # Metric IDs
    
    # Time settings
    time_range: int = 3600  # seconds
    time_granularity: TimeGranularity = TimeGranularity.MINUTE
    
    # Visualization settings
    x_axis_label: str = "Time"
    y_axis_label: str = "Value"
    color_scheme: List[str] = field(default_factory=lambda: [
        "#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd"
    ])
    
    # Interactivity
    is_real_time: bool = True
    refresh_interval: int = 30  # seconds
    
    # Filters
    dimension_filters: Dict[str, Any] = field(default_factory=dict)
    
    # Layout
    width: int = 800
    height: int = 400
    position: Dict[str, int] = field(default_factory=lambda: {"x": 0, "y": 0})


@dataclass
class Dashboard:
    """Dashboard definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    charts: List[str] = field(default_factory=list)  # Chart IDs
    
    # Layout
    layout: Dict[str, Any] = field(default_factory=dict)
    theme: str = "light"
    
    # Access control
    is_public: bool = False
    allowed_users: List[str] = field(default_factory=list)
    
    # Metadata
    created_at: datetime = field(default_factory=datetime.now)
    created_by: str = ""
    last_modified: datetime = field(default_factory=datetime.now)


@dataclass
class KPI:
    """Key Performance Indicator"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    metric_id: str = ""
    current_value: float = 0.0
    previous_value: float = 0.0
    
    # Targets
    target_value: Optional[float] = None
    min_value: Optional[float] = None
    max_value: Optional[float] = None
    
    # Status
    status: str = "normal"  # normal, warning, critical
    trend: str = "stable"   # up, down, stable
    
    # Display
    display_format: str = "{:.2f}"
    unit: str = ""
    
    # Metadata
    updated_at: datetime = field(default_factory=datetime.now)


class DataStore:
    """Time-series data storage"""
    
    def __init__(self, retention_days: int = 30):
        self.data: Dict[str, deque] = defaultdict(lambda: deque())
        self.retention_seconds = retention_days * 24 * 3600
        self.lock = threading.RLock()
        self.logger = logging.getLogger(__name__)
    
    def add_data_point(self, metric_id: str, data_point: DataPoint):
        """Add data point"""
        with self.lock:
            self.data[metric_id].append(data_point)
            
            # Clean old data
            cutoff_time = datetime.now() - timedelta(seconds=self.retention_seconds)
            while (self.data[metric_id] and 
                   self.data[metric_id][0].timestamp < cutoff_time):
                self.data[metric_id].popleft()
    
    def add_data_points(self, metric_id: str, data_points: List[DataPoint]):
        """Add multiple data points"""
        with self.lock:
            for point in data_points:
                self.add_data_point(metric_id, point)
    
    def get_data_points(self, metric_id: str, start_time: datetime, 
                       end_time: datetime) -> List[DataPoint]:
        """Get data points in time range"""
        with self.lock:
            points = self.data.get(metric_id, deque())
            return [
                point for point in points
                if start_time <= point.timestamp <= end_time
            ]
    
    def get_latest_data_point(self, metric_id: str) -> Optional[DataPoint]:
        """Get latest data point"""
        with self.lock:
            points = self.data.get(metric_id, deque())
            return points[-1] if points else None
    
    def get_aggregated_data(self, metric_id: str, start_time: datetime,
                           end_time: datetime, granularity: TimeGranularity,
                           aggregation: AggregationType) -> List[Dict[str, Any]]:
        """Get aggregated data"""
        points = self.get_data_points(metric_id, start_time, end_time)
        if not points:
            return []
        
        # Group by time buckets
        buckets = self._create_time_buckets(start_time, end_time, granularity)
        bucket_data = {bucket: [] for bucket in buckets}
        
        for point in points:
            bucket = self._get_time_bucket(point.timestamp, granularity)
            if bucket in bucket_data:
                bucket_data[bucket].append(point.value)
        
        # Aggregate each bucket
        result = []
        for bucket in sorted(buckets):
            values = bucket_data[bucket]
            if values:
                agg_value = self._aggregate_values(values, aggregation)
                result.append({
                    'timestamp': bucket.isoformat(),
                    'value': agg_value,
                    'count': len(values)
                })
        
        return result
    
    def _create_time_buckets(self, start_time: datetime, end_time: datetime,
                            granularity: TimeGranularity) -> List[datetime]:
        """Create time buckets"""
        buckets = []
        current = start_time
        
        while current <= end_time:
            buckets.append(current)
            current = self._next_time_bucket(current, granularity)
        
        return buckets
    
    def _get_time_bucket(self, timestamp: datetime, 
                        granularity: TimeGranularity) -> datetime:
        """Get time bucket for timestamp"""
        if granularity == TimeGranularity.SECOND:
            return timestamp.replace(microsecond=0)
        elif granularity == TimeGranularity.MINUTE:
            return timestamp.replace(second=0, microsecond=0)
        elif granularity == TimeGranularity.HOUR:
            return timestamp.replace(minute=0, second=0, microsecond=0)
        elif granularity == TimeGranularity.DAY:
            return timestamp.replace(hour=0, minute=0, second=0, microsecond=0)
        elif granularity == TimeGranularity.WEEK:
            days_since_monday = timestamp.weekday()
            monday = timestamp - timedelta(days=days_since_monday)
            return monday.replace(hour=0, minute=0, second=0, microsecond=0)
        elif granularity == TimeGranularity.MONTH:
            return timestamp.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
        elif granularity == TimeGranularity.YEAR:
            return timestamp.replace(month=1, day=1, hour=0, minute=0, second=0, microsecond=0)
        
        return timestamp
    
    def _next_time_bucket(self, current: datetime, 
                         granularity: TimeGranularity) -> datetime:
        """Get next time bucket"""
        if granularity == TimeGranularity.SECOND:
            return current + timedelta(seconds=1)
        elif granularity == TimeGranularity.MINUTE:
            return current + timedelta(minutes=1)
        elif granularity == TimeGranularity.HOUR:
            return current + timedelta(hours=1)
        elif granularity == TimeGranularity.DAY:
            return current + timedelta(days=1)
        elif granularity == TimeGranularity.WEEK:
            return current + timedelta(weeks=1)
        elif granularity == TimeGranularity.MONTH:
            if current.month == 12:
                return current.replace(year=current.year + 1, month=1)
            else:
                return current.replace(month=current.month + 1)
        elif granularity == TimeGranularity.YEAR:
            return current.replace(year=current.year + 1)
        
        return current
    
    def _aggregate_values(self, values: List[float], 
                         aggregation: AggregationType) -> float:
        """Aggregate values"""
        if not values:
            return 0.0
        
        if aggregation == AggregationType.SUM:
            return sum(values)
        elif aggregation == AggregationType.COUNT:
            return len(values)
        elif aggregation == AggregationType.AVG:
            return statistics.mean(values)
        elif aggregation == AggregationType.MIN:
            return min(values)
        elif aggregation == AggregationType.MAX:
            return max(values)
        elif aggregation == AggregationType.MEDIAN:
            return statistics.median(values)
        elif aggregation == AggregationType.VARIANCE:
            return statistics.variance(values) if len(values) > 1 else 0.0
        elif aggregation == AggregationType.STD_DEV:
            return statistics.stdev(values) if len(values) > 1 else 0.0
        elif aggregation == AggregationType.DISTINCT_COUNT:
            return len(set(values))
        elif aggregation == AggregationType.PERCENTILE:
            # Default to 95th percentile
            return self._percentile(values, 95.0)
        
        return sum(values)  # Default to sum
    
    def _percentile(self, values: List[float], percentile: float) -> float:
        """Calculate percentile"""
        if not values:
            return 0.0
        
        sorted_values = sorted(values)
        k = (len(sorted_values) - 1) * (percentile / 100.0)
        f = math.floor(k)
        c = math.ceil(k)
        
        if f == c:
            return sorted_values[int(k)]
        
        d0 = sorted_values[int(f)] * (c - k)
        d1 = sorted_values[int(c)] * (k - f)
        return d0 + d1


class ChartRenderer:
    """Chart rendering engine"""
    
    def __init__(self, data_store: DataStore):
        self.data_store = data_store
        self.logger = logging.getLogger(__name__)
    
    def render_chart(self, chart_config: ChartConfiguration, 
                    metric_definitions: Dict[str, MetricDefinition]) -> Dict[str, Any]:
        """Render chart data"""
        end_time = datetime.now()
        start_time = end_time - timedelta(seconds=chart_config.time_range)
        
        # Get data for each metric
        chart_data = {
            'config': asdict(chart_config),
            'data': {},
            'metadata': {
                'generated_at': datetime.now().isoformat(),
                'time_range': {
                    'start': start_time.isoformat(),
                    'end': end_time.isoformat()
                }
            }
        }
        
        for metric_id in chart_config.metrics:
            metric_def = metric_definitions.get(metric_id)
            if not metric_def:
                continue
            
            try:
                if chart_config.chart_type == ChartType.METRIC:
                    # Single value metric
                    chart_data['data'][metric_id] = self._render_metric(
                        metric_id, metric_def, end_time
                    )
                else:
                    # Time series data
                    chart_data['data'][metric_id] = self._render_time_series(
                        metric_id, metric_def, start_time, end_time, chart_config
                    )
                
            except Exception as e:
                self.logger.error(f"Error rendering metric {metric_id}: {e}")
                chart_data['data'][metric_id] = {'error': str(e)}
        
        return chart_data
    
    def _render_metric(self, metric_id: str, metric_def: MetricDefinition,
                      end_time: datetime) -> Dict[str, Any]:
        """Render single metric value"""
        start_time = end_time - timedelta(seconds=metric_def.time_window)
        points = self.data_store.get_data_points(metric_id, start_time, end_time)
        
        if not points:
            return {'value': 0, 'status': 'no_data'}
        
        values = [point.value for point in points]
        aggregated_value = self.data_store._aggregate_values(values, metric_def.aggregation)
        
        # Determine status based on thresholds
        status = 'normal'
        if metric_def.critical_threshold is not None:
            if aggregated_value >= metric_def.critical_threshold:
                status = 'critical'
        elif metric_def.warning_threshold is not None:
            if aggregated_value >= metric_def.warning_threshold:
                status = 'warning'
        
        return {
            'value': aggregated_value,
            'formatted_value': metric_def.format.format(aggregated_value),
            'unit': metric_def.unit,
            'status': status,
            'data_points': len(points)
        }
    
    def _render_time_series(self, metric_id: str, metric_def: MetricDefinition,
                           start_time: datetime, end_time: datetime,
                           chart_config: ChartConfiguration) -> Dict[str, Any]:
        """Render time series data"""
        aggregated_data = self.data_store.get_aggregated_data(
            metric_id, start_time, end_time,
            chart_config.time_granularity, metric_def.aggregation
        )
        
        return {
            'name': metric_def.name,
            'unit': metric_def.unit,
            'aggregation': metric_def.aggregation.value,
            'data_points': aggregated_data,
            'total_points': len(aggregated_data)
        }


class KPICalculator:
    """KPI calculation engine"""
    
    def __init__(self, data_store: DataStore):
        self.data_store = data_store
        self.logger = logging.getLogger(__name__)
    
    def calculate_kpi(self, kpi: KPI, metric_def: MetricDefinition) -> KPI:
        """Calculate KPI value and status"""
        end_time = datetime.now()
        start_time = end_time - timedelta(seconds=metric_def.time_window)
        
        # Get current period data
        current_points = self.data_store.get_data_points(metric_def.id, start_time, end_time)
        
        if current_points:
            values = [point.value for point in current_points]
            kpi.current_value = self.data_store._aggregate_values(values, metric_def.aggregation)
        else:
            kpi.current_value = 0.0
        
        # Get previous period data for comparison
        prev_end = start_time
        prev_start = prev_end - timedelta(seconds=metric_def.time_window)
        prev_points = self.data_store.get_data_points(metric_def.id, prev_start, prev_end)
        
        if prev_points:
            prev_values = [point.value for point in prev_points]
            kpi.previous_value = self.data_store._aggregate_values(prev_values, metric_def.aggregation)
        else:
            kpi.previous_value = 0.0
        
        # Calculate trend
        if kpi.previous_value > 0:
            change_percent = ((kpi.current_value - kpi.previous_value) / kpi.previous_value) * 100
            if abs(change_percent) < 5:  # Within 5% is considered stable
                kpi.trend = "stable"
            elif change_percent > 0:
                kpi.trend = "up"
            else:
                kpi.trend = "down"
        else:
            kpi.trend = "stable"
        
        # Calculate status
        kpi.status = self._calculate_status(kpi, metric_def)
        kpi.updated_at = datetime.now()
        
        return kpi
    
    def _calculate_status(self, kpi: KPI, metric_def: MetricDefinition) -> str:
        """Calculate KPI status"""
        value = kpi.current_value
        
        # Check metric thresholds first
        if metric_def.critical_threshold is not None and value >= metric_def.critical_threshold:
            return "critical"
        elif metric_def.warning_threshold is not None and value >= metric_def.warning_threshold:
            return "warning"
        
        # Check KPI-specific targets
        if kpi.target_value is not None:
            # For targets, we assume closer is better
            target_variance = abs(value - kpi.target_value) / max(kpi.target_value, 1.0)
            
            if target_variance > 0.2:  # More than 20% from target
                return "critical"
            elif target_variance > 0.1:  # More than 10% from target
                return "warning"
        
        # Check bounds
        if kpi.min_value is not None and value < kpi.min_value:
            return "critical"
        if kpi.max_value is not None and value > kpi.max_value:
            return "critical"
        
        return "normal"


class AnalyticsEngine:
    """Main analytics engine"""
    
    def __init__(self, retention_days: int = 30):
        self.data_store = DataStore(retention_days)
        self.chart_renderer = ChartRenderer(self.data_store)
        self.kpi_calculator = KPICalculator(self.data_store)
        
        # Configuration storage
        self.metrics: Dict[str, MetricDefinition] = {}
        self.charts: Dict[str, ChartConfiguration] = {}
        self.dashboards: Dict[str, Dashboard] = {}
        self.kpis: Dict[str, KPI] = {}
        
        self.logger = logging.getLogger(__name__)
        
        # Real-time processing
        self.is_running = False
        self.update_task: Optional[asyncio.Task] = None
        
        # Event handlers
        self.alert_handlers: List[Callable[[str, Dict[str, Any]], None]] = []
    
    async def start(self):
        """Start analytics engine"""
        if self.is_running:
            return
        
        self.is_running = True
        self.update_task = asyncio.create_task(self._update_loop())
        self.logger.info("Analytics engine started")
    
    async def stop(self):
        """Stop analytics engine"""
        if not self.is_running:
            return
        
        self.is_running = False
        
        if self.update_task:
            self.update_task.cancel()
        
        self.logger.info("Analytics engine stopped")
    
    def create_metric(self, metric_def: MetricDefinition) -> str:
        """Create metric definition"""
        self.metrics[metric_def.id] = metric_def
        self.logger.info(f"Created metric: {metric_def.name} ({metric_def.id})")
        return metric_def.id
    
    def create_chart(self, chart_config: ChartConfiguration) -> str:
        """Create chart configuration"""
        self.charts[chart_config.id] = chart_config
        self.logger.info(f"Created chart: {chart_config.title} ({chart_config.id})")
        return chart_config.id
    
    def create_dashboard(self, dashboard: Dashboard) -> str:
        """Create dashboard"""
        self.dashboards[dashboard.id] = dashboard
        self.logger.info(f"Created dashboard: {dashboard.name} ({dashboard.id})")
        return dashboard.id
    
    def create_kpi(self, kpi: KPI) -> str:
        """Create KPI"""
        self.kpis[kpi.id] = kpi
        self.logger.info(f"Created KPI: {kpi.name} ({kpi.id})")
        return kpi.id
    
    def ingest_data(self, metric_id: str, value: float, 
                   dimensions: Optional[Dict[str, Any]] = None,
                   timestamp: Optional[datetime] = None):
        """Ingest single data point"""
        data_point = DataPoint(
            timestamp=timestamp or datetime.now(),
            value=value,
            dimensions=dimensions or {}
        )
        
        self.data_store.add_data_point(metric_id, data_point)
    
    def ingest_batch(self, metric_id: str, data_points: List[Tuple[float, datetime]]):
        """Ingest batch of data points"""
        points = [
            DataPoint(timestamp=timestamp, value=value)
            for value, timestamp in data_points
        ]
        
        self.data_store.add_data_points(metric_id, points)
    
    def render_chart(self, chart_id: str) -> Dict[str, Any]:
        """Render chart"""
        chart_config = self.charts.get(chart_id)
        if not chart_config:
            raise ValueError(f"Chart not found: {chart_id}")
        
        return self.chart_renderer.render_chart(chart_config, self.metrics)
    
    def render_dashboard(self, dashboard_id: str) -> Dict[str, Any]:
        """Render complete dashboard"""
        dashboard = self.dashboards.get(dashboard_id)
        if not dashboard:
            raise ValueError(f"Dashboard not found: {dashboard_id}")
        
        # Render all charts
        chart_data = {}
        for chart_id in dashboard.charts:
            try:
                chart_data[chart_id] = self.render_chart(chart_id)
            except Exception as e:
                self.logger.error(f"Error rendering chart {chart_id}: {e}")
                chart_data[chart_id] = {'error': str(e)}
        
        return {
            'dashboard': asdict(dashboard),
            'charts': chart_data,
            'generated_at': datetime.now().isoformat()
        }
    
    def get_kpi_values(self, kpi_ids: Optional[List[str]] = None) -> Dict[str, KPI]:
        """Get current KPI values"""
        target_kpis = kpi_ids or list(self.kpis.keys())
        results = {}
        
        for kpi_id in target_kpis:
            kpi = self.kpis.get(kpi_id)
            if not kpi:
                continue
            
            metric_def = self.metrics.get(kpi.metric_id)
            if not metric_def:
                continue
            
            try:
                updated_kpi = self.kpi_calculator.calculate_kpi(kpi, metric_def)
                results[kpi_id] = updated_kpi
            except Exception as e:
                self.logger.error(f"Error calculating KPI {kpi_id}: {e}")
        
        return results
    
    def add_alert_handler(self, handler: Callable[[str, Dict[str, Any]], None]):
        """Add alert handler"""
        self.alert_handlers.append(handler)
    
    async def _update_loop(self):
        """Background update loop"""
        while self.is_running:
            try:
                # Update KPIs
                await self._update_kpis()
                
                # Check for alerts
                await self._check_alerts()
                
                await asyncio.sleep(30)  # Update every 30 seconds
                
            except Exception as e:
                self.logger.error(f"Update loop error: {e}")
                await asyncio.sleep(60)
    
    async def _update_kpis(self):
        """Update all KPIs"""
        for kpi_id, kpi in self.kpis.items():
            metric_def = self.metrics.get(kpi.metric_id)
            if metric_def:
                try:
                    self.kpi_calculator.calculate_kpi(kpi, metric_def)
                except Exception as e:
                    self.logger.error(f"Error updating KPI {kpi_id}: {e}")
    
    async def _check_alerts(self):
        """Check for alerts"""
        for kpi_id, kpi in self.kpis.items():
            if kpi.status in ['warning', 'critical']:
                alert_data = {
                    'kpi_id': kpi_id,
                    'kpi_name': kpi.name,
                    'status': kpi.status,
                    'current_value': kpi.current_value,
                    'target_value': kpi.target_value,
                    'timestamp': datetime.now().isoformat()
                }
                
                # Send alerts
                for handler in self.alert_handlers:
                    try:
                        handler(kpi.status, alert_data)
                    except Exception as e:
                        self.logger.error(f"Alert handler error: {e}")
    
    def get_metric_stats(self, metric_id: str, hours: int = 24) -> Dict[str, Any]:
        """Get metric statistics"""
        end_time = datetime.now()
        start_time = end_time - timedelta(hours=hours)
        
        points = self.data_store.get_data_points(metric_id, start_time, end_time)
        
        if not points:
            return {'error': 'No data available'}
        
        values = [point.value for point in points]
        
        return {
            'metric_id': metric_id,
            'time_range': f'{hours} hours',
            'data_points': len(values),
            'min': min(values),
            'max': max(values),
            'avg': statistics.mean(values),
            'median': statistics.median(values),
            'std_dev': statistics.stdev(values) if len(values) > 1 else 0.0,
            'total': sum(values),
            'first_value': values[0],
            'last_value': values[-1]
        }
    
    def export_data(self, metric_id: str, start_time: datetime, 
                   end_time: datetime) -> List[Dict[str, Any]]:
        """Export raw data"""
        points = self.data_store.get_data_points(metric_id, start_time, end_time)
        
        return [
            {
                'timestamp': point.timestamp.isoformat(),
                'value': point.value,
                'dimensions': point.dimensions,
                'metadata': point.metadata
            }
            for point in points
        ]


if __name__ == "__main__":
    async def example_alert_handler(status: str, alert_data: Dict[str, Any]):
        """Example alert handler"""
        print(f"ALERT [{status.upper()}]: {alert_data['kpi_name']} = {alert_data['current_value']}")
    
    async def main():
        # Create analytics engine
        analytics = AnalyticsEngine(retention_days=7)
        
        # Add alert handler
        analytics.add_alert_handler(example_alert_handler)
        
        # Create metrics
        cpu_metric = MetricDefinition(
            name="CPU Usage",
            description="System CPU utilization",
            data_source="system_metrics",
            aggregation=AggregationType.AVG,
            unit="%",
            format="{:.1f}%",
            warning_threshold=70.0,
            critical_threshold=90.0
        )
        
        request_metric = MetricDefinition(
            name="Request Count",
            description="HTTP requests per minute",
            aggregation=AggregationType.SUM,
            unit="req/min",
            format="{:.0f}"
        )
        
        analytics.create_metric(cpu_metric)
        analytics.create_metric(request_metric)
        
        # Create charts
        cpu_chart = ChartConfiguration(
            title="CPU Usage Over Time",
            chart_type=ChartType.LINE,
            metrics=[cpu_metric.id],
            time_range=3600,  # 1 hour
            time_granularity=TimeGranularity.MINUTE,
            y_axis_label="CPU %"
        )
        
        request_chart = ChartConfiguration(
            title="Request Rate",
            chart_type=ChartType.BAR,
            metrics=[request_metric.id],
            time_range=1800,  # 30 minutes
            time_granularity=TimeGranularity.MINUTE,
            y_axis_label="Requests"
        )
        
        analytics.create_chart(cpu_chart)
        analytics.create_chart(request_chart)
        
        # Create dashboard
        main_dashboard = Dashboard(
            name="System Overview",
            description="Main system metrics dashboard",
            charts=[cpu_chart.id, request_chart.id]
        )
        
        analytics.create_dashboard(main_dashboard)
        
        # Create KPIs
        cpu_kpi = KPI(
            name="Average CPU Usage",
            metric_id=cpu_metric.id,
            target_value=50.0,
            max_value=80.0,
            display_format="{:.1f}%",
            unit="%"
        )
        
        analytics.create_kpi(cpu_kpi)
        
        # Start analytics engine
        await analytics.start()
        
        # Simulate data ingestion
        print("Ingesting sample data...")
        
        # Generate sample CPU data
        import random
        now = datetime.now()
        for i in range(60):  # 60 minutes of data
            timestamp = now - timedelta(minutes=60-i)
            cpu_value = 30 + random.uniform(-10, 40)  # 20-70% CPU
            analytics.ingest_data(cpu_metric.id, cpu_value, timestamp=timestamp)
            
            # Requests (higher CPU = more requests)
            request_value = max(0, (cpu_value - 20) * 2 + random.uniform(-20, 20))
            analytics.ingest_data(request_metric.id, request_value, timestamp=timestamp)
        
        # Wait for processing
        await asyncio.sleep(2)
        
        # Render charts
        print("\nRendering CPU chart...")
        cpu_chart_data = analytics.render_chart(cpu_chart.id)
        print(f"CPU chart has {len(cpu_chart_data['data'][cpu_metric.id]['data_points'])} data points")
        
        # Render dashboard
        print("\nRendering dashboard...")
        dashboard_data = analytics.render_dashboard(main_dashboard.id)
        print(f"Dashboard rendered with {len(dashboard_data['charts'])} charts")
        
        # Get KPI values
        print("\nCalculating KPIs...")
        kpi_values = analytics.get_kpi_values()
        for kpi_id, kpi in kpi_values.items():
            print(f"KPI {kpi.name}: {kpi.current_value:.1f}{kpi.unit} ({kpi.status}, trend: {kpi.trend})")
        
        # Get metric statistics
        print(f"\nMetric statistics:")
        stats = analytics.get_metric_stats(cpu_metric.id, hours=1)
        print(f"CPU Stats: avg={stats['avg']:.1f}%, min={stats['min']:.1f}%, max={stats['max']:.1f}%")
        
        # Stop analytics engine
        await analytics.stop()
        
        print("\ng12.1: Analytics Engine with Real-time Dashboards - COMPLETED ✅")
    
    asyncio.run(main()) 