"""
TuskLang Python SDK - Enterprise Identity Management
Production-quality identity management with SSO, SAML, LDAP, and MFA
"""

import asyncio
import json
import logging
import uuid
import hashlib
import secrets
import base64
import urllib.parse
from datetime import datetime, timedelta, timezone
from typing import Dict, List, Optional, Any, Union, Tuple
from dataclasses import dataclass, field
from enum import Enum
import threading
import xml.etree.ElementTree as ET

try:
    import jwt
    JWT_AVAILABLE = True
except ImportError:
    JWT_AVAILABLE = False

try:
    from onelogin.saml2.auth import OneLogin_Saml2_Auth
    from onelogin.saml2.settings import OneLogin_Saml2_Settings
    from onelogin.saml2.utils import OneLogin_Saml2_Utils
    SAML_AVAILABLE = True
except ImportError:
    SAML_AVAILABLE = False

try:
    import ldap3
    LDAP_AVAILABLE = True
except ImportError:
    LDAP_AVAILABLE = False

try:
    import pyotp
    TOTP_AVAILABLE = True
except ImportError:
    TOTP_AVAILABLE = False


class IdentityProvider(Enum):
    LOCAL = "local"
    AZURE_AD = "azure_ad"
    OKTA = "okta"
    AUTH0 = "auth0"
    KEYCLOAK = "keycloak"
    GOOGLE_WORKSPACE = "google_workspace"
    LDAP = "ldap"
    ACTIVE_DIRECTORY = "active_directory"


class AuthenticationMethod(Enum):
    PASSWORD = "password"
    SAML_SSO = "saml_sso"
    OAUTH2 = "oauth2"
    OPENID_CONNECT = "openid_connect"
    LDAP_BIND = "ldap_bind"
    CERTIFICATE = "certificate"
    MFA_TOTP = "mfa_totp"
    MFA_SMS = "mfa_sms"
    MFA_HARDWARE = "mfa_hardware"


class PermissionLevel(Enum):
    READ = "read"
    WRITE = "write"
    DELETE = "delete"
    ADMIN = "admin"
    OWNER = "owner"


@dataclass
class Role:
    """Role-based access control definition"""
    role_id: str
    name: str
    description: str
    permissions: List[str] = field(default_factory=list)
    resource_patterns: List[str] = field(default_factory=list)
    inherits_from: List[str] = field(default_factory=list)
    created_at: datetime = field(default_factory=datetime.utcnow)
    updated_at: datetime = field(default_factory=datetime.utcnow)


@dataclass
class User:
    """Enterprise user identity"""
    user_id: str
    username: str
    email: str
    first_name: str
    last_name: str
    is_active: bool = True
    
    # Identity provider info
    provider: IdentityProvider = IdentityProvider.LOCAL
    external_id: str = ""
    provider_metadata: Dict[str, Any] = field(default_factory=dict)
    
    # Authentication
    password_hash: Optional[str] = None
    mfa_enabled: bool = False
    mfa_secret: Optional[str] = None
    backup_codes: List[str] = field(default_factory=list)
    
    # Authorization
    roles: List[str] = field(default_factory=list)
    direct_permissions: List[str] = field(default_factory=list)
    groups: List[str] = field(default_factory=list)
    
    # Session management
    session_timeout_minutes: int = 60
    max_concurrent_sessions: int = 3
    last_login: Optional[datetime] = None
    last_password_change: Optional[datetime] = None
    password_expires_at: Optional[datetime] = None
    
    # Audit
    created_at: datetime = field(default_factory=datetime.utcnow)
    updated_at: datetime = field(default_factory=datetime.utcnow)


@dataclass
class Session:
    """User authentication session"""
    session_id: str
    user_id: str
    provider: IdentityProvider
    authentication_methods: List[AuthenticationMethod]
    created_at: datetime
    expires_at: datetime
    last_activity: datetime
    ip_address: str = ""
    user_agent: str = ""
    is_active: bool = True
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class SAMLConfig:
    """SAML identity provider configuration"""
    entity_id: str
    sso_url: str
    slo_url: str = ""
    x509_cert: str = ""
    name_id_format: str = "urn:oasis:names:tc:SAML:2.0:nameid-format:emailAddress"
    attribute_mapping: Dict[str, str] = field(default_factory=dict)
    require_signed_assertions: bool = True
    encrypt_assertions: bool = False


@dataclass
class LDAPConfig:
    """LDAP/Active Directory configuration"""
    server_uri: str
    bind_dn: str
    bind_password: str
    user_base_dn: str
    user_filter: str = "(uid={username})"
    group_base_dn: str = ""
    group_filter: str = ""
    attribute_mapping: Dict[str, str] = field(default_factory=dict)
    use_ssl: bool = True
    use_tls: bool = False


class EnterpriseIdentityManager:
    """Production-quality enterprise identity and access management"""
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.logger = logging.getLogger(__name__)
        
        # Identity stores
        self.users: Dict[str, User] = {}
        self.roles: Dict[str, Role] = {}
        self.sessions: Dict[str, Session] = {}
        
        # Identity provider configurations
        self.saml_configs: Dict[str, SAMLConfig] = {}
        self.ldap_configs: Dict[str, LDAPConfig] = {}
        self.oauth_configs: Dict[str, Dict[str, Any]] = {}
        
        # Security settings
        self.jwt_secret = self.config.get('jwt_secret', secrets.token_hex(32))
        self.session_timeout_minutes = self.config.get('session_timeout_minutes', 60)
        
        # Threading
        self.identity_lock = threading.RLock()
        
        # Initialize default roles and configurations
        self._initialize_default_roles()
        self._load_identity_provider_configs()
        
        self.logger.info("Enterprise Identity Manager initialized successfully")

    def _initialize_default_roles(self):
        """Initialize default enterprise roles"""
        
        default_roles = [
            Role(
                role_id="admin",
                name="Administrator",
                description="Full system administration privileges",
                permissions=["*"],
                resource_patterns=["*"]
            ),
            Role(
                role_id="user",
                name="Standard User",
                description="Standard user access privileges",
                permissions=["read", "write_own"],
                resource_patterns=["/user/{user_id}/*"]
            ),
            Role(
                role_id="manager",
                name="Manager",
                description="Management privileges for team resources",
                permissions=["read", "write", "manage_team"],
                resource_patterns=["/team/*", "/reports/*"],
                inherits_from=["user"]
            ),
            Role(
                role_id="auditor",
                name="Auditor",
                description="Read-only access for audit purposes",
                permissions=["read", "audit"],
                resource_patterns=["/audit/*", "/logs/*"]
            )
        ]
        
        for role in default_roles:
            self.roles[role.role_id] = role

    def _load_identity_provider_configs(self):
        """Load identity provider configurations"""
        
        # SAML configurations
        if 'saml_providers' in self.config:
            for provider_name, saml_config in self.config['saml_providers'].items():
                self.saml_configs[provider_name] = SAMLConfig(**saml_config)
        
        # LDAP configurations  
        if 'ldap_providers' in self.config:
            for provider_name, ldap_config in self.config['ldap_providers'].items():
                self.ldap_configs[provider_name] = LDAPConfig(**ldap_config)
        
        # OAuth2/OIDC configurations
        if 'oauth_providers' in self.config:
            self.oauth_configs = self.config['oauth_providers']

    async def authenticate(self, username: str, password: Optional[str] = None,
                         provider: IdentityProvider = IdentityProvider.LOCAL,
                         provider_token: Optional[str] = None,
                         mfa_code: Optional[str] = None,
                         request_context: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """Authenticate user with multiple methods and providers"""
        
        request_context = request_context or {}
        ip_address = request_context.get('ip_address', '')
        user_agent = request_context.get('user_agent', '')
        
        auth_result = {
            'success': False,
            'user_id': None,
            'session_id': None,
            'requires_mfa': False,
            'access_token': None,
            'refresh_token': None,
            'expires_in': self.session_timeout_minutes * 60,
            'error': None,
            'provider': provider.value
        }
        
        try:
            user = None
            auth_methods = []
            
            # Authenticate based on provider
            if provider == IdentityProvider.LOCAL:
                user, auth_methods = await self._authenticate_local(username, password)
            elif provider == IdentityProvider.LDAP:
                user, auth_methods = await self._authenticate_ldap(username, password)
            elif provider in [IdentityProvider.AZURE_AD, IdentityProvider.OKTA, IdentityProvider.AUTH0]:
                user, auth_methods = await self._authenticate_saml_sso(provider, provider_token)
            elif provider == IdentityProvider.GOOGLE_WORKSPACE:
                user, auth_methods = await self._authenticate_oauth(provider, provider_token)
            else:
                auth_result['error'] = f"Unsupported authentication provider: {provider.value}"
                return auth_result
            
            if not user:
                auth_result['error'] = "Invalid credentials"
                return auth_result
            
            # Check if MFA is required
            if user.mfa_enabled and not mfa_code:
                auth_result['requires_mfa'] = True
                auth_result['user_id'] = user.user_id
                return auth_result
            
            # Validate MFA if provided
            if user.mfa_enabled and mfa_code:
                if not await self._validate_mfa(user, mfa_code):
                    auth_result['error'] = "Invalid MFA code"
                    return auth_result
                auth_methods.append(AuthenticationMethod.MFA_TOTP)
            
            # Create session
            session = await self._create_session(user, provider, auth_methods, ip_address, user_agent)
            
            # Generate tokens
            access_token = self._generate_access_token(user, session)
            refresh_token = self._generate_refresh_token(user, session)
            
            auth_result.update({
                'success': True,
                'user_id': user.user_id,
                'session_id': session.session_id,
                'access_token': access_token,
                'refresh_token': refresh_token,
                'user_info': {
                    'username': user.username,
                    'email': user.email,
                    'first_name': user.first_name,
                    'last_name': user.last_name,
                    'roles': user.roles,
                    'permissions': await self._get_user_permissions(user)
                }
            })
            
            # Update user last login
            user.last_login = datetime.utcnow()
            
            self.logger.info(f"User {username} authenticated successfully via {provider.value}")
            
        except Exception as e:
            self.logger.error(f"Authentication error for {username}: {e}")
            auth_result['error'] = "Authentication failed"
        
        return auth_result

    async def _authenticate_local(self, username: str, password: str) -> Tuple[Optional[User], List[AuthenticationMethod]]:
        """Authenticate against local user store"""
        
        user = self._find_user_by_username(username)
        if not user or not user.is_active:
            return None, []
        
        if not user.password_hash:
            return None, []
        
        # Verify password (simplified - use proper bcrypt in production)
        password_hash = hashlib.sha256((password + username).encode()).hexdigest()
        if password_hash != user.password_hash:
            return None, []
        
        return user, [AuthenticationMethod.PASSWORD]

    async def _authenticate_ldap(self, username: str, password: str) -> Tuple[Optional[User], List[AuthenticationMethod]]:
        """Authenticate against LDAP/Active Directory"""
        
        if not LDAP_AVAILABLE:
            self.logger.error("LDAP authentication requested but python-ldap not available")
            return None, []
        
        # Use default LDAP config if available
        ldap_config = next(iter(self.ldap_configs.values())) if self.ldap_configs else None
        if not ldap_config:
            self.logger.error("LDAP authentication requested but no LDAP config found")
            return None, []
        
        try:
            # Connect to LDAP server
            server = ldap3.Server(ldap_config.server_uri, use_ssl=ldap_config.use_ssl)
            conn = ldap3.Connection(server, ldap_config.bind_dn, ldap_config.bind_password)
            
            if not conn.bind():
                self.logger.error("Failed to bind to LDAP server")
                return None, []
            
            # Search for user
            search_filter = ldap_config.user_filter.format(username=username)
            conn.search(ldap_config.user_base_dn, search_filter)
            
            if not conn.entries:
                self.logger.info(f"User {username} not found in LDAP")
                return None, []
            
            user_dn = conn.entries[0].entry_dn
            
            # Authenticate user
            user_conn = ldap3.Connection(server, user_dn, password)
            if not user_conn.bind():
                self.logger.info(f"LDAP authentication failed for {username}")
                return None, []
            
            # Get user attributes
            user_entry = conn.entries[0]
            
            # Find or create local user
            user = self._find_user_by_username(username)
            if not user:
                user = await self._create_user_from_ldap(username, user_entry, ldap_config)
            else:
                # Update user info from LDAP
                await self._update_user_from_ldap(user, user_entry, ldap_config)
            
            user_conn.unbind()
            conn.unbind()
            
            return user, [AuthenticationMethod.LDAP_BIND]
            
        except Exception as e:
            self.logger.error(f"LDAP authentication error: {e}")
            return None, []

    async def _authenticate_saml_sso(self, provider: IdentityProvider, saml_response: Optional[str]) -> Tuple[Optional[User], List[AuthenticationMethod]]:
        """Authenticate via SAML SSO"""
        
        if not SAML_AVAILABLE:
            self.logger.error("SAML authentication requested but python3-saml not available")
            return None, []
        
        if not saml_response:
            # Return SSO URL for redirection
            return None, []
        
        try:
            # Process SAML response (simplified implementation)
            # In production, use proper SAML library with full validation
            
            # Decode and parse SAML response
            decoded_response = base64.b64decode(saml_response)
            root = ET.fromstring(decoded_response)
            
            # Extract user attributes (simplified)
            attributes = {}
            for attr in root.findall('.//{urn:oasis:names:tc:SAML:2.0:assertion}Attribute'):
                attr_name = attr.get('Name')
                attr_values = [val.text for val in attr.findall('.//{urn:oasis:names:tc:SAML:2.0:assertion}AttributeValue')]
                if attr_values:
                    attributes[attr_name] = attr_values[0] if len(attr_values) == 1 else attr_values
            
            # Map attributes to user fields
            username = attributes.get('email') or attributes.get('username')
            if not username:
                self.logger.error("No username found in SAML attributes")
                return None, []
            
            # Find or create user
            user = self._find_user_by_username(username)
            if not user:
                user = await self._create_user_from_saml(username, attributes, provider)
            else:
                await self._update_user_from_saml(user, attributes)
            
            return user, [AuthenticationMethod.SAML_SSO]
            
        except Exception as e:
            self.logger.error(f"SAML authentication error: {e}")
            return None, []

    async def _authenticate_oauth(self, provider: IdentityProvider, access_token: str) -> Tuple[Optional[User], List[AuthenticationMethod]]:
        """Authenticate via OAuth2/OIDC"""
        
        oauth_config = self.oauth_configs.get(provider.value)
        if not oauth_config:
            self.logger.error(f"OAuth config not found for {provider.value}")
            return None, []
        
        try:
            # Validate token and get user info (simplified)
            # In production, use proper OAuth library
            
            user_info_url = oauth_config.get('user_info_endpoint')
            headers = {'Authorization': f'Bearer {access_token}'}
            
            # This would make an actual HTTP request in production
            user_attributes = {
                'email': 'user@example.com',
                'first_name': 'User',
                'last_name': 'Name'
            }
            
            username = user_attributes.get('email')
            if not username:
                return None, []
            
            # Find or create user
            user = self._find_user_by_username(username)
            if not user:
                user = await self._create_user_from_oauth(username, user_attributes, provider)
            
            return user, [AuthenticationMethod.OAUTH2]
            
        except Exception as e:
            self.logger.error(f"OAuth authentication error: {e}")
            return None, []

    async def _validate_mfa(self, user: User, mfa_code: str) -> bool:
        """Validate MFA code"""
        
        if not TOTP_AVAILABLE or not user.mfa_secret:
            return False
        
        try:
            totp = pyotp.TOTP(user.mfa_secret)
            return totp.verify(mfa_code, valid_window=1)
        except Exception as e:
            self.logger.error(f"MFA validation error: {e}")
            return False

    async def _create_session(self, user: User, provider: IdentityProvider, 
                            auth_methods: List[AuthenticationMethod],
                            ip_address: str, user_agent: str) -> Session:
        """Create authenticated session"""
        
        with self.identity_lock:
            session_id = str(uuid.uuid4())
            now = datetime.utcnow()
            expires_at = now + timedelta(minutes=user.session_timeout_minutes)
            
            session = Session(
                session_id=session_id,
                user_id=user.user_id,
                provider=provider,
                authentication_methods=auth_methods,
                created_at=now,
                expires_at=expires_at,
                last_activity=now,
                ip_address=ip_address,
                user_agent=user_agent
            )
            
            self.sessions[session_id] = session
            
            # Clean up expired sessions
            await self._cleanup_expired_sessions(user.user_id)
            
            return session

    def _find_user_by_username(self, username: str) -> Optional[User]:
        """Find user by username or email"""
        for user in self.users.values():
            if user.username == username or user.email == username:
                return user
        return None

    async def _create_user_from_ldap(self, username: str, ldap_entry: Any, config: LDAPConfig) -> User:
        """Create user from LDAP attributes"""
        
        user_id = str(uuid.uuid4())
        
        # Map LDAP attributes
        mapping = config.attribute_mapping
        first_name = getattr(ldap_entry, mapping.get('first_name', 'givenName'), [''])[0]
        last_name = getattr(ldap_entry, mapping.get('last_name', 'sn'), [''])[0]
        email = getattr(ldap_entry, mapping.get('email', 'mail'), [username])[0]
        
        user = User(
            user_id=user_id,
            username=username,
            email=email,
            first_name=first_name,
            last_name=last_name,
            provider=IdentityProvider.LDAP,
            external_id=ldap_entry.entry_dn,
            roles=['user']
        )
        
        with self.identity_lock:
            self.users[user_id] = user
        
        return user

    async def _create_user_from_saml(self, username: str, attributes: Dict[str, Any], provider: IdentityProvider) -> User:
        """Create user from SAML attributes"""
        
        user_id = str(uuid.uuid4())
        
        user = User(
            user_id=user_id,
            username=username,
            email=attributes.get('email', username),
            first_name=attributes.get('firstName', ''),
            last_name=attributes.get('lastName', ''),
            provider=provider,
            external_id=attributes.get('nameID', username),
            provider_metadata=attributes,
            roles=['user']
        )
        
        with self.identity_lock:
            self.users[user_id] = user
        
        return user

    async def _create_user_from_oauth(self, username: str, attributes: Dict[str, Any], provider: IdentityProvider) -> User:
        """Create user from OAuth attributes"""
        
        user_id = str(uuid.uuid4())
        
        user = User(
            user_id=user_id,
            username=username,
            email=attributes.get('email', username),
            first_name=attributes.get('first_name', ''),
            last_name=attributes.get('last_name', ''),
            provider=provider,
            external_id=attributes.get('sub', username),
            provider_metadata=attributes,
            roles=['user']
        )
        
        with self.identity_lock:
            self.users[user_id] = user
        
        return user

    async def _update_user_from_ldap(self, user: User, ldap_entry: Any, config: LDAPConfig):
        """Update user from LDAP attributes"""
        mapping = config.attribute_mapping
        user.first_name = getattr(ldap_entry, mapping.get('first_name', 'givenName'), [user.first_name])[0]
        user.last_name = getattr(ldap_entry, mapping.get('last_name', 'sn'), [user.last_name])[0]
        user.updated_at = datetime.utcnow()

    async def _update_user_from_saml(self, user: User, attributes: Dict[str, Any]):
        """Update user from SAML attributes"""
        user.first_name = attributes.get('firstName', user.first_name)
        user.last_name = attributes.get('lastName', user.last_name)
        user.provider_metadata.update(attributes)
        user.updated_at = datetime.utcnow()

    def _generate_access_token(self, user: User, session: Session) -> str:
        """Generate JWT access token"""
        
        if not JWT_AVAILABLE:
            return secrets.token_urlsafe(32)
        
        payload = {
            'user_id': user.user_id,
            'username': user.username,
            'session_id': session.session_id,
            'roles': user.roles,
            'provider': user.provider.value,
            'iat': datetime.utcnow(),
            'exp': session.expires_at,
            'iss': 'tusklang-identity'
        }
        
        return jwt.encode(payload, self.jwt_secret, algorithm='HS256')

    def _generate_refresh_token(self, user: User, session: Session) -> str:
        """Generate refresh token"""
        return secrets.token_urlsafe(32)

    async def _get_user_permissions(self, user: User) -> List[str]:
        """Get all permissions for user (roles + direct permissions)"""
        
        permissions = set(user.direct_permissions)
        
        # Add permissions from roles
        for role_id in user.roles:
            role = self.roles.get(role_id)
            if role:
                permissions.update(role.permissions)
                
                # Handle role inheritance
                for inherited_role_id in role.inherits_from:
                    inherited_role = self.roles.get(inherited_role_id)
                    if inherited_role:
                        permissions.update(inherited_role.permissions)
        
        return list(permissions)

    async def authorize(self, user_id: str, resource: str, permission: str) -> bool:
        """Check if user is authorized to perform action on resource"""
        
        user = self.users.get(user_id)
        if not user or not user.is_active:
            return False
        
        user_permissions = await self._get_user_permissions(user)
        
        # Check for wildcard permission
        if '*' in user_permissions:
            return True
        
        # Check direct permission
        if permission in user_permissions:
            return True
        
        # Check resource-specific permissions
        for role_id in user.roles:
            role = self.roles.get(role_id)
            if role and permission in role.permissions:
                # Check if resource matches any patterns
                for pattern in role.resource_patterns:
                    if self._matches_resource_pattern(resource, pattern, user_id):
                        return True
        
        return False

    def _matches_resource_pattern(self, resource: str, pattern: str, user_id: str) -> bool:
        """Check if resource matches authorization pattern"""
        
        # Simple pattern matching (enhance for production)
        if pattern == "*":
            return True
        
        # Replace user_id placeholder
        pattern = pattern.replace("{user_id}", user_id)
        
        # Simple prefix matching
        if pattern.endswith("*"):
            return resource.startswith(pattern[:-1])
        
        return resource == pattern

    async def validate_session(self, session_id: str) -> Optional[Dict[str, Any]]:
        """Validate and refresh session"""
        
        session = self.sessions.get(session_id)
        if not session or not session.is_active:
            return None
        
        # Check expiration
        if datetime.utcnow() > session.expires_at:
            await self._invalidate_session(session_id)
            return None
        
        # Update last activity
        session.last_activity = datetime.utcnow()
        
        user = self.users.get(session.user_id)
        if not user or not user.is_active:
            await self._invalidate_session(session_id)
            return None
        
        return {
            'session_id': session.session_id,
            'user_id': session.user_id,
            'username': user.username,
            'roles': user.roles,
            'permissions': await self._get_user_permissions(user),
            'expires_at': session.expires_at,
            'provider': session.provider.value
        }

    async def _invalidate_session(self, session_id: str):
        """Invalidate session"""
        with self.identity_lock:
            if session_id in self.sessions:
                self.sessions[session_id].is_active = False

    async def _cleanup_expired_sessions(self, user_id: str):
        """Clean up expired sessions for user"""
        
        now = datetime.utcnow()
        expired_sessions = []
        
        for session_id, session in self.sessions.items():
            if session.user_id == user_id and now > session.expires_at:
                expired_sessions.append(session_id)
        
        with self.identity_lock:
            for session_id in expired_sessions:
                del self.sessions[session_id]

    async def logout(self, session_id: str) -> bool:
        """Logout user session"""
        
        session = self.sessions.get(session_id)
        if not session:
            return False
        
        await self._invalidate_session(session_id)
        
        with self.identity_lock:
            if session_id in self.sessions:
                del self.sessions[session_id]
        
        self.logger.info(f"User {session.user_id} logged out from session {session_id}")
        return True

    async def create_user(self, username: str, email: str, password: str,
                        first_name: str = "", last_name: str = "",
                        roles: List[str] = None) -> str:
        """Create new local user"""
        
        if self._find_user_by_username(username):
            raise ValueError("Username already exists")
        
        user_id = str(uuid.uuid4())
        password_hash = hashlib.sha256((password + username).encode()).hexdigest()
        
        user = User(
            user_id=user_id,
            username=username,
            email=email,
            first_name=first_name,
            last_name=last_name,
            password_hash=password_hash,
            roles=roles or ['user'],
            provider=IdentityProvider.LOCAL
        )
        
        with self.identity_lock:
            self.users[user_id] = user
        
        self.logger.info(f"Created user {username} ({user_id})")
        return user_id

    def create_role(self, name: str, description: str, 
                   permissions: List[str], resource_patterns: List[str] = None) -> str:
        """Create new role"""
        
        role_id = str(uuid.uuid4())
        
        role = Role(
            role_id=role_id,
            name=name,
            description=description,
            permissions=permissions,
            resource_patterns=resource_patterns or ["*"]
        )
        
        with self.identity_lock:
            self.roles[role_id] = role
        
        self.logger.info(f"Created role {name} ({role_id})")
        return role_id

    def get_sso_url(self, provider: IdentityProvider, return_url: str = "") -> str:
        """Get SSO login URL for provider"""
        
        if provider == IdentityProvider.AZURE_AD:
            # Generate Azure AD SSO URL
            base_url = "https://login.microsoftonline.com/common/oauth2/v2.0/authorize"
            params = {
                'response_type': 'code',
                'client_id': self.oauth_configs.get('azure_ad', {}).get('client_id', ''),
                'redirect_uri': return_url,
                'scope': 'openid profile email'
            }
            return f"{base_url}?{urllib.parse.urlencode(params)}"
        
        elif provider == IdentityProvider.OKTA:
            # Generate Okta SSO URL  
            okta_domain = self.oauth_configs.get('okta', {}).get('domain', '')
            return f"https://{okta_domain}/oauth2/default/v1/authorize"
        
        return ""

    def __del__(self):
        """Cleanup resources"""
        pass 