"""
TuskLang Python SDK - Multi-Tenant Architecture Manager
Production-quality multi-tenancy with isolation, resource management, and tenant provisioning
"""

import asyncio
import json
import logging
import uuid
import hashlib
import secrets
from datetime import datetime, timedelta, timezone
from typing import Dict, List, Optional, Any, Union, Tuple, Set
from dataclasses import dataclass, field
from enum import Enum
from pathlib import Path
import threading
from concurrent.futures import ThreadPoolExecutor
import time
import os

try:
    import cryptography.fernet
    from cryptography.hazmat.primitives import hashes
    from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
    CRYPTO_AVAILABLE = True
except ImportError:
    CRYPTO_AVAILABLE = False

try:
    import sqlite3
    SQLITE_AVAILABLE = True
except ImportError:
    SQLITE_AVAILABLE = False

try:
    import psutil
    PSUTIL_AVAILABLE = True
except ImportError:
    PSUTIL_AVAILABLE = False


class TenantStatus(Enum):
    ACTIVE = "active"
    INACTIVE = "inactive"
    SUSPENDED = "suspended"
    PROVISIONING = "provisioning"
    DEPROVISIONING = "deprovisioning"
    MAINTENANCE = "maintenance"
    DELETED = "deleted"


class TenantTier(Enum):
    BASIC = "basic"
    PROFESSIONAL = "professional"
    ENTERPRISE = "enterprise"
    PREMIUM = "premium"


class IsolationLevel(Enum):
    DATABASE_PER_TENANT = "database_per_tenant"
    SCHEMA_PER_TENANT = "schema_per_tenant"
    ROW_LEVEL_SECURITY = "row_level_security"
    APPLICATION_LEVEL = "application_level"


class ResourceType(Enum):
    CPU_CORES = "cpu_cores"
    MEMORY_MB = "memory_mb"
    STORAGE_GB = "storage_gb"
    NETWORK_MBPS = "network_mbps"
    DATABASE_CONNECTIONS = "database_connections"
    API_REQUESTS_PER_HOUR = "api_requests_per_hour"
    CONCURRENT_USERS = "concurrent_users"


@dataclass
class ResourceQuota:
    """Resource quota definition for tenant"""
    resource_type: ResourceType
    allocated: float
    consumed: float = 0.0
    reserved: float = 0.0
    soft_limit: float = 0.9  # 90% warning threshold
    hard_limit: float = 1.0  # 100% enforcement threshold
    burst_allowed: bool = False
    burst_limit: float = 1.2  # 120% burst capacity
    monitoring_enabled: bool = True
    alert_threshold: float = 0.8  # 80% alert threshold


@dataclass
class TenantConfiguration:
    """Tenant-specific configuration settings"""
    tenant_id: str
    custom_domain: Optional[str] = None
    branding_config: Dict[str, Any] = field(default_factory=dict)
    feature_flags: Dict[str, bool] = field(default_factory=dict)
    security_policies: Dict[str, Any] = field(default_factory=dict)
    integration_settings: Dict[str, Any] = field(default_factory=dict)
    notification_preferences: Dict[str, Any] = field(default_factory=dict)
    backup_retention_days: int = 30
    data_residency_region: str = "us-east-1"
    compliance_requirements: List[str] = field(default_factory=list)
    custom_workflows: Dict[str, Any] = field(default_factory=dict)


@dataclass
class Tenant:
    """Multi-tenant entity with complete isolation and configuration"""
    tenant_id: str
    name: str
    status: TenantStatus
    tier: TenantTier
    isolation_level: IsolationLevel
    
    # Basic information
    display_name: str
    description: str = ""
    contact_email: str = ""
    contact_phone: str = ""
    
    # Dates and tracking
    created_at: datetime = field(default_factory=datetime.utcnow)
    updated_at: datetime = field(default_factory=datetime.utcnow)
    activated_at: Optional[datetime] = None
    suspended_at: Optional[datetime] = None
    deleted_at: Optional[datetime] = None
    
    # Resource management
    resource_quotas: Dict[ResourceType, ResourceQuota] = field(default_factory=dict)
    current_usage: Dict[ResourceType, float] = field(default_factory=dict)
    
    # Security and isolation
    encryption_key: str = ""
    database_connection_string: str = ""
    schema_name: str = ""
    security_context: Dict[str, Any] = field(default_factory=dict)
    
    # Configuration
    configuration: TenantConfiguration = field(default_factory=lambda: TenantConfiguration(""))
    
    # Billing and metrics
    billing_account_id: str = ""
    subscription_id: str = ""
    billing_cycle_start: Optional[datetime] = None
    total_users: int = 0
    active_users: int = 0
    storage_used_gb: float = 0.0
    bandwidth_used_gb: float = 0.0


@dataclass
class TenantProvisioningTask:
    """Tenant provisioning task tracking"""
    task_id: str
    tenant_id: str
    task_type: str  # provision, deprovision, upgrade, downgrade
    status: str
    progress_percentage: float = 0.0
    steps_completed: List[str] = field(default_factory=list)
    steps_remaining: List[str] = field(default_factory=list)
    error_messages: List[str] = field(default_factory=list)
    started_at: datetime = field(default_factory=datetime.utcnow)
    completed_at: Optional[datetime] = None
    estimated_completion: Optional[datetime] = None


@dataclass
class UsageMetrics:
    """Tenant usage metrics for billing and monitoring"""
    tenant_id: str
    timestamp: datetime
    resource_usage: Dict[ResourceType, float]
    active_sessions: int
    api_calls: int
    data_transferred_gb: float
    features_used: Set[str]
    performance_metrics: Dict[str, float]  # latency, throughput, etc.


class MultiTenantManager:
    """Production-quality multi-tenant architecture with complete isolation and management"""
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.logger = logging.getLogger(__name__)
        self.db_path = self.config.get('database_path', 'tenants.db')
        
        # Encryption for tenant data
        self.master_key = self._generate_master_key()
        self.cipher = None
        if CRYPTO_AVAILABLE:
            self.cipher = cryptography.fernet.Fernet(self.master_key)
        
        # In-memory tenant registry
        self.tenants: Dict[str, Tenant] = {}
        self.provisioning_tasks: Dict[str, TenantProvisioningTask] = {}
        self.usage_metrics: List[UsageMetrics] = []
        
        # Threading for resource monitoring
        self.executor = ThreadPoolExecutor(max_workers=20)
        self.monitoring_active = True
        self.tenant_lock = threading.RLock()
        
        # Resource monitoring thread
        self.monitoring_thread = threading.Thread(target=self._resource_monitoring_loop, daemon=True)
        
        # Initialize system
        self._initialize_database()
        self._load_tenants_from_database()
        self._start_monitoring()
        
        self.logger.info("Multi-Tenant Manager initialized successfully")

    def _generate_master_key(self) -> bytes:
        """Generate or load master encryption key"""
        key_file = Path(self.config.get('master_key_file', 'tenant_master.key'))
        
        if key_file.exists():
            with open(key_file, 'rb') as f:
                return f.read()
        
        # Generate new master key
        if CRYPTO_AVAILABLE:
            key = cryptography.fernet.Fernet.generate_key()
        else:
            key = secrets.token_bytes(32)
        
        # Secure key storage
        with open(key_file, 'wb') as f:
            f.write(key)
        os.chmod(key_file, 0o600)  # Restrict to owner only
        
        return key

    def _initialize_database(self):
        """Initialize database schema for multi-tenant management"""
        if not SQLITE_AVAILABLE:
            self.logger.warning("SQLite not available, using in-memory storage")
            return
        
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.executescript("""
                CREATE TABLE IF NOT EXISTS tenants (
                    tenant_id TEXT PRIMARY KEY,
                    name TEXT NOT NULL,
                    status TEXT NOT NULL,
                    tier TEXT NOT NULL,
                    isolation_level TEXT NOT NULL,
                    display_name TEXT NOT NULL,
                    description TEXT,
                    contact_email TEXT,
                    contact_phone TEXT,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    activated_at TIMESTAMP,
                    suspended_at TIMESTAMP,
                    deleted_at TIMESTAMP,
                    encryption_key TEXT,
                    database_connection_string TEXT,
                    schema_name TEXT,
                    security_context TEXT,
                    configuration TEXT,
                    billing_account_id TEXT,
                    subscription_id TEXT,
                    billing_cycle_start TIMESTAMP,
                    total_users INTEGER DEFAULT 0,
                    active_users INTEGER DEFAULT 0,
                    storage_used_gb REAL DEFAULT 0.0,
                    bandwidth_used_gb REAL DEFAULT 0.0
                );
                
                CREATE TABLE IF NOT EXISTS resource_quotas (
                    quota_id TEXT PRIMARY KEY,
                    tenant_id TEXT NOT NULL,
                    resource_type TEXT NOT NULL,
                    allocated REAL NOT NULL,
                    consumed REAL DEFAULT 0.0,
                    reserved REAL DEFAULT 0.0,
                    soft_limit REAL DEFAULT 0.9,
                    hard_limit REAL DEFAULT 1.0,
                    burst_allowed INTEGER DEFAULT 0,
                    burst_limit REAL DEFAULT 1.2,
                    monitoring_enabled INTEGER DEFAULT 1,
                    alert_threshold REAL DEFAULT 0.8,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (tenant_id) REFERENCES tenants(tenant_id)
                );
                
                CREATE TABLE IF NOT EXISTS provisioning_tasks (
                    task_id TEXT PRIMARY KEY,
                    tenant_id TEXT NOT NULL,
                    task_type TEXT NOT NULL,
                    status TEXT NOT NULL,
                    progress_percentage REAL DEFAULT 0.0,
                    steps_completed TEXT,
                    steps_remaining TEXT,
                    error_messages TEXT,
                    started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    completed_at TIMESTAMP,
                    estimated_completion TIMESTAMP,
                    FOREIGN KEY (tenant_id) REFERENCES tenants(tenant_id)
                );
                
                CREATE TABLE IF NOT EXISTS usage_metrics (
                    metric_id TEXT PRIMARY KEY,
                    tenant_id TEXT NOT NULL,
                    timestamp TIMESTAMP NOT NULL,
                    resource_usage TEXT,
                    active_sessions INTEGER,
                    api_calls INTEGER,
                    data_transferred_gb REAL,
                    features_used TEXT,
                    performance_metrics TEXT,
                    FOREIGN KEY (tenant_id) REFERENCES tenants(tenant_id)
                );
                
                CREATE INDEX IF NOT EXISTS idx_tenants_status ON tenants(status);
                CREATE INDEX IF NOT EXISTS idx_tenants_tier ON tenants(tier);
                CREATE INDEX IF NOT EXISTS idx_quotas_tenant ON resource_quotas(tenant_id);
                CREATE INDEX IF NOT EXISTS idx_tasks_tenant ON provisioning_tasks(tenant_id);
                CREATE INDEX IF NOT EXISTS idx_metrics_tenant_timestamp ON usage_metrics(tenant_id, timestamp);
            """)
            
            conn.commit()
            conn.close()
            self.logger.info("Multi-tenant database initialized successfully")
            
        except Exception as e:
            self.logger.error(f"Database initialization failed: {e}")

    def _load_tenants_from_database(self):
        """Load existing tenants from database"""
        if not SQLITE_AVAILABLE:
            return
        
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute("SELECT * FROM tenants")
            rows = cursor.fetchall()
            
            for row in rows:
                tenant = self._row_to_tenant(row)
                if tenant:
                    self.tenants[tenant.tenant_id] = tenant
            
            # Load resource quotas
            for tenant_id in self.tenants:
                self._load_tenant_quotas(tenant_id)
            
            conn.close()
            self.logger.info(f"Loaded {len(self.tenants)} tenants from database")
            
        except Exception as e:
            self.logger.error(f"Failed to load tenants: {e}")

    def _row_to_tenant(self, row) -> Optional[Tenant]:
        """Convert database row to Tenant object"""
        try:
            tenant = Tenant(
                tenant_id=row[0],
                name=row[1],
                status=TenantStatus(row[2]),
                tier=TenantTier(row[3]),
                isolation_level=IsolationLevel(row[4]),
                display_name=row[5],
                description=row[6] or "",
                contact_email=row[7] or "",
                contact_phone=row[8] or "",
                encryption_key=row[15] or "",
                database_connection_string=row[16] or "",
                schema_name=row[17] or "",
                billing_account_id=row[20] or "",
                subscription_id=row[21] or "",
                total_users=row[23] or 0,
                active_users=row[24] or 0,
                storage_used_gb=row[25] or 0.0,
                bandwidth_used_gb=row[26] or 0.0
            )
            
            # Parse JSON fields
            if row[18]:  # security_context
                tenant.security_context = json.loads(row[18])
            if row[19]:  # configuration
                config_data = json.loads(row[19])
                tenant.configuration = TenantConfiguration(
                    tenant_id=tenant.tenant_id,
                    **config_data
                )
            
            return tenant
        except Exception as e:
            self.logger.error(f"Error parsing tenant row: {e}")
            return None

    def _load_tenant_quotas(self, tenant_id: str):
        """Load resource quotas for a tenant"""
        if not SQLITE_AVAILABLE:
            return
        
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute("SELECT * FROM resource_quotas WHERE tenant_id = ?", (tenant_id,))
            rows = cursor.fetchall()
            
            tenant = self.tenants.get(tenant_id)
            if not tenant:
                return
            
            for row in rows:
                quota = ResourceQuota(
                    resource_type=ResourceType(row[2]),
                    allocated=row[3],
                    consumed=row[4],
                    reserved=row[5],
                    soft_limit=row[6],
                    hard_limit=row[7],
                    burst_allowed=bool(row[8]),
                    burst_limit=row[9],
                    monitoring_enabled=bool(row[10]),
                    alert_threshold=row[11]
                )
                tenant.resource_quotas[quota.resource_type] = quota
            
            conn.close()
            
        except Exception as e:
            self.logger.error(f"Failed to load quotas for tenant {tenant_id}: {e}")

    def _start_monitoring(self):
        """Start background resource monitoring"""
        if self.monitoring_thread and not self.monitoring_thread.is_alive():
            self.monitoring_thread.start()
            self.logger.info("Resource monitoring started")

    def _resource_monitoring_loop(self):
        """Background thread for monitoring tenant resource usage"""
        while self.monitoring_active:
            try:
                for tenant_id in list(self.tenants.keys()):
                    self._update_tenant_usage(tenant_id)
                
                # Check for quota violations
                self._check_quota_violations()
                
                # Sleep for monitoring interval
                time.sleep(self.config.get('monitoring_interval_seconds', 60))
                
            except Exception as e:
                self.logger.error(f"Error in resource monitoring: {e}")
                time.sleep(30)  # Shorter sleep on error

    def _update_tenant_usage(self, tenant_id: str):
        """Update current resource usage for tenant"""
        with self.tenant_lock:
            tenant = self.tenants.get(tenant_id)
            if not tenant or tenant.status != TenantStatus.ACTIVE:
                return
            
            # Simulate resource usage updates (in real implementation, gather from monitoring systems)
            if PSUTIL_AVAILABLE:
                tenant.current_usage[ResourceType.CPU_CORES] = psutil.cpu_percent() / 100.0
                tenant.current_usage[ResourceType.MEMORY_MB] = psutil.virtual_memory().used / (1024 * 1024)
                tenant.current_usage[ResourceType.STORAGE_GB] = psutil.disk_usage('/').used / (1024 * 1024 * 1024)
            
            # Update consumed quotas
            for resource_type, quota in tenant.resource_quotas.items():
                current_usage = tenant.current_usage.get(resource_type, 0.0)
                quota.consumed = min(current_usage, quota.allocated * quota.hard_limit)

    def _check_quota_violations(self):
        """Check for and handle quota violations"""
        for tenant in self.tenants.values():
            if tenant.status != TenantStatus.ACTIVE:
                continue
            
            for resource_type, quota in tenant.resource_quotas.items():
                usage_percentage = quota.consumed / quota.allocated if quota.allocated > 0 else 0
                
                # Check for violations
                if usage_percentage >= quota.hard_limit and not quota.burst_allowed:
                    self._handle_quota_violation(tenant, resource_type, usage_percentage, "hard_limit_exceeded")
                elif usage_percentage >= quota.alert_threshold:
                    self._handle_quota_violation(tenant, resource_type, usage_percentage, "alert_threshold_reached")

    def _handle_quota_violation(self, tenant: Tenant, resource_type: ResourceType, 
                              usage_percentage: float, violation_type: str):
        """Handle quota violation"""
        self.logger.warning(
            f"Quota violation for tenant {tenant.tenant_id}: "
            f"{resource_type.value} at {usage_percentage:.1%} - {violation_type}"
        )
        
        # Implement violation handling logic (throttling, notifications, etc.)
        # This would integrate with the compliance framework for audit logging

    async def create_tenant(self, name: str, tier: TenantTier, 
                          isolation_level: IsolationLevel,
                          contact_email: str = "",
                          configuration: Optional[Dict[str, Any]] = None) -> str:
        """Create new tenant with complete provisioning"""
        
        tenant_id = str(uuid.uuid4())
        task_id = str(uuid.uuid4())
        
        # Create tenant object
        tenant = Tenant(
            tenant_id=tenant_id,
            name=name,
            status=TenantStatus.PROVISIONING,
            tier=tier,
            isolation_level=isolation_level,
            display_name=name,
            contact_email=contact_email,
            encryption_key=self._generate_tenant_encryption_key(),
            schema_name=f"tenant_{tenant_id.replace('-', '_')}",
            configuration=TenantConfiguration(tenant_id, **(configuration or {}))
        )
        
        # Set default resource quotas based on tier
        tenant.resource_quotas = self._get_default_quotas(tier)
        
        # Create provisioning task
        provisioning_task = TenantProvisioningTask(
            task_id=task_id,
            tenant_id=tenant_id,
            task_type="provision",
            status="in_progress",
            steps_remaining=[
                "create_database_resources",
                "setup_security_context", 
                "configure_isolation",
                "apply_resource_quotas",
                "validate_tenant_setup",
                "activate_tenant"
            ]
        )
        
        with self.tenant_lock:
            self.tenants[tenant_id] = tenant
            self.provisioning_tasks[task_id] = provisioning_task
        
        # Start async provisioning
        self.executor.submit(self._provision_tenant_async, tenant_id, task_id)
        
        self.logger.info(f"Tenant creation initiated: {tenant_id} ({name})")
        return tenant_id

    def _generate_tenant_encryption_key(self) -> str:
        """Generate unique encryption key for tenant"""
        if CRYPTO_AVAILABLE:
            return cryptography.fernet.Fernet.generate_key().decode()
        else:
            return secrets.token_hex(32)

    def _get_default_quotas(self, tier: TenantTier) -> Dict[ResourceType, ResourceQuota]:
        """Get default resource quotas based on tenant tier"""
        
        quota_configs = {
            TenantTier.BASIC: {
                ResourceType.CPU_CORES: 2.0,
                ResourceType.MEMORY_MB: 4096,
                ResourceType.STORAGE_GB: 10,
                ResourceType.API_REQUESTS_PER_HOUR: 1000,
                ResourceType.CONCURRENT_USERS: 10,
                ResourceType.DATABASE_CONNECTIONS: 5
            },
            TenantTier.PROFESSIONAL: {
                ResourceType.CPU_CORES: 4.0,
                ResourceType.MEMORY_MB: 8192,
                ResourceType.STORAGE_GB: 50,
                ResourceType.API_REQUESTS_PER_HOUR: 10000,
                ResourceType.CONCURRENT_USERS: 100,
                ResourceType.DATABASE_CONNECTIONS: 20
            },
            TenantTier.ENTERPRISE: {
                ResourceType.CPU_CORES: 8.0,
                ResourceType.MEMORY_MB: 16384,
                ResourceType.STORAGE_GB: 200,
                ResourceType.API_REQUESTS_PER_HOUR: 100000,
                ResourceType.CONCURRENT_USERS: 1000,
                ResourceType.DATABASE_CONNECTIONS: 50
            },
            TenantTier.PREMIUM: {
                ResourceType.CPU_CORES: 16.0,
                ResourceType.MEMORY_MB: 32768,
                ResourceType.STORAGE_GB: 500,
                ResourceType.API_REQUESTS_PER_HOUR: 1000000,
                ResourceType.CONCURRENT_USERS: 10000,
                ResourceType.DATABASE_CONNECTIONS: 100
            }
        }
        
        quotas = {}
        for resource_type, allocated in quota_configs.get(tier, {}).items():
            quotas[resource_type] = ResourceQuota(
                resource_type=resource_type,
                allocated=allocated,
                burst_allowed=(tier in [TenantTier.ENTERPRISE, TenantTier.PREMIUM])
            )
        
        return quotas

    def _provision_tenant_async(self, tenant_id: str, task_id: str):
        """Asynchronously provision tenant resources"""
        
        tenant = self.tenants.get(tenant_id)
        task = self.provisioning_tasks.get(task_id)
        
        if not tenant or not task:
            return
        
        try:
            total_steps = len(task.steps_remaining)
            
            for i, step in enumerate(task.steps_remaining.copy()):
                self.logger.info(f"Provisioning {tenant_id}: {step}")
                
                # Execute provisioning step
                success = self._execute_provisioning_step(tenant, step)
                
                if success:
                    task.steps_completed.append(step)
                    task.steps_remaining.remove(step)
                    task.progress_percentage = (i + 1) / total_steps * 100
                else:
                    task.error_messages.append(f"Failed to execute step: {step}")
                    task.status = "failed"
                    tenant.status = TenantStatus.INACTIVE
                    return
                
                # Small delay between steps
                time.sleep(1)
            
            # Provisioning completed successfully
            task.status = "completed"
            task.completed_at = datetime.utcnow()
            task.progress_percentage = 100.0
            
            tenant.status = TenantStatus.ACTIVE
            tenant.activated_at = datetime.utcnow()
            
            # Persist tenant to database
            self._persist_tenant(tenant)
            
            self.logger.info(f"Tenant {tenant_id} provisioned successfully")
            
        except Exception as e:
            self.logger.error(f"Tenant provisioning failed for {tenant_id}: {e}")
            task.status = "failed"
            task.error_messages.append(str(e))
            tenant.status = TenantStatus.INACTIVE

    def _execute_provisioning_step(self, tenant: Tenant, step: str) -> bool:
        """Execute individual provisioning step"""
        
        try:
            if step == "create_database_resources":
                # Create tenant-specific database resources
                if tenant.isolation_level == IsolationLevel.DATABASE_PER_TENANT:
                    # Create separate database
                    db_name = f"tenant_{tenant.tenant_id.replace('-', '_')}"
                    tenant.database_connection_string = f"sqlite:///{db_name}.db"
                elif tenant.isolation_level == IsolationLevel.SCHEMA_PER_TENANT:
                    # Create schema in shared database
                    tenant.database_connection_string = f"sqlite:///shared.db"
                    # Schema name already set in tenant creation
                
                return True
                
            elif step == "setup_security_context":
                # Setup security context and encryption
                tenant.security_context = {
                    'encryption_enabled': True,
                    'key_rotation_days': 90,
                    'access_logging': True,
                    'data_masking': tenant.tier in [TenantTier.ENTERPRISE, TenantTier.PREMIUM]
                }
                return True
                
            elif step == "configure_isolation":
                # Configure isolation mechanisms
                isolation_config = {
                    'level': tenant.isolation_level.value,
                    'network_isolation': tenant.tier in [TenantTier.ENTERPRISE, TenantTier.PREMIUM],
                    'compute_isolation': True,
                    'data_encryption': True
                }
                tenant.security_context.update(isolation_config)
                return True
                
            elif step == "apply_resource_quotas":
                # Apply resource quotas at system level
                for resource_type, quota in tenant.resource_quotas.items():
                    self.logger.debug(f"Applied quota for {resource_type.value}: {quota.allocated}")
                return True
                
            elif step == "validate_tenant_setup":
                # Validate tenant setup
                if not tenant.encryption_key or not tenant.schema_name:
                    return False
                if not tenant.resource_quotas:
                    return False
                return True
                
            elif step == "activate_tenant":
                # Final activation
                tenant.status = TenantStatus.ACTIVE
                return True
                
            else:
                self.logger.warning(f"Unknown provisioning step: {step}")
                return False
                
        except Exception as e:
            self.logger.error(f"Error executing step {step}: {e}")
            return False

    def _persist_tenant(self, tenant: Tenant):
        """Persist tenant to database"""
        if not SQLITE_AVAILABLE:
            return
        
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute("""
                INSERT OR REPLACE INTO tenants (
                    tenant_id, name, status, tier, isolation_level, display_name,
                    description, contact_email, contact_phone, created_at, updated_at,
                    activated_at, suspended_at, deleted_at, encryption_key,
                    database_connection_string, schema_name, security_context,
                    configuration, billing_account_id, subscription_id,
                    billing_cycle_start, total_users, active_users,
                    storage_used_gb, bandwidth_used_gb
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            """, (
                tenant.tenant_id, tenant.name, tenant.status.value, tenant.tier.value,
                tenant.isolation_level.value, tenant.display_name, tenant.description,
                tenant.contact_email, tenant.contact_phone, tenant.created_at, tenant.updated_at,
                tenant.activated_at, tenant.suspended_at, tenant.deleted_at,
                tenant.encryption_key, tenant.database_connection_string, tenant.schema_name,
                json.dumps(tenant.security_context), json.dumps(tenant.configuration.__dict__),
                tenant.billing_account_id, tenant.subscription_id, tenant.billing_cycle_start,
                tenant.total_users, tenant.active_users, tenant.storage_used_gb, tenant.bandwidth_used_gb
            ))
            
            # Persist resource quotas
            for resource_type, quota in tenant.resource_quotas.items():
                cursor.execute("""
                    INSERT OR REPLACE INTO resource_quotas (
                        quota_id, tenant_id, resource_type, allocated, consumed, reserved,
                        soft_limit, hard_limit, burst_allowed, burst_limit,
                        monitoring_enabled, alert_threshold
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                """, (
                    f"{tenant.tenant_id}_{resource_type.value}", tenant.tenant_id,
                    resource_type.value, quota.allocated, quota.consumed, quota.reserved,
                    quota.soft_limit, quota.hard_limit, int(quota.burst_allowed),
                    quota.burst_limit, int(quota.monitoring_enabled), quota.alert_threshold
                ))
            
            conn.commit()
            conn.close()
            
        except Exception as e:
            self.logger.error(f"Failed to persist tenant {tenant.tenant_id}: {e}")

    async def get_tenant(self, tenant_id: str) -> Optional[Tenant]:
        """Get tenant by ID"""
        with self.tenant_lock:
            return self.tenants.get(tenant_id)

    async def list_tenants(self, status: Optional[TenantStatus] = None,
                          tier: Optional[TenantTier] = None) -> List[Tenant]:
        """List tenants with optional filtering"""
        with self.tenant_lock:
            tenants = list(self.tenants.values())
        
        if status:
            tenants = [t for t in tenants if t.status == status]
        
        if tier:
            tenants = [t for t in tenants if t.tier == tier]
        
        return tenants

    async def update_tenant_tier(self, tenant_id: str, new_tier: TenantTier) -> bool:
        """Update tenant tier and adjust quotas accordingly"""
        
        tenant = await self.get_tenant(tenant_id)
        if not tenant:
            return False
        
        old_tier = tenant.tier
        tenant.tier = new_tier
        tenant.updated_at = datetime.utcnow()
        
        # Update resource quotas
        new_quotas = self._get_default_quotas(new_tier)
        tenant.resource_quotas.update(new_quotas)
        
        # Persist changes
        self._persist_tenant(tenant)
        
        self.logger.info(f"Tenant {tenant_id} tier updated from {old_tier.value} to {new_tier.value}")
        return True

    async def suspend_tenant(self, tenant_id: str, reason: str = "") -> bool:
        """Suspend tenant operations"""
        
        tenant = await self.get_tenant(tenant_id)
        if not tenant or tenant.status == TenantStatus.SUSPENDED:
            return False
        
        tenant.status = TenantStatus.SUSPENDED
        tenant.suspended_at = datetime.utcnow()
        tenant.updated_at = datetime.utcnow()
        
        # Implement suspension logic (disable access, stop services, etc.)
        
        self._persist_tenant(tenant)
        
        self.logger.info(f"Tenant {tenant_id} suspended: {reason}")
        return True

    async def reactivate_tenant(self, tenant_id: str) -> bool:
        """Reactivate suspended tenant"""
        
        tenant = await self.get_tenant(tenant_id)
        if not tenant or tenant.status != TenantStatus.SUSPENDED:
            return False
        
        tenant.status = TenantStatus.ACTIVE
        tenant.suspended_at = None
        tenant.updated_at = datetime.utcnow()
        
        # Implement reactivation logic (restore access, restart services, etc.)
        
        self._persist_tenant(tenant)
        
        self.logger.info(f"Tenant {tenant_id} reactivated")
        return True

    async def delete_tenant(self, tenant_id: str, permanent: bool = False) -> bool:
        """Delete tenant (soft delete by default)"""
        
        tenant = await self.get_tenant(tenant_id)
        if not tenant:
            return False
        
        if permanent:
            # Permanent deletion - remove all data
            with self.tenant_lock:
                del self.tenants[tenant_id]
            
            # Remove from database
            if SQLITE_AVAILABLE:
                conn = sqlite3.connect(self.db_path)
                cursor = conn.cursor()
                cursor.execute("DELETE FROM tenants WHERE tenant_id = ?", (tenant_id,))
                cursor.execute("DELETE FROM resource_quotas WHERE tenant_id = ?", (tenant_id,))
                cursor.execute("DELETE FROM usage_metrics WHERE tenant_id = ?", (tenant_id,))
                conn.commit()
                conn.close()
            
            self.logger.info(f"Tenant {tenant_id} permanently deleted")
        else:
            # Soft delete
            tenant.status = TenantStatus.DELETED
            tenant.deleted_at = datetime.utcnow()
            tenant.updated_at = datetime.utcnow()
            
            self._persist_tenant(tenant)
            
            self.logger.info(f"Tenant {tenant_id} soft deleted")
        
        return True

    async def get_tenant_usage_report(self, tenant_id: str, 
                                    start_date: Optional[datetime] = None,
                                    end_date: Optional[datetime] = None) -> Dict[str, Any]:
        """Generate comprehensive tenant usage report"""
        
        tenant = await self.get_tenant(tenant_id)
        if not tenant:
            return {}
        
        if not start_date:
            start_date = datetime.utcnow() - timedelta(days=30)
        if not end_date:
            end_date = datetime.utcnow()
        
        # Calculate usage statistics
        usage_report = {
            'tenant_id': tenant_id,
            'tenant_name': tenant.name,
            'report_period': {
                'start': start_date.isoformat(),
                'end': end_date.isoformat()
            },
            'resource_usage': {},
            'billing_summary': {
                'total_cost': 0.0,
                'resource_costs': {}
            },
            'performance_metrics': {
                'avg_response_time': 0.0,
                'uptime_percentage': 99.9,
                'error_rate': 0.1
            },
            'recommendations': []
        }
        
        # Resource usage analysis
        for resource_type, quota in tenant.resource_quotas.items():
            usage_percentage = (quota.consumed / quota.allocated * 100) if quota.allocated > 0 else 0
            
            usage_report['resource_usage'][resource_type.value] = {
                'allocated': quota.allocated,
                'consumed': quota.consumed,
                'usage_percentage': usage_percentage,
                'quota_violations': 0,  # Would track from monitoring data
                'peak_usage': quota.consumed * 1.2  # Simulated peak
            }
            
            # Generate recommendations
            if usage_percentage > 80:
                usage_report['recommendations'].append(
                    f"Consider upgrading {resource_type.value} allocation - currently at {usage_percentage:.1f}%"
                )
            elif usage_percentage < 20:
                usage_report['recommendations'].append(
                    f"Consider reducing {resource_type.value} allocation to optimize costs - only using {usage_percentage:.1f}%"
                )
        
        return usage_report

    async def get_provisioning_status(self, task_id: str) -> Optional[Dict[str, Any]]:
        """Get provisioning task status"""
        
        task = self.provisioning_tasks.get(task_id)
        if not task:
            return None
        
        return {
            'task_id': task.task_id,
            'tenant_id': task.tenant_id,
            'task_type': task.task_type,
            'status': task.status,
            'progress_percentage': task.progress_percentage,
            'steps_completed': task.steps_completed,
            'steps_remaining': task.steps_remaining,
            'error_messages': task.error_messages,
            'started_at': task.started_at.isoformat(),
            'completed_at': task.completed_at.isoformat() if task.completed_at else None,
            'estimated_completion': task.estimated_completion.isoformat() if task.estimated_completion else None
        }

    def create_tenant_context(self, tenant_id: str) -> Dict[str, Any]:
        """Create security context for tenant operations"""
        
        tenant = self.tenants.get(tenant_id)
        if not tenant:
            return {}
        
        context = {
            'tenant_id': tenant_id,
            'isolation_level': tenant.isolation_level.value,
            'encryption_key': tenant.encryption_key,
            'database_connection': tenant.database_connection_string,
            'schema_name': tenant.schema_name,
            'security_policies': tenant.security_context,
            'resource_limits': {
                rt.value: quota.allocated 
                for rt, quota in tenant.resource_quotas.items()
            },
            'timestamp': datetime.utcnow().isoformat()
        }
        
        return context

    def __del__(self):
        """Cleanup resources"""
        self.monitoring_active = False
        if hasattr(self, 'monitoring_thread') and self.monitoring_thread.is_alive():
            self.monitoring_thread.join(timeout=5)
        if hasattr(self, 'executor'):
            self.executor.shutdown(wait=False) 