"""
TuskLang Python SDK - Database Migration Manager (g8.3)
Automated migrations with versioning and rollback capabilities
"""

import os
import re
import json
import logging
import hashlib
from datetime import datetime
from typing import Dict, List, Optional, Tuple, Any
from dataclasses import dataclass
from pathlib import Path
from abc import ABC, abstractmethod


@dataclass
class Migration:
    """Represents a database migration"""
    version: str
    name: str
    up_sql: str
    down_sql: str
    checksum: str
    created_at: datetime
    applied_at: Optional[datetime] = None
    
    def generate_checksum(self) -> str:
        """Generate checksum for migration integrity"""
        content = f"{self.version}{self.name}{self.up_sql}{self.down_sql}"
        return hashlib.sha256(content.encode()).hexdigest()


@dataclass
class MigrationResult:
    """Result of migration operation"""
    success: bool
    version: str
    message: str
    execution_time: float
    error: Optional[Exception] = None


class MigrationError(Exception):
    """Migration-specific exception"""
    pass


class MigrationRunner(ABC):
    """Abstract migration runner for different database engines"""
    
    @abstractmethod
    def execute_sql(self, sql: str) -> None:
        """Execute SQL statement"""
        pass
    
    @abstractmethod
    def create_migration_table(self) -> None:
        """Create migration tracking table"""
        pass
    
    @abstractmethod
    def get_applied_migrations(self) -> List[str]:
        """Get list of applied migration versions"""
        pass
    
    @abstractmethod
    def record_migration(self, migration: Migration) -> None:
        """Record applied migration"""
        pass
    
    @abstractmethod
    def remove_migration_record(self, version: str) -> None:
        """Remove migration record (for rollback)"""
        pass


class PostgreSQLMigrationRunner(MigrationRunner):
    """PostgreSQL migration runner"""
    
    def __init__(self, db_adapter):
        self.db = db_adapter
    
    def execute_sql(self, sql: str) -> None:
        """Execute SQL statement"""
        self.db.execute_command(sql)
    
    def create_migration_table(self) -> None:
        """Create schema_migrations table"""
        sql = """
        CREATE TABLE IF NOT EXISTS schema_migrations (
            version VARCHAR(255) PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            checksum VARCHAR(64) NOT NULL,
            applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            execution_time FLOAT DEFAULT 0,
            INDEX idx_applied_at (applied_at)
        );
        """
        self.execute_sql(sql)
    
    def get_applied_migrations(self) -> List[str]:
        """Get applied migrations"""
        try:
            results = self.db.execute_query(
                "SELECT version FROM schema_migrations ORDER BY version"
            )
            return [row['version'] for row in results]
        except:
            return []
    
    def record_migration(self, migration: Migration) -> None:
        """Record migration as applied"""
        sql = """
        INSERT INTO schema_migrations (version, name, checksum, applied_at)
        VALUES (%s, %s, %s, %s)
        """
        self.db.execute_command(sql, (
            migration.version,
            migration.name,
            migration.checksum,
            migration.applied_at or datetime.now()
        ))
    
    def remove_migration_record(self, version: str) -> None:
        """Remove migration record"""
        self.db.execute_command(
            "DELETE FROM schema_migrations WHERE version = %s",
            (version,)
        )


class MySQLMigrationRunner(MigrationRunner):
    """MySQL migration runner"""
    
    def __init__(self, db_adapter):
        self.db = db_adapter
    
    def execute_sql(self, sql: str) -> None:
        """Execute SQL statement"""
        self.db.execute_command(sql)
    
    def create_migration_table(self) -> None:
        """Create schema_migrations table"""
        sql = """
        CREATE TABLE IF NOT EXISTS schema_migrations (
            version VARCHAR(255) PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            checksum VARCHAR(64) NOT NULL,
            applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            execution_time FLOAT DEFAULT 0,
            KEY idx_applied_at (applied_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        """
        self.execute_sql(sql)
    
    def get_applied_migrations(self) -> List[str]:
        """Get applied migrations"""
        try:
            results = self.db.execute_query(
                "SELECT version FROM schema_migrations ORDER BY version"
            )
            return [row['version'] for row in results]
        except:
            return []
    
    def record_migration(self, migration: Migration) -> None:
        """Record migration as applied"""
        sql = """
        INSERT INTO schema_migrations (version, name, checksum, applied_at)
        VALUES (%s, %s, %s, %s)
        """
        self.db.execute_command(sql, (
            migration.version,
            migration.name,
            migration.checksum,
            migration.applied_at or datetime.now()
        ))
    
    def remove_migration_record(self, version: str) -> None:
        """Remove migration record"""
        self.db.execute_command(
            "DELETE FROM schema_migrations WHERE version = %s",
            (version,)
        )


class SQLiteMigrationRunner(MigrationRunner):
    """SQLite migration runner"""
    
    def __init__(self, db_adapter):
        self.db = db_adapter
    
    def execute_sql(self, sql: str) -> None:
        """Execute SQL statement"""
        self.db.execute_command(sql)
    
    def create_migration_table(self) -> None:
        """Create schema_migrations table"""
        sql = """
        CREATE TABLE IF NOT EXISTS schema_migrations (
            version TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            checksum TEXT NOT NULL,
            applied_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            execution_time REAL DEFAULT 0
        );
        CREATE INDEX IF NOT EXISTS idx_applied_at ON schema_migrations(applied_at);
        """
        self.execute_sql(sql)
    
    def get_applied_migrations(self) -> List[str]:
        """Get applied migrations"""
        try:
            results = self.db.execute_query(
                "SELECT version FROM schema_migrations ORDER BY version"
            )
            return [row['version'] for row in results]
        except:
            return []
    
    def record_migration(self, migration: Migration) -> None:
        """Record migration as applied"""
        sql = """
        INSERT INTO schema_migrations (version, name, checksum, applied_at)
        VALUES (?, ?, ?, ?)
        """
        self.db.execute_command(sql, (
            migration.version,
            migration.name,
            migration.checksum,
            migration.applied_at or datetime.now()
        ))
    
    def remove_migration_record(self, version: str) -> None:
        """Remove migration record"""
        self.db.execute_command(
            "DELETE FROM schema_migrations WHERE version = ?",
            (version,)
        )


class MigrationManager:
    """Main migration management system"""
    
    def __init__(self, db_adapter, migrations_path: str = "migrations"):
        self.db_adapter = db_adapter
        self.migrations_path = Path(migrations_path)
        self.migrations_path.mkdir(exist_ok=True)
        self.logger = logging.getLogger(__name__)
        
        # Create appropriate runner based on database type
        self.runner = self._create_runner()
        self.runner.create_migration_table()
    
    def _create_runner(self) -> MigrationRunner:
        """Create appropriate migration runner"""
        # Detect database type from adapter
        adapter_class = self.db_adapter.__class__.__name__
        
        if "PostgreSQL" in adapter_class:
            return PostgreSQLMigrationRunner(self.db_adapter)
        elif "MySQL" in adapter_class:
            return MySQLMigrationRunner(self.db_adapter)
        elif "SQLite" in adapter_class:
            return SQLiteMigrationRunner(self.db_adapter)
        else:
            # Default to SQLite runner
            return SQLiteMigrationRunner(self.db_adapter)
    
    def create_migration(self, name: str, up_sql: str = "", down_sql: str = "") -> str:
        """Create a new migration file"""
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        version = f"{timestamp}_{self._sanitize_name(name)}"
        
        migration_file = self.migrations_path / f"{version}.sql"
        
        content = f"""-- Migration: {name}
-- Created: {datetime.now().isoformat()}
-- Version: {version}

-- +migrate Up
{up_sql or "-- Add your UP migration SQL here"}

-- +migrate Down
{down_sql or "-- Add your DOWN migration SQL here"}
"""
        
        with open(migration_file, 'w') as f:
            f.write(content)
        
        self.logger.info(f"Created migration: {migration_file}")
        return version
    
    def _sanitize_name(self, name: str) -> str:
        """Sanitize migration name for filename"""
        return re.sub(r'[^\w\s-]', '', name).strip().replace(' ', '_').lower()
    
    def load_migrations(self) -> List[Migration]:
        """Load all migration files"""
        migrations = []
        
        for file_path in sorted(self.migrations_path.glob("*.sql")):
            try:
                migration = self._parse_migration_file(file_path)
                migrations.append(migration)
            except Exception as e:
                self.logger.error(f"Error loading migration {file_path}: {e}")
        
        return migrations
    
    def _parse_migration_file(self, file_path: Path) -> Migration:
        """Parse migration file into Migration object"""
        with open(file_path, 'r') as f:
            content = f.read()
        
        # Extract version from filename
        version = file_path.stem
        
        # Parse name from comment
        name_match = re.search(r'-- Migration: (.+)', content)
        name = name_match.group(1) if name_match else version
        
        # Split UP and DOWN migrations
        up_sql = ""
        down_sql = ""
        
        sections = re.split(r'-- \+migrate (Up|Down)', content)
        
        for i in range(1, len(sections), 2):
            if i + 1 < len(sections):
                section_type = sections[i].strip()
                section_sql = sections[i + 1].strip()
                
                if section_type == "Up":
                    up_sql = section_sql
                elif section_type == "Down":
                    down_sql = section_sql
        
        migration = Migration(
            version=version,
            name=name,
            up_sql=up_sql,
            down_sql=down_sql,
            checksum="",
            created_at=datetime.now()
        )
        
        migration.checksum = migration.generate_checksum()
        return migration
    
    def get_pending_migrations(self) -> List[Migration]:
        """Get migrations that haven't been applied"""
        all_migrations = self.load_migrations()
        applied_versions = set(self.runner.get_applied_migrations())
        
        return [m for m in all_migrations if m.version not in applied_versions]
    
    def migrate_up(self, target_version: Optional[str] = None) -> List[MigrationResult]:
        """Apply pending migrations"""
        pending = self.get_pending_migrations()
        
        if target_version:
            # Filter to migrations up to target version
            pending = [m for m in pending if m.version <= target_version]
        
        results = []
        
        for migration in pending:
            result = self._apply_migration(migration)
            results.append(result)
            
            if not result.success:
                self.logger.error(f"Migration failed: {result.message}")
                break
        
        return results
    
    def migrate_down(self, target_version: Optional[str] = None, steps: int = 1) -> List[MigrationResult]:
        """Rollback migrations"""
        applied_versions = self.runner.get_applied_migrations()
        
        if target_version:
            # Rollback to specific version
            to_rollback = [v for v in applied_versions if v > target_version]
        else:
            # Rollback specified number of steps
            to_rollback = applied_versions[-steps:] if steps <= len(applied_versions) else applied_versions
        
        to_rollback.reverse()  # Rollback in reverse order
        
        results = []
        all_migrations = {m.version: m for m in self.load_migrations()}
        
        for version in to_rollback:
            if version in all_migrations:
                migration = all_migrations[version]
                result = self._rollback_migration(migration)
                results.append(result)
                
                if not result.success:
                    self.logger.error(f"Rollback failed: {result.message}")
                    break
        
        return results
    
    def _apply_migration(self, migration: Migration) -> MigrationResult:
        """Apply a single migration"""
        start_time = datetime.now()
        
        try:
            if not migration.up_sql.strip():
                raise MigrationError("Empty UP migration SQL")
            
            self.logger.info(f"Applying migration: {migration.version}")
            
            # Execute UP migration
            self.runner.execute_sql(migration.up_sql)
            
            # Record migration
            migration.applied_at = datetime.now()
            self.runner.record_migration(migration)
            
            execution_time = (datetime.now() - start_time).total_seconds()
            
            return MigrationResult(
                success=True,
                version=migration.version,
                message=f"Migration {migration.version} applied successfully",
                execution_time=execution_time
            )
            
        except Exception as e:
            execution_time = (datetime.now() - start_time).total_seconds()
            
            return MigrationResult(
                success=False,
                version=migration.version,
                message=f"Migration {migration.version} failed: {str(e)}",
                execution_time=execution_time,
                error=e
            )
    
    def _rollback_migration(self, migration: Migration) -> MigrationResult:
        """Rollback a single migration"""
        start_time = datetime.now()
        
        try:
            if not migration.down_sql.strip():
                raise MigrationError("Empty DOWN migration SQL")
            
            self.logger.info(f"Rolling back migration: {migration.version}")
            
            # Execute DOWN migration
            self.runner.execute_sql(migration.down_sql)
            
            # Remove migration record
            self.runner.remove_migration_record(migration.version)
            
            execution_time = (datetime.now() - start_time).total_seconds()
            
            return MigrationResult(
                success=True,
                version=migration.version,
                message=f"Migration {migration.version} rolled back successfully",
                execution_time=execution_time
            )
            
        except Exception as e:
            execution_time = (datetime.now() - start_time).total_seconds()
            
            return MigrationResult(
                success=False,
                version=migration.version,
                message=f"Rollback {migration.version} failed: {str(e)}",
                execution_time=execution_time,
                error=e
            )
    
    def status(self) -> Dict[str, Any]:
        """Get migration status"""
        all_migrations = self.load_migrations()
        applied_versions = set(self.runner.get_applied_migrations())
        pending_migrations = self.get_pending_migrations()
        
        return {
            'total_migrations': len(all_migrations),
            'applied_migrations': len(applied_versions),
            'pending_migrations': len(pending_migrations),
            'last_applied': max(applied_versions) if applied_versions else None,
            'next_pending': pending_migrations[0].version if pending_migrations else None,
            'status': 'up_to_date' if not pending_migrations else 'pending_migrations'
        }
    
    def validate_migrations(self) -> List[Dict[str, Any]]:
        """Validate migration integrity"""
        issues = []
        all_migrations = self.load_migrations()
        
        # Check for duplicate versions
        versions = [m.version for m in all_migrations]
        duplicates = set([v for v in versions if versions.count(v) > 1])
        
        for duplicate in duplicates:
            issues.append({
                'type': 'duplicate_version',
                'version': duplicate,
                'message': f"Duplicate migration version: {duplicate}"
            })
        
        # Check for missing DOWN migrations
        for migration in all_migrations:
            if not migration.down_sql.strip():
                issues.append({
                    'type': 'missing_down_migration',
                    'version': migration.version,
                    'message': f"Missing DOWN migration for: {migration.version}"
                })
        
        return issues


# CLI-like interface functions
def create_migration_manager(db_adapter, migrations_path: str = "migrations") -> MigrationManager:
    """Create migration manager instance"""
    return MigrationManager(db_adapter, migrations_path)


if __name__ == "__main__":
    # Example usage
    import sqlite3
    from database_connector import SQLiteAdapter, ConnectionConfig
    
    # Create SQLite connection for testing
    config = ConnectionConfig(engine="sqlite", database="test_migrations.db")
    adapter = SQLiteAdapter(config)
    
    # Create migration manager
    manager = MigrationManager(adapter)
    
    # Create sample migration
    version1 = manager.create_migration(
        "create_users_table",
        up_sql="""
        CREATE TABLE users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            email TEXT UNIQUE NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        );
        """,
        down_sql="DROP TABLE IF EXISTS users;"
    )
    
    version2 = manager.create_migration(
        "add_users_index",
        up_sql="CREATE INDEX idx_users_email ON users(email);",
        down_sql="DROP INDEX IF EXISTS idx_users_email;"
    )
    
    print(f"Created migrations: {version1}, {version2}")
    
    # Check status
    status = manager.status()
    print(f"Migration status: {status}")
    
    # Apply migrations
    results = manager.migrate_up()
    for result in results:
        print(f"Migration result: {result.message}")
    
    # Check status again
    status = manager.status()
    print(f"Migration status after apply: {status}")
    
    # Rollback one migration
    rollback_results = manager.migrate_down(steps=1)
    for result in rollback_results:
        print(f"Rollback result: {result.message}")
    
    # Final status
    status = manager.status()
    print(f"Final migration status: {status}")
    
    print("\ng8.3: Database Migration Manager - COMPLETED ✅") 