"""
AMQP Operator - Advanced Message Queuing Protocol Integration
Production-ready AMQP integration with queue management, delivery guarantees, and consumer patterns.
"""

import asyncio
import json
import logging
import ssl
import time
import uuid
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from typing import Any, Callable, Dict, List, Optional, Union
from enum import Enum

# AMQP Support
try:
    import pika
    from pika.exchange_type import ExchangeType
    from pika.spec import BasicProperties
    PIKA_AVAILABLE = True
except ImportError:
    PIKA_AVAILABLE = False
    print("pika library not available. @amqp operator will be limited.")

# Async AMQP Support
try:
    import aio_pika
    from aio_pika import Message, DeliveryMode, ExchangeType as AsyncExchangeType
    from aio_pika.abc import AbstractChannel, AbstractConnection, AbstractExchange, AbstractQueue
    AIO_PIKA_AVAILABLE = True
except ImportError:
    AIO_PIKA_AVAILABLE = False
    print("aio-pika library not available. Async AMQP features limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class MessagePriority(Enum):
    """Message priority levels."""
    LOW = 1
    NORMAL = 5
    HIGH = 10

class ExchangeType(Enum):
    """Exchange types."""
    DIRECT = "direct"
    FANOUT = "fanout"
    TOPIC = "topic"
    HEADERS = "headers"

@dataclass
class AMQPConfig:
    """AMQP connection configuration."""
    host: str = "localhost"
    port: int = 5672
    username: str = "guest"
    password: str = "guest"
    virtual_host: str = "/"
    ssl_enabled: bool = False
    ssl_context: Optional[ssl.SSLContext] = None
    connection_timeout: int = 30
    heartbeat: int = 600
    blocked_connection_timeout: int = 300
    retry_attempts: int = 3
    retry_delay: float = 1.0

@dataclass
class AMQPQueue:
    """AMQP queue configuration."""
    name: str
    durable: bool = True
    exclusive: bool = False
    auto_delete: bool = False
    arguments: Dict[str, Any] = field(default_factory=dict)
    bind_to_exchange: Optional[str] = None
    routing_key: str = ""

@dataclass
class AMQPExchange:
    """AMQP exchange configuration."""
    name: str
    type: ExchangeType = ExchangeType.DIRECT
    durable: bool = True
    auto_delete: bool = False
    internal: bool = False
    arguments: Dict[str, Any] = field(default_factory=dict)

@dataclass
class AMQPMessage:
    """AMQP message structure."""
    body: Union[str, bytes, Dict[str, Any]]
    routing_key: str = ""
    exchange: str = ""
    properties: Optional[Dict[str, Any]] = None
    priority: MessagePriority = MessagePriority.NORMAL
    ttl: Optional[int] = None  # milliseconds
    persistent: bool = True
    correlation_id: Optional[str] = None
    reply_to: Optional[str] = None
    message_id: Optional[str] = None
    timestamp: Optional[datetime] = None
    headers: Dict[str, Any] = field(default_factory=dict)

@dataclass
class AMQPConsumer:
    """AMQP consumer configuration."""
    queue_name: str
    callback: Callable
    auto_ack: bool = False
    exclusive: bool = False
    consumer_tag: Optional[str] = None
    prefetch_count: int = 1
    prefetch_size: int = 0

class AMQPConnectionManager:
    """Manages AMQP connections with failover and recovery."""
    
    def __init__(self, config: AMQPConfig):
        self.config = config
        self.connection: Optional[pika.BlockingConnection] = None
        self.async_connection: Optional[aio_pika.Connection] = None
        self.channel: Optional[pika.channel.Channel] = None
        self.async_channel: Optional[AbstractChannel] = None
        self.is_connected = False
        self._connection_lock = asyncio.Lock()
        
    async def connect(self) -> bool:
        """Establish AMQP connection."""
        if not PIKA_AVAILABLE:
            logger.error("pika library not available")
            return False
        
        async with self._connection_lock:
            try:
                # Connection parameters
                credentials = pika.PlainCredentials(self.config.username, self.config.password)
                
                connection_params = pika.ConnectionParameters(
                    host=self.config.host,
                    port=self.config.port,
                    virtual_host=self.config.virtual_host,
                    credentials=credentials,
                    connection_attempts=self.config.retry_attempts,
                    retry_delay=self.config.retry_delay,
                    heartbeat=self.config.heartbeat,
                    blocked_connection_timeout=self.config.blocked_connection_timeout
                )
                
                if self.config.ssl_enabled:
                    connection_params.ssl_options = pika.SSLOptions(
                        context=self.config.ssl_context or ssl.create_default_context()
                    )
                
                # Establish synchronous connection
                self.connection = pika.BlockingConnection(connection_params)
                self.channel = self.connection.channel()
                
                # Establish async connection if available
                if AIO_PIKA_AVAILABLE:
                    connection_string = self._build_connection_string()
                    self.async_connection = await aio_pika.connect_robust(connection_string)
                    self.async_channel = await self.async_connection.channel()
                    await self.async_channel.set_qos(prefetch_count=100)
                
                self.is_connected = True
                logger.info(f"Connected to AMQP broker: {self.config.host}:{self.config.port}")
                
                # Start connection monitoring
                asyncio.create_task(self._monitor_connection())
                
                return True
                
            except Exception as e:
                logger.error(f"Error connecting to AMQP: {str(e)}")
                return False
    
    def _build_connection_string(self) -> str:
        """Build connection string for async connection."""
        scheme = "amqps" if self.config.ssl_enabled else "amqp"
        return (f"{scheme}://{self.config.username}:{self.config.password}@"
                f"{self.config.host}:{self.config.port}/{self.config.virtual_host}")
    
    async def _monitor_connection(self):
        """Monitor connection health."""
        while self.is_connected:
            try:
                await asyncio.sleep(30)  # Check every 30 seconds
                
                # Check sync connection
                if self.connection and not self.connection.is_closed:
                    self.connection.process_data_events(time_limit=0)
                else:
                    logger.warning("Sync AMQP connection lost")
                    await self._reconnect()
                
            except Exception as e:
                logger.error(f"Connection monitoring error: {str(e)}")
                await self._reconnect()
    
    async def _reconnect(self):
        """Attempt to reconnect."""
        logger.info("Attempting AMQP reconnection...")
        await self.close()
        await asyncio.sleep(5)
        await self.connect()
    
    async def close(self):
        """Close connections."""
        self.is_connected = False
        
        try:
            if self.async_connection and not self.async_connection.is_closed:
                await self.async_connection.close()
            
            if self.connection and not self.connection.is_closed:
                self.connection.close()
        except Exception as e:
            logger.warning(f"Error closing connections: {str(e)}")

class AMQPOperator:
    """@amqp operator implementation with full production features."""
    
    def __init__(self):
        self.connection_manager: Optional[AMQPConnectionManager] = None
        self.declared_exchanges = {}
        self.declared_queues = {}
        self.active_consumers = {}
        self.operation_stats = {
            'messages_published': 0,
            'messages_consumed': 0,
            'exchanges_declared': 0,
            'queues_declared': 0,
            'bindings_created': 0,
            'dlq_messages': 0
        }
        self._executor = ThreadPoolExecutor(max_workers=10)
    
    async def connect(self, config: Optional[AMQPConfig] = None) -> bool:
        """Connect to AMQP broker."""
        if config is None:
            config = AMQPConfig()
        
        self.connection_manager = AMQPConnectionManager(config)
        return await self.connection_manager.connect()
    
    @property
    def channel(self) -> Optional[Any]:
        """Get synchronous channel."""
        return self.connection_manager.channel if self.connection_manager else None
    
    @property
    def async_channel(self) -> Optional[Any]:
        """Get asynchronous channel."""
        return self.connection_manager.async_channel if self.connection_manager else None
    
    # Exchange Operations
    async def declare_exchange(self, exchange: AMQPExchange) -> bool:
        """Declare exchange."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            self.channel.exchange_declare(
                exchange=exchange.name,
                exchange_type=exchange.type.value,
                durable=exchange.durable,
                auto_delete=exchange.auto_delete,
                internal=exchange.internal,
                arguments=exchange.arguments
            )
            
            self.declared_exchanges[exchange.name] = exchange
            self.operation_stats['exchanges_declared'] += 1
            logger.info(f"Declared exchange: {exchange.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error declaring exchange {exchange.name}: {str(e)}")
            raise
    
    async def delete_exchange(self, exchange_name: str, if_unused: bool = False) -> bool:
        """Delete exchange."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            self.channel.exchange_delete(exchange=exchange_name, if_unused=if_unused)
            
            if exchange_name in self.declared_exchanges:
                del self.declared_exchanges[exchange_name]
            
            logger.info(f"Deleted exchange: {exchange_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error deleting exchange {exchange_name}: {str(e)}")
            raise
    
    # Queue Operations
    async def declare_queue(self, queue: AMQPQueue) -> bool:
        """Declare queue."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            # Declare queue
            result = self.channel.queue_declare(
                queue=queue.name,
                durable=queue.durable,
                exclusive=queue.exclusive,
                auto_delete=queue.auto_delete,
                arguments=queue.arguments
            )
            
            # Bind to exchange if specified
            if queue.bind_to_exchange:
                self.channel.queue_bind(
                    exchange=queue.bind_to_exchange,
                    queue=queue.name,
                    routing_key=queue.routing_key
                )
                self.operation_stats['bindings_created'] += 1
            
            self.declared_queues[queue.name] = queue
            self.operation_stats['queues_declared'] += 1
            logger.info(f"Declared queue: {queue.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error declaring queue {queue.name}: {str(e)}")
            raise
    
    async def delete_queue(self, queue_name: str, if_unused: bool = False, if_empty: bool = False) -> bool:
        """Delete queue."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            self.channel.queue_delete(
                queue=queue_name,
                if_unused=if_unused,
                if_empty=if_empty
            )
            
            if queue_name in self.declared_queues:
                del self.declared_queues[queue_name]
            
            logger.info(f"Deleted queue: {queue_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error deleting queue {queue_name}: {str(e)}")
            raise
    
    async def purge_queue(self, queue_name: str) -> int:
        """Purge queue messages."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            result = self.channel.queue_purge(queue=queue_name)
            logger.info(f"Purged {result.method.message_count} messages from queue: {queue_name}")
            return result.method.message_count
            
        except Exception as e:
            logger.error(f"Error purging queue {queue_name}: {str(e)}")
            raise
    
    async def get_queue_info(self, queue_name: str) -> Dict[str, Any]:
        """Get queue information."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            result = self.channel.queue_declare(queue=queue_name, passive=True)
            return {
                'name': queue_name,
                'message_count': result.method.message_count,
                'consumer_count': result.method.consumer_count
            }
            
        except Exception as e:
            logger.error(f"Error getting queue info {queue_name}: {str(e)}")
            raise
    
    # Binding Operations
    async def bind_queue(self, queue_name: str, exchange_name: str, routing_key: str = "",
                        arguments: Optional[Dict[str, Any]] = None) -> bool:
        """Bind queue to exchange."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            self.channel.queue_bind(
                exchange=exchange_name,
                queue=queue_name,
                routing_key=routing_key,
                arguments=arguments or {}
            )
            
            self.operation_stats['bindings_created'] += 1
            logger.info(f"Bound queue {queue_name} to exchange {exchange_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error binding queue: {str(e)}")
            raise
    
    async def unbind_queue(self, queue_name: str, exchange_name: str, routing_key: str = "",
                          arguments: Optional[Dict[str, Any]] = None) -> bool:
        """Unbind queue from exchange."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            self.channel.queue_unbind(
                exchange=exchange_name,
                queue=queue_name,
                routing_key=routing_key,
                arguments=arguments or {}
            )
            
            logger.info(f"Unbound queue {queue_name} from exchange {exchange_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error unbinding queue: {str(e)}")
            raise
    
    # Message Publishing
    async def publish_message(self, message: AMQPMessage) -> bool:
        """Publish message."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            # Prepare message body
            if isinstance(message.body, dict):
                body = json.dumps(message.body)
            elif isinstance(message.body, str):
                body = message.body.encode('utf-8')
            else:
                body = message.body
            
            # Prepare properties
            properties = BasicProperties(
                delivery_mode=2 if message.persistent else 1,
                priority=message.priority.value,
                correlation_id=message.correlation_id,
                reply_to=message.reply_to,
                message_id=message.message_id or str(uuid.uuid4()),
                timestamp=int(message.timestamp.timestamp()) if message.timestamp else None,
                headers=message.headers,
                expiration=str(message.ttl) if message.ttl else None
            )
            
            # Publish message
            self.channel.basic_publish(
                exchange=message.exchange,
                routing_key=message.routing_key,
                body=body,
                properties=properties,
                mandatory=True
            )
            
            self.operation_stats['messages_published'] += 1
            logger.debug(f"Published message to {message.exchange}/{message.routing_key}")
            return True
            
        except Exception as e:
            logger.error(f"Error publishing message: {str(e)}")
            raise
    
    async def publish_batch(self, messages: List[AMQPMessage]) -> int:
        """Publish multiple messages."""
        published_count = 0
        
        for message in messages:
            try:
                await self.publish_message(message)
                published_count += 1
            except Exception as e:
                logger.error(f"Error in batch publish: {str(e)}")
        
        logger.info(f"Published {published_count}/{len(messages)} messages in batch")
        return published_count
    
    # Message Consumption
    async def start_consumer(self, consumer: AMQPConsumer) -> str:
        """Start message consumer."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            # Set QoS
            self.channel.basic_qos(
                prefetch_count=consumer.prefetch_count,
                prefetch_size=consumer.prefetch_size
            )
            
            # Create consumer wrapper
            def consumer_wrapper(channel, method, properties, body):
                try:
                    # Decode body
                    if isinstance(body, bytes):
                        try:
                            decoded_body = json.loads(body.decode('utf-8'))
                        except json.JSONDecodeError:
                            decoded_body = body.decode('utf-8')
                    else:
                        decoded_body = body
                    
                    # Create message object
                    message = AMQPMessage(
                        body=decoded_body,
                        routing_key=method.routing_key,
                        exchange=method.exchange,
                        correlation_id=properties.correlation_id,
                        message_id=properties.message_id,
                        headers=properties.headers or {}
                    )
                    
                    # Call user callback
                    try:
                        result = consumer.callback(message)
                        
                        # Handle acknowledgment
                        if not consumer.auto_ack:
                            if result is not False:  # Ack unless explicitly False
                                channel.basic_ack(delivery_tag=method.delivery_tag)
                            else:
                                channel.basic_nack(delivery_tag=method.delivery_tag, requeue=True)
                        
                        self.operation_stats['messages_consumed'] += 1
                        
                    except Exception as callback_error:
                        logger.error(f"Consumer callback error: {str(callback_error)}")
                        if not consumer.auto_ack:
                            # Send to DLQ or reject
                            channel.basic_nack(delivery_tag=method.delivery_tag, requeue=False)
                            self.operation_stats['dlq_messages'] += 1
                
                except Exception as e:
                    logger.error(f"Consumer wrapper error: {str(e)}")
            
            # Start consuming
            consumer_tag = self.channel.basic_consume(
                queue=consumer.queue_name,
                on_message_callback=consumer_wrapper,
                auto_ack=consumer.auto_ack,
                exclusive=consumer.exclusive,
                consumer_tag=consumer.consumer_tag
            )
            
            self.active_consumers[consumer_tag] = consumer
            logger.info(f"Started consumer: {consumer_tag} for queue {consumer.queue_name}")
            
            # Start consuming in background
            asyncio.create_task(self._run_consumer())
            
            return consumer_tag
            
        except Exception as e:
            logger.error(f"Error starting consumer: {str(e)}")
            raise
    
    async def _run_consumer(self):
        """Run consumer in background."""
        try:
            while self.connection_manager and self.connection_manager.is_connected:
                self.connection_manager.connection.process_data_events(time_limit=1)
                await asyncio.sleep(0.1)
        except Exception as e:
            logger.error(f"Consumer run error: {str(e)}")
    
    async def stop_consumer(self, consumer_tag: str) -> bool:
        """Stop consumer."""
        if not self.channel:
            raise RuntimeError("Not connected to AMQP broker")
        
        try:
            self.channel.basic_cancel(consumer_tag)
            
            if consumer_tag in self.active_consumers:
                del self.active_consumers[consumer_tag]
            
            logger.info(f"Stopped consumer: {consumer_tag}")
            return True
            
        except Exception as e:
            logger.error(f"Error stopping consumer: {str(e)}")
            raise
    
    # Advanced Features
    async def setup_dead_letter_queue(self, queue_name: str, dlx_name: str = "dlx",
                                     dlq_name: str = None) -> bool:
        """Setup dead letter queue for a queue."""
        if dlq_name is None:
            dlq_name = f"{queue_name}.dlq"
        
        try:
            # Declare DLX
            dlx = AMQPExchange(name=dlx_name, type=ExchangeType.DIRECT)
            await self.declare_exchange(dlx)
            
            # Declare DLQ
            dlq = AMQPQueue(
                name=dlq_name,
                bind_to_exchange=dlx_name,
                routing_key=queue_name
            )
            await self.declare_queue(dlq)
            
            # Configure main queue with DLX
            main_queue = AMQPQueue(
                name=queue_name,
                arguments={
                    'x-dead-letter-exchange': dlx_name,
                    'x-dead-letter-routing-key': queue_name
                }
            )
            await self.declare_queue(main_queue)
            
            logger.info(f"Setup DLQ for queue {queue_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error setting up DLQ: {str(e)}")
            raise
    
    async def setup_delayed_message_exchange(self, exchange_name: str = "delayed-exchange") -> bool:
        """Setup delayed message exchange using rabbitmq-delayed-message-exchange plugin."""
        try:
            delayed_exchange = AMQPExchange(
                name=exchange_name,
                type=ExchangeType.DIRECT,
                arguments={'x-delayed-type': 'direct'}
            )
            
            # Note: This requires the rabbitmq-delayed-message-exchange plugin
            self.channel.exchange_declare(
                exchange=exchange_name,
                exchange_type='x-delayed-message',
                arguments={'x-delayed-type': 'direct'}
            )
            
            logger.info(f"Setup delayed message exchange: {exchange_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error setting up delayed exchange: {str(e)}")
            raise
    
    async def publish_delayed_message(self, message: AMQPMessage, delay_ms: int) -> bool:
        """Publish delayed message (requires delayed message plugin)."""
        message.headers['x-delay'] = delay_ms
        return await self.publish_message(message)
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get operation statistics."""
        return {
            'operations': self.operation_stats.copy(),
            'connected': self.connection_manager.is_connected if self.connection_manager else False,
            'declared_exchanges': len(self.declared_exchanges),
            'declared_queues': len(self.declared_queues),
            'active_consumers': len(self.active_consumers)
        }
    
    async def close(self):
        """Close connections and cleanup."""
        # Stop all consumers
        for consumer_tag in list(self.active_consumers.keys()):
            await self.stop_consumer(consumer_tag)
        
        # Close connections
        if self.connection_manager:
            await self.connection_manager.close()
        
        # Shutdown executor
        self._executor.shutdown(wait=True)
        
        logger.info("AMQP operator closed")

# Export the operator
__all__ = [
    'AMQPOperator', 'AMQPConfig', 'AMQPQueue', 'AMQPExchange', 'AMQPMessage', 
    'AMQPConsumer', 'MessagePriority', 'ExchangeType'
] 