"""
Istio Operator - Service Mesh Management Integration
Production-ready Istio integration with traffic management, security policies, and observability.
"""

import asyncio
import base64
import json
import logging
import ssl
import subprocess
import time
import yaml
from concurrent.futures import ThreadPoolExecutor
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Union
from pathlib import Path

# Kubernetes Client
try:
    from kubernetes import client, config
    from kubernetes.client.rest import ApiException
    KUBERNETES_AVAILABLE = True
except ImportError:
    KUBERNETES_AVAILABLE = False
    print("kubernetes library not available. @istio operator will be limited.")

# HTTP Client for Istio APIs
try:
    import aiohttp
    import requests
    HTTP_AVAILABLE = True
except ImportError:
    HTTP_AVAILABLE = False
    print("HTTP libraries not available. @istio operator will be limited.")

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class IstioConfig:
    """Istio configuration."""
    kubeconfig_path: Optional[str] = None
    namespace: str = "istio-system"
    pilot_endpoint: Optional[str] = None
    citadel_endpoint: Optional[str] = None
    galley_endpoint: Optional[str] = None
    mixer_endpoint: Optional[str] = None
    timeout: int = 30
    verify_ssl: bool = True

@dataclass
class VirtualService:
    """Istio Virtual Service configuration."""
    name: str
    namespace: str = "default"
    hosts: List[str] = field(default_factory=list)
    gateways: List[str] = field(default_factory=list)
    http_routes: List[Dict[str, Any]] = field(default_factory=list)
    tcp_routes: List[Dict[str, Any]] = field(default_factory=list)
    tls_routes: List[Dict[str, Any]] = field(default_factory=list)

@dataclass
class DestinationRule:
    """Istio Destination Rule configuration."""
    name: str
    namespace: str = "default"
    host: str = ""
    traffic_policy: Optional[Dict[str, Any]] = None
    port_level_settings: List[Dict[str, Any]] = field(default_factory=list)
    subsets: List[Dict[str, Any]] = field(default_factory=list)

@dataclass
class Gateway:
    """Istio Gateway configuration."""
    name: str
    namespace: str = "default"
    selector: Dict[str, str] = field(default_factory=dict)
    servers: List[Dict[str, Any]] = field(default_factory=list)

@dataclass
class ServiceEntry:
    """Istio Service Entry configuration."""
    name: str
    namespace: str = "default"
    hosts: List[str] = field(default_factory=list)
    ports: List[Dict[str, Any]] = field(default_factory=list)
    location: str = "MESH_EXTERNAL"
    resolution: str = "DNS"
    addresses: List[str] = field(default_factory=list)
    endpoints: List[Dict[str, Any]] = field(default_factory=list)

@dataclass
class AuthorizationPolicy:
    """Istio Authorization Policy configuration."""
    name: str
    namespace: str = "default"
    selector: Optional[Dict[str, Any]] = None
    action: str = "ALLOW"  # ALLOW, DENY, AUDIT, CUSTOM
    rules: List[Dict[str, Any]] = field(default_factory=list)

@dataclass
class PeerAuthentication:
    """Istio Peer Authentication configuration."""
    name: str
    namespace: str = "default"
    selector: Optional[Dict[str, Any]] = None
    mtls: Dict[str, str] = field(default_factory=lambda: {"mode": "STRICT"})
    port_level_mtls: Dict[int, Dict[str, str]] = field(default_factory=dict)

class IstioKubernetesClient:
    """Kubernetes client for Istio resources."""
    
    def __init__(self, config: IstioConfig):
        self.config = config
        self.api_client = None
        self.custom_objects_api = None
        
    async def initialize(self) -> bool:
        """Initialize Kubernetes client."""
        if not KUBERNETES_AVAILABLE:
            logger.error("Kubernetes client library not available")
            return False
        
        try:
            # Load kubeconfig
            if self.config.kubeconfig_path:
                config.load_kube_config(config_file=self.config.kubeconfig_path)
            else:
                try:
                    config.load_incluster_config()
                except config.ConfigException:
                    config.load_kube_config()
            
            self.api_client = client.ApiClient()
            self.custom_objects_api = client.CustomObjectsApi()
            
            logger.info("Kubernetes client initialized")
            return True
            
        except Exception as e:
            logger.error(f"Error initializing Kubernetes client: {str(e)}")
            return False
    
    async def create_custom_resource(self, group: str, version: str, namespace: str, 
                                   plural: str, body: Dict[str, Any]) -> Dict[str, Any]:
        """Create custom resource."""
        try:
            result = self.custom_objects_api.create_namespaced_custom_object(
                group=group,
                version=version,
                namespace=namespace,
                plural=plural,
                body=body
            )
            return result
        except ApiException as e:
            logger.error(f"Error creating {plural}: {str(e)}")
            raise
    
    async def update_custom_resource(self, group: str, version: str, namespace: str,
                                   plural: str, name: str, body: Dict[str, Any]) -> Dict[str, Any]:
        """Update custom resource."""
        try:
            result = self.custom_objects_api.patch_namespaced_custom_object(
                group=group,
                version=version,
                namespace=namespace,
                plural=plural,
                name=name,
                body=body
            )
            return result
        except ApiException as e:
            logger.error(f"Error updating {plural}/{name}: {str(e)}")
            raise
    
    async def delete_custom_resource(self, group: str, version: str, namespace: str,
                                   plural: str, name: str) -> bool:
        """Delete custom resource."""
        try:
            self.custom_objects_api.delete_namespaced_custom_object(
                group=group,
                version=version,
                namespace=namespace,
                plural=plural,
                name=name
            )
            return True
        except ApiException as e:
            if e.status == 404:
                return True  # Already deleted
            logger.error(f"Error deleting {plural}/{name}: {str(e)}")
            raise
    
    async def get_custom_resource(self, group: str, version: str, namespace: str,
                                plural: str, name: str) -> Optional[Dict[str, Any]]:
        """Get custom resource."""
        try:
            result = self.custom_objects_api.get_namespaced_custom_object(
                group=group,
                version=version,
                namespace=namespace,
                plural=plural,
                name=name
            )
            return result
        except ApiException as e:
            if e.status == 404:
                return None
            logger.error(f"Error getting {plural}/{name}: {str(e)}")
            raise
    
    async def list_custom_resources(self, group: str, version: str, namespace: str,
                                  plural: str) -> List[Dict[str, Any]]:
        """List custom resources."""
        try:
            result = self.custom_objects_api.list_namespaced_custom_object(
                group=group,
                version=version,
                namespace=namespace,
                plural=plural
            )
            return result.get('items', [])
        except ApiException as e:
            logger.error(f"Error listing {plural}: {str(e)}")
            raise

class IstioOperator:
    """@istio operator implementation with full production features."""
    
    def __init__(self):
        self.k8s_client: Optional[IstioKubernetesClient] = None
        self.config: Optional[IstioConfig] = None
        self.operation_stats = {
            'virtual_service_operations': 0,
            'destination_rule_operations': 0,
            'gateway_operations': 0,
            'service_entry_operations': 0,
            'auth_policy_operations': 0,
            'peer_auth_operations': 0
        }
        self._executor = ThreadPoolExecutor(max_workers=5)
        
        # Istio API groups and versions
        self.istio_networking_group = "networking.istio.io"
        self.istio_security_group = "security.istio.io"
        self.istio_api_version = "v1beta1"
    
    async def connect(self, config: Optional[IstioConfig] = None) -> bool:
        """Connect to Istio via Kubernetes."""
        if config is None:
            config = IstioConfig()
        
        self.config = config
        self.k8s_client = IstioKubernetesClient(config)
        
        success = await self.k8s_client.initialize()
        
        if success:
            # Verify Istio installation
            await self._verify_istio_installation()
        
        return success
    
    async def _verify_istio_installation(self) -> bool:
        """Verify Istio is installed in the cluster."""
        try:
            # Check if Istio system namespace exists
            v1 = client.CoreV1Api()
            try:
                v1.read_namespace(name=self.config.namespace)
                logger.info(f"Istio system namespace '{self.config.namespace}' found")
            except ApiException as e:
                if e.status == 404:
                    logger.warning(f"Istio system namespace '{self.config.namespace}' not found")
                    return False
            
            # Check for Istio CRDs
            crd_api = client.ApiextensionsV1Api()
            crds = crd_api.list_custom_resource_definition()
            istio_crds = [crd for crd in crds.items if 'istio.io' in crd.spec.group]
            
            if istio_crds:
                logger.info(f"Found {len(istio_crds)} Istio CRDs")
                return True
            else:
                logger.warning("No Istio CRDs found")
                return False
                
        except Exception as e:
            logger.error(f"Error verifying Istio installation: {str(e)}")
            return False
    
    # Virtual Service Operations
    async def create_virtual_service(self, vs: VirtualService) -> bool:
        """Create Virtual Service."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        body = {
            "apiVersion": f"{self.istio_networking_group}/{self.istio_api_version}",
            "kind": "VirtualService",
            "metadata": {
                "name": vs.name,
                "namespace": vs.namespace
            },
            "spec": {
                "hosts": vs.hosts
            }
        }
        
        if vs.gateways:
            body["spec"]["gateways"] = vs.gateways
        if vs.http_routes:
            body["spec"]["http"] = vs.http_routes
        if vs.tcp_routes:
            body["spec"]["tcp"] = vs.tcp_routes
        if vs.tls_routes:
            body["spec"]["tls"] = vs.tls_routes
        
        try:
            await self.k8s_client.create_custom_resource(
                group=self.istio_networking_group,
                version=self.istio_api_version,
                namespace=vs.namespace,
                plural="virtualservices",
                body=body
            )
            
            self.operation_stats['virtual_service_operations'] += 1
            logger.info(f"Created VirtualService: {vs.namespace}/{vs.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating VirtualService: {str(e)}")
            raise
    
    async def update_virtual_service(self, vs: VirtualService) -> bool:
        """Update Virtual Service."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        body = {
            "spec": {
                "hosts": vs.hosts
            }
        }
        
        if vs.gateways:
            body["spec"]["gateways"] = vs.gateways
        if vs.http_routes:
            body["spec"]["http"] = vs.http_routes
        if vs.tcp_routes:
            body["spec"]["tcp"] = vs.tcp_routes
        if vs.tls_routes:
            body["spec"]["tls"] = vs.tls_routes
        
        try:
            await self.k8s_client.update_custom_resource(
                group=self.istio_networking_group,
                version=self.istio_api_version,
                namespace=vs.namespace,
                plural="virtualservices",
                name=vs.name,
                body=body
            )
            
            self.operation_stats['virtual_service_operations'] += 1
            logger.info(f"Updated VirtualService: {vs.namespace}/{vs.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error updating VirtualService: {str(e)}")
            raise
    
    async def delete_virtual_service(self, name: str, namespace: str = "default") -> bool:
        """Delete Virtual Service."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        try:
            success = await self.k8s_client.delete_custom_resource(
                group=self.istio_networking_group,
                version=self.istio_api_version,
                namespace=namespace,
                plural="virtualservices",
                name=name
            )
            
            if success:
                self.operation_stats['virtual_service_operations'] += 1
                logger.info(f"Deleted VirtualService: {namespace}/{name}")
            
            return success
            
        except Exception as e:
            logger.error(f"Error deleting VirtualService: {str(e)}")
            raise
    
    # Destination Rule Operations
    async def create_destination_rule(self, dr: DestinationRule) -> bool:
        """Create Destination Rule."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        body = {
            "apiVersion": f"{self.istio_networking_group}/{self.istio_api_version}",
            "kind": "DestinationRule",
            "metadata": {
                "name": dr.name,
                "namespace": dr.namespace
            },
            "spec": {
                "host": dr.host
            }
        }
        
        if dr.traffic_policy:
            body["spec"]["trafficPolicy"] = dr.traffic_policy
        if dr.port_level_settings:
            body["spec"]["portLevelSettings"] = dr.port_level_settings
        if dr.subsets:
            body["spec"]["subsets"] = dr.subsets
        
        try:
            await self.k8s_client.create_custom_resource(
                group=self.istio_networking_group,
                version=self.istio_api_version,
                namespace=dr.namespace,
                plural="destinationrules",
                body=body
            )
            
            self.operation_stats['destination_rule_operations'] += 1
            logger.info(f"Created DestinationRule: {dr.namespace}/{dr.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating DestinationRule: {str(e)}")
            raise
    
    # Gateway Operations
    async def create_gateway(self, gw: Gateway) -> bool:
        """Create Gateway."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        body = {
            "apiVersion": f"{self.istio_networking_group}/{self.istio_api_version}",
            "kind": "Gateway",
            "metadata": {
                "name": gw.name,
                "namespace": gw.namespace
            },
            "spec": {
                "selector": gw.selector,
                "servers": gw.servers
            }
        }
        
        try:
            await self.k8s_client.create_custom_resource(
                group=self.istio_networking_group,
                version=self.istio_api_version,
                namespace=gw.namespace,
                plural="gateways",
                body=body
            )
            
            self.operation_stats['gateway_operations'] += 1
            logger.info(f"Created Gateway: {gw.namespace}/{gw.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating Gateway: {str(e)}")
            raise
    
    # Service Entry Operations
    async def create_service_entry(self, se: ServiceEntry) -> bool:
        """Create Service Entry."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        body = {
            "apiVersion": f"{self.istio_networking_group}/{self.istio_api_version}",
            "kind": "ServiceEntry",
            "metadata": {
                "name": se.name,
                "namespace": se.namespace
            },
            "spec": {
                "hosts": se.hosts,
                "ports": se.ports,
                "location": se.location,
                "resolution": se.resolution
            }
        }
        
        if se.addresses:
            body["spec"]["addresses"] = se.addresses
        if se.endpoints:
            body["spec"]["endpoints"] = se.endpoints
        
        try:
            await self.k8s_client.create_custom_resource(
                group=self.istio_networking_group,
                version=self.istio_api_version,
                namespace=se.namespace,
                plural="serviceentries",
                body=body
            )
            
            self.operation_stats['service_entry_operations'] += 1
            logger.info(f"Created ServiceEntry: {se.namespace}/{se.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating ServiceEntry: {str(e)}")
            raise
    
    # Authorization Policy Operations
    async def create_authorization_policy(self, ap: AuthorizationPolicy) -> bool:
        """Create Authorization Policy."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        body = {
            "apiVersion": f"{self.istio_security_group}/{self.istio_api_version}",
            "kind": "AuthorizationPolicy",
            "metadata": {
                "name": ap.name,
                "namespace": ap.namespace
            },
            "spec": {
                "action": ap.action
            }
        }
        
        if ap.selector:
            body["spec"]["selector"] = ap.selector
        if ap.rules:
            body["spec"]["rules"] = ap.rules
        
        try:
            await self.k8s_client.create_custom_resource(
                group=self.istio_security_group,
                version=self.istio_api_version,
                namespace=ap.namespace,
                plural="authorizationpolicies",
                body=body
            )
            
            self.operation_stats['auth_policy_operations'] += 1
            logger.info(f"Created AuthorizationPolicy: {ap.namespace}/{ap.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating AuthorizationPolicy: {str(e)}")
            raise
    
    # Peer Authentication Operations
    async def create_peer_authentication(self, pa: PeerAuthentication) -> bool:
        """Create Peer Authentication."""
        if not self.k8s_client:
            raise RuntimeError("Not connected to Istio")
        
        body = {
            "apiVersion": f"{self.istio_security_group}/{self.istio_api_version}",
            "kind": "PeerAuthentication",
            "metadata": {
                "name": pa.name,
                "namespace": pa.namespace
            },
            "spec": {
                "mtls": pa.mtls
            }
        }
        
        if pa.selector:
            body["spec"]["selector"] = pa.selector
        if pa.port_level_mtls:
            body["spec"]["portLevelMtls"] = pa.port_level_mtls
        
        try:
            await self.k8s_client.create_custom_resource(
                group=self.istio_security_group,
                version=self.istio_api_version,
                namespace=pa.namespace,
                plural="peerauthentications",
                body=body
            )
            
            self.operation_stats['peer_auth_operations'] += 1
            logger.info(f"Created PeerAuthentication: {pa.namespace}/{pa.name}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating PeerAuthentication: {str(e)}")
            raise
    
    # Traffic Management Helpers
    async def configure_canary_deployment(self, service_name: str, namespace: str,
                                         stable_version: str, canary_version: str,
                                         canary_weight: int = 10) -> bool:
        """Configure canary deployment with traffic splitting."""
        try:
            # Create Virtual Service for canary traffic
            vs = VirtualService(
                name=f"{service_name}-canary",
                namespace=namespace,
                hosts=[service_name],
                http_routes=[
                    {
                        "match": [{"headers": {"canary": {"exact": "true"}}}],
                        "route": [{"destination": {"host": service_name, "subset": "canary"}}]
                    },
                    {
                        "route": [
                            {"destination": {"host": service_name, "subset": "stable"}, "weight": 100 - canary_weight},
                            {"destination": {"host": service_name, "subset": "canary"}, "weight": canary_weight}
                        ]
                    }
                ]
            )
            
            # Create Destination Rule with subsets
            dr = DestinationRule(
                name=f"{service_name}-destination",
                namespace=namespace,
                host=service_name,
                subsets=[
                    {"name": "stable", "labels": {"version": stable_version}},
                    {"name": "canary", "labels": {"version": canary_version}}
                ]
            )
            
            await self.create_virtual_service(vs)
            await self.create_destination_rule(dr)
            
            logger.info(f"Configured canary deployment for {service_name} with {canary_weight}% traffic")
            return True
            
        except Exception as e:
            logger.error(f"Error configuring canary deployment: {str(e)}")
            raise
    
    async def configure_circuit_breaker(self, service_name: str, namespace: str,
                                      max_connections: int = 10, max_pending_requests: int = 10,
                                      max_requests: int = 10, consecutive_errors: int = 5) -> bool:
        """Configure circuit breaker for service."""
        try:
            dr = DestinationRule(
                name=f"{service_name}-circuit-breaker",
                namespace=namespace,
                host=service_name,
                traffic_policy={
                    "connectionPool": {
                        "tcp": {"maxConnections": max_connections},
                        "http": {
                            "http1MaxPendingRequests": max_pending_requests,
                            "maxRequestsPerConnection": max_requests
                        }
                    },
                    "outlierDetection": {
                        "consecutiveErrors": consecutive_errors,
                        "interval": "30s",
                        "baseEjectionTime": "30s",
                        "maxEjectionPercent": 50
                    }
                }
            )
            
            await self.create_destination_rule(dr)
            logger.info(f"Configured circuit breaker for {service_name}")
            return True
            
        except Exception as e:
            logger.error(f"Error configuring circuit breaker: {str(e)}")
            raise
    
    # Observability Integration
    async def get_service_metrics(self, service_name: str, namespace: str) -> Dict[str, Any]:
        """Get service metrics from Istio telemetry."""
        # This would integrate with Prometheus/Grafana
        # For now, return placeholder structure
        return {
            "service": service_name,
            "namespace": namespace,
            "request_total": 0,
            "request_duration": 0.0,
            "success_rate": 1.0,
            "p50_latency": 0.0,
            "p95_latency": 0.0,
            "p99_latency": 0.0
        }
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get operation statistics."""
        return {
            'operations': self.operation_stats.copy(),
            'connected': self.k8s_client is not None,
            'istio_namespace': self.config.namespace if self.config else None
        }
    
    async def close(self):
        """Close connections and cleanup."""
        if self.k8s_client and self.k8s_client.api_client:
            self.k8s_client.api_client.close()
        
        self._executor.shutdown(wait=True)
        logger.info("Istio operator closed")

# Export the operator
__all__ = [
    'IstioOperator', 'IstioConfig', 'VirtualService', 'DestinationRule', 
    'Gateway', 'ServiceEntry', 'AuthorizationPolicy', 'PeerAuthentication'
] 