"""
Azure Functions Integration for TuskLang Python SDK
Provides serverless Azure Functions runtime with triggers and monitoring
"""

import asyncio
import json
import os
import zipfile
import tempfile
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime, timedelta
import logging
import base64

try:
    from azure.functions import FunctionApp, HttpRequest, HttpResponse
    from azure.storage.blob import BlobServiceClient
    from azure.monitor.query import LogsQueryClient
    from azure.identity import DefaultAzureCredential
    from azure.mgmt.web import WebSiteManagementClient
    from azure.mgmt.resource import ResourceManagementClient
    AZURE_SDK_AVAILABLE = True
except ImportError:
    AZURE_SDK_AVAILABLE = False


class AzureFunctionsIntegration:
    """
    Main Azure Functions integration class for TuskLang Python SDK.
    Enables TuskLang to run as serverless Azure Functions.
    """
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        if not AZURE_SDK_AVAILABLE:
            self.logger.warning("Azure SDK not available - some features will be limited")
        
        self.config = config or {}
        self.subscription_id = self.config.get('subscription_id')
        self.resource_group = self.config.get('resource_group', 'tusklang-rg')
        self.function_app_name = self.config.get('function_app_name', 'tusklang-functions')
        self.location = self.config.get('location', 'eastus')
        
        # Function configuration
        self.functions = {}
        self.triggers = {}
        self.bindings = {}
        
        # Performance tracking
        self.stats = {
            'functions_executed': 0,
            'http_requests': 0,
            'timer_triggers': 0,
            'blob_triggers': 0,
            'queue_triggers': 0,
            'errors': 0,
            'cold_starts': 0,
            'total_execution_time_ms': 0
        }
        
        # Azure clients
        self.credential = None
        self.web_client = None
        self.resource_client = None
        self.logs_client = None
        self.blob_client = None
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Initialize Azure integration
        self._initialize_azure()
    
    def _initialize_azure(self):
        """Initialize Azure SDK clients"""
        try:
            if AZURE_SDK_AVAILABLE:
                # Set up authentication
                self.credential = DefaultAzureCredential()
                
                if self.subscription_id:
                    # Initialize management clients
                    self.web_client = WebSiteManagementClient(
                        self.credential, self.subscription_id
                    )
                    self.resource_client = ResourceManagementClient(
                        self.credential, self.subscription_id
                    )
                    
                    # Initialize monitoring
                    self.logs_client = LogsQueryClient(self.credential)
                
                self.logger.info("Azure integration initialized")
            else:
                self.logger.info("Azure integration initialized in simulation mode")
                
        except Exception as e:
            self.logger.error(f"Azure initialization failed: {e}")
            raise
    
    def create_function_app(self, app_name: Optional[str] = None) -> Dict[str, Any]:
        """Create Azure Function App with ARM template"""
        app_name = app_name or self.function_app_name
        
        try:
            # Create ARM template
            arm_template = self._create_arm_template(app_name)
            
            # Deploy to Azure (simulated if Azure SDK not available)
            if AZURE_SDK_AVAILABLE and self.resource_client:
                deployment_result = self._deploy_arm_template(arm_template, app_name)
            else:
                deployment_result = self._simulate_deployment(arm_template, app_name)
            
            return {
                'success': True,
                'app_name': app_name,
                'resource_group': self.resource_group,
                'location': self.location,
                'deployment': deployment_result,
                'timestamp': datetime.now().isoformat()
            }
            
        except Exception as e:
            self.logger.error(f"Function App creation failed: {e}")
            return {
                'success': False,
                'error': str(e),
                'app_name': app_name,
                'timestamp': datetime.now().isoformat()
            }
    
    def _create_arm_template(self, app_name: str) -> Dict[str, Any]:
        """Create ARM template for Function App deployment"""
        template = {
            "$schema": "https://schema.management.azure.com/schemas/2019-04-01/deploymentTemplate.json#",
            "contentVersion": "1.0.0.0",
            "parameters": {
                "functionAppName": {
                    "type": "string",
                    "defaultValue": app_name
                },
                "location": {
                    "type": "string",
                    "defaultValue": self.location
                }
            },
            "variables": {
                "storageAccountName": f"{app_name.replace('-', '')}storage",
                "hostingPlanName": f"{app_name}-plan",
                "applicationInsightsName": f"{app_name}-insights"
            },
            "resources": [
                {
                    "type": "Microsoft.Storage/storageAccounts",
                    "apiVersion": "2021-02-01",
                    "name": "[variables('storageAccountName')]",
                    "location": "[parameters('location')]",
                    "sku": {
                        "name": "Standard_LRS"
                    },
                    "kind": "Storage"
                },
                {
                    "type": "Microsoft.Web/serverfarms",
                    "apiVersion": "2021-02-01",
                    "name": "[variables('hostingPlanName')]",
                    "location": "[parameters('location')]",
                    "sku": {
                        "name": "Y1",
                        "tier": "Dynamic"
                    },
                    "properties": {}
                },
                {
                    "type": "Microsoft.Insights/components",
                    "apiVersion": "2020-02-02",
                    "name": "[variables('applicationInsightsName')]",
                    "location": "[parameters('location')]",
                    "kind": "web",
                    "properties": {
                        "Application_Type": "web"
                    }
                },
                {
                    "type": "Microsoft.Web/sites",
                    "apiVersion": "2021-02-01",
                    "name": "[parameters('functionAppName')]",
                    "location": "[parameters('location')]",
                    "kind": "functionapp",
                    "dependsOn": [
                        "[resourceId('Microsoft.Storage/storageAccounts', variables('storageAccountName'))]",
                        "[resourceId('Microsoft.Web/serverfarms', variables('hostingPlanName'))]",
                        "[resourceId('Microsoft.Insights/components', variables('applicationInsightsName'))]"
                    ],
                    "properties": {
                        "serverFarmId": "[resourceId('Microsoft.Web/serverfarms', variables('hostingPlanName'))]",
                        "siteConfig": {
                            "appSettings": [
                                {
                                    "name": "AzureWebJobsStorage",
                                    "value": "[concat('DefaultEndpointsProtocol=https;AccountName=', variables('storageAccountName'), ';EndpointSuffix=', environment().suffixes.storage, ';AccountKey=',listKeys(resourceId('Microsoft.Storage/storageAccounts', variables('storageAccountName')), '2021-02-01').keys[0].value)]"
                                },
                                {
                                    "name": "WEBSITE_CONTENTAZUREFILECONNECTIONSTRING",
                                    "value": "[concat('DefaultEndpointsProtocol=https;AccountName=', variables('storageAccountName'), ';EndpointSuffix=', environment().suffixes.storage, ';AccountKey=',listKeys(resourceId('Microsoft.Storage/storageAccounts', variables('storageAccountName')), '2021-02-01').keys[0].value)]"
                                },
                                {
                                    "name": "WEBSITE_CONTENTSHARE",
                                    "value": "[toLower(parameters('functionAppName'))]"
                                },
                                {
                                    "name": "FUNCTIONS_EXTENSION_VERSION",
                                    "value": "~4"
                                },
                                {
                                    "name": "FUNCTIONS_WORKER_RUNTIME",
                                    "value": "python"
                                },
                                {
                                    "name": "APPINSIGHTS_INSTRUMENTATIONKEY",
                                    "value": "[reference(resourceId('Microsoft.Insights/components', variables('applicationInsightsName'))).InstrumentationKey]"
                                },
                                {
                                    "name": "TUSKLANG_ENABLED",
                                    "value": "true"
                                }
                            ]
                        }
                    }
                }
            ],
            "outputs": {
                "functionAppUrl": {
                    "type": "string",
                    "value": "[concat('https://', parameters('functionAppName'), '.azurewebsites.net')]"
                }
            }
        }
        
        return template
    
    def register_http_function(self, name: str, operation: str, 
                             auth_level: str = 'function',
                             methods: List[str] = None) -> Dict[str, Any]:
        """Register HTTP trigger function"""
        methods = methods or ['GET', 'POST']
        
        function_config = {
            'name': name,
            'type': 'http',
            'operation': operation,
            'auth_level': auth_level,
            'methods': methods,
            'bindings': [
                {
                    "authLevel": auth_level,
                    "type": "httpTrigger",
                    "direction": "in",
                    "name": "req",
                    "methods": methods
                },
                {
                    "type": "http",
                    "direction": "out",
                    "name": "$return"
                }
            ]
        }
        
        # Create function code
        function_code = self._generate_http_function_code(name, operation, auth_level)
        
        self.functions[name] = {
            'config': function_config,
            'code': function_code,
            'type': 'http'
        }
        
        return {
            'success': True,
            'function_name': name,
            'trigger_type': 'http',
            'operation': operation,
            'methods': methods
        }
    
    def register_timer_function(self, name: str, operation: str, 
                              schedule: str) -> Dict[str, Any]:
        """Register timer trigger function"""
        function_config = {
            'name': name,
            'type': 'timer',
            'operation': operation,
            'schedule': schedule,
            'bindings': [
                {
                    "name": "mytimer",
                    "type": "timerTrigger",
                    "direction": "in",
                    "schedule": schedule
                }
            ]
        }
        
        # Create function code
        function_code = self._generate_timer_function_code(name, operation, schedule)
        
        self.functions[name] = {
            'config': function_config,
            'code': function_code,
            'type': 'timer'
        }
        
        return {
            'success': True,
            'function_name': name,
            'trigger_type': 'timer',
            'operation': operation,
            'schedule': schedule
        }
    
    def register_blob_function(self, name: str, operation: str,
                             container: str, path_pattern: str = "{name}") -> Dict[str, Any]:
        """Register blob trigger function"""
        function_config = {
            'name': name,
            'type': 'blob',
            'operation': operation,
            'container': container,
            'path_pattern': path_pattern,
            'bindings': [
                {
                    "name": "myblob",
                    "type": "blobTrigger",
                    "direction": "in",
                    "path": f"{container}/{path_pattern}",
                    "connection": "AzureWebJobsStorage"
                }
            ]
        }
        
        # Create function code
        function_code = self._generate_blob_function_code(name, operation, container)
        
        self.functions[name] = {
            'config': function_config,
            'code': function_code,
            'type': 'blob'
        }
        
        return {
            'success': True,
            'function_name': name,
            'trigger_type': 'blob',
            'operation': operation,
            'container': container
        }
    
    def _generate_http_function_code(self, name: str, operation: str, auth_level: str) -> str:
        """Generate Python code for HTTP function"""
        return f'''
import logging
import json
import azure.functions as func
from datetime import datetime

def main(req: func.HttpRequest) -> func.HttpResponse:
    logging.info('TuskLang HTTP function {name} processed a request.')
    
    try:
        # Get request data
        method = req.method
        url = req.url
        headers = dict(req.headers)
        
        # Get request body
        try:
            req_body = req.get_json()
        except ValueError:
            req_body = {{"raw": req.get_body().decode('utf-8')}}
        
        # Create TuskLang context
        context = {{
            "method": method,
            "url": url,
            "headers": headers,
            "body": req_body,
            "function_name": "{name}",
            "auth_level": "{auth_level}",
            "timestamp": datetime.now().isoformat()
        }}
        
        # Execute TuskLang operation
        result = execute_tusklang_operation("{operation}", context)
        
        # Return response
        return func.HttpResponse(
            json.dumps({{
                "success": True,
                "result": result,
                "function": "{name}",
                "operation": "{operation}",
                "execution_time": result.get("execution_time_ms", 0),
                "timestamp": datetime.now().isoformat()
            }}),
            status_code=200,
            headers={{"Content-Type": "application/json"}}
        )
        
    except Exception as e:
        logging.error(f"Function execution failed: {{e}}")
        
        return func.HttpResponse(
            json.dumps({{
                "success": False,
                "error": str(e),
                "function": "{name}",
                "operation": "{operation}",
                "timestamp": datetime.now().isoformat()
            }}),
            status_code=500,
            headers={{"Content-Type": "application/json"}}
        )

def execute_tusklang_operation(operation: str, context: dict):
    """Execute TuskLang operation within Azure Function"""
    try:
        # This would integrate with the actual TuskLang SDK
        # For now, simulate operation execution
        
        import time
        start_time = time.time()
        
        # Basic operation handling
        if operation.startswith("@http"):
            result = handle_http_operation(operation, context)
        elif "=" in operation and not operation.startswith("@"):
            # Variable assignment
            parts = operation.split("=", 1)
            var_name = parts[0].strip()
            value = parts[1].strip()
            result = {{"variable": var_name, "value": value, "type": "assignment"}}
        else:
            result = {{"operation": operation, "context": context, "type": "generic"}}
        
        execution_time = (time.time() - start_time) * 1000
        result["execution_time_ms"] = execution_time
        
        return result
        
    except Exception as e:
        return {{"error": str(e), "type": "execution_error"}}

def handle_http_operation(operation: str, context: dict):
    """Handle HTTP-specific operations"""
    return {{
        "type": "http_response",
        "method": context.get("method"),
        "operation": operation,
        "headers_received": len(context.get("headers", {{}})),
        "has_body": bool(context.get("body")),
        "processed": True
    }}
'''
    
    def _generate_timer_function_code(self, name: str, operation: str, schedule: str) -> str:
        """Generate Python code for timer function"""
        return f'''
import logging
import json
import azure.functions as func
from datetime import datetime

def main(mytimer: func.TimerRequest) -> None:
    logging.info('TuskLang timer function {name} executed.')
    
    try:
        # Get timer information
        is_past_due = mytimer.past_due
        
        # Create TuskLang context
        context = {{
            "schedule": "{schedule}",
            "past_due": is_past_due,
            "function_name": "{name}",
            "timestamp": datetime.now().isoformat()
        }}
        
        # Execute TuskLang operation
        result = execute_tusklang_operation("{operation}", context)
        
        logging.info(f"Timer function result: {{json.dumps(result)}}")
        
    except Exception as e:
        logging.error(f"Timer function execution failed: {{e}}")

def execute_tusklang_operation(operation: str, context: dict):
    """Execute TuskLang operation within timer function"""
    try:
        import time
        start_time = time.time()
        
        # Basic operation handling for timer
        if operation.startswith("@timer"):
            result = handle_timer_operation(operation, context)
        elif operation.startswith("@schedule"):
            result = handle_schedule_operation(operation, context)
        else:
            result = {{
                "operation": operation,
                "context": context,
                "type": "timer_execution"
            }}
        
        execution_time = (time.time() - start_time) * 1000
        result["execution_time_ms"] = execution_time
        
        return result
        
    except Exception as e:
        return {{"error": str(e), "type": "execution_error"}}

def handle_timer_operation(operation: str, context: dict):
    """Handle timer-specific operations"""
    return {{
        "type": "timer_response",
        "schedule": context.get("schedule"),
        "past_due": context.get("past_due"),
        "operation": operation,
        "executed_at": datetime.now().isoformat()
    }}

def handle_schedule_operation(operation: str, context: dict):
    """Handle scheduled operations"""
    return {{
        "type": "scheduled_response",
        "operation": operation,
        "schedule": context.get("schedule"),
        "next_execution": "calculated_based_on_schedule"
    }}
'''
    
    def _generate_blob_function_code(self, name: str, operation: str, container: str) -> str:
        """Generate Python code for blob function"""
        return f'''
import logging
import json
import azure.functions as func
from datetime import datetime

def main(myblob: func.InputStream) -> None:
    logging.info('TuskLang blob function {name} processed blob: %s', myblob.name)
    
    try:
        # Get blob information
        blob_name = myblob.name
        blob_length = myblob.length
        
        # Read blob content (first 1KB for processing)
        blob_content = myblob.read(1024).decode('utf-8', errors='ignore')
        
        # Create TuskLang context
        context = {{
            "blob_name": blob_name,
            "blob_length": blob_length,
            "container": "{container}",
            "content_preview": blob_content,
            "function_name": "{name}",
            "timestamp": datetime.now().isoformat()
        }}
        
        # Execute TuskLang operation
        result = execute_tusklang_operation("{operation}", context)
        
        logging.info(f"Blob function result: {{json.dumps(result)}}")
        
    except Exception as e:
        logging.error(f"Blob function execution failed: {{e}}")

def execute_tusklang_operation(operation: str, context: dict):
    """Execute TuskLang operation within blob function"""
    try:
        import time
        start_time = time.time()
        
        # Basic operation handling for blob
        if operation.startswith("@blob"):
            result = handle_blob_operation(operation, context)
        elif operation.startswith("@storage"):
            result = handle_storage_operation(operation, context)
        else:
            result = {{
                "operation": operation,
                "context": context,
                "type": "blob_execution"
            }}
        
        execution_time = (time.time() - start_time) * 1000
        result["execution_time_ms"] = execution_time
        
        return result
        
    except Exception as e:
        return {{"error": str(e), "type": "execution_error"}}

def handle_blob_operation(operation: str, context: dict):
    """Handle blob-specific operations"""
    return {{
        "type": "blob_response",
        "blob_name": context.get("blob_name"),
        "blob_size": context.get("blob_length"),
        "container": context.get("container"),
        "operation": operation,
        "processed": True
    }}

def handle_storage_operation(operation: str, context: dict):
    """Handle storage operations"""
    return {{
        "type": "storage_response",
        "operation": operation,
        "blob_info": {{
            "name": context.get("blob_name"),
            "size": context.get("blob_length"),
            "container": context.get("container")
        }}
    }}
'''
    
    def deploy_functions(self, output_dir: Optional[str] = None) -> str:
        """Deploy functions to Azure or create deployment package"""
        output_dir = output_dir or tempfile.mkdtemp(prefix='tusklang_azure_')
        
        try:
            # Create function app structure
            function_app_dir = os.path.join(output_dir, self.function_app_name)
            os.makedirs(function_app_dir, exist_ok=True)
            
            # Create host.json
            self._create_host_json(function_app_dir)
            
            # Create requirements.txt
            self._create_requirements_txt(function_app_dir)
            
            # Create functions
            for func_name, func_data in self.functions.items():
                func_dir = os.path.join(function_app_dir, func_name)
                os.makedirs(func_dir, exist_ok=True)
                
                # Write function.json
                with open(os.path.join(func_dir, 'function.json'), 'w') as f:
                    json.dump(func_data['config']['bindings'], f, indent=2)
                
                # Write __init__.py
                with open(os.path.join(func_dir, '__init__.py'), 'w') as f:
                    f.write(func_data['code'])
            
            # Create deployment package
            zip_path = f"{function_app_dir}.zip"
            self._create_deployment_zip(function_app_dir, zip_path)
            
            self.logger.info(f"Function deployment package created: {zip_path}")
            return zip_path
            
        except Exception as e:
            self.logger.error(f"Function deployment failed: {e}")
            raise
    
    def _create_host_json(self, app_dir: str):
        """Create host.json configuration file"""
        host_config = {
            "version": "2.0",
            "functionTimeout": "00:05:00",
            "extensionBundle": {
                "id": "Microsoft.Azure.Functions.ExtensionBundle",
                "version": "[2.*, 3.0.0)"
            },
            "logging": {
                "applicationInsights": {
                    "samplingSettings": {
                        "isEnabled": True
                    }
                }
            }
        }
        
        with open(os.path.join(app_dir, 'host.json'), 'w') as f:
            json.dump(host_config, f, indent=2)
    
    def _create_requirements_txt(self, app_dir: str):
        """Create requirements.txt file"""
        requirements = [
            "azure-functions>=1.11.0",
            "azure-storage-blob>=12.0.0",
            "azure-identity>=1.0.0",
            "requests>=2.25.0"
        ]
        
        with open(os.path.join(app_dir, 'requirements.txt'), 'w') as f:
            f.write('\n'.join(requirements))
    
    def _create_deployment_zip(self, source_dir: str, zip_path: str):
        """Create deployment ZIP package"""
        with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
            for root, dirs, files in os.walk(source_dir):
                for file in files:
                    file_path = os.path.join(root, file)
                    arcname = os.path.relpath(file_path, source_dir)
                    zipf.write(file_path, arcname)
    
    async def execute_operation(self, operation: str, context: Optional[Dict] = None,
                              trigger_type: str = 'manual') -> Dict[str, Any]:
        """Execute TuskLang operation in Azure Functions context"""
        start_time = datetime.now()
        self.stats['functions_executed'] += 1
        
        try:
            context = context or {}
            
            # Add Azure-specific context
            azure_context = {
                **context,
                'subscription_id': self.subscription_id,
                'resource_group': self.resource_group,
                'function_app': self.function_app_name,
                'location': self.location,
                'trigger_type': trigger_type
            }
            
            # Route based on trigger type
            if trigger_type == 'http':
                result = await self._execute_http_operation(operation, azure_context)
                self.stats['http_requests'] += 1
            elif trigger_type == 'timer':
                result = await self._execute_timer_operation(operation, azure_context)
                self.stats['timer_triggers'] += 1
            elif trigger_type == 'blob':
                result = await self._execute_blob_operation(operation, azure_context)
                self.stats['blob_triggers'] += 1
            else:
                result = await self._execute_manual_operation(operation, azure_context)
            
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            self.stats['total_execution_time_ms'] += execution_time
            
            return {
                'success': True,
                'result': result,
                'execution_time_ms': execution_time,
                'trigger_type': trigger_type,
                'azure_context': True,
                'timestamp': datetime.now().isoformat()
            }
            
        except Exception as e:
            self.stats['errors'] += 1
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            
            self.logger.error(f"Azure Functions operation failed: {e}")
            
            return {
                'success': False,
                'error': str(e),
                'error_type': type(e).__name__,
                'execution_time_ms': execution_time,
                'trigger_type': trigger_type,
                'azure_context': True,
                'timestamp': datetime.now().isoformat()
            }
    
    async def _execute_http_operation(self, operation: str, context: Dict) -> Any:
        """Execute HTTP trigger operation"""
        # Simulate HTTP request processing
        http_result = {
            'operation': operation,
            'context': context,
            'trigger': 'http',
            'method': context.get('method', 'GET'),
            'status_code': 200,
            'response_type': 'application/json'
        }
        
        return http_result
    
    async def _execute_timer_operation(self, operation: str, context: Dict) -> Any:
        """Execute timer trigger operation"""
        timer_result = {
            'operation': operation,
            'context': context,
            'trigger': 'timer',
            'scheduled': True,
            'next_run': 'calculated'
        }
        
        return timer_result
    
    async def _execute_blob_operation(self, operation: str, context: Dict) -> Any:
        """Execute blob trigger operation"""
        blob_result = {
            'operation': operation,
            'context': context,
            'trigger': 'blob',
            'blob_processed': True,
            'storage_account': 'configured'
        }
        
        return blob_result
    
    async def _execute_manual_operation(self, operation: str, context: Dict) -> Any:
        """Execute manual operation"""
        try:
            # Try advanced operators if available
            from ...core.base.advanced_operators_integration import AdvancedOperatorIntegration
            
            operator_integration = AdvancedOperatorIntegration()
            
            if operation.startswith('@azure'):
                return await self._execute_azure_specific_operation(operation, context)
            else:
                # Delegate to advanced operators
                return await self._delegate_to_advanced_operators(
                    operator_integration, operation, context
                )
                
        except ImportError:
            # Fallback execution
            return await self._execute_basic_operation(operation, context)
    
    def get_stats(self) -> Dict[str, Any]:
        """Get Azure Functions integration statistics"""
        return {
            **self.stats,
            'subscription_id': self.subscription_id,
            'resource_group': self.resource_group,
            'function_app_name': self.function_app_name,
            'location': self.location,
            'registered_functions': len(self.functions),
            'azure_sdk_available': AZURE_SDK_AVAILABLE,
            'function_types': {
                func_name: func_data['type'] 
                for func_name, func_data in self.functions.items()
            }
        }


# Convenience functions
async def execute_in_azure_functions(operation: str, context: Optional[Dict] = None,
                                   trigger_type: str = 'manual',
                                   config: Optional[Dict] = None) -> Dict[str, Any]:
    """
    Convenience function to execute TuskLang operation in Azure Functions
    """
    integration = AzureFunctionsIntegration(config)
    return await integration.execute_operation(operation, context, trigger_type)


def create_azure_function_app(app_name: str, resource_group: str, 
                            subscription_id: str) -> str:
    """Create Azure Function App with TuskLang integration"""
    integration = AzureFunctionsIntegration({
        'function_app_name': app_name,
        'resource_group': resource_group,
        'subscription_id': subscription_id
    })
    
    # Create function app
    result = integration.create_function_app()
    
    if result['success']:
        return f"Function app created: {result['deployment']}"
    else:
        return f"Function app creation failed: {result['error']}" 