"""
Browser Extension Integration for TuskLang Python SDK
Provides browser extension framework with Manifest v3 compatibility
"""

import asyncio
import json
import base64
import hashlib
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime
import logging
import os
import zipfile
import tempfile


class BrowserExtensionIntegration:
    """
    Main browser extension integration class for TuskLang Python SDK.
    Enables TuskLang to run as browser extensions across multiple browsers.
    """
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.extension_id = self.config.get('extension_id', 'tusklang-extension')
        self.manifest_version = self.config.get('manifest_version', 3)
        
        # Browser compatibility settings
        self.target_browsers = self.config.get('browsers', ['chrome', 'firefox', 'safari', 'edge'])
        self.min_versions = {
            'chrome': '90',
            'firefox': '88', 
            'safari': '14',
            'edge': '90'
        }
        
        # Extension architecture
        self.content_scripts = {}
        self.background_scripts = {}
        self.popup_scripts = {}
        self.storage_handlers = {}
        
        # Performance tracking
        self.stats = {
            'messages_processed': 0,
            'storage_operations': 0,
            'content_script_injections': 0,
            'cross_origin_requests': 0,
            'errors': 0,
            'active_tabs': 0
        }
        
        # Communication channels
        self.message_handlers = {}
        self.port_connections = {}
        self.event_listeners = {}
        
        # Security settings
        self.permissions = set()
        self.host_permissions = set()
        self.content_security_policy = None
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Set up extension framework
        self._initialize_extension()
    
    def _initialize_extension(self):
        """Initialize browser extension framework"""
        try:
            # Configure permissions
            self._configure_permissions()
            
            # Set up message handling
            self._setup_message_system()
            
            # Initialize storage
            self._initialize_storage()
            
            # Set up security policies
            self._configure_security()
            
            self.logger.info("Browser extension integration initialized")
            
        except Exception as e:
            self.logger.error(f"Extension initialization failed: {e}")
            raise
    
    def _configure_permissions(self):
        """Configure extension permissions for Manifest v3"""
        # Standard permissions
        self.permissions.update([
            'storage',
            'activeTab',
            'scripting',
            'tabs'
        ])
        
        # Optional permissions based on features
        if self.config.get('network_access', True):
            self.permissions.add('webRequest')
            
        if self.config.get('cross_origin', False):
            self.permissions.add('webRequestBlocking')
            
        if self.config.get('background_sync', False):
            self.permissions.add('background')
            
        # Host permissions for cross-origin requests
        if self.config.get('host_permissions'):
            self.host_permissions.update(self.config['host_permissions'])
    
    def _setup_message_system(self):
        """Set up cross-script messaging system"""
        # Register default message handlers
        self.message_handlers.update({
            'execute_operation': self._handle_execute_operation,
            'get_stats': self._handle_get_stats,
            'configure': self._handle_configure,
            'storage_get': self._handle_storage_get,
            'storage_set': self._handle_storage_set,
            'inject_content': self._handle_inject_content
        })
    
    def _initialize_storage(self):
        """Initialize extension storage systems"""
        # Storage areas: local, sync, session (Manifest v3)
        self.storage_handlers = {
            'local': LocalStorageHandler(),
            'sync': SyncStorageHandler(),  
            'session': SessionStorageHandler()
        }
    
    def _configure_security(self):
        """Configure Content Security Policy and other security settings"""
        if self.manifest_version == 3:
            # Manifest v3 CSP - more restrictive
            self.content_security_policy = {
                "extension_pages": "script-src 'self' 'wasm-unsafe-eval'; object-src 'self';"
            }
        else:
            # Manifest v2 CSP
            self.content_security_policy = {
                "content_security_policy": "script-src 'self' 'unsafe-eval'; object-src 'self';"
            }
    
    async def execute_operation(self, operation: str, context: Optional[Dict] = None,
                              target: str = 'background') -> Dict[str, Any]:
        """Execute TuskLang operation in browser extension context"""
        start_time = datetime.now()
        self.stats['messages_processed'] += 1
        
        try:
            # Validate operation
            if not operation or not isinstance(operation, str):
                raise ValueError("Operation must be a non-empty string")
            
            context = context or {}
            
            # Route to appropriate execution context
            if target == 'content':
                result = await self._execute_in_content_script(operation, context)
            elif target == 'background':
                result = await self._execute_in_background(operation, context)
            elif target == 'popup':
                result = await self._execute_in_popup(operation, context)
            else:
                raise ValueError(f"Unknown execution target: {target}")
            
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            
            return {
                'success': True,
                'result': result,
                'execution_time_ms': execution_time,
                'target': target,
                'timestamp': datetime.now().isoformat()
            }
            
        except Exception as e:
            self.stats['errors'] += 1
            execution_time = (datetime.now() - start_time).total_seconds() * 1000
            
            self.logger.error(f"Extension operation failed: {e}")
            
            return {
                'success': False,
                'error': str(e),
                'error_type': type(e).__name__,
                'execution_time_ms': execution_time,
                'target': target,
                'timestamp': datetime.now().isoformat()
            }
    
    async def _execute_in_content_script(self, operation: str, context: Dict) -> Any:
        """Execute operation in content script context"""
        try:
            # Import TuskLang core
            from ...core.base.advanced_operators_integration import AdvancedOperatorIntegration
            
            # Create operator integration
            operator_integration = AdvancedOperatorIntegration()
            
            # Execute based on operation type
            if operation.startswith('@dom'):
                return await self._execute_dom_operation(operation, context)
            elif operation.startswith('@inject'):
                return await self._execute_injection_operation(operation, context)
            elif operation.startswith('@storage'):
                return await self._execute_storage_operation(operation, context)
            else:
                # Use advanced operator integration
                return await self._delegate_to_advanced_operators(
                    operator_integration, operation, context
                )
                
        except ImportError:
            # Fallback execution without advanced operators
            return await self._execute_basic_operation(operation, context)
    
    async def _execute_in_background(self, operation: str, context: Dict) -> Any:
        """Execute operation in background script context"""
        try:
            # Background script operations
            if operation.startswith('@network'):
                return await self._execute_network_operation(operation, context)
            elif operation.startswith('@tabs'):
                return await self._execute_tabs_operation(operation, context)  
            elif operation.startswith('@storage'):
                return await self._execute_storage_operation(operation, context)
            elif operation.startswith('@schedule'):
                return await self._execute_scheduled_operation(operation, context)
            else:
                # Generic background operation
                return await self._execute_basic_operation(operation, context)
                
        except Exception as e:
            raise RuntimeError(f"Background execution failed: {e}")
    
    async def _execute_in_popup(self, operation: str, context: Dict) -> Any:
        """Execute operation in popup script context"""
        try:
            # Popup-specific operations
            if operation.startswith('@ui'):
                return await self._execute_ui_operation(operation, context)
            elif operation.startswith('@settings'):
                return await self._execute_settings_operation(operation, context)
            else:
                # Generic popup operation
                return await self._execute_basic_operation(operation, context)
                
        except Exception as e:
            raise RuntimeError(f"Popup execution failed: {e}")
    
    async def _execute_dom_operation(self, operation: str, context: Dict) -> Any:
        """Execute DOM manipulation operations"""
        # Parse DOM operation: @dom("selector", "action", data)
        import re
        match = re.match(r'@dom\("([^"]+)"\s*,\s*"([^"]+)"(?:\s*,\s*(.+))?\)', operation)
        
        if not match:
            raise ValueError("Invalid @dom operation syntax")
        
        selector = match.group(1)
        action = match.group(2)
        data = match.group(3)
        
        # Simulate DOM operations (would use browser APIs in real extension)
        dom_result = {
            'selector': selector,
            'action': action,
            'data': data,
            'elements_found': 1,  # Simulated
            'success': True
        }
        
        return dom_result
    
    async def _execute_injection_operation(self, operation: str, context: Dict) -> Any:
        """Execute script injection operations"""
        self.stats['content_script_injections'] += 1
        
        # Parse injection: @inject("script", "target")
        import re
        match = re.match(r'@inject\("([^"]+)"\s*,\s*"([^"]+)"\)', operation)
        
        if not match:
            raise ValueError("Invalid @inject operation syntax")
        
        script = match.group(1)
        target = match.group(2)
        
        # Simulate script injection
        injection_result = {
            'script': script,
            'target': target,
            'injected': True,
            'execution_id': f"inject_{hash(script)}_{int(datetime.now().timestamp())}"
        }
        
        return injection_result
    
    async def _execute_storage_operation(self, operation: str, context: Dict) -> Any:
        """Execute extension storage operations"""
        self.stats['storage_operations'] += 1
        
        # Parse storage operation: @storage("area", "action", key, value)
        import re
        match = re.match(r'@storage\("([^"]+)"\s*,\s*"([^"]+)"\s*,\s*"([^"]+)"(?:\s*,\s*(.+))?\)', operation)
        
        if not match:
            raise ValueError("Invalid @storage operation syntax")
        
        area = match.group(1)  # local, sync, session
        action = match.group(2)  # get, set, remove, clear
        key = match.group(3)
        value = match.group(4)
        
        storage_handler = self.storage_handlers.get(area)
        if not storage_handler:
            raise ValueError(f"Unknown storage area: {area}")
        
        if action == 'get':
            result = await storage_handler.get(key)
        elif action == 'set':
            result = await storage_handler.set(key, json.loads(value) if value else None)
        elif action == 'remove':
            result = await storage_handler.remove(key)
        elif action == 'clear':
            result = await storage_handler.clear()
        else:
            raise ValueError(f"Unknown storage action: {action}")
        
        return result
    
    async def _execute_network_operation(self, operation: str, context: Dict) -> Any:
        """Execute network request operations"""
        self.stats['cross_origin_requests'] += 1
        
        # Parse network operation: @network("method", "url", headers, data)
        import re
        match = re.match(r'@network\("([^"]+)"\s*,\s*"([^"]+)"(?:\s*,\s*({[^}]+}))(?:\s*,\s*(.+))?\)', operation)
        
        if not match:
            raise ValueError("Invalid @network operation syntax")
        
        method = match.group(1)
        url = match.group(2)
        headers = json.loads(match.group(3)) if match.group(3) else {}
        data = match.group(4)
        
        # Simulate network request (would use fetch API in real extension)
        network_result = {
            'method': method,
            'url': url,
            'headers': headers,
            'data': data,
            'status': 200,
            'response': f'Simulated response for {method} {url}',
            'timestamp': datetime.now().isoformat()
        }
        
        return network_result
    
    async def _execute_tabs_operation(self, operation: str, context: Dict) -> Any:
        """Execute browser tabs operations"""
        # Parse tabs operation: @tabs("action", data)
        import re
        match = re.match(r'@tabs\("([^"]+)"(?:\s*,\s*(.+))?\)', operation)
        
        if not match:
            raise ValueError("Invalid @tabs operation syntax")
        
        action = match.group(1)
        data = match.group(2)
        
        # Simulate tabs operations
        if action == 'create':
            result = {'action': 'create', 'tab_id': 12345, 'url': data or 'about:blank'}
        elif action == 'query':
            result = {'action': 'query', 'tabs': [{'id': 1, 'url': 'https://example.com', 'active': True}]}
        elif action == 'update':
            result = {'action': 'update', 'tab_id': int(data) if data else 1, 'updated': True}
        else:
            result = {'action': action, 'data': data, 'simulated': True}
        
        self.stats['active_tabs'] += 1
        return result
    
    async def _execute_basic_operation(self, operation: str, context: Dict) -> Any:
        """Execute basic operation without advanced features"""
        # Basic expression evaluation
        if '=' in operation and not operation.startswith('@'):
            # Variable assignment
            parts = operation.split('=', 1)
            var_name = parts[0].strip()
            value = parts[1].strip()
            
            return {
                'type': 'assignment',
                'variable': var_name,
                'value': value,
                'context': 'extension'
            }
        else:
            return {
                'type': 'expression',
                'operation': operation,
                'context': context,
                'evaluated': True
            }
    
    def create_manifest(self, browser: str = 'chrome') -> Dict[str, Any]:
        """Create browser-specific manifest file"""
        base_manifest = {
            'manifest_version': self.manifest_version,
            'name': self.config.get('name', 'TuskLang Extension'),
            'version': self.config.get('version', '1.0.0'),
            'description': self.config.get('description', 'TuskLang Python SDK Browser Extension'),
            'permissions': list(self.permissions),
            'host_permissions': list(self.host_permissions)
        }
        
        if self.manifest_version == 3:
            # Manifest v3 format
            base_manifest.update({
                'action': {
                    'default_popup': 'popup.html',
                    'default_title': 'TuskLang Extension'
                },
                'background': {
                    'service_worker': 'background.js'
                },
                'content_scripts': [{
                    'matches': ['<all_urls>'],
                    'js': ['content.js'],
                    'run_at': 'document_idle'
                }]
            })
            
            if self.content_security_policy:
                base_manifest.update(self.content_security_policy)
                
        else:
            # Manifest v2 format (for Firefox compatibility)
            base_manifest.update({
                'browser_action': {
                    'default_popup': 'popup.html',
                    'default_title': 'TuskLang Extension'
                },
                'background': {
                    'scripts': ['background.js'],
                    'persistent': False
                },
                'content_scripts': [{
                    'matches': ['<all_urls>'],
                    'js': ['content.js'],
                    'run_at': 'document_idle'
                }]
            })
            
            if self.content_security_policy:
                base_manifest.update(self.content_security_policy)
        
        # Browser-specific adjustments
        if browser == 'firefox':
            base_manifest['browser_specific_settings'] = {
                'gecko': {
                    'id': f'{self.extension_id}@tusklang.io',
                    'strict_min_version': '88.0'
                }
            }
        elif browser == 'safari':
            # Safari-specific settings
            base_manifest['safari_specific_settings'] = {
                'minimum_version': '14.0'
            }
        
        return base_manifest
    
    def create_extension_package(self, output_dir: str, browser: str = 'chrome') -> str:
        """Create extension package for distribution"""
        package_dir = os.path.join(output_dir, f'{self.extension_id}-{browser}')
        os.makedirs(package_dir, exist_ok=True)
        
        try:
            # Create manifest
            manifest = self.create_manifest(browser)
            with open(os.path.join(package_dir, 'manifest.json'), 'w') as f:
                json.dump(manifest, f, indent=2)
            
            # Create background script
            self._create_background_script(package_dir)
            
            # Create content script
            self._create_content_script(package_dir)
            
            # Create popup files
            self._create_popup_files(package_dir)
            
            # Create ZIP package
            zip_path = f'{package_dir}.zip'
            with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
                for root, dirs, files in os.walk(package_dir):
                    for file in files:
                        file_path = os.path.join(root, file)
                        arcname = os.path.relpath(file_path, package_dir)
                        zipf.write(file_path, arcname)
            
            self.logger.info(f"Extension package created: {zip_path}")
            return zip_path
            
        except Exception as e:
            self.logger.error(f"Package creation failed: {e}")
            raise
    
    def _create_background_script(self, package_dir: str):
        """Create background script file"""
        background_js = f'''
// TuskLang Extension Background Script
// Generated for extension: {self.extension_id}

class TuskLangBackground {{
    constructor() {{
        this.messageHandlers = new Map();
        this.setupMessageHandling();
        this.setupContextMenus();
    }}
    
    setupMessageHandling() {{
        chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {{
            this.handleMessage(request, sender, sendResponse);
            return true; // Keep message channel open for async response
        }});
        
        chrome.runtime.onConnect.addListener((port) => {{
            this.handlePortConnection(port);
        }});
    }}
    
    async handleMessage(request, sender, sendResponse) {{
        try {{
            const {{ action, data }} = request;
            
            switch (action) {{
                case 'execute_operation':
                    const result = await this.executeOperation(data.operation, data.context);
                    sendResponse(result);
                    break;
                case 'get_stats':
                    sendResponse(this.getStats());
                    break;
                case 'network_request':
                    const response = await this.makeNetworkRequest(data);
                    sendResponse(response);
                    break;
                default:
                    sendResponse({{ success: false, error: `Unknown action: ${{action}}` }});
            }}
        }} catch (error) {{
            sendResponse({{
                success: false,
                error: error.message,
                timestamp: new Date().toISOString()
            }});
        }}
    }}
    
    async executeOperation(operation, context = {{}}) {{
        const startTime = Date.now();
        
        try {{
            // TuskLang operation execution logic would go here
            // This is a simplified simulation
            
            let result;
            if (operation.startsWith('@network')) {{
                result = await this.handleNetworkOperation(operation, context);
            }} else if (operation.startsWith('@tabs')) {{
                result = await this.handleTabsOperation(operation, context);
            }} else if (operation.startsWith('@storage')) {{
                result = await this.handleStorageOperation(operation, context);
            }} else {{
                result = {{ operation, context, executed: true }};
            }}
            
            return {{
                success: true,
                result: result,
                execution_time_ms: Date.now() - startTime,
                timestamp: new Date().toISOString()
            }};
            
        }} catch (error) {{
            return {{
                success: false,
                error: error.message,
                execution_time_ms: Date.now() - startTime,
                timestamp: new Date().toISOString()
            }};
        }}
    }}
    
    async handleNetworkOperation(operation, context) {{
        // Network operation simulation
        return {{
            type: 'network',
            operation,
            context,
            simulated: true
        }};
    }}
    
    async handleTabsOperation(operation, context) {{
        // Tabs operation simulation  
        return {{
            type: 'tabs',
            operation,
            context,
            simulated: true
        }};
    }}
    
    async handleStorageOperation(operation, context) {{
        // Storage operation simulation
        return {{
            type: 'storage',
            operation,
            context,
            simulated: true
        }};
    }}
    
    setupContextMenus() {{
        chrome.contextMenus.create({{
            id: 'tusklang-execute',
            title: 'Execute TuskLang',
            contexts: ['all']
        }});
        
        chrome.contextMenus.onClicked.addListener((info, tab) => {{
            if (info.menuItemId === 'tusklang-execute') {{
                this.executeFromContextMenu(info, tab);
            }}
        }});
    }}
    
    getStats() {{
        return {{
            extension_id: '{self.extension_id}',
            version: chrome.runtime.getManifest().version,
            uptime: Date.now(),
            active: true
        }};
    }}
}}

// Initialize background script
const tuskLangBackground = new TuskLangBackground();
console.log('TuskLang Extension Background Script initialized');
'''
        
        with open(os.path.join(package_dir, 'background.js'), 'w') as f:
            f.write(background_js)
    
    def _create_content_script(self, package_dir: str):
        """Create content script file"""
        content_js = f'''
// TuskLang Extension Content Script
// Generated for extension: {self.extension_id}

class TuskLangContent {{
    constructor() {{
        this.setupMessageHandling();
        this.injectTuskLangAPI();
        this.observePageChanges();
    }}
    
    setupMessageHandling() {{
        chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {{
            this.handleMessage(request, sender, sendResponse);
            return true;
        }});
    }}
    
    async handleMessage(request, sender, sendResponse) {{
        try {{
            const {{ action, data }} = request;
            
            switch (action) {{
                case 'execute_dom_operation':
                    const result = await this.executeDOMOperation(data.operation, data.context);
                    sendResponse(result);
                    break;
                case 'inject_script':
                    const injected = this.injectScript(data.script);
                    sendResponse({{ success: injected }});
                    break;
                case 'get_page_info':
                    sendResponse(this.getPageInfo());
                    break;
                default:
                    sendResponse({{ success: false, error: `Unknown action: ${{action}}` }});
            }}
        }} catch (error) {{
            sendResponse({{
                success: false,
                error: error.message,
                timestamp: new Date().toISOString()
            }});
        }}
    }}
    
    async executeDOMOperation(operation, context = {{}}) {{
        const startTime = Date.now();
        
        try {{
            // Parse DOM operation
            const domMatch = operation.match(/@dom\\("([^"]+)"\\s*,\\s*"([^"]+)"(?:\\s*,\\s*(.+))?\\)/);
            if (!domMatch) {{
                throw new Error('Invalid DOM operation syntax');
            }}
            
            const [, selector, action, data] = domMatch;
            const elements = document.querySelectorAll(selector);
            
            let result = {{
                selector,
                action,
                elements_found: elements.length,
                results: []
            }};
            
            elements.forEach((element, index) => {{
                let elementResult = {{ index }};
                
                switch (action) {{
                    case 'text':
                        elementResult.value = element.textContent;
                        break;
                    case 'html':
                        elementResult.value = element.innerHTML;
                        break;
                    case 'attribute':
                        const attr = data ? JSON.parse(data) : 'id';
                        elementResult.value = element.getAttribute(attr);
                        break;
                    case 'style':
                        const style = data ? JSON.parse(data) : 'display';
                        elementResult.value = window.getComputedStyle(element)[style];
                        break;
                    case 'click':
                        element.click();
                        elementResult.value = 'clicked';
                        break;
                    default:
                        elementResult.value = 'unknown_action';
                }}
                
                result.results.push(elementResult);
            }});
            
            return {{
                success: true,
                result: result,
                execution_time_ms: Date.now() - startTime,
                timestamp: new Date().toISOString()
            }};
            
        }} catch (error) {{
            return {{
                success: false,
                error: error.message,
                execution_time_ms: Date.now() - startTime,
                timestamp: new Date().toISOString()
            }};
        }}
    }}
    
    injectTuskLangAPI() {{
        // Inject TuskLang API into page context
        const script = document.createElement('script');
        script.textContent = `
            window.TuskLang = {{
                execute: function(operation, context) {{
                    return new Promise((resolve) => {{
                        window.postMessage({{
                            type: 'TUSKLANG_EXECUTE',
                            operation: operation,
                            context: context
                        }}, '*');
                        
                        const handleResponse = (event) => {{
                            if (event.data.type === 'TUSKLANG_RESPONSE') {{
                                window.removeEventListener('message', handleResponse);
                                resolve(event.data.result);
                            }}
                        }};
                        
                        window.addEventListener('message', handleResponse);
                    }});
                }}
            }};
        `;
        
        document.documentElement.appendChild(script);
        script.remove();
        
        // Listen for messages from injected API
        window.addEventListener('message', async (event) => {{
            if (event.data.type === 'TUSKLANG_EXECUTE') {{
                const result = await this.executeDOMOperation(
                    event.data.operation, 
                    event.data.context
                );
                
                window.postMessage({{
                    type: 'TUSKLANG_RESPONSE',
                    result: result
                }}, '*');
            }}
        }});
    }}
    
    getPageInfo() {{
        return {{
            url: window.location.href,
            title: document.title,
            domain: window.location.hostname,
            elements: {{
                total: document.querySelectorAll('*').length,
                scripts: document.querySelectorAll('script').length,
                links: document.querySelectorAll('a').length,
                images: document.querySelectorAll('img').length
            }},
            timestamp: new Date().toISOString()
        }};
    }}
    
    observePageChanges() {{
        // Observe DOM changes for dynamic content
        const observer = new MutationObserver((mutations) => {{
            mutations.forEach((mutation) => {{
                if (mutation.type === 'childList') {{
                    // Handle dynamic content changes
                    console.log('DOM changed:', mutation);
                }}
            }});
        }});
        
        observer.observe(document.body, {{
            childList: true,
            subtree: true
        }});
    }}
}}

// Initialize content script
const tuskLangContent = new TuskLangContent();
console.log('TuskLang Extension Content Script initialized');
'''
        
        with open(os.path.join(package_dir, 'content.js'), 'w') as f:
            f.write(content_js)
    
    def _create_popup_files(self, package_dir: str):
        """Create popup HTML and JavaScript files"""
        # popup.html
        popup_html = f'''
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <style>
        body {{
            width: 350px;
            padding: 20px;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }}
        
        .header {{
            text-align: center;
            margin-bottom: 20px;
            border-bottom: 1px solid #ddd;
            padding-bottom: 10px;
        }}
        
        .operation-input {{
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            margin-bottom: 10px;
            font-family: monospace;
        }}
        
        .execute-button {{
            width: 100%;
            padding: 10px;
            background: #007cba;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
        }}
        
        .execute-button:hover {{
            background: #005a87;
        }}
        
        .result-container {{
            margin-top: 15px;
            padding: 10px;
            background: #f5f5f5;
            border-radius: 4px;
            max-height: 200px;
            overflow-y: auto;
        }}
        
        .stats {{
            margin-top: 10px;
            font-size: 12px;
            color: #666;
        }}
    </style>
</head>
<body>
    <div class="header">
        <h2>TuskLang Extension</h2>
        <p>Execute TuskLang operations</p>
    </div>
    
    <div>
        <textarea class="operation-input" id="operationInput" 
                  placeholder="Enter TuskLang operation (e.g., @dom(&quot;h1&quot;, &quot;text&quot;))"></textarea>
        <button class="execute-button" id="executeButton">Execute Operation</button>
    </div>
    
    <div class="result-container" id="resultContainer" style="display:none;">
        <h4>Result:</h4>
        <pre id="resultOutput"></pre>
    </div>
    
    <div class="stats" id="stats">
        Extension ID: {self.extension_id}
    </div>
    
    <script src="popup.js"></script>
</body>
</html>
'''
        
        with open(os.path.join(package_dir, 'popup.html'), 'w') as f:
            f.write(popup_html)
        
        # popup.js
        popup_js = '''
document.addEventListener('DOMContentLoaded', function() {
    const operationInput = document.getElementById('operationInput');
    const executeButton = document.getElementById('executeButton');
    const resultContainer = document.getElementById('resultContainer');
    const resultOutput = document.getElementById('resultOutput');
    const statsElement = document.getElementById('stats');
    
    executeButton.addEventListener('click', executeOperation);
    operationInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter' && (e.ctrlKey || e.metaKey)) {
            executeOperation();
        }
    });
    
    async function executeOperation() {
        const operation = operationInput.value.trim();
        if (!operation) {
            alert('Please enter an operation');
            return;
        }
        
        executeButton.disabled = true;
        executeButton.textContent = 'Executing...';
        
        try {
            // Get current tab
            const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
            const currentTab = tabs[0];
            
            // Send operation to background script
            const result = await chrome.runtime.sendMessage({
                action: 'execute_operation',
                data: {
                    operation: operation,
                    context: {
                        tab_id: currentTab.id,
                        url: currentTab.url
                    }
                }
            });
            
            // Display result
            resultOutput.textContent = JSON.stringify(result, null, 2);
            resultContainer.style.display = 'block';
            
            // Update stats
            updateStats();
            
        } catch (error) {
            resultOutput.textContent = `Error: ${error.message}`;
            resultContainer.style.display = 'block';
        } finally {
            executeButton.disabled = false;
            executeButton.textContent = 'Execute Operation';
        }
    }
    
    async function updateStats() {
        try {
            const stats = await chrome.runtime.sendMessage({ action: 'get_stats' });
            statsElement.textContent = `Stats: ${JSON.stringify(stats)}`;
        } catch (error) {
            console.error('Failed to get stats:', error);
        }
    }
    
    // Initialize
    updateStats();
});
'''
        
        with open(os.path.join(package_dir, 'popup.js'), 'w') as f:
            f.write(popup_js)
    
    def get_stats(self) -> Dict[str, Any]:
        """Get extension statistics"""
        return {
            **self.stats,
            'extension_id': self.extension_id,
            'manifest_version': self.manifest_version,
            'target_browsers': self.target_browsers,
            'permissions': list(self.permissions),
            'host_permissions': list(self.host_permissions),
            'active_handlers': len(self.message_handlers),
            'storage_areas': list(self.storage_handlers.keys())
        }


# Storage handler classes
class LocalStorageHandler:
    """Handle local extension storage"""
    
    def __init__(self):
        self.storage = {}
    
    async def get(self, key: str) -> Any:
        return self.storage.get(key)
    
    async def set(self, key: str, value: Any) -> bool:
        self.storage[key] = value
        return True
    
    async def remove(self, key: str) -> bool:
        return self.storage.pop(key, None) is not None
    
    async def clear(self) -> bool:
        self.storage.clear()
        return True


class SyncStorageHandler:
    """Handle sync extension storage"""
    
    def __init__(self):
        self.storage = {}
    
    async def get(self, key: str) -> Any:
        return self.storage.get(key)
    
    async def set(self, key: str, value: Any) -> bool:
        self.storage[key] = value
        return True
    
    async def remove(self, key: str) -> bool:
        return self.storage.pop(key, None) is not None
    
    async def clear(self) -> bool:
        self.storage.clear()
        return True


class SessionStorageHandler:
    """Handle session extension storage (Manifest v3)"""
    
    def __init__(self):
        self.storage = {}
    
    async def get(self, key: str) -> Any:
        return self.storage.get(key)
    
    async def set(self, key: str, value: Any) -> bool:
        self.storage[key] = value
        return True
    
    async def remove(self, key: str) -> bool:
        return self.storage.pop(key, None) is not None
    
    async def clear(self) -> bool:
        self.storage.clear()
        return True


# Convenience functions
async def execute_in_browser_extension(operation: str, context: Optional[Dict] = None,
                                     target: str = 'background',
                                     config: Optional[Dict] = None) -> Dict[str, Any]:
    """
    Convenience function to execute TuskLang operation in browser extension
    """
    integration = BrowserExtensionIntegration(config)
    return await integration.execute_operation(operation, context, target)


def create_browser_extension(extension_id: str, output_dir: str, 
                           browsers: List[str] = None) -> List[str]:
    """Create browser extension packages for multiple browsers"""
    browsers = browsers or ['chrome', 'firefox', 'edge']
    
    integration = BrowserExtensionIntegration({
        'extension_id': extension_id,
        'browsers': browsers
    })
    
    packages = []
    for browser in browsers:
        package_path = integration.create_extension_package(output_dir, browser)
        packages.append(package_path)
    
    return packages 