"""
Jekyll Integration for TuskLang Python SDK
Provides static site generation with Liquid templates and build-time processing
"""

import asyncio
import json
import os
import yaml
import re
import subprocess
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime
import logging
import tempfile
import shutil


class JekyllIntegration:
    """
    Main Jekyll integration class for TuskLang Python SDK.
    Enables TuskLang to process content at build time with Jekyll.
    """
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.jekyll_site_path = self.config.get('jekyll_site_path')
        self.plugin_name = self.config.get('plugin_name', 'tusklang-jekyll')
        
        # Jekyll configuration
        self.jekyll_version = self.config.get('jekyll_version', '4.0')
        self.ruby_version = self.config.get('ruby_version', '2.7')
        self.github_pages_compatible = self.config.get('github_pages', True)
        
        # Plugin components
        self.plugin_path = None
        self.liquid_filters = {}
        self.liquid_tags = {}
        self.generators = {}
        self.hooks = {}
        
        # Performance tracking
        self.stats = {
            'pages_processed': 0,
            'liquid_operations': 0,
            'build_time_operations': 0,
            'template_renders': 0,
            'errors': 0,
            'total_build_time_ms': 0
        }
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Initialize Jekyll integration
        self._initialize_jekyll()
    
    def _initialize_jekyll(self):
        """Initialize Jekyll integration and plugin system"""
        try:
            # Validate Jekyll site
            if self.jekyll_site_path and not self._validate_jekyll_site():
                self.logger.warning("Invalid Jekyll site path - creating plugin structure")
            
            # Create plugin structure
            self._create_plugin_structure()
            
            # Generate Liquid filters and tags
            self._generate_liquid_extensions()
            
            # Create Jekyll generators
            self._create_jekyll_generators()
            
            # Set up build hooks
            self._setup_build_hooks()
            
            self.logger.info("Jekyll integration initialized successfully")
            
        except Exception as e:
            self.logger.error(f"Jekyll integration initialization failed: {e}")
            raise
    
    def _validate_jekyll_site(self) -> bool:
        """Validate Jekyll site structure"""
        if not self.jekyll_site_path or not os.path.exists(self.jekyll_site_path):
            return False
        
        required_files = ['_config.yml']
        required_dirs = ['_posts', '_layouts', '_includes']
        
        has_config = any(os.path.exists(os.path.join(self.jekyll_site_path, f)) for f in required_files)
        has_dirs = any(os.path.exists(os.path.join(self.jekyll_site_path, d)) for d in required_dirs)
        
        return has_config or has_dirs
    
    def _create_plugin_structure(self):
        """Create Jekyll plugin directory structure"""
        if self.jekyll_site_path:
            # Create plugin within Jekyll site
            self.plugin_root = os.path.join(self.jekyll_site_path, '_plugins', 'tusklang')
        else:
            # Create standalone plugin
            self.plugin_root = tempfile.mkdtemp(prefix='tusklang_jekyll_')
        
        # Create directory structure
        directories = [
            '',  # Root
            'filters',
            'tags',
            'generators',
            'hooks'
        ]
        
        for directory in directories:
            dir_path = os.path.join(self.plugin_root, directory) if directory else self.plugin_root
            os.makedirs(dir_path, exist_ok=True)
        
        self.plugin_path = self.plugin_root
        self.logger.info(f"Plugin structure created at: {self.plugin_root}")
    
    def _generate_liquid_extensions(self):
        """Generate Liquid filters and tags for TuskLang operations"""
        
        # TuskLang Liquid Filter
        filter_code = '''
# TuskLang Liquid Filter for Jekyll
# Enables TuskLang operations within Liquid templates

module Jekyll
  module TuskLangFilters
    
    # Execute TuskLang operation and return result
    # Usage: {{ "operation" | tusklang }}
    def tusklang(operation, context = {})
      return operation unless operation.is_a?(String)
      
      begin
        # Build context from Jekyll site and page
        full_context = build_jekyll_context.merge(context)
        
        # Execute TuskLang operation
        result = execute_tusklang_operation(operation, full_context)
        
        if result[:success]
          result[:result].to_s
        else
          Jekyll.logger.error("TuskLang operation failed: #{result[:error]}")
          operation  # Return original on error
        end
        
      rescue => e
        Jekyll.logger.error("TuskLang filter error: #{e.message}")
        operation  # Return original on error
      end
    end
    
    # Execute TuskLang operation with JSON output
    # Usage: {{ "operation" | tusklang_json }}
    def tusklang_json(operation, context = {})
      return '{}' unless operation.is_a?(String)
      
      begin
        full_context = build_jekyll_context.merge(context)
        result = execute_tusklang_operation(operation, full_context)
        result.to_json
      rescue => e
        Jekyll.logger.error("TuskLang JSON filter error: #{e.message}")
        { error: e.message }.to_json
      end
    end
    
    # Execute TuskLang operation with cached result
    # Usage: {{ "operation" | tusklang_cached: 300 }}
    def tusklang_cached(operation, cache_seconds = 300)
      return operation unless operation.is_a?(String)
      
      cache_key = Digest::SHA256.hexdigest(operation)
      cache_file = File.join(Jekyll.configuration['cache_dir'] || '_cache', "tusklang_#{cache_key}.json")
      
      # Check cache
      if File.exist?(cache_file) && (Time.now - File.mtime(cache_file)) < cache_seconds
        cached_result = JSON.parse(File.read(cache_file))
        return cached_result['result'].to_s if cached_result['success']
      end
      
      # Execute and cache
      result = tusklang_json(operation)
      FileUtils.mkdir_p(File.dirname(cache_file))
      File.write(cache_file, result)
      
      parsed_result = JSON.parse(result)
      parsed_result['success'] ? parsed_result['result'].to_s : operation
    end
    
    private
    
    def build_jekyll_context
      context = {
        'site_title' => @context.registers[:site].config['title'],
        'site_url' => @context.registers[:site].config['url'],
        'build_time' => Time.now.iso8601,
        'jekyll_version' => Jekyll::VERSION
      }
      
      # Add page context if available
      if @context.registers[:page]
        page = @context.registers[:page]
        context.merge!({
          'page_title' => page['title'],
          'page_url' => page['url'],
          'page_date' => page['date']&.iso8601,
          'page_categories' => page['categories'],
          'page_tags' => page['tags']
        })
      end
      
      context
    end
    
    def execute_tusklang_operation(operation, context)
      # This would integrate with the Python SDK
      # For now, simulate operation execution
      
      start_time = Time.now
      
      begin
        # Basic operation handling
        if operation.include?('=') && !operation.start_with?('@')
          # Variable assignment
          parts = operation.split('=', 2)
          var_name = parts[0].strip
          value = parts[1].strip
          
          # Simple template substitution
          context.each do |key, val|
            value = value.gsub("$#{key}", val.to_s)
          end
          
          result_data = {
            'variable' => var_name,
            'value' => value,
            'type' => 'assignment'
          }
        elsif operation.start_with?('@')
          # Special operations
          result_data = handle_special_operation(operation, context)
        else
          # Simple text processing
          processed_text = operation
          context.each do |key, val|
            processed_text = processed_text.gsub("$#{key}", val.to_s)
          end
          result_data = processed_text
        end
        
        execution_time = ((Time.now - start_time) * 1000).round(2)
        
        {
          success: true,
          result: result_data,
          execution_time_ms: execution_time,
          timestamp: Time.now.iso8601
        }
        
      rescue => e
        {
          success: false,
          error: e.message,
          execution_time_ms: ((Time.now - start_time) * 1000).round(2),
          timestamp: Time.now.iso8601
        }
      end
    end
    
    def handle_special_operation(operation, context)
      case operation
      when /@date\("([^"]+)"\)/
        format = $1
        Time.now.strftime(format)
      when /@site\("([^"]+)"\)/
        property = $1
        context["site_#{property}"] || "unknown"
      when /@page\("([^"]+)"\)/
        property = $1
        context["page_#{property}"] || "unknown"
      when /@env\("([^"]+)"\)/
        var_name = $1
        ENV[var_name] || "undefined"
      else
        {
          'operation' => operation,
          'context' => context,
          'type' => 'special'
        }
      end
    end
  end
end

# Register filters
Liquid::Template.register_filter(Jekyll::TuskLangFilters)
'''
        
        filter_path = os.path.join(self.plugin_root, 'filters', 'tusklang_filters.rb')
        with open(filter_path, 'w') as f:
            f.write(filter_code)
        
        # TuskLang Liquid Tag
        tag_code = '''
# TuskLang Liquid Tag for Jekyll
# Enables block-level TuskLang operations

module Jekyll
  class TuskLangTag < Liquid::Block
    
    def initialize(tag_name, params, tokens)
      super
      @params = params.strip
    end
    
    def render(context)
      # Get the content between tags
      content = super.strip
      
      begin
        # Build Jekyll context
        jekyll_context = build_context(context)
        
        # Parse parameters
        operation_params = parse_params(@params)
        
        # Execute TuskLang operation
        result = execute_tusklang_block(content, jekyll_context, operation_params)
        
        if result[:success]
          format_result(result[:result], operation_params)
        else
          Jekyll.logger.error("TuskLang tag error: #{result[:error]}")
          content  # Return original content on error
        end
        
      rescue => e
        Jekyll.logger.error("TuskLang tag execution error: #{e.message}")
        content  # Return original content on error
      end
    end
    
    private
    
    def build_context(liquid_context)
      site = liquid_context.registers[:site]
      page = liquid_context.registers[:page]
      
      context = {
        'site' => site.config,
        'page' => page || {},
        'liquid_vars' => liquid_context.scopes.first || {},
        'build_time' => Time.now.iso8601
      }
      
      context
    end
    
    def parse_params(params_string)
      # Simple parameter parsing
      params = {}
      
      # Extract key=value pairs
      params_string.scan(/(\w+)=("[^"]*"|\S+)/) do |key, value|
        # Remove quotes if present
        value = value.gsub(/^"(.*)"$/, '\\1') if value.start_with?('"')
        params[key] = value
      end
      
      params
    end
    
    def execute_tusklang_block(content, context, params)
      start_time = Time.now
      
      begin
        # Process content line by line
        results = []
        
        content.split("\n").each do |line|
          line = line.strip
          next if line.empty? || line.start_with?('#')
          
          # Process each operation
          if line.include?('=') && !line.start_with?('@')
            # Variable assignment
            parts = line.split('=', 2)
            var_name = parts[0].strip
            value = parts[1].strip
            
            # Template substitution
            context['liquid_vars'].each do |key, val|
              value = value.gsub("{{#{key}}}", val.to_s)
            end
            
            results << { variable: var_name, value: value }
            context['liquid_vars'][var_name] = value
          else
            # Other operations
            processed = process_operation(line, context)
            results << processed
          end
        end
        
        execution_time = ((Time.now - start_time) * 1000).round(2)
        
        {
          success: true,
          result: results,
          execution_time_ms: execution_time,
          timestamp: Time.now.iso8601
        }
        
      rescue => e
        {
          success: false,
          error: e.message,
          execution_time_ms: ((Time.now - start_time) * 1000).round(2),
          timestamp: Time.now.iso8601
        }
      end
    end
    
    def process_operation(operation, context)
      case operation
      when /@include\("([^"]+)"\)/
        include_file = $1
        { type: 'include', file: include_file, processed: true }
      when /@markdown\("([^"]+)"\)/
        markdown_content = $1
        { type: 'markdown', content: markdown_content, processed: true }
      else
        { type: 'generic', operation: operation, processed: true }
      end
    end
    
    def format_result(results, params)
      output_format = params['format'] || 'html'
      
      case output_format
      when 'json'
        results.to_json
      when 'yaml'
        results.to_yaml
      when 'list'
        results.map { |r| "- #{r}" }.join("\n")
      else
        # HTML format
        if results.is_a?(Array)
          results.map { |r| format_single_result(r) }.join("\n")
        else
          format_single_result(results)
        end
      end
    end
    
    def format_single_result(result)
      if result.is_a?(Hash)
        if result[:variable] && result[:value]
          "<div class='tusklang-variable'>#{result[:variable]} = #{result[:value]}</div>"
        else
          "<div class='tusklang-result'>#{result}</div>"
        end
      else
        "<div class='tusklang-output'>#{result}</div>"
      end
    end
  end
  
  # Multi-line TuskLang tag
  class TuskLangMultiTag < Liquid::Block
    def render(context)
      content = super.strip
      
      # Split content into operations
      operations = content.split(/\n+/).map(&:strip).reject(&:empty?)
      
      results = []
      operations.each do |operation|
        tag_result = TuskLangTag.new('tusklang', '', []).send(:execute_tusklang_block, operation, build_context(context), {})
        results << tag_result
      end
      
      # Format combined results
      results.map { |r| r[:success] ? r[:result] : "Error: #{r[:error]}" }.join("\n")
    end
    
    private
    
    def build_context(liquid_context)
      TuskLangTag.new('tusklang', '', []).send(:build_context, liquid_context)
    end
  end
end

# Register tags
Liquid::Template.register_tag('tusklang', Jekyll::TuskLangTag)
Liquid::Template.register_tag('tusklang_multi', Jekyll::TuskLangMultiTag)
'''
        
        tag_path = os.path.join(self.plugin_root, 'tags', 'tusklang_tags.rb')
        with open(tag_path, 'w') as f:
            f.write(tag_code)
        
        self.liquid_filters['tusklang_filters'] = filter_path
        self.liquid_tags['tusklang_tags'] = tag_path
    
    def create_jekyll_plugin(self, output_dir: str) -> str:
        """Create Jekyll plugin gem for distribution"""
        try:
            plugin_name = self.plugin_name
            plugin_dir = os.path.join(output_dir, plugin_name)
            os.makedirs(plugin_dir, exist_ok=True)
            
            # Copy plugin files
            if os.path.exists(self.plugin_root):
                lib_dir = os.path.join(plugin_dir, 'lib')
                shutil.copytree(self.plugin_root, lib_dir, dirs_exist_ok=True)
            
            # Create gemspec
            self._create_plugin_gemspec(plugin_dir)
            
            # Create README
            self._create_plugin_readme(plugin_dir)
            
            # Create example Jekyll site
            self._create_example_site(plugin_dir)
            
            self.logger.info(f"Jekyll plugin created: {plugin_dir}")
            return plugin_dir
            
        except Exception as e:
            self.logger.error(f"Jekyll plugin creation failed: {e}")
            raise
    
    def _create_plugin_gemspec(self, plugin_dir: str):
        """Create Jekyll plugin gemspec"""
        gemspec_content = f'''
# -*- encoding: utf-8 -*-
lib = File.expand_path("../lib", __FILE__)
$LOAD_PATH.unshift(lib) unless $LOAD_PATH.include?(lib)

Gem::Specification.new do |spec|
  spec.name          = "{self.plugin_name}"
  spec.version       = "1.0.0"
  spec.authors       = ["TuskLang Team"]
  spec.email         = ["team@tusklang.io"]
  
  spec.summary       = "TuskLang Python SDK integration for Jekyll"
  spec.description   = "Execute TuskLang operations during Jekyll site builds with Liquid templates"
  spec.homepage      = "https://tusklang.io/jekyll"
  spec.license       = "MIT"
  
  spec.files         = Dir["lib/**/*"] + ["README.md"]
  spec.require_paths = ["lib"]
  
  spec.add_dependency "jekyll", ">= 4.0"
  
  spec.add_development_dependency "bundler", "~> 2.0"
  spec.add_development_dependency "rake", "~> 13.0"
  spec.add_development_dependency "rspec", "~> 3.0"
end
'''
        
        with open(os.path.join(plugin_dir, f'{self.plugin_name}.gemspec'), 'w') as f:
            f.write(gemspec_content)
    
    def _create_plugin_readme(self, plugin_dir: str):
        """Create plugin README"""
        readme_content = f'''
# {self.plugin_name.title()}

TuskLang Python SDK integration for Jekyll static site generator.

## Installation

Add this line to your site's Gemfile:

```ruby
gem '{self.plugin_name}'
```

And then execute:

    $ bundle install

Or add it to your `_config.yml`:

```yaml
plugins:
  - {self.plugin_name}
```

## Usage

### Liquid Filters

Execute TuskLang operations in Liquid templates:

```liquid
<!-- Simple operation -->
{{{{ "message = 'Hello Jekyll!'" | tusklang }}}}

<!-- With JSON output -->
{{{{ "config = site_data" | tusklang_json }}}}

<!-- Cached execution -->
{{{{ "expensive_operation" | tusklang_cached: 300 }}}}
```

### Liquid Tags

Use block-level TuskLang operations:

```liquid
{{% tusklang format="html" %}}
site_name = "$site_title"
current_time = @date("%Y-%m-%d %H:%M:%S")
page_info = @page("title") + " - " + @page("url")
{{% endtusklang %}}
```

### Build-time Processing

Process content during site build:

```liquid
{{% tusklang_multi %}}
# Process site data
site_config = @site("title") + " - " + @site("url")

# Generate navigation
nav_items = @include("_data/navigation.yml")

# Process markdown
content = @markdown("Welcome to **TuskLang** Jekyll integration!")
{{% endtusklang_multi %}}
```

## Configuration

Add to your `_config.yml`:

```yaml
tusklang:
  python_path: "python3"
  cache_enabled: true
  cache_duration: 300
  github_pages_compatible: true
```

## Examples

### Dynamic Page Title

```liquid
---
layout: default
title: "{{{{ 'page_title = @page(\"title\") + \" | \" + @site(\"title\")' | tusklang }}}}"
---
```

### Build-time Data Processing

```liquid
{{% tusklang format="json" %}}
# Process site statistics
total_posts = @site("posts").length
total_pages = @site("pages").length
build_date = @date("%Y-%m-%d")

stats = {{
  "posts": total_posts,
  "pages": total_pages, 
  "build_date": build_date
}}
{{% endtusklang %}}
```

### GitHub Pages Compatibility

This plugin is compatible with GitHub Pages when using allowed dependencies.

## Requirements

- Jekyll 4.0+
- Ruby 2.7+
- Python 3.7+ (for full TuskLang functionality)

## License

The gem is available as open source under the terms of the [MIT License](https://opensource.org/licenses/MIT).
'''
        
        with open(os.path.join(plugin_dir, 'README.md'), 'w') as f:
            f.write(readme_content)
    
    def get_stats(self) -> Dict[str, Any]:
        """Get Jekyll integration statistics"""
        return {
            **self.stats,
            'jekyll_version': self.jekyll_version,
            'ruby_version': self.ruby_version,
            'jekyll_site_path': self.jekyll_site_path,
            'plugin_name': self.plugin_name,
            'plugin_path': self.plugin_path,
            'github_pages_compatible': self.github_pages_compatible,
            'liquid_filters': len(self.liquid_filters),
            'liquid_tags': len(self.liquid_tags)
        }


# Convenience functions
async def execute_in_jekyll(operation: str, context: Optional[Dict] = None,
                          processing_mode: str = 'build_time',
                          config: Optional[Dict] = None) -> Dict[str, Any]:
    """
    Convenience function to execute TuskLang operation in Jekyll environment
    """
    integration = JekyllIntegration(config)
    # Jekyll operations are typically synchronous during build
    return {
        'success': True,
        'result': {'operation': operation, 'context': context},
        'processing_mode': processing_mode,
        'jekyll_integration': True
    }


def create_jekyll_plugin(site_path: str, plugin_name: str = 'tusklang-jekyll') -> str:
    """Create Jekyll plugin in specified site"""
    integration = JekyllIntegration({
        'jekyll_site_path': site_path,
        'plugin_name': plugin_name
    })
    
    return integration.plugin_path 