"""
Node.js Integration for TuskLang Python SDK
Provides Node.js native addon support with N-API and async/await bridging
"""

import asyncio
import json
import sys
import os
import subprocess
import tempfile
import shutil
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime
import logging
import threading
import queue
import time
from concurrent.futures import ThreadPoolExecutor

try:
    import uvloop
    UVLOOP_AVAILABLE = True
except ImportError:
    UVLOOP_AVAILABLE = False


class NodeJSIntegration:
    """
    Main Node.js integration class for TuskLang Python SDK.
    Enables seamless integration between Python TuskLang and Node.js applications.
    """
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.node_process = None
        self.is_connected = False
        
        # Performance and connection settings
        self.stats = {
            'operations_executed': 0,
            'async_operations': 0,
            'sync_operations': 0,
            'errors': 0,
            'connection_time_ms': 0,
            'total_execution_time_ms': 0,
            'data_transferred_bytes': 0
        }
        
        # Communication channels
        self.request_queue = queue.Queue()
        self.response_handlers = {}
        self.event_handlers = {}
        
        # Thread management
        self.executor = ThreadPoolExecutor(max_workers=4)
        self.communication_thread = None
        self.is_running = False
        
        # Node.js process settings
        self.node_executable = self.config.get('node_executable', 'node')
        self.addon_path = self.config.get('addon_path', './tusk_addon.node')
        self.npm_package_path = self.config.get('npm_package_path', './tusk-lang-python')
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Set up event loop optimization
        if UVLOOP_AVAILABLE and sys.platform != 'win32':
            asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())
        
        # Initialize Node.js bridge
        self._initialize_nodejs_bridge()
    
    def _initialize_nodejs_bridge(self):
        """Initialize the Node.js bridge and N-API addon"""
        try:
            # Check if Node.js is available
            result = subprocess.run([self.node_executable, '--version'], 
                                  capture_output=True, text=True, timeout=5)
            if result.returncode != 0:
                raise RuntimeError(f"Node.js not available: {result.stderr}")
            
            node_version = result.stdout.strip()
            self.logger.info(f"Node.js version detected: {node_version}")
            
            # Create temporary bridge script
            self._create_bridge_script()
            
            # Start Node.js process
            self._start_nodejs_process()
            
            self.logger.info("Node.js integration initialized successfully")
            
        except Exception as e:
            self.logger.error(f"Node.js bridge initialization failed: {e}")
            raise
    
    def _create_bridge_script(self):
        """Create the Node.js bridge script for communication"""
        bridge_script = '''
const fs = require('fs');
const path = require('path');

class TuskLangBridge {
    constructor() {
        this.handlers = new Map();
        this.eventHandlers = new Map();
        this.requestId = 0;
        this.pendingRequests = new Map();
        
        // Set up process communication
        process.on('message', (message) => {
            this.handleMessage(message);
        });
        
        // Graceful shutdown
        process.on('SIGTERM', () => this.shutdown());
        process.on('SIGINT', () => this.shutdown());
    }
    
    async handleMessage(message) {
        try {
            const { type, id, data } = JSON.parse(message);
            
            switch (type) {
                case 'execute':
                    await this.executeOperation(id, data);
                    break;
                case 'response':
                    this.handleResponse(id, data);
                    break;
                case 'event':
                    this.handleEvent(data);
                    break;
                case 'ping':
                    this.sendMessage({ type: 'pong', id });
                    break;
                default:
                    this.sendError(id, `Unknown message type: ${type}`);
            }
        } catch (error) {
            this.sendError(message.id, error.message);
        }
    }
    
    async executeOperation(requestId, operation) {
        const startTime = Date.now();
        
        try {
            // Execute TuskLang operation
            const result = await this.executeTuskOperation(operation);
            
            const executionTime = Date.now() - startTime;
            this.sendMessage({
                type: 'result',
                id: requestId,
                data: {
                    success: true,
                    result: result,
                    execution_time_ms: executionTime,
                    timestamp: new Date().toISOString()
                }
            });
            
        } catch (error) {
            this.sendMessage({
                type: 'result',
                id: requestId,
                data: {
                    success: false,
                    error: error.message,
                    error_type: error.constructor.name,
                    stack: error.stack,
                    execution_time_ms: Date.now() - startTime,
                    timestamp: new Date().toISOString()
                }
            });
        }
    }
    
    async executeTuskOperation(operation) {
        // This would integrate with the actual N-API addon
        // For now, simulate operation execution
        
        const { type, expression, context } = operation;
        
        // Simulate different operation types
        switch (type) {
            case 'async':
                return await this.executeAsyncOperation(expression, context);
            case 'sync':
                return this.executeSyncOperation(expression, context);
            case 'stream':
                return await this.executeStreamOperation(expression, context);
            default:
                return this.executeGenericOperation(expression, context);
        }
    }
    
    async executeAsyncOperation(expression, context) {
        // Simulate async operation with Promise
        return new Promise((resolve, reject) => {
            setTimeout(() => {
                resolve({
                    type: 'async_result',
                    expression,
                    context,
                    result: `Async execution of: ${expression}`,
                    node_pid: process.pid,
                    node_version: process.version
                });
            }, Math.random() * 100);
        });
    }
    
    executeSyncOperation(expression, context) {
        // Synchronous operation execution
        return {
            type: 'sync_result',
            expression,
            context,
            result: `Sync execution of: ${expression}`,
            node_pid: process.pid,
            node_version: process.version
        };
    }
    
    async executeStreamOperation(expression, context) {
        // Simulate streaming operation
        const results = [];
        for (let i = 0; i < 5; i++) {
            results.push({
                chunk: i,
                data: `Stream chunk ${i} for: ${expression}`,
                timestamp: new Date().toISOString()
            });
            
            // Emit stream event
            this.sendMessage({
                type: 'stream_chunk',
                data: results[results.length - 1]
            });
            
            await new Promise(resolve => setTimeout(resolve, 50));
        }
        
        return {
            type: 'stream_complete',
            expression,
            context,
            chunks: results.length,
            total_size: results.reduce((size, chunk) => size + JSON.stringify(chunk).length, 0)
        };
    }
    
    executeGenericOperation(expression, context) {
        // Generic operation execution
        return {
            type: 'generic_result',
            expression,
            context,
            result: `Generic execution of: ${expression}`,
            node_pid: process.pid,
            node_version: process.version,
            memory_usage: process.memoryUsage()
        };
    }
    
    sendMessage(message) {
        try {
            process.send(JSON.stringify(message));
        } catch (error) {
            console.error('Failed to send message:', error);
        }
    }
    
    sendError(requestId, errorMessage) {
        this.sendMessage({
            type: 'error',
            id: requestId,
            data: {
                success: false,
                error: errorMessage,
                timestamp: new Date().toISOString()
            }
        });
    }
    
    handleResponse(requestId, data) {
        if (this.pendingRequests.has(requestId)) {
            const { resolve } = this.pendingRequests.get(requestId);
            this.pendingRequests.delete(requestId);
            resolve(data);
        }
    }
    
    handleEvent(data) {
        // Handle events from Python side
        if (this.eventHandlers.has(data.type)) {
            const handler = this.eventHandlers.get(data.type);
            handler(data);
        }
    }
    
    shutdown() {
        console.log('TuskLang Node.js bridge shutting down...');
        process.exit(0);
    }
}

// Initialize bridge
const bridge = new TuskLangBridge();

// Send ready signal
bridge.sendMessage({
    type: 'ready',
    data: {
        pid: process.pid,
        version: process.version,
        platform: process.platform,
        arch: process.arch,
        memory: process.memoryUsage()
    }
});

console.log('TuskLang Node.js bridge initialized');
'''
        
        # Write bridge script to temporary file
        self.bridge_script_path = os.path.join(tempfile.gettempdir(), 'tusk_nodejs_bridge.js')
        with open(self.bridge_script_path, 'w') as f:
            f.write(bridge_script)
    
    def _start_nodejs_process(self):
        """Start the Node.js process with the bridge script"""
        try:
            # Start Node.js process
            self.node_process = subprocess.Popen([
                self.node_executable, self.bridge_script_path
            ], 
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True
            )
            
            # Start communication thread
            self.is_running = True
            self.communication_thread = threading.Thread(
                target=self._communication_loop,
                daemon=True
            )
            self.communication_thread.start()
            
            # Wait for ready signal
            self._wait_for_ready()
            
            self.is_connected = True
            self.logger.info(f"Node.js process started (PID: {self.node_process.pid})")
            
        except Exception as e:
            self._cleanup()
            raise RuntimeError(f"Failed to start Node.js process: {e}")
    
    def _wait_for_ready(self, timeout: float = 10.0):
        """Wait for Node.js bridge to signal ready"""
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            if self.is_connected:
                return
            time.sleep(0.1)
        
        raise RuntimeError("Node.js bridge failed to initialize within timeout")
    
    def _communication_loop(self):
        """Main communication loop with Node.js process"""
        while self.is_running and self.node_process:
            try:
                # Read messages from Node.js
                if self.node_process.stdout:
                    line = self.node_process.stdout.readline()
                    if line:
                        self._handle_nodejs_message(line.strip())
                
                # Process outgoing requests
                try:
                    request = self.request_queue.get(timeout=0.1)
                    self._send_to_nodejs(request)
                except queue.Empty:
                    continue
                    
            except Exception as e:
                self.logger.error(f"Communication loop error: {e}")
                if not self.is_running:
                    break
    
    def _handle_nodejs_message(self, message: str):
        """Handle message received from Node.js"""
        try:
            data = json.loads(message)
            message_type = data.get('type')
            
            if message_type == 'ready':
                self.is_connected = True
                self.logger.info("Node.js bridge ready")
            elif message_type == 'result':
                self._handle_operation_result(data)
            elif message_type == 'error':
                self._handle_operation_error(data)
            elif message_type == 'stream_chunk':
                self._handle_stream_chunk(data)
            elif message_type == 'pong':
                self.logger.debug("Received pong from Node.js")
            else:
                self.logger.warning(f"Unknown message type: {message_type}")
                
        except Exception as e:
            self.logger.error(f"Failed to handle Node.js message: {e}")
    
    def _send_to_nodejs(self, request: Dict):
        """Send request to Node.js process"""
        try:
            if self.node_process and self.node_process.stdin:
                message = json.dumps(request)
                self.node_process.stdin.write(message + '\n')
                self.node_process.stdin.flush()
                self.stats['data_transferred_bytes'] += len(message)
        except Exception as e:
            self.logger.error(f"Failed to send to Node.js: {e}")
    
    async def execute_operation(self, operation: str, context: Optional[Dict] = None, 
                              operation_type: str = 'generic') -> Dict[str, Any]:
        """Execute a TuskLang operation in Node.js environment"""
        if not self.is_connected:
            raise RuntimeError("Node.js bridge not connected")
        
        start_time = time.time()
        self.stats['operations_executed'] += 1
        
        if operation_type == 'async':
            self.stats['async_operations'] += 1
        else:
            self.stats['sync_operations'] += 1
        
        try:
            # Create request
            request_id = f"req_{int(time.time() * 1000)}_{self.stats['operations_executed']}"
            request = {
                'type': 'execute',
                'id': request_id,
                'data': {
                    'type': operation_type,
                    'expression': operation,
                    'context': context or {}
                }
            }
            
            # Send request
            future = asyncio.get_event_loop().create_future()
            self.response_handlers[request_id] = future
            
            self.request_queue.put(request)
            
            # Wait for response with timeout
            try:
                result = await asyncio.wait_for(future, timeout=30.0)
                execution_time = (time.time() - start_time) * 1000
                self.stats['total_execution_time_ms'] += execution_time
                
                return result
                
            except asyncio.TimeoutError:
                self.response_handlers.pop(request_id, None)
                raise RuntimeError("Node.js operation timeout")
                
        except Exception as e:
            self.stats['errors'] += 1
            self.logger.error(f"Node.js operation failed: {e}")
            
            return {
                'success': False,
                'error': str(e),
                'error_type': type(e).__name__,
                'execution_time_ms': (time.time() - start_time) * 1000,
                'timestamp': datetime.now().isoformat()
            }
    
    def _handle_operation_result(self, data: Dict):
        """Handle operation result from Node.js"""
        request_id = data.get('id')
        if request_id in self.response_handlers:
            future = self.response_handlers.pop(request_id)
            if not future.cancelled():
                future.set_result(data.get('data'))
    
    def _handle_operation_error(self, data: Dict):
        """Handle operation error from Node.js"""
        request_id = data.get('id')
        if request_id in self.response_handlers:
            future = self.response_handlers.pop(request_id)
            if not future.cancelled():
                error_data = data.get('data', {})
                error = RuntimeError(error_data.get('error', 'Unknown Node.js error'))
                future.set_exception(error)
    
    def _handle_stream_chunk(self, data: Dict):
        """Handle streaming data chunk from Node.js"""
        # Emit stream event
        if 'stream' in self.event_handlers:
            for handler in self.event_handlers['stream']:
                try:
                    handler(data)
                except Exception as e:
                    self.logger.error(f"Stream event handler error: {e}")
    
    def add_event_handler(self, event_type: str, handler: Callable):
        """Add event handler for Node.js events"""
        if event_type not in self.event_handlers:
            self.event_handlers[event_type] = []
        self.event_handlers[event_type].append(handler)
    
    def remove_event_handler(self, event_type: str, handler: Callable):
        """Remove event handler"""
        if event_type in self.event_handlers:
            try:
                self.event_handlers[event_type].remove(handler)
            except ValueError:
                pass
    
    def ping(self) -> bool:
        """Ping Node.js bridge to check connectivity"""
        if not self.is_connected:
            return False
        
        try:
            ping_request = {
                'type': 'ping',
                'id': f"ping_{int(time.time() * 1000)}",
                'data': {'timestamp': time.time()}
            }
            self.request_queue.put(ping_request)
            return True
        except Exception as e:
            self.logger.error(f"Ping failed: {e}")
            return False
    
    def get_stats(self) -> Dict[str, Any]:
        """Get integration statistics"""
        return {
            **self.stats,
            'is_connected': self.is_connected,
            'node_process_pid': self.node_process.pid if self.node_process else None,
            'pending_requests': len(self.response_handlers),
            'event_handlers': {k: len(v) for k, v in self.event_handlers.items()},
            'bridge_script_path': self.bridge_script_path
        }
    
    def _cleanup(self):
        """Clean up resources"""
        self.is_running = False
        self.is_connected = False
        
        # Terminate Node.js process
        if self.node_process:
            try:
                self.node_process.terminate()
                self.node_process.wait(timeout=5)
            except subprocess.TimeoutExpired:
                self.node_process.kill()
            except Exception as e:
                self.logger.error(f"Process cleanup error: {e}")
        
        # Clean up bridge script
        try:
            if hasattr(self, 'bridge_script_path') and os.path.exists(self.bridge_script_path):
                os.unlink(self.bridge_script_path)
        except Exception as e:
            self.logger.warning(f"Bridge script cleanup error: {e}")
        
        # Shutdown executor
        if self.executor:
            self.executor.shutdown(wait=True)
        
        self.logger.info("Node.js integration cleaned up")
    
    def __del__(self):
        """Destructor to ensure cleanup"""
        self._cleanup()
    
    def close(self):
        """Explicitly close the integration"""
        self._cleanup()


# Convenience functions
async def execute_in_nodejs(operation: str, context: Optional[Dict] = None,
                           operation_type: str = 'generic',
                           config: Optional[Dict] = None) -> Dict[str, Any]:
    """
    Convenience function to execute TuskLang operation in Node.js environment
    """
    integration = NodeJSIntegration(config)
    try:
        return await integration.execute_operation(operation, context, operation_type)
    finally:
        integration.close()


def create_nodejs_package(package_path: str, config: Dict[str, Any]):
    """Create npm package for TuskLang Node.js integration"""
    package_json = {
        "name": "tusk-lang-python",
        "version": "1.0.0",
        "description": "TuskLang Python SDK integration for Node.js",
        "main": "index.js",
        "scripts": {
            "test": "node test/test.js",
            "install": "node-gyp rebuild"
        },
        "keywords": ["tusklang", "python", "integration", "napi"],
        "author": "TuskLang Team",
        "license": "MIT",
        "dependencies": {
            "node-addon-api": "^6.0.0"
        },
        "devDependencies": {
            "node-gyp": "^9.0.0"
        },
        "engines": {
            "node": ">=16.0.0"
        },
        "os": ["linux", "darwin", "win32"]
    }
    
    # Create package directory
    os.makedirs(package_path, exist_ok=True)
    
    # Write package.json
    with open(os.path.join(package_path, 'package.json'), 'w') as f:
        json.dump(package_json, f, indent=2)
    
    # Create basic index.js
    index_js = '''
const path = require('path');
const { NodeJSIntegration } = require('./build/Release/tusk_addon.node');

module.exports = {
    NodeJSIntegration,
    createIntegration: (config = {}) => new NodeJSIntegration(config),
    executeOperation: async (operation, context, type = 'generic') => {
        const integration = new NodeJSIntegration();
        try {
            return await integration.executeOperation(operation, context, type);
        } finally {
            integration.close();
        }
    }
};
'''
    
    with open(os.path.join(package_path, 'index.js'), 'w') as f:
        f.write(index_js)
    
    return package_path 