"""
WebAssembly Integration for TuskLang Python SDK
Provides WebAssembly runtime support using Pyodide with browser and server-side execution
"""

import asyncio
import json
import sys
import traceback
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime
import logging
import time

try:
    import pyodide
    import js
    from pyodide.ffi import create_proxy, to_js
    PYODIDE_AVAILABLE = True
except ImportError:
    PYODIDE_AVAILABLE = False
    pyodide = None
    js = None


class WebAssemblyIntegration:
    """
    Main WebAssembly integration class for TuskLang Python SDK.
    Enables running TuskLang operations in WebAssembly environments.
    """
    
    def __init__(self, config: Optional[Dict[str, Any]] = None):
        self.config = config or {}
        self.is_browser = self._detect_browser_environment()
        self.is_pyodide = PYODIDE_AVAILABLE
        
        # Performance tracking
        self.stats = {
            'operations_executed': 0,
            'memory_usage': 0,
            'execution_time_ms': 0,
            'errors': 0,
            'gc_collections': 0
        }
        
        # Memory management settings
        self.memory_limit = self.config.get('memory_limit', 128 * 1024 * 1024)  # 128MB
        self.gc_threshold = self.config.get('gc_threshold', 50 * 1024 * 1024)  # 50MB
        
        # JavaScript bridge callbacks
        self.js_callbacks = {}
        self.event_listeners = {}
        
        # Initialize logging
        self.logger = logging.getLogger(__name__)
        
        # Initialize if Pyodide is available
        if self.is_pyodide:
            self._initialize_pyodide()
    
    def _detect_browser_environment(self) -> bool:
        """Detect if running in a browser environment"""
        try:
            # Check for browser-specific globals
            if PYODIDE_AVAILABLE and hasattr(js, 'window') and hasattr(js, 'document'):
                return True
            return False
        except:
            return False
    
    def _initialize_pyodide(self):
        """Initialize Pyodide-specific features"""
        if not self.is_pyodide:
            return
        
        try:
            # Set up memory monitoring
            if self.is_browser:
                self._setup_browser_integration()
            
            # Configure garbage collection
            self._setup_memory_management()
            
            self.logger.info("Pyodide WebAssembly integration initialized successfully")
        except Exception as e:
            self.logger.error(f"Pyodide initialization failed: {e}")
            raise
    
    def _setup_browser_integration(self):
        """Set up browser-specific integrations"""
        if not (self.is_pyodide and self.is_browser):
            return
        
        try:
            # Create JavaScript proxy for Python callbacks
            self.js_proxy = create_proxy(self._handle_js_callback)
            
            # Expose Python functions to JavaScript
            js.globalThis.tuskLangWasm = {
                'execute': create_proxy(self.execute_operation),
                'getStats': create_proxy(lambda: to_js(self.get_stats())),
                'addEventListener': create_proxy(self.add_event_listener),
                'removeEventListener': create_proxy(self.remove_event_listener)
            }
            
            # Set up window event listeners
            self._setup_window_events()
            
            self.logger.info("Browser integration setup complete")
        except Exception as e:
            self.logger.error(f"Browser integration setup failed: {e}")
    
    def _setup_window_events(self):
        """Set up window event listeners for browser integration"""
        if not (self.is_pyodide and self.is_browser):
            return
        
        try:
            # Memory pressure monitoring
            if hasattr(js.window, 'addEventListener'):
                js.window.addEventListener('beforeunload', create_proxy(self._cleanup_resources))
                js.window.addEventListener('unload', create_proxy(self._cleanup_resources))
            
            # Page visibility for optimization
            if hasattr(js.document, 'addEventListener'):
                js.document.addEventListener('visibilitychange', 
                                           create_proxy(self._handle_visibility_change))
        except Exception as e:
            self.logger.warning(f"Window events setup failed: {e}")
    
    def _setup_memory_management(self):
        """Set up memory management and garbage collection"""
        import gc
        
        # Configure garbage collection thresholds
        gc.set_threshold(700, 10, 10)
        
        # Schedule periodic memory checks
        if self.is_browser and hasattr(js, 'setInterval'):
            js.setInterval(create_proxy(self._check_memory_usage), 5000)  # Every 5 seconds
    
    def _check_memory_usage(self):
        """Check and manage memory usage"""
        import gc
        import sys
        
        try:
            # Get current memory usage
            memory_info = sys.getsizeof(gc.get_objects())
            self.stats['memory_usage'] = memory_info
            
            # Trigger garbage collection if needed
            if memory_info > self.gc_threshold:
                collected = gc.collect()
                self.stats['gc_collections'] += 1
                self.logger.debug(f"Garbage collection freed {collected} objects")
                
                # Emit memory event to JavaScript if in browser
                if self.is_browser and hasattr(js, 'globalThis'):
                    event_data = {
                        'type': 'memory_gc',
                        'collected': collected,
                        'memory_usage': memory_info
                    }
                    self._emit_event('memory', event_data)
        except Exception as e:
            self.logger.warning(f"Memory check failed: {e}")
    
    async def execute_operation(self, operation: str, context: Optional[Dict] = None) -> Dict[str, Any]:
        """Execute a TuskLang operation in WebAssembly environment"""
        start_time = time.time()
        self.stats['operations_executed'] += 1
        
        try:
            # Validate input
            if not operation or not isinstance(operation, str):
                raise ValueError("Operation must be a non-empty string")
            
            context = context or {}
            
            # Parse operation
            result = await self._execute_tusk_operation(operation, context)
            
            # Track execution time
            execution_time = (time.time() - start_time) * 1000
            self.stats['execution_time_ms'] += execution_time
            
            # Return structured result
            return {
                'success': True,
                'result': result,
                'execution_time_ms': execution_time,
                'memory_usage': self.stats['memory_usage'],
                'timestamp': datetime.now().isoformat()
            }
            
        except Exception as e:
            self.stats['errors'] += 1
            error_details = {
                'success': False,
                'error': str(e),
                'error_type': type(e).__name__,
                'traceback': traceback.format_exc(),
                'execution_time_ms': (time.time() - start_time) * 1000,
                'timestamp': datetime.now().isoformat()
            }
            
            self.logger.error(f"Operation execution failed: {e}")
            return error_details
    
    async def _execute_tusk_operation(self, operation: str, context: Dict) -> Any:
        """Execute the actual TuskLang operation"""
        # Import TuskLang core here to avoid circular imports
        try:
            from ...core.base.advanced_operators_integration import AdvancedOperatorIntegration
            
            # Create operator integration instance
            operator_integration = AdvancedOperatorIntegration()
            
            # Determine operation type and execute
            if operation.startswith('@graphql'):
                return await operator_integration.execute_graphql(operation, context)
            elif operation.startswith('@grpc'):
                return await operator_integration.execute_grpc(operation, context)
            elif operation.startswith('@websocket'):
                return await operator_integration.execute_websocket(operation, context)
            elif operation.startswith('@sse'):
                return await operator_integration.execute_sse(operation, context)
            else:
                # Generic operation execution
                return await self._execute_generic_operation(operation, context)
                
        except ImportError as e:
            self.logger.warning(f"Advanced operators not available: {e}")
            return await self._execute_generic_operation(operation, context)
    
    async def _execute_generic_operation(self, operation: str, context: Dict) -> Any:
        """Execute generic TuskLang operation"""
        # Basic operation execution - this would integrate with the main TSK engine
        try:
            # Simple expression evaluation for now
            if '=' in operation and not operation.startswith('@'):
                # Variable assignment
                parts = operation.split('=', 1)
                var_name = parts[0].strip()
                value = parts[1].strip()
                
                # Store in context
                context[var_name] = value
                return {'variable': var_name, 'value': value, 'type': 'assignment'}
            else:
                # Return operation as-is for now
                return {'operation': operation, 'context': context, 'type': 'generic'}
                
        except Exception as e:
            raise RuntimeError(f"Generic operation execution failed: {e}")
    
    def _handle_js_callback(self, callback_id: str, *args):
        """Handle JavaScript callback invocation"""
        try:
            if callback_id in self.js_callbacks:
                callback = self.js_callbacks[callback_id]
                return callback(*args)
            else:
                self.logger.warning(f"Unknown callback ID: {callback_id}")
                return None
        except Exception as e:
            self.logger.error(f"JavaScript callback error: {e}")
            return None
    
    def add_event_listener(self, event_type: str, callback: Callable):
        """Add event listener for WebAssembly events"""
        if event_type not in self.event_listeners:
            self.event_listeners[event_type] = []
        
        callback_id = f"{event_type}_{len(self.event_listeners[event_type])}"
        self.event_listeners[event_type].append(callback)
        self.js_callbacks[callback_id] = callback
        
        return callback_id
    
    def remove_event_listener(self, event_type: str, callback_id: str):
        """Remove event listener"""
        if callback_id in self.js_callbacks:
            del self.js_callbacks[callback_id]
        
        if event_type in self.event_listeners:
            # Remove from event listeners list (simplified)
            self.event_listeners[event_type] = [
                cb for cb in self.event_listeners[event_type] 
                if id(cb) != callback_id
            ]
    
    def _emit_event(self, event_type: str, data: Any):
        """Emit event to registered listeners"""
        if event_type in self.event_listeners:
            for callback in self.event_listeners[event_type]:
                try:
                    callback(data)
                except Exception as e:
                    self.logger.error(f"Event callback error: {e}")
    
    def _handle_visibility_change(self, event):
        """Handle page visibility change for optimization"""
        if not (self.is_pyodide and self.is_browser):
            return
        
        try:
            is_hidden = js.document.hidden
            self._emit_event('visibility', {'hidden': is_hidden})
            
            # Adjust performance based on visibility
            if is_hidden:
                self.logger.debug("Page hidden - reducing background activity")
            else:
                self.logger.debug("Page visible - resuming normal activity")
        except Exception as e:
            self.logger.warning(f"Visibility change handling failed: {e}")
    
    def _cleanup_resources(self, event=None):
        """Clean up resources before unload"""
        try:
            # Clear JavaScript callbacks
            self.js_callbacks.clear()
            self.event_listeners.clear()
            
            # Force garbage collection
            import gc
            gc.collect()
            
            self.logger.info("Resources cleaned up successfully")
        except Exception as e:
            self.logger.error(f"Resource cleanup failed: {e}")
    
    def get_stats(self) -> Dict[str, Any]:
        """Get integration statistics"""
        return {
            **self.stats,
            'is_browser': self.is_browser,
            'is_pyodide': self.is_pyodide,
            'memory_limit': self.memory_limit,
            'gc_threshold': self.gc_threshold,
            'active_callbacks': len(self.js_callbacks),
            'event_listeners': {k: len(v) for k, v in self.event_listeners.items()}
        }
    
    def configure(self, config: Dict[str, Any]):
        """Update configuration"""
        self.config.update(config)
        
        # Apply memory settings
        if 'memory_limit' in config:
            self.memory_limit = config['memory_limit']
        if 'gc_threshold' in config:
            self.gc_threshold = config['gc_threshold']
        
        self.logger.info("Configuration updated")


# Convenience function for direct usage
async def execute_in_wasm(operation: str, context: Optional[Dict] = None, 
                         config: Optional[Dict] = None) -> Dict[str, Any]:
    """
    Convenience function to execute TuskLang operation in WebAssembly environment
    """
    integration = WebAssemblyIntegration(config)
    return await integration.execute_operation(operation, context)


# Browser initialization function
def initialize_browser_tusk():
    """Initialize TuskLang in browser environment"""
    if not PYODIDE_AVAILABLE:
        raise RuntimeError("Pyodide not available - WebAssembly integration requires Pyodide")
    
    integration = WebAssemblyIntegration({
        'memory_limit': 256 * 1024 * 1024,  # 256MB for browser
        'gc_threshold': 100 * 1024 * 1024   # 100MB GC threshold
    })
    
    return integration 