#!/usr/bin/env python3
"""
Simple Test Implementation Verification
======================================
Tests that the implementation works correctly
"""

import sys
import os
import json
from datetime import datetime

def test_file_creation():
    """Test that all required files were created"""
    required_files = [
        'commandsneeded/tests/test_all_commands.py',
        'commandsneeded/tests/run_tests.py',
        'commandsneeded/validation/validate_commands.py',
        'commandsneeded/docs/sync_documentation.py',
        'commandsneeded/ci/.github/workflows/cli-tests.yml',
        'commandsneeded/qa/lint_commands.py',
        'commandsneeded/performance/test_performance.py',
        'commandsneeded/a5-summary.json',
        'commandsneeded/a5-ideas.json'
    ]
    
    results = {
        'files_created': [],
        'files_missing': [],
        'total_required': len(required_files),
        'total_created': 0
    }
    
    for file_path in required_files:
        if os.path.exists(file_path):
            results['files_created'].append(file_path)
            results['total_created'] += 1
        else:
            results['files_missing'].append(file_path)
    
    return results

def test_file_content():
    """Test that files have meaningful content"""
    content_tests = {
        'commandsneeded/tests/test_all_commands.py': {
            'min_size': 1000,
            'keywords': ['unittest', 'TestTuskLangCLI', 'test_']
        },
        'commandsneeded/validation/validate_commands.py': {
            'min_size': 500,
            'keywords': ['CommandValidator', 'validate_', 'discover_']
        },
        'commandsneeded/docs/sync_documentation.py': {
            'min_size': 500,
            'keywords': ['DocumentationSync', 'generate_', 'sync_']
        },
        'commandsneeded/ci/.github/workflows/cli-tests.yml': {
            'min_size': 200,
            'keywords': ['name:', 'on:', 'jobs:', 'steps:']
        },
        'commandsneeded/qa/lint_commands.py': {
            'min_size': 500,
            'keywords': ['CommandLinter', 'lint_', 'check_']
        },
        'commandsneeded/performance/test_performance.py': {
            'min_size': 500,
            'keywords': ['PerformanceTester', 'measure_', 'test_']
        }
    }
    
    results = {
        'files_tested': 0,
        'files_passed': 0,
        'files_failed': 0,
        'details': []
    }
    
    for file_path, criteria in content_tests.items():
        if not os.path.exists(file_path):
            continue
        
        results['files_tested'] += 1
        file_result = {'file': file_path, 'passed': True, 'issues': []}
        
        try:
            with open(file_path, 'r') as f:
                content = f.read()
            
            # Check file size
            if len(content) < criteria['min_size']:
                file_result['passed'] = False
                file_result['issues'].append(f"File too small: {len(content)} chars (min: {criteria['min_size']})")
            
            # Check for keywords
            for keyword in criteria['keywords']:
                if keyword not in content:
                    file_result['passed'] = False
                    file_result['issues'].append(f"Missing keyword: {keyword}")
            
            if file_result['passed']:
                results['files_passed'] += 1
            else:
                results['files_failed'] += 1
            
        except Exception as e:
            file_result['passed'] = False
            file_result['issues'].append(f"Error reading file: {e}")
            results['files_failed'] += 1
        
        results['details'].append(file_result)
    
    return results

def test_json_files():
    """Test that JSON files are valid"""
    json_files = [
        'commandsneeded/a5-summary.json',
        'commandsneeded/a5-ideas.json'
    ]
    
    results = {
        'files_tested': 0,
        'files_valid': 0,
        'files_invalid': 0,
        'details': []
    }
    
    for file_path in json_files:
        if not os.path.exists(file_path):
            continue
        
        results['files_tested'] += 1
        file_result = {'file': file_path, 'valid': False, 'error': None}
        
        try:
            with open(file_path, 'r') as f:
                data = json.load(f)
            
            # Check for required fields
            if file_path.endswith('a5-summary.json'):
                required_fields = ['agent_id', 'status', 'goals_performance']
            else:
                required_fields = ['agent_id', 'brainstorming_ideas']
            
            missing_fields = [field for field in required_fields if field not in data]
            
            if missing_fields:
                file_result['error'] = f"Missing required fields: {missing_fields}"
            else:
                file_result['valid'] = True
                results['files_valid'] += 1
            
        except json.JSONDecodeError as e:
            file_result['error'] = f"Invalid JSON: {e}"
        except Exception as e:
            file_result['error'] = f"Error reading file: {e}"
        
        if not file_result['valid']:
            results['files_invalid'] += 1
        
        results['details'].append(file_result)
    
    return results

def main():
    """Run all tests"""
    print("🧪 Testing TuskTSK CLI Implementation")
    print("=" * 50)
    
    # Test file creation
    print("\n📁 Testing file creation...")
    file_results = test_file_creation()
    print(f"   Files created: {file_results['total_created']}/{file_results['total_required']}")
    
    if file_results['files_missing']:
        print("   Missing files:")
        for file in file_results['files_missing']:
            print(f"     - {file}")
    
    # Test file content
    print("\n📝 Testing file content...")
    content_results = test_file_content()
    print(f"   Files tested: {content_results['files_tested']}")
    print(f"   Files passed: {content_results['files_passed']}")
    print(f"   Files failed: {content_results['files_failed']}")
    
    if content_results['files_failed'] > 0:
        print("   Failed files:")
        for detail in content_results['details']:
            if not detail['passed']:
                print(f"     - {detail['file']}: {', '.join(detail['issues'])}")
    
    # Test JSON files
    print("\n📄 Testing JSON files...")
    json_results = test_json_files()
    print(f"   Files tested: {json_results['files_tested']}")
    print(f"   Files valid: {json_results['files_valid']}")
    print(f"   Files invalid: {json_results['files_invalid']}")
    
    if json_results['files_invalid'] > 0:
        print("   Invalid files:")
        for detail in json_results['details']:
            if not detail['valid']:
                print(f"     - {detail['file']}: {detail['error']}")
    
    # Overall results
    print("\n" + "=" * 50)
    print("📊 Overall Results")
    print("=" * 50)
    
    total_files_created = file_results['total_created']
    total_files_required = file_results['total_required']
    content_success_rate = (content_results['files_passed'] / content_results['files_tested'] * 100) if content_results['files_tested'] > 0 else 0
    json_success_rate = (json_results['files_valid'] / json_results['files_tested'] * 100) if json_results['files_tested'] > 0 else 0
    
    print(f"File Creation: {total_files_created}/{total_files_required} ({total_files_created/total_files_required*100:.1f}%)")
    print(f"Content Quality: {content_results['files_passed']}/{content_results['files_tested']} ({content_success_rate:.1f}%)")
    print(f"JSON Validity: {json_results['files_valid']}/{json_results['files_tested']} ({json_success_rate:.1f}%)")
    
    overall_success = (
        total_files_created == total_files_required and
        content_results['files_failed'] == 0 and
        json_results['files_invalid'] == 0
    )
    
    if overall_success:
        print("\n✅ All tests passed! Implementation is complete and working.")
        return 0
    else:
        print("\n❌ Some tests failed. Implementation needs attention.")
        return 1

if __name__ == '__main__':
    sys.exit(main()) 