#!/usr/bin/env python3
"""
Automated Test Runner for TuskTSK CLI
=====================================
Runs all command tests with comprehensive reporting and CI integration
"""

import sys
import os
import time
import json
import subprocess
import argparse
from pathlib import Path
from datetime import datetime
from typing import Dict, List, Any, Optional
import unittest
import coverage

# Add parent directory to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from test_all_commands import run_all_tests


class TestRunner:
    """Automated test runner with comprehensive reporting"""
    
    def __init__(self, verbose: bool = False, coverage_enabled: bool = True):
        self.verbose = verbose
        self.coverage_enabled = coverage_enabled
        self.results = {
            'timestamp': datetime.now().isoformat(),
            'total_tests': 0,
            'passed': 0,
            'failed': 0,
            'errors': 0,
            'skipped': 0,
            'coverage': None,
            'test_details': [],
            'performance_metrics': {}
        }
        
    def run_with_coverage(self) -> bool:
        """Run tests with coverage analysis"""
        if not self.coverage_enabled:
            return self.run_tests()
        
        # Start coverage measurement
        cov = coverage.Coverage()
        cov.start()
        
        start_time = time.time()
        success = self.run_tests()
        end_time = time.time()
        
        # Stop coverage and generate report
        cov.stop()
        cov.save()
        
        # Generate coverage report
        coverage_data = cov.get_data()
        total_lines = 0
        covered_lines = 0
        
        for filename in coverage_data.measured_files():
            if 'tusktsk' in filename:  # Only count our code
                file_coverage = cov.analysis2(filename)
                total_lines += len(file_coverage[1])
                covered_lines += len(file_coverage[2])
        
        coverage_percentage = (covered_lines / total_lines * 100) if total_lines > 0 else 0
        
        self.results['coverage'] = {
            'percentage': round(coverage_percentage, 2),
            'covered_lines': covered_lines,
            'total_lines': total_lines
        }
        
        self.results['performance_metrics']['execution_time'] = round(end_time - start_time, 2)
        
        if self.verbose:
            print(f"\n📊 Coverage Report:")
            print(f"   Coverage: {coverage_percentage:.2f}%")
            print(f"   Covered Lines: {covered_lines}")
            print(f"   Total Lines: {total_lines}")
            print(f"   Execution Time: {end_time - start_time:.2f}s")
        
        return success
    
    def run_tests(self) -> bool:
        """Run all test suites"""
        print("🚀 Starting TuskTSK CLI Test Suite...")
        print("=" * 60)
        
        # Create test suite
        test_suite = unittest.TestSuite()
        
        # Import and add all test modules
        test_modules = [
            'test_all_commands',
            'test_arguments',
            'test_error_handling',
            'test_edge_cases',
            'test_integration'
        ]
        
        for module_name in test_modules:
            try:
                module = __import__(module_name)
                if hasattr(module, 'run_all_tests'):
                    # Run module-specific tests
                    module_success = module.run_all_tests()
                    self.results['test_details'].append({
                        'module': module_name,
                        'status': 'passed' if module_success else 'failed'
                    })
                else:
                    # Add all test cases from module
                    loader = unittest.TestLoader()
                    tests = loader.loadTestsFromModule(module)
                    test_suite.addTests(tests)
            except ImportError as e:
                if self.verbose:
                    print(f"⚠️  Warning: Could not import {module_name}: {e}")
                self.results['test_details'].append({
                    'module': module_name,
                    'status': 'skipped',
                    'error': str(e)
                })
        
        # Run the test suite
        runner = unittest.TextTestRunner(verbosity=2 if self.verbose else 1)
        result = runner.run(test_suite)
        
        # Update results
        self.results['total_tests'] = result.testsRun
        self.results['passed'] = result.testsRun - len(result.failures) - len(result.errors)
        self.results['failed'] = len(result.failures)
        self.results['errors'] = len(result.errors)
        self.results['skipped'] = result.skipped if hasattr(result, 'skipped') else 0
        
        # Add detailed failure information
        for failure in result.failures:
            self.results['test_details'].append({
                'test': failure[0],
                'status': 'failed',
                'error': failure[1]
            })
        
        for error in result.errors:
            self.results['test_details'].append({
                'test': error[0],
                'status': 'error',
                'error': error[1]
            })
        
        success = result.wasSuccessful()
        
        # Print summary
        self.print_summary()
        
        return success
    
    def print_summary(self):
        """Print test execution summary"""
        print("\n" + "=" * 60)
        print("📋 Test Execution Summary")
        print("=" * 60)
        print(f"Total Tests: {self.results['total_tests']}")
        print(f"✅ Passed: {self.results['passed']}")
        print(f"❌ Failed: {self.results['failed']}")
        print(f"⚠️  Errors: {self.results['errors']}")
        print(f"⏭️  Skipped: {self.results['skipped']}")
        
        if self.results['coverage']:
            print(f"📊 Coverage: {self.results['coverage']['percentage']}%")
        
        if self.results['performance_metrics'].get('execution_time'):
            print(f"⏱️  Execution Time: {self.results['performance_metrics']['execution_time']}s")
        
        # Print detailed results
        if self.verbose and self.results['test_details']:
            print("\n📝 Detailed Results:")
            for detail in self.results['test_details']:
                status_icon = {
                    'passed': '✅',
                    'failed': '❌',
                    'error': '⚠️',
                    'skipped': '⏭️'
                }.get(detail['status'], '❓')
                
                print(f"   {status_icon} {detail.get('module', detail.get('test', 'Unknown'))}")
                if 'error' in detail:
                    print(f"      Error: {detail['error'][:100]}...")
    
    def save_results(self, output_file: str = None):
        """Save test results to JSON file"""
        if not output_file:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_file = f"test_results_{timestamp}.json"
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        with open(output_path, 'w') as f:
            json.dump(self.results, f, indent=2, default=str)
        
        if self.verbose:
            print(f"\n💾 Results saved to: {output_path}")
        
        return output_path
    
    def generate_junit_xml(self, output_file: str = None):
        """Generate JUnit XML report for CI integration"""
        if not output_file:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_file = f"junit_results_{timestamp}.xml"
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        # Generate JUnit XML format
        xml_content = f"""<?xml version="1.0" encoding="UTF-8"?>
<testsuites name="TuskTSK CLI Tests" tests="{self.results['total_tests']}" failures="{self.results['failed']}" errors="{self.results['errors']}" skipped="{self.results['skipped']}">
  <testsuite name="TuskTSK CLI" tests="{self.results['total_tests']}" failures="{self.results['failed']}" errors="{self.results['errors']}" skipped="{self.results['skipped']}">
"""
        
        # Add test cases
        for detail in self.results['test_details']:
            test_name = detail.get('test', detail.get('module', 'Unknown'))
            if isinstance(test_name, unittest.TestCase):
                test_name = f"{test_name.__class__.__name__}.{test_name._testMethodName}"
            
            xml_content += f'    <testcase name="{test_name}" classname="TuskTSK CLI">'
            
            if detail['status'] in ['failed', 'error']:
                xml_content += f"""
      <failure type="{detail['status']}" message="Test {detail['status']}">
        <![CDATA[{detail.get('error', 'Unknown error')}]]>
      </failure>"""
            
            xml_content += '</testcase>\n'
        
        xml_content += """  </testsuite>
</testsuites>"""
        
        with open(output_path, 'w') as f:
            f.write(xml_content)
        
        if self.verbose:
            print(f"📄 JUnit XML report saved to: {output_path}")
        
        return output_path


def run_performance_tests():
    """Run performance-specific tests"""
    print("\n⚡ Running Performance Tests...")
    
    performance_tests = [
        'test_performance',
        'test_memory',
        'test_cleanup',
        'test_concurrent',
        'test_large_data',
        'test_network'
    ]
    
    results = {}
    
    for test_module in performance_tests:
        try:
            module = __import__(test_module)
            if hasattr(module, 'run_performance_tests'):
                start_time = time.time()
                success = module.run_performance_tests()
                end_time = time.time()
                
                results[test_module] = {
                    'status': 'passed' if success else 'failed',
                    'execution_time': round(end_time - start_time, 2)
                }
        except ImportError:
            results[test_module] = {
                'status': 'skipped',
                'execution_time': 0
            }
    
    return results


def run_security_tests():
    """Run security-specific tests"""
    print("\n🔒 Running Security Tests...")
    
    security_tests = [
        'test_security',
        'test_accessibility',
        'test_i18n'
    ]
    
    results = {}
    
    for test_module in security_tests:
        try:
            module = __import__(test_module)
            if hasattr(module, 'run_security_tests'):
                success = module.run_security_tests()
                results[test_module] = {
                    'status': 'passed' if success else 'failed'
                }
        except ImportError:
            results[test_module] = {
                'status': 'skipped'
            }
    
    return results


def main():
    """Main entry point for test runner"""
    parser = argparse.ArgumentParser(description='TuskTSK CLI Test Runner')
    parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose output')
    parser.add_argument('--no-coverage', action='store_true', help='Disable coverage analysis')
    parser.add_argument('--output', '-o', help='Output file for results')
    parser.add_argument('--junit', help='Generate JUnit XML report')
    parser.add_argument('--performance', action='store_true', help='Run performance tests')
    parser.add_argument('--security', action='store_true', help='Run security tests')
    parser.add_argument('--all', action='store_true', help='Run all test types')
    
    args = parser.parse_args()
    
    # Create test runner
    runner = TestRunner(
        verbose=args.verbose,
        coverage_enabled=not args.no_coverage
    )
    
    # Run main tests
    success = runner.run_with_coverage()
    
    # Run additional test types if requested
    if args.all or args.performance:
        performance_results = run_performance_tests()
        runner.results['performance_tests'] = performance_results
    
    if args.all or args.security:
        security_results = run_security_tests()
        runner.results['security_tests'] = security_results
    
    # Save results
    if args.output:
        runner.save_results(args.output)
    else:
        runner.save_results()
    
    # Generate JUnit XML if requested
    if args.junit:
        runner.generate_junit_xml(args.junit)
    
    # Exit with appropriate code
    sys.exit(0 if success else 1)


if __name__ == '__main__':
    main() 