#!/usr/bin/env python3
"""
Comprehensive Test Suite for TuskTSK CLI Commands
=================================================
Tests all CLI commands with full coverage and edge case handling
"""

import unittest
import sys
import os
import tempfile
import shutil
import json
import subprocess
from pathlib import Path
from unittest.mock import patch, MagicMock, mock_open
from io import StringIO

# Add parent directory to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from cli.main import TuskLangCLI
from cli.commands import (
    db_commands, dev_commands, test_commands, service_commands,
    cache_commands, config_commands, binary_commands, ai_commands,
    utility_commands, peanuts_commands, css_commands, license_commands,
    dependency_commands
)


class TestTuskLangCLI(unittest.TestCase):
    """Test suite for main CLI functionality"""
    
    def setUp(self):
        """Set up test environment"""
        self.cli = TuskLangCLI()
        self.temp_dir = tempfile.mkdtemp()
        self.original_cwd = os.getcwd()
        os.chdir(self.temp_dir)
        
    def tearDown(self):
        """Clean up test environment"""
        os.chdir(self.original_cwd)
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    def test_cli_initialization(self):
        """Test CLI initialization"""
        self.assertIsNotNone(self.cli.parser)
        self.assertIsNotNone(self.cli.parser.description)
    
    def test_version_argument(self):
        """Test version argument"""
        with patch('sys.stdout', new=StringIO()) as fake_out:
            with self.assertRaises(SystemExit):
                self.cli.parser.parse_args(['--version'])
    
    def test_verbose_argument(self):
        """Test verbose argument"""
        args = self.cli.parser.parse_args(['--verbose'])
        self.assertTrue(args.verbose)
    
    def test_quiet_argument(self):
        """Test quiet argument"""
        args = self.cli.parser.parse_args(['--quiet'])
        self.assertTrue(args.quiet)
    
    def test_json_argument(self):
        """Test JSON output argument"""
        args = self.cli.parser.parse_args(['--json'])
        self.assertTrue(args.json)
    
    def test_config_argument(self):
        """Test config file argument"""
        args = self.cli.parser.parse_args(['--config', 'test.conf'])
        self.assertEqual(args.config, 'test.conf')


class TestDatabaseCommands(unittest.TestCase):
    """Test suite for database commands"""
    
    def setUp(self):
        """Set up test environment"""
        self.temp_dir = tempfile.mkdtemp()
        self.original_cwd = os.getcwd()
        os.chdir(self.temp_dir)
        
    def tearDown(self):
        """Clean up test environment"""
        os.chdir(self.original_cwd)
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    @patch('cli.commands.db_commands.TSK')
    def test_db_status_command(self, mock_tsk):
        """Test database status command"""
        mock_tsk_instance = MagicMock()
        mock_tsk_instance.check_connection.return_value = True
        mock_tsk.return_value = mock_tsk_instance
        
        result = db_commands.db_status()
        self.assertTrue(result)
        mock_tsk_instance.check_connection.assert_called_once()
    
    @patch('cli.commands.db_commands.TSK')
    def test_db_init_command(self, mock_tsk):
        """Test database initialization command"""
        mock_tsk_instance = MagicMock()
        mock_tsk.return_value = mock_tsk_instance
        
        result = db_commands.db_init()
        self.assertTrue(result)
        mock_tsk_instance.init_database.assert_called_once()
    
    @patch('cli.commands.db_commands.TSK')
    def test_db_migrate_command(self, mock_tsk):
        """Test database migration command"""
        mock_tsk_instance = MagicMock()
        mock_tsk.return_value = mock_tsk_instance
        
        # Create a test migration file
        migration_file = os.path.join(self.temp_dir, 'test_migration.sql')
        with open(migration_file, 'w') as f:
            f.write('CREATE TABLE test (id INTEGER PRIMARY KEY);')
        
        result = db_commands.db_migrate(migration_file)
        self.assertTrue(result)
        mock_tsk_instance.run_migration.assert_called_once_with(migration_file)
    
    @patch('cli.commands.db_commands.TSK')
    def test_db_backup_command(self, mock_tsk):
        """Test database backup command"""
        mock_tsk_instance = MagicMock()
        mock_tsk.return_value = mock_tsk_instance
        
        backup_file = os.path.join(self.temp_dir, 'backup.db')
        result = db_commands.db_backup(backup_file)
        self.assertTrue(result)
        mock_tsk_instance.backup_database.assert_called_once_with(backup_file)
    
    @patch('cli.commands.db_commands.TSK')
    def test_db_restore_command(self, mock_tsk):
        """Test database restore command"""
        mock_tsk_instance = MagicMock()
        mock_tsk.return_value = mock_tsk_instance
        
        backup_file = os.path.join(self.temp_dir, 'backup.db')
        with open(backup_file, 'w') as f:
            f.write('test backup data')
        
        result = db_commands.db_restore(backup_file)
        self.assertTrue(result)
        mock_tsk_instance.restore_database.assert_called_once_with(backup_file)


class TestDevelopmentCommands(unittest.TestCase):
    """Test suite for development commands"""
    
    @patch('cli.commands.dev_commands.subprocess.Popen')
    def test_serve_command(self, mock_popen):
        """Test development server command"""
        mock_process = MagicMock()
        mock_popen.return_value = mock_process
        
        result = dev_commands.serve(8080)
        self.assertTrue(result)
        mock_popen.assert_called_once()
    
    @patch('cli.commands.dev_commands.subprocess.run')
    def test_build_command(self, mock_run):
        """Test build command"""
        mock_run.return_value.returncode = 0
        
        result = dev_commands.build()
        self.assertTrue(result)
        mock_run.assert_called_once()


class TestTestCommands(unittest.TestCase):
    """Test suite for test commands"""
    
    @patch('cli.commands.test_commands.subprocess.run')
    def test_test_all_command(self, mock_run):
        """Test running all tests"""
        mock_run.return_value.returncode = 0
        
        result = test_commands.test_all()
        self.assertTrue(result)
        mock_run.assert_called_once()
    
    @patch('cli.commands.test_commands.subprocess.run')
    def test_test_unit_command(self, mock_run):
        """Test running unit tests"""
        mock_run.return_value.returncode = 0
        
        result = test_commands.test_unit()
        self.assertTrue(result)
        mock_run.assert_called_once()
    
    @patch('cli.commands.test_commands.subprocess.run')
    def test_test_integration_command(self, mock_run):
        """Test running integration tests"""
        mock_run.return_value.returncode = 0
        
        result = test_commands.test_integration()
        self.assertTrue(result)
        mock_run.assert_called_once()


class TestServiceCommands(unittest.TestCase):
    """Test suite for service commands"""
    
    @patch('cli.commands.service_commands.subprocess.Popen')
    def test_service_start_command(self, mock_popen):
        """Test service start command"""
        mock_process = MagicMock()
        mock_popen.return_value = mock_process
        
        result = service_commands.service_start('test-service')
        self.assertTrue(result)
        mock_popen.assert_called_once()
    
    @patch('cli.commands.service_commands.subprocess.run')
    def test_service_stop_command(self, mock_run):
        """Test service stop command"""
        mock_run.return_value.returncode = 0
        
        result = service_commands.service_stop('test-service')
        self.assertTrue(result)
        mock_run.assert_called_once()
    
    @patch('cli.commands.service_commands.subprocess.run')
    def test_service_status_command(self, mock_run):
        """Test service status command"""
        mock_run.return_value.returncode = 0
        mock_run.return_value.stdout = b'service is running'
        
        result = service_commands.service_status('test-service')
        self.assertTrue(result)
        mock_run.assert_called_once()


class TestCacheCommands(unittest.TestCase):
    """Test suite for cache commands"""
    
    def setUp(self):
        """Set up test environment"""
        self.temp_dir = tempfile.mkdtemp()
        
    def tearDown(self):
        """Clean up test environment"""
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    @patch('cli.commands.cache_commands.CacheManager')
    def test_cache_clear_command(self, mock_cache_manager):
        """Test cache clear command"""
        mock_cache = MagicMock()
        mock_cache_manager.return_value = mock_cache
        
        result = cache_commands.cache_clear()
        self.assertTrue(result)
        mock_cache.clear_all.assert_called_once()
    
    @patch('cli.commands.cache_commands.CacheManager')
    def test_cache_status_command(self, mock_cache_manager):
        """Test cache status command"""
        mock_cache = MagicMock()
        mock_cache.get_stats.return_value = {'size': 1024, 'items': 10}
        mock_cache_manager.return_value = mock_cache
        
        result = cache_commands.cache_status()
        self.assertTrue(result)
        mock_cache.get_stats.assert_called_once()


class TestConfigCommands(unittest.TestCase):
    """Test suite for config commands"""
    
    def setUp(self):
        """Set up test environment"""
        self.temp_dir = tempfile.mkdtemp()
        self.config_file = os.path.join(self.temp_dir, 'test_config.json')
        
    def tearDown(self):
        """Clean up test environment"""
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    def test_config_get_command(self):
        """Test config get command"""
        config_data = {'server': {'port': 8080}}
        with open(self.config_file, 'w') as f:
            json.dump(config_data, f)
        
        with patch('cli.commands.config_commands.CONFIG_FILE', self.config_file):
            result = config_commands.config_get('server.port')
            self.assertEqual(result, 8080)
    
    def test_config_set_command(self):
        """Test config set command"""
        config_data = {'server': {'port': 8080}}
        with open(self.config_file, 'w') as f:
            json.dump(config_data, f)
        
        with patch('cli.commands.config_commands.CONFIG_FILE', self.config_file):
            result = config_commands.config_set('server.port', 9090)
            self.assertTrue(result)
            
            # Verify the change was saved
            with open(self.config_file, 'r') as f:
                updated_config = json.load(f)
            self.assertEqual(updated_config['server']['port'], 9090)
    
    def test_config_list_command(self):
        """Test config list command"""
        config_data = {'server': {'port': 8080}, 'database': {'url': 'sqlite:///test.db'}}
        with open(self.config_file, 'w') as f:
            json.dump(config_data, f)
        
        with patch('cli.commands.config_commands.CONFIG_FILE', self.config_file):
            result = config_commands.config_list()
            self.assertIsInstance(result, dict)
            self.assertEqual(result, config_data)


class TestBinaryCommands(unittest.TestCase):
    """Test suite for binary commands"""
    
    def setUp(self):
        """Set up test environment"""
        self.temp_dir = tempfile.mkdtemp()
        
    def tearDown(self):
        """Clean up test environment"""
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    def test_binary_compile_command(self):
        """Test binary compilation command"""
        # Create a test TuskLang file
        tusk_file = os.path.join(self.temp_dir, 'test.tsk')
        with open(tusk_file, 'w') as f:
            f.write('print("Hello, World!");')
        
        result = binary_commands.binary_compile(tusk_file)
        self.assertTrue(result)
        
        # Check if binary file was created
        binary_file = os.path.join(self.temp_dir, 'test.bin')
        self.assertTrue(os.path.exists(binary_file))
    
    def test_binary_decompile_command(self):
        """Test binary decompilation command"""
        # Create a test binary file
        binary_file = os.path.join(self.temp_dir, 'test.bin')
        with open(binary_file, 'wb') as f:
            f.write(b'test binary data')
        
        result = binary_commands.binary_decompile(binary_file)
        self.assertTrue(result)
        
        # Check if decompiled file was created
        decompiled_file = os.path.join(self.temp_dir, 'test_decompiled.tsk')
        self.assertTrue(os.path.exists(decompiled_file))


class TestAICommands(unittest.TestCase):
    """Test suite for AI commands"""
    
    @patch('cli.commands.ai_commands.AIProcessor')
    def test_ai_analyze_command(self, mock_ai_processor):
        """Test AI analysis command"""
        mock_ai = MagicMock()
        mock_ai.analyze.return_value = {'sentiment': 'positive', 'confidence': 0.95}
        mock_ai_processor.return_value = mock_ai
        
        result = ai_commands.ai_analyze('test input')
        self.assertIsInstance(result, dict)
        self.assertEqual(result['sentiment'], 'positive')
        mock_ai.analyze.assert_called_once_with('test input')
    
    @patch('cli.commands.ai_commands.AIProcessor')
    def test_ai_generate_command(self, mock_ai_processor):
        """Test AI generation command"""
        mock_ai = MagicMock()
        mock_ai.generate.return_value = 'Generated content'
        mock_ai_processor.return_value = mock_ai
        
        result = ai_commands.ai_generate('test prompt')
        self.assertEqual(result, 'Generated content')
        mock_ai.generate.assert_called_once_with('test prompt')


class TestUtilityCommands(unittest.TestCase):
    """Test suite for utility commands"""
    
    def setUp(self):
        """Set up test environment"""
        self.temp_dir = tempfile.mkdtemp()
        
    def tearDown(self):
        """Clean up test environment"""
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    def test_utility_format_command(self):
        """Test utility format command"""
        # Create a test file
        test_file = os.path.join(self.temp_dir, 'test.tsk')
        with open(test_file, 'w') as f:
            f.write('print("test");\nprint("another test");')
        
        result = utility_commands.utility_format(test_file)
        self.assertTrue(result)
        
        # Check if file was formatted
        with open(test_file, 'r') as f:
            content = f.read()
        self.assertIn('print("test");', content)
    
    def test_utility_lint_command(self):
        """Test utility lint command"""
        # Create a test file
        test_file = os.path.join(self.temp_dir, 'test.tsk')
        with open(test_file, 'w') as f:
            f.write('print("test");')
        
        result = utility_commands.utility_lint(test_file)
        self.assertTrue(result)
    
    def test_utility_validate_command(self):
        """Test utility validate command"""
        # Create a test file
        test_file = os.path.join(self.temp_dir, 'test.tsk')
        with open(test_file, 'w') as f:
            f.write('print("test");')
        
        result = utility_commands.utility_validate(test_file)
        self.assertTrue(result)


class TestPeanutsCommands(unittest.TestCase):
    """Test suite for peanuts commands"""
    
    @patch('cli.commands.peanuts_commands.PeanutManager')
    def test_peanuts_create_command(self, mock_peanut_manager):
        """Test peanuts create command"""
        mock_manager = MagicMock()
        mock_peanut_manager.return_value = mock_manager
        
        result = peanuts_commands.peanuts_create('test_peanut')
        self.assertTrue(result)
        mock_manager.create_peanut.assert_called_once_with('test_peanut')
    
    @patch('cli.commands.peanuts_commands.PeanutManager')
    def test_peanuts_list_command(self, mock_peanut_manager):
        """Test peanuts list command"""
        mock_manager = MagicMock()
        mock_manager.list_peanuts.return_value = ['peanut1', 'peanut2']
        mock_peanut_manager.return_value = mock_manager
        
        result = peanuts_commands.peanuts_list()
        self.assertEqual(result, ['peanut1', 'peanut2'])
        mock_manager.list_peanuts.assert_called_once()


class TestCSSCommands(unittest.TestCase):
    """Test suite for CSS commands"""
    
    def setUp(self):
        """Set up test environment"""
        self.temp_dir = tempfile.mkdtemp()
        
    def tearDown(self):
        """Clean up test environment"""
        shutil.rmtree(self.temp_dir, ignore_errors=True)
    
    def test_css_compile_command(self):
        """Test CSS compilation command"""
        # Create a test CSS file
        css_file = os.path.join(self.temp_dir, 'test.css')
        with open(css_file, 'w') as f:
            f.write('.test { color: red; }')
        
        result = css_commands.css_compile(css_file)
        self.assertTrue(result)
        
        # Check if compiled file was created
        compiled_file = os.path.join(self.temp_dir, 'test.min.css')
        self.assertTrue(os.path.exists(compiled_file))
    
    def test_css_validate_command(self):
        """Test CSS validation command"""
        # Create a test CSS file
        css_file = os.path.join(self.temp_dir, 'test.css')
        with open(css_file, 'w') as f:
            f.write('.test { color: red; }')
        
        result = css_commands.css_validate(css_file)
        self.assertTrue(result)


class TestLicenseCommands(unittest.TestCase):
    """Test suite for license commands"""
    
    @patch('cli.commands.license_commands.LicenseManager')
    def test_license_check_command(self, mock_license_manager):
        """Test license check command"""
        mock_manager = MagicMock()
        mock_manager.check_license.return_value = True
        mock_license_manager.return_value = mock_manager
        
        result = license_commands.license_check()
        self.assertTrue(result)
        mock_manager.check_license.assert_called_once()
    
    @patch('cli.commands.license_commands.LicenseManager')
    def test_license_generate_command(self, mock_license_manager):
        """Test license generate command"""
        mock_manager = MagicMock()
        mock_manager.generate_license.return_value = 'test-license-key'
        mock_license_manager.return_value = mock_manager
        
        result = license_commands.license_generate('test-user')
        self.assertEqual(result, 'test-license-key')
        mock_manager.generate_license.assert_called_once_with('test-user')


class TestDependencyCommands(unittest.TestCase):
    """Test suite for dependency commands"""
    
    @patch('cli.commands.dependency_commands.subprocess.run')
    def test_dependency_install_command(self, mock_run):
        """Test dependency install command"""
        mock_run.return_value.returncode = 0
        
        result = dependency_commands.dependency_install('test-package')
        self.assertTrue(result)
        mock_run.assert_called_once()
    
    @patch('cli.commands.dependency_commands.subprocess.run')
    def test_dependency_update_command(self, mock_run):
        """Test dependency update command"""
        mock_run.return_value.returncode = 0
        
        result = dependency_commands.dependency_update('test-package')
        self.assertTrue(result)
        mock_run.assert_called_once()
    
    @patch('cli.commands.dependency_commands.subprocess.run')
    def test_dependency_list_command(self, mock_run):
        """Test dependency list command"""
        mock_run.return_value.returncode = 0
        mock_run.return_value.stdout = b'test-package==1.0.0'
        
        result = dependency_commands.dependency_list()
        self.assertTrue(result)
        mock_run.assert_called_once()


def run_all_tests():
    """Run all test suites"""
    # Create test suite
    test_suite = unittest.TestSuite()
    
    # Add all test classes
    test_classes = [
        TestTuskLangCLI,
        TestDatabaseCommands,
        TestDevelopmentCommands,
        TestTestCommands,
        TestServiceCommands,
        TestCacheCommands,
        TestConfigCommands,
        TestBinaryCommands,
        TestAICommands,
        TestUtilityCommands,
        TestPeanutsCommands,
        TestCSSCommands,
        TestLicenseCommands,
        TestDependencyCommands
    ]
    
    for test_class in test_classes:
        tests = unittest.TestLoader().loadTestsFromTestCase(test_class)
        test_suite.addTests(tests)
    
    # Run tests
    runner = unittest.TextTestRunner(verbosity=2)
    result = runner.run(test_suite)
    
    return result.wasSuccessful()


if __name__ == '__main__':
    success = run_all_tests()
    sys.exit(0 if success else 1) 