#!/usr/bin/env python3
"""
Command Validation Script for TuskTSK CLI
=========================================
Validates all documented commands exist and work correctly
"""

import sys
import os
import json
import subprocess
import argparse
from pathlib import Path
from typing import Dict, List, Any, Optional, Tuple
from datetime import datetime
import importlib
import inspect

# Add parent directory to path for imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from cli.main import TuskLangCLI


class CommandValidator:
    """Validates CLI commands and their functionality"""
    
    def __init__(self, verbose: bool = False):
        self.verbose = verbose
        self.cli = TuskLangCLI()
        self.validation_results = {
            'timestamp': datetime.now().isoformat(),
            'total_commands': 0,
            'valid_commands': 0,
            'invalid_commands': 0,
            'missing_commands': 0,
            'command_details': [],
            'errors': []
        }
        
    def discover_all_commands(self) -> Dict[str, Any]:
        """Discover all available commands from CLI parser"""
        commands = {}
        
        # Get all subparsers
        for action in self.cli.parser._subparsers._group_actions:
            if hasattr(action, 'choices'):
                for command_name, subparser in action.choices.items():
                    commands[command_name] = {
                        'help': subparser.description or subparser.help,
                        'subcommands': {},
                        'arguments': []
                    }
                    
                    # Get subcommands
                    for subaction in subparser._subparsers._group_actions:
                        if hasattr(subaction, 'choices'):
                            for subcommand_name, subsubparser in subaction.choices.items():
                                commands[command_name]['subcommands'][subcommand_name] = {
                                    'help': subsubparser.description or subsubparser.help,
                                    'arguments': []
                                }
                                
                                # Get arguments
                                for arg in subsubparser._actions:
                                    if arg.dest != 'help':
                                        commands[command_name]['subcommands'][subcommand_name]['arguments'].append({
                                            'name': arg.dest,
                                            'type': type(arg.type).__name__ if arg.type else 'str',
                                            'required': arg.required,
                                            'default': arg.default,
                                            'help': arg.help
                                        })
        
        return commands
    
    def validate_command_help(self, command_name: str, subcommand_name: str = None) -> bool:
        """Validate command has proper help text"""
        try:
            if subcommand_name:
                # Test subcommand help
                result = subprocess.run(
                    ['python', '-m', 'tusktsk.cli.main', command_name, subcommand_name, '--help'],
                    capture_output=True,
                    text=True,
                    timeout=10
                )
            else:
                # Test main command help
                result = subprocess.run(
                    ['python', '-m', 'tusktsk.cli.main', command_name, '--help'],
                    capture_output=True,
                    text=True,
                    timeout=10
                )
            
            return result.returncode == 0 and len(result.stdout.strip()) > 0
        except Exception as e:
            if self.verbose:
                print(f"Error validating help for {command_name}: {e}")
            return False
    
    def validate_command_execution(self, command_name: str, subcommand_name: str = None, args: List[str] = None) -> bool:
        """Validate command can be executed"""
        try:
            cmd = ['python', '-m', 'tusktsk.cli.main', command_name]
            
            if subcommand_name:
                cmd.append(subcommand_name)
            
            if args:
                cmd.extend(args)
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                timeout=30
            )
            
            # Command should not crash (return code 0 or 1 is acceptable for validation)
            return result.returncode in [0, 1, 2]  # 0=success, 1=error, 2=usage error
        except subprocess.TimeoutExpired:
            if self.verbose:
                print(f"Timeout validating execution for {command_name}")
            return False
        except Exception as e:
            if self.verbose:
                print(f"Error validating execution for {command_name}: {e}")
            return False
    
    def validate_command_arguments(self, command_name: str, subcommand_name: str = None) -> List[Dict[str, Any]]:
        """Validate command arguments work correctly"""
        argument_results = []
        
        try:
            # Get command parser
            if subcommand_name:
                parser = self.cli.parser._subparsers._group_actions[0].choices[command_name]._subparsers._group_actions[0].choices[subcommand_name]
            else:
                parser = self.cli.parser._subparsers._group_actions[0].choices[command_name]
            
            # Test each argument
            for action in parser._actions:
                if action.dest == 'help':
                    continue
                
                arg_result = {
                    'name': action.dest,
                    'type': type(action.type).__name__ if action.type else 'str',
                    'required': action.required,
                    'valid': True,
                    'error': None
                }
                
                # Test argument parsing
                try:
                    if action.type == int:
                        test_value = "123"
                    elif action.type == float:
                        test_value = "123.45"
                    elif action.type == bool:
                        test_value = "true"
                    else:
                        test_value = "test_value"
                    
                    # Test that argument can be parsed
                    test_args = [command_name]
                    if subcommand_name:
                        test_args.append(subcommand_name)
                    test_args.extend([f"--{action.dest}", test_value])
                    
                    parsed_args = self.cli.parser.parse_args(test_args)
                    arg_result['valid'] = True
                    
                except Exception as e:
                    arg_result['valid'] = False
                    arg_result['error'] = str(e)
                
                argument_results.append(arg_result)
        
        except Exception as e:
            if self.verbose:
                print(f"Error validating arguments for {command_name}: {e}")
        
        return argument_results
    
    def validate_command_output(self, command_name: str, subcommand_name: str = None) -> Dict[str, Any]:
        """Validate command output formatting and structure"""
        output_result = {
            'valid': False,
            'has_output': False,
            'output_format': None,
            'error': None
        }
        
        try:
            cmd = ['python', '-m', 'tusktsk.cli.main', command_name]
            
            if subcommand_name:
                cmd.append(subcommand_name)
            
            # Test with JSON output
            cmd.append('--json')
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                text=True,
                timeout=30
            )
            
            if result.returncode in [0, 1, 2]:
                output_result['has_output'] = len(result.stdout.strip()) > 0 or len(result.stderr.strip()) > 0
                
                # Check if output is valid JSON
                if result.stdout.strip():
                    try:
                        json.loads(result.stdout)
                        output_result['output_format'] = 'json'
                        output_result['valid'] = True
                    except json.JSONDecodeError:
                        output_result['output_format'] = 'text'
                        output_result['valid'] = True
                
                # Check stderr for errors
                if result.stderr.strip():
                    output_result['error'] = result.stderr.strip()
        
        except Exception as e:
            output_result['error'] = str(e)
        
        return output_result
    
    def validate_all_commands(self) -> Dict[str, Any]:
        """Validate all discovered commands"""
        print("🔍 Discovering all CLI commands...")
        commands = self.discover_all_commands()
        
        print(f"📋 Found {len(commands)} main command categories")
        
        for command_name, command_info in commands.items():
            print(f"\n🔧 Validating command: {command_name}")
            
            # Validate main command
            main_command_valid = self.validate_command_help(command_name)
            main_command_executable = self.validate_command_execution(command_name)
            
            command_detail = {
                'command': command_name,
                'help_valid': main_command_valid,
                'executable': main_command_executable,
                'subcommands': {},
                'arguments': self.validate_command_arguments(command_name),
                'output': self.validate_command_output(command_name)
            }
            
            # Validate subcommands
            for subcommand_name in command_info['subcommands']:
                print(f"  📝 Validating subcommand: {subcommand_name}")
                
                subcommand_valid = self.validate_command_help(command_name, subcommand_name)
                subcommand_executable = self.validate_command_execution(command_name, subcommand_name)
                
                command_detail['subcommands'][subcommand_name] = {
                    'help_valid': subcommand_valid,
                    'executable': subcommand_executable,
                    'arguments': self.validate_command_arguments(command_name, subcommand_name),
                    'output': self.validate_command_output(command_name, subcommand_name)
                }
            
            # Update counters
            self.validation_results['total_commands'] += 1
            
            if main_command_valid and main_command_executable:
                self.validation_results['valid_commands'] += 1
            else:
                self.validation_results['invalid_commands'] += 1
            
            self.validation_results['command_details'].append(command_detail)
        
        return self.validation_results
    
    def validate_documented_commands(self, documentation_file: str = None) -> Dict[str, Any]:
        """Validate that all documented commands exist and work"""
        documented_commands = self.load_documented_commands(documentation_file)
        
        print(f"📚 Validating {len(documented_commands)} documented commands...")
        
        for doc_command in documented_commands:
            command_name = doc_command.get('command')
            subcommand_name = doc_command.get('subcommand')
            
            print(f"🔍 Validating documented command: {command_name} {subcommand_name or ''}")
            
            # Check if command exists
            command_exists = self.command_exists_in_cli(command_name, subcommand_name)
            
            if not command_exists:
                self.validation_results['missing_commands'] += 1
                self.validation_results['errors'].append({
                    'type': 'missing_command',
                    'command': command_name,
                    'subcommand': subcommand_name,
                    'message': f"Documented command '{command_name} {subcommand_name or ''}' not found in CLI"
                })
                continue
            
            # Validate command functionality
            help_valid = self.validate_command_help(command_name, subcommand_name)
            executable = self.validate_command_execution(command_name, subcommand_name)
            
            if not help_valid or not executable:
                self.validation_results['invalid_commands'] += 1
                self.validation_results['errors'].append({
                    'type': 'invalid_command',
                    'command': command_name,
                    'subcommand': subcommand_name,
                    'help_valid': help_valid,
                    'executable': executable
                })
            else:
                self.validation_results['valid_commands'] += 1
        
        return self.validation_results
    
    def command_exists_in_cli(self, command_name: str, subcommand_name: str = None) -> bool:
        """Check if command exists in CLI parser"""
        try:
            if command_name not in self.cli.parser._subparsers._group_actions[0].choices:
                return False
            
            if subcommand_name:
                main_parser = self.cli.parser._subparsers._group_actions[0].choices[command_name]
                return subcommand_name in main_parser._subparsers._group_actions[0].choices
            
            return True
        except Exception:
            return False
    
    def load_documented_commands(self, documentation_file: str = None) -> List[Dict[str, Any]]:
        """Load documented commands from file or generate from code"""
        if documentation_file and os.path.exists(documentation_file):
            with open(documentation_file, 'r') as f:
                return json.load(f)
        
        # Generate from CLI parser as fallback
        commands = []
        discovered_commands = self.discover_all_commands()
        
        for command_name, command_info in discovered_commands.items():
            commands.append({
                'command': command_name,
                'help': command_info['help'],
                'category': 'main'
            })
            
            for subcommand_name, subcommand_info in command_info['subcommands'].items():
                commands.append({
                    'command': command_name,
                    'subcommand': subcommand_name,
                    'help': subcommand_info['help'],
                    'category': 'subcommand'
                })
        
        return commands
    
    def generate_validation_report(self, output_file: str = None) -> str:
        """Generate comprehensive validation report"""
        if not output_file:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_file = f"command_validation_{timestamp}.json"
        
        output_path = os.path.join(os.path.dirname(__file__), output_file)
        
        with open(output_path, 'w') as f:
            json.dump(self.validation_results, f, indent=2, default=str)
        
        return output_path
    
    def print_summary(self):
        """Print validation summary"""
        print("\n" + "=" * 60)
        print("📋 Command Validation Summary")
        print("=" * 60)
        print(f"Total Commands: {self.validation_results['total_commands']}")
        print(f"✅ Valid Commands: {self.validation_results['valid_commands']}")
        print(f"❌ Invalid Commands: {self.validation_results['invalid_commands']}")
        print(f"🔍 Missing Commands: {self.validation_results['missing_commands']}")
        
        if self.validation_results['errors']:
            print(f"\n⚠️  Errors Found:")
            for error in self.validation_results['errors']:
                print(f"   - {error['type']}: {error.get('message', 'Unknown error')}")


def main():
    """Main entry point for command validation"""
    parser = argparse.ArgumentParser(description='TuskTSK CLI Command Validator')
    parser.add_argument('--verbose', '-v', action='store_true', help='Enable verbose output')
    parser.add_argument('--discover', action='store_true', help='Discover all commands')
    parser.add_argument('--validate-docs', action='store_true', help='Validate documented commands')
    parser.add_argument('--docs-file', help='Documentation file to validate against')
    parser.add_argument('--output', '-o', help='Output file for validation results')
    parser.add_argument('--all', action='store_true', help='Run all validations')
    
    args = parser.parse_args()
    
    # Create validator
    validator = CommandValidator(verbose=args.verbose)
    
    # Run validations
    if args.all or args.discover:
        print("🚀 Running command discovery and validation...")
        validator.validate_all_commands()
    
    if args.all or args.validate_docs:
        print("📚 Validating documented commands...")
        validator.validate_documented_commands(args.docs_file)
    
    # Print summary
    validator.print_summary()
    
    # Save results
    if args.output:
        output_file = validator.generate_validation_report(args.output)
    else:
        output_file = validator.generate_validation_report()
    
    if args.verbose:
        print(f"\n💾 Validation results saved to: {output_file}")
    
    # Exit with appropriate code
    total_issues = validator.validation_results['invalid_commands'] + validator.validation_results['missing_commands']
    sys.exit(0 if total_issues == 0 else 1)


if __name__ == '__main__':
    main() 