# Flask-TSK Deployment Guide

**Complete Guide to Deploying Flask-TSK in Production**

## 🚀 Overview

Flask-TSK is a revolutionary Flask extension that provides:

- **Performance Engine** - 10x faster template rendering
- **TuskLang Integration** - Full SDK access and configuration
- **Configuration Management** - Load from `peanu.tsk` files
- **Database Adapters** - PostgreSQL, MySQL, MongoDB, Redis
- **REST API Endpoints** - Built-in TuskLang operations
- **FastAPI Support** - Complete FastAPI router integration

## 📦 Installation

### Production Installation

```bash
# Install with all dependencies
pip install flask-tsk[performance,databases,fastapi]

# Or install specific components
pip install flask-tsk[performance]  # Performance optimizations only
pip install flask-tsk[databases]    # Database adapters only
pip install flask-tsk[fastapi]      # FastAPI integration only
```

### Development Installation

```bash
# Clone the repository
git clone https://github.com/grim-project/flask-tsk.git
cd flask-tsk

# Install in development mode
pip install -e .

# Install development dependencies
pip install -e .[dev]
```

## 🔧 Configuration

### Flask Configuration

```python
from flask import Flask
from tsk_flask import FlaskTSK

app = Flask(__name__)

# Flask-TSK configuration
app.config.update({
    'TSK_CONFIG_PATH': '/path/to/config.tsk',  # Custom config path
    'TSK_AUTO_LOAD': True,                     # Auto-load peanu.tsk
    'TSK_ENABLE_BLUEPRINT': True,              # Enable API endpoints
    'TSK_ENABLE_CONTEXT': True,                # Enable template context
    'TSK_ENABLE_FULL_SDK': True,               # Enable full TuskLang SDK
})

# Initialize Flask-TSK
tsk = FlaskTSK(app)
```

### TuskLang Configuration (peanu.tsk)

```ini
[database]
type = "postgresql"
host = "localhost"
port = 5432
name = "myapp"
username = "user"
password = "pass"

[security]
encryption_key = "your-encryption-key"
jwt_secret = "your-jwt-secret"

[ui]
theme = "dark"
component_cache = true
minify_assets = true

[performance]
turbo_engine = true
cache_enabled = true
parallel_processing = true
```

## 🚀 Quick Start Examples

### Basic Flask Application

```python
from flask import Flask, render_template
from tsk_flask import FlaskTSK

app = Flask(__name__)
tsk = FlaskTSK(app)

@app.route('/')
def index():
    # Get configuration from TuskLang
    app_name = tsk.get_config('app', 'name', 'Flask-TSK')
    return render_template('index.html', app_name=app_name)

@app.route('/config')
def show_config():
    # Display all configuration
    config = tsk.get_all_sections()
    return {'config': config}

if __name__ == '__main__':
    app.run(debug=True)
```

### Template Integration

```html
<!-- templates/index.html -->
<!DOCTYPE html>
<html>
<head>
    <title>{{ app_name }}</title>
</head>
<body>
    <h1>{{ tsk_config('app.name', 'Flask-TSK') }}</h1>
    <p>{{ tsk_section('welcome.message', 'Welcome!') }}</p>
    
    {% if tsk_available %}
        <p>TuskLang v{{ tsk_version }} is available</p>
    {% endif %}
</body>
</html>
```

### FastAPI Integration

```python
from fastapi import FastAPI
from tsk_flask import FastAPIRouter

app = FastAPI()

# Add TuskLang routes
tsk_router = FastAPIRouter()
app.include_router(tsk_router, prefix="/tsk")

@app.get("/")
def read_root():
    return {"message": "Flask-TSK with FastAPI"}
```

## 🔥 Advanced Usage

### Performance Optimization

```python
from tsk_flask import optimize_flask_app, render_turbo_template

# Apply all performance optimizations
optimize_flask_app(app)

# High-performance template rendering
result = render_turbo_template(template_content, context)
```

### Async Operations

```python
import asyncio
from tsk_flask import render_turbo_template_async

async def render_templates():
    results = await asyncio.gather(*[
        render_turbo_template_async(template, context)
        for context in contexts
    ])
    return results
```

### Database Integration

```python
# Get database configuration
db_config = tsk.get_database_config()

# Use with SQLAlchemy
from sqlalchemy import create_engine
engine = create_engine(db_config['connection_string'])

# Use with TuskLang adapters
from tusktsk.adapters import get_adapter
adapter = get_adapter(db_config['type'])
connection = adapter.connect()
```

## 🛠️ API Endpoints

Flask-TSK provides REST API endpoints when enabled:

### Configuration Endpoints

```bash
# Get all configuration
GET /tsk/config

# Get specific section
GET /tsk/config/database

# Get specific key
GET /tsk/config/database/host
```

### Function Execution

```bash
# Execute TuskLang function
POST /tsk/function
Content-Type: application/json

{
    "section": "utils",
    "key": "format_date",
    "args": ["2024-01-01"]
}
```

### TuskLang Operations

```bash
# Parse TuskLang content
POST /tsk/parse
Content-Type: application/json

{
    "content": "[database]\ntype = \"postgresql\""
}

# Stringify data to TuskLang
POST /tsk/stringify
Content-Type: application/json

{
    "data": {"database": {"type": "postgresql"}}
}
```

### Utility Endpoints

```bash
# Get capabilities
GET /tsk/capabilities

# Get performance stats
GET /tsk/performance

# Health check
GET /tsk/health
```

## 🧪 Testing

### Run Tests

```bash
# Run all tests
pytest

# Run with coverage
pytest --cov=tsk_flask

# Run specific test file
pytest test_integration.py
```

### Performance Testing

```bash
# Run performance benchmarks
python tsk_flask/performance_benchmark.py

# Run performance demo
python tsk_flask/performance_demo.py
```

### Integration Testing

```bash
# Test Flask integration
python tsk_flask/test_example.py

# Test FastAPI integration
python tsk_flask/test_fastapi.py
```

## 🚀 Production Deployment

### Gunicorn Deployment

```bash
# Install Gunicorn
pip install gunicorn

# Run with Gunicorn
gunicorn -w 4 -b 0.0.0.0:8000 app:app
```

### Docker Deployment

```dockerfile
FROM python:3.11-slim

WORKDIR /app

COPY requirements.txt .
RUN pip install -r requirements.txt

COPY . .

EXPOSE 8000

CMD ["gunicorn", "-w", "4", "-b", "0.0.0.0:8000", "app:app"]
```

### Environment Variables

```bash
export TSK_CONFIG_PATH="/etc/app/config.tsk"
export TSK_ENABLE_BLUEPRINT="true"
export TSK_ENABLE_FULL_SDK="true"
export FLASK_ENV="production"
```

## 📊 Monitoring

### Performance Metrics

```python
# Get performance statistics
stats = tsk.get_performance_stats()

# Monitor cache hit rates
cache_stats = stats.get('cache', {})
hit_rate = cache_stats.get('hit_rate', 0)
```

### Health Checks

```python
@app.route('/health')
def health_check():
    return {
        'status': 'healthy',
        'tusk_available': tsk.is_tusk_available(),
        'tusk_version': tsk.get_tusk_version(),
        'performance_stats': tsk.get_performance_stats()
    }
```

## 🔒 Security

### Configuration Security

```python
# Secure configuration loading
app.config.update({
    'TSK_CONFIG_PATH': '/secure/config.tsk',
    'TSK_ENABLE_SECURITY': True,
    'TSK_ENCRYPTION_KEY': os.environ.get('TSK_ENCRYPTION_KEY'),
})
```

### API Security

```python
# Enable API authentication
app.config.update({
    'TSK_API_AUTH': True,
    'TSK_API_KEY': os.environ.get('TSK_API_KEY'),
})
```

## 🆘 Troubleshooting

### Common Issues

1. **TuskLang not available**
   ```bash
   pip install tusktsk>=2.0.3
   ```

2. **Configuration not loading**
   ```bash
   # Check file permissions
   ls -la peanu.tsk
   
   # Check file syntax
   python -c "from tusktsk import parse; parse(open('peanu.tsk').read())"
   ```

3. **Performance issues**
   ```bash
   # Enable performance optimizations
   pip install flask-tsk[performance]
   
   # Check performance stats
   curl http://localhost:5000/tsk/performance
   ```

### Debug Mode

```python
# Enable debug logging
app.config.update({
    'TSK_DEBUG': True,
    'TSK_LOG_LEVEL': 'DEBUG',
})
```

## 📚 Additional Resources

- [Flask-TSK Documentation](https://flask-tsk.readthedocs.io/)
- [TuskLang Documentation](https://tusklang.org)
- [Performance Guide](PERFORMANCE_REVOLUTION.md)
- [API Reference](API_REFERENCE.md)
- [Examples](examples/)

## 🤝 Support

- **GitHub Issues**: [https://github.com/grim-project/flask-tsk/issues](https://github.com/grim-project/flask-tsk/issues)
- **GitHub Discussions**: [https://github.com/grim-project/flask-tsk/discussions](https://github.com/grim-project/flask-tsk/discussions)
- **Documentation**: [https://flask-tsk.readthedocs.io/](https://flask-tsk.readthedocs.io/)

---

**Flask-TSK** - Making Flask faster than Django, more powerful than ever! 🚀 