#!/usr/bin/env python3
"""
TuskLang Package Management for Python
======================================
Cross-language package management integration

Features:
- crates.io (Rust) integration
- npm (Node.js) integration
- go.mod (Go) integration
- Maven Central (Java) integration
- NuGet (.NET) integration
- RubyGems (Ruby) integration
- Composer (PHP) integration
"""

import json
import subprocess
import os
import sys
import tempfile
import shutil
import requests
from typing import Dict, List, Optional, Any, Union
from pathlib import Path
import xml.etree.ElementTree as ET


class CargoIntegration:
    """Rust crates.io integration"""
    
    def __init__(self):
        self.cargo_path = shutil.which("cargo")
        self.registry_url = "https://crates.io/api/v1"
    
    def search_crates(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search crates on crates.io"""
        try:
            url = f"{self.registry_url}/crates"
            params = {'q': query, 'per_page': limit}
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                return data.get('crates', [])
            return []
        except Exception as e:
            print(f"Crates.io search error: {e}")
            return []
    
    def get_crate_info(self, crate_name: str) -> Optional[Dict[str, Any]]:
        """Get crate information"""
        try:
            url = f"{self.registry_url}/crates/{crate_name}"
            response = requests.get(url)
            
            if response.status_code == 200:
                return response.json()
            return None
        except Exception as e:
            print(f"Crate info error: {e}")
            return None
    
    def create_cargo_toml(self, project_name: str, dependencies: Dict[str, str]) -> str:
        """Create Cargo.toml file"""
        toml_content = f"""
[package]
name = "{project_name}"
version = "0.1.0"
edition = "2021"

[dependencies]
"""
        
        for dep_name, dep_version in dependencies.items():
            toml_content += f'{dep_name} = "{dep_version}"\n'
        
        return toml_content
    
    def install_dependency(self, crate_name: str, version: str = None) -> bool:
        """Install Rust dependency"""
        try:
            if not self.cargo_path:
                return False
            
            cmd = [self.cargo_path, "add", crate_name]
            if version:
                cmd.append(f"--version={version}")
            
            result = subprocess.run(cmd, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Cargo install error: {e}")
            return False
    
    def build_project(self, project_path: str) -> bool:
        """Build Rust project"""
        try:
            if not self.cargo_path:
                return False
            
            cmd = [self.cargo_path, "build"]
            result = subprocess.run(cmd, cwd=project_path, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Cargo build error: {e}")
            return False


class NPMIntegration:
    """Node.js npm integration"""
    
    def __init__(self):
        self.npm_path = shutil.which("npm")
        self.registry_url = "https://registry.npmjs.org"
    
    def search_packages(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search npm packages"""
        try:
            url = f"{self.registry_url}/-/v1/search"
            params = {'text': query, 'size': limit}
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                return data.get('objects', [])
            return []
        except Exception as e:
            print(f"NPM search error: {e}")
            return []
    
    def get_package_info(self, package_name: str) -> Optional[Dict[str, Any]]:
        """Get package information"""
        try:
            url = f"{self.registry_url}/{package_name}"
            response = requests.get(url)
            
            if response.status_code == 200:
                return response.json()
            return None
        except Exception as e:
            print(f"Package info error: {e}")
            return None
    
    def create_package_json(self, project_name: str, dependencies: Dict[str, str]) -> str:
        """Create package.json file"""
        package_json = {
            "name": project_name,
            "version": "1.0.0",
            "description": "TuskLang Node.js project",
            "main": "index.js",
            "scripts": {
                "start": "node index.js",
                "test": "echo \"Error: no test specified\" && exit 1"
            },
            "dependencies": dependencies,
            "keywords": ["tusklang", "configuration"],
            "author": "TuskLang Team",
            "license": "MIT"
        }
        
        return json.dumps(package_json, indent=2)
    
    def install_dependency(self, package_name: str, version: str = None) -> bool:
        """Install npm dependency"""
        try:
            if not self.npm_path:
                return False
            
            cmd = [self.npm_path, "install"]
            if version:
                cmd.append(f"{package_name}@{version}")
            else:
                cmd.append(package_name)
            
            result = subprocess.run(cmd, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"NPM install error: {e}")
            return False
    
    def publish_package(self, package_path: str) -> bool:
        """Publish npm package"""
        try:
            if not self.npm_path:
                return False
            
            cmd = [self.npm_path, "publish"]
            result = subprocess.run(cmd, cwd=package_path, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"NPM publish error: {e}")
            return False


class GoModIntegration:
    """Go module integration"""
    
    def __init__(self):
        self.go_path = shutil.which("go")
        self.proxy_url = "https://proxy.golang.org"
    
    def search_modules(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search Go modules"""
        try:
            url = f"{self.proxy_url}/search"
            params = {'q': query, 'limit': limit}
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                return data.get('results', [])
            return []
        except Exception as e:
            print(f"Go module search error: {e}")
            return []
    
    def get_module_info(self, module_name: str) -> Optional[Dict[str, Any]]:
        """Get module information"""
        try:
            url = f"{self.proxy_url}/{module_name}"
            response = requests.get(url)
            
            if response.status_code == 200:
                return response.json()
            return None
        except Exception as e:
            print(f"Module info error: {e}")
            return None
    
    def create_go_mod(self, module_name: str, dependencies: Dict[str, str]) -> str:
        """Create go.mod file"""
        go_mod_content = f"""
module {module_name}

go 1.21

require (
"""
        
        for dep_name, dep_version in dependencies.items():
            go_mod_content += f"\t{dep_name} {dep_version}\n"
        
        go_mod_content += ")\n"
        return go_mod_content
    
    def install_dependency(self, module_name: str, version: str = None) -> bool:
        """Install Go dependency"""
        try:
            if not self.go_path:
                return False
            
            cmd = [self.go_path, "get"]
            if version:
                cmd.append(f"{module_name}@{version}")
            else:
                cmd.append(module_name)
            
            result = subprocess.run(cmd, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Go get error: {e}")
            return False
    
    def build_project(self, project_path: str) -> bool:
        """Build Go project"""
        try:
            if not self.go_path:
                return False
            
            cmd = [self.go_path, "build"]
            result = subprocess.run(cmd, cwd=project_path, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Go build error: {e}")
            return False


class MavenIntegration:
    """Java Maven Central integration"""
    
    def __init__(self):
        self.mvn_path = shutil.which("mvn")
        self.central_url = "https://search.maven.org/solrsearch/select"
    
    def search_artifacts(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search Maven artifacts"""
        try:
            url = self.central_url
            params = {
                'q': query,
                'rows': limit,
                'wt': 'json'
            }
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                return data.get('response', {}).get('docs', [])
            return []
        except Exception as e:
            print(f"Maven search error: {e}")
            return []
    
    def get_artifact_info(self, group_id: str, artifact_id: str) -> Optional[Dict[str, Any]]:
        """Get artifact information"""
        try:
            query = f"g:{group_id} AND a:{artifact_id}"
            url = self.central_url
            params = {
                'q': query,
                'wt': 'json'
            }
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                docs = data.get('response', {}).get('docs', [])
                return docs[0] if docs else None
            return None
        except Exception as e:
            print(f"Artifact info error: {e}")
            return None
    
    def create_pom_xml(self, group_id: str, artifact_id: str, version: str, dependencies: Dict[str, Dict[str, str]]) -> str:
        """Create pom.xml file"""
        pom_xml = f"""<?xml version="1.0" encoding="UTF-8"?>
<project xmlns="http://maven.apache.org/POM/4.0.0"
         xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
         xsi:schemaLocation="http://maven.apache.org/POM/4.0.0 
         http://maven.apache.org/xsd/maven-4.0.0.xsd">
    <modelVersion>4.0.0</modelVersion>
    
    <groupId>{group_id}</groupId>
    <artifactId>{artifact_id}</artifactId>
    <version>{version}</version>
    <packaging>jar</packaging>
    
    <name>TuskLang Java Project</name>
    <description>TuskLang configuration for Java</description>
    
    <properties>
        <maven.compiler.source>11</maven.compiler.source>
        <maven.compiler.target>11</maven.compiler.target>
        <project.build.sourceEncoding>UTF-8</project.build.sourceEncoding>
    </properties>
    
    <dependencies>
"""
        
        for dep_group, dep_artifact, dep_version in dependencies.values():
            pom_xml += f"""
        <dependency>
            <groupId>{dep_group}</groupId>
            <artifactId>{dep_artifact}</artifactId>
            <version>{dep_version}</version>
        </dependency>
"""
        
        pom_xml += """
    </dependencies>
    
    <build>
        <plugins>
            <plugin>
                <groupId>org.apache.maven.plugins</groupId>
                <artifactId>maven-compiler-plugin</artifactId>
                <version>3.8.1</version>
                <configuration>
                    <source>11</source>
                    <target>11</target>
                </configuration>
            </plugin>
        </plugins>
    </build>
</project>
"""
        return pom_xml
    
    def install_dependency(self, group_id: str, artifact_id: str, version: str) -> bool:
        """Install Maven dependency"""
        try:
            if not self.mvn_path:
                return False
            
            cmd = [self.mvn_path, "dependency:get", 
                   f"-Dartifact={group_id}:{artifact_id}:{version}"]
            result = subprocess.run(cmd, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Maven dependency error: {e}")
            return False
    
    def build_project(self, project_path: str) -> bool:
        """Build Maven project"""
        try:
            if not self.mvn_path:
                return False
            
            cmd = [self.mvn_path, "clean", "compile"]
            result = subprocess.run(cmd, cwd=project_path, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Maven build error: {e}")
            return False


class NuGetIntegration:
    """NuGet package manager integration"""
    
    def __init__(self):
        self.dotnet_path = shutil.which("dotnet")
        self.nuget_url = "https://api.nuget.org/v3"
    
    def search_packages(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search NuGet packages"""
        try:
            url = f"{self.nuget_url}/search"
            params = {'q': query, 'take': limit}
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                return data.get('data', [])
            return []
        except Exception as e:
            print(f"NuGet search error: {e}")
            return []
    
    def get_package_info(self, package_id: str) -> Optional[Dict[str, Any]]:
        """Get package information"""
        try:
            url = f"{self.nuget_url}/registration5-semver1/{package_id}/index.json"
            response = requests.get(url)
            
            if response.status_code == 200:
                return response.json()
            return None
        except Exception as e:
            print(f"Package info error: {e}")
            return None
    
    def create_csproj(self, project_name: str, dependencies: Dict[str, str]) -> str:
        """Create .csproj file"""
        csproj_content = f"""<Project Sdk="Microsoft.NET.Sdk">

  <PropertyGroup>
    <OutputType>Exe</OutputType>
    <TargetFramework>net6.0</TargetFramework>
    <ImplicitUsings>enable</ImplicitUsings>
    <Nullable>enable</Nullable>
  </PropertyGroup>

  <ItemGroup>
"""
        
        for package_id, version in dependencies.items():
            csproj_content += f'    <PackageReference Include="{package_id}" Version="{version}" />\n'
        
        csproj_content += """  </ItemGroup>

</Project>
"""
        return csproj_content
    
    def install_dependency(self, package_id: str, version: str = None) -> bool:
        """Install NuGet package"""
        try:
            if not self.dotnet_path:
                return False
            
            cmd = [self.dotnet_path, "add", "package", package_id]
            if version:
                cmd.append(f"--version={version}")
            
            result = subprocess.run(cmd, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"NuGet install error: {e}")
            return False
    
    def build_project(self, project_path: str) -> bool:
        """Build .NET project"""
        try:
            if not self.dotnet_path:
                return False
            
            cmd = [self.dotnet_path, "build"]
            result = subprocess.run(cmd, cwd=project_path, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Dotnet build error: {e}")
            return False


class RubyGemsIntegration:
    """RubyGems integration"""
    
    def __init__(self):
        self.gem_path = shutil.which("gem")
        self.ruby_path = shutil.which("ruby")
        self.gems_url = "https://rubygems.org/api/v1"
    
    def search_gems(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search RubyGems"""
        try:
            url = f"{self.gems_url}/search.json"
            params = {'query': query}
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                return data[:limit]
            return []
        except Exception as e:
            print(f"RubyGems search error: {e}")
            return []
    
    def get_gem_info(self, gem_name: str) -> Optional[Dict[str, Any]]:
        """Get gem information"""
        try:
            url = f"{self.gems_url}/gems/{gem_name}.json"
            response = requests.get(url)
            
            if response.status_code == 200:
                return response.json()
            return None
        except Exception as e:
            print(f"Gem info error: {e}")
            return None
    
    def create_gemfile(self, dependencies: Dict[str, str]) -> str:
        """Create Gemfile"""
        gemfile_content = """source 'https://rubygems.org'

# TuskLang Ruby project
"""
        
        for gem_name, version in dependencies.items():
            gemfile_content += f"gem '{gem_name}', '~> {version}'\n"
        
        return gemfile_content
    
    def install_dependency(self, gem_name: str, version: str = None) -> bool:
        """Install Ruby gem"""
        try:
            if not self.gem_path:
                return False
            
            cmd = [self.gem_path, "install", gem_name]
            if version:
                cmd.append(f"-v={version}")
            
            result = subprocess.run(cmd, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Gem install error: {e}")
            return False
    
    def create_gemspec(self, gem_name: str, version: str, dependencies: Dict[str, str]) -> str:
        """Create .gemspec file"""
        gemspec_content = f"""# -*- encoding: utf-8 -*-
# frozen_string_literal: true

Gem::Specification.new do |spec|
  spec.name          = "{gem_name}"
  spec.version       = "{version}"
  spec.authors       = ["TuskLang Team"]
  spec.email         = ["team@tusklang.org"]

  spec.summary       = "TuskLang configuration for Ruby"
  spec.description   = "Ruby integration for TuskLang configuration language"
  spec.homepage      = "https://tusklang.org"
  spec.license       = "MIT"
  spec.required_ruby_version = Gem::Requirement.new(">= 2.6.0")

  spec.metadata["homepage_uri"] = spec.homepage
  spec.metadata["source_code_uri"] = "https://github.com/tusklang/ruby"

  spec.files = Dir.glob("lib/**/*") + %w[README.md LICENSE.txt]
  spec.bindir        = "exe"
  spec.executables   = spec.files.grep(%r{\\Aexe/}) {{ |f| File.basename(f) }}
  spec.require_paths = ["lib"]

  spec.add_development_dependency "bundler", "~> 2.0"
  spec.add_development_dependency "rake", "~> 13.0"
  spec.add_development_dependency "rspec", "~> 3.0"
"""
        
        for dep_name, dep_version in dependencies.items():
            gemspec_content += f'  spec.add_dependency "{dep_name}", "~> {dep_version}"\n'
        
        return gemspec_content


class ComposerIntegration:
    """PHP Composer integration"""
    
    def __init__(self):
        self.composer_path = shutil.which("composer")
        self.packagist_url = "https://packagist.org"
    
    def search_packages(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search Packagist packages"""
        try:
            url = f"{self.packagist_url}/search.json"
            params = {'q': query}
            response = requests.get(url, params=params)
            
            if response.status_code == 200:
                data = response.json()
                return data.get('results', [])[:limit]
            return []
        except Exception as e:
            print(f"Packagist search error: {e}")
            return []
    
    def get_package_info(self, package_name: str) -> Optional[Dict[str, Any]]:
        """Get package information"""
        try:
            url = f"{self.packagist_url}/packages/{package_name}.json"
            response = requests.get(url)
            
            if response.status_code == 200:
                return response.json()
            return None
        except Exception as e:
            print(f"Package info error: {e}")
            return None
    
    def create_composer_json(self, project_name: str, dependencies: Dict[str, str]) -> str:
        """Create composer.json file"""
        composer_json = {
            "name": f"tusklang/{project_name}",
            "description": "TuskLang PHP project",
            "type": "library",
            "license": "MIT",
            "authors": [
                {
                    "name": "TuskLang Team",
                    "email": "team@tusklang.org"
                }
            ],
            "require": dependencies,
            "require-dev": {
                "phpunit/phpunit": "^9.0"
            },
            "autoload": {
                "psr-4": {
                    f"TuskLang\\{project_name}\\": "src/"
                }
            },
            "autoload-dev": {
                "psr-4": {
                    f"TuskLang\\{project_name}\\Tests\\": "tests/"
                }
            },
            "minimum-stability": "stable",
            "prefer-stable": True
        }
        
        return json.dumps(composer_json, indent=2)
    
    def install_dependency(self, package_name: str, version: str = None) -> bool:
        """Install Composer dependency"""
        try:
            if not self.composer_path:
                return False
            
            cmd = [self.composer_path, "require", package_name]
            if version:
                cmd.append(f":{version}")
            
            result = subprocess.run(cmd, capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Composer install error: {e}")
            return False
    
    def install_dependencies(self, composer_json_path: str) -> bool:
        """Install all dependencies from composer.json"""
        try:
            if not self.composer_path:
                return False
            
            cmd = [self.composer_path, "install"]
            result = subprocess.run(cmd, cwd=os.path.dirname(composer_json_path), 
                                  capture_output=True, text=True)
            return result.returncode == 0
        except Exception as e:
            print(f"Composer install error: {e}")
            return False


class PackageManagement:
    """Main package management class"""
    
    def __init__(self):
        self.cargo = CargoIntegration()
        self.npm = NPMIntegration()
        self.go_mod = GoModIntegration()
        self.maven = MavenIntegration()
        self.nuget = NuGetIntegration()
        self.rubygems = RubyGemsIntegration()
        self.composer = ComposerIntegration()
    
    def get_package_manager_status(self) -> Dict[str, bool]:
        """Get status of all package managers"""
        return {
            'cargo': self.cargo.cargo_path is not None,
            'npm': self.npm.npm_path is not None,
            'go_mod': self.go_mod.go_path is not None,
            'maven': self.maven.mvn_path is not None,
            'nuget': self.nuget.dotnet_path is not None,
            'rubygems': self.rubygems.gem_path is not None,
            'composer': self.composer.composer_path is not None
        }
    
    def search_packages(self, manager: str, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Search packages across different package managers"""
        if manager == 'cargo':
            return self.cargo.search_crates(query, limit)
        elif manager == 'npm':
            return self.npm.search_packages(query, limit)
        elif manager == 'go_mod':
            return self.go_mod.search_modules(query, limit)
        elif manager == 'maven':
            return self.maven.search_artifacts(query, limit)
        elif manager == 'nuget':
            return self.nuget.search_packages(query, limit)
        elif manager == 'rubygems':
            return self.rubygems.search_gems(query, limit)
        elif manager == 'composer':
            return self.composer.search_packages(query, limit)
        else:
            return []
    
    def create_project_files(self, manager: str, project_name: str, dependencies: Dict[str, str]) -> Dict[str, str]:
        """Create project files for different package managers"""
        if manager == 'cargo':
            return {
                'Cargo.toml': self.cargo.create_cargo_toml(project_name, dependencies)
            }
        elif manager == 'npm':
            return {
                'package.json': self.npm.create_package_json(project_name, dependencies)
            }
        elif manager == 'go_mod':
            return {
                'go.mod': self.go_mod.create_go_mod(project_name, dependencies)
            }
        elif manager == 'maven':
            return {
                'pom.xml': self.maven.create_pom_xml('org.tusklang', project_name, '1.0.0', dependencies)
            }
        elif manager == 'nuget':
            return {
                f'{project_name}.csproj': self.nuget.create_csproj(project_name, dependencies)
            }
        elif manager == 'rubygems':
            return {
                'Gemfile': self.rubygems.create_gemfile(dependencies),
                f'{project_name}.gemspec': self.rubygems.create_gemspec(project_name, '1.0.0', dependencies)
            }
        elif manager == 'composer':
            return {
                'composer.json': self.composer.create_composer_json(project_name, dependencies)
            }
        else:
            return {'error': f'Unknown package manager: {manager}'}
    
    def install_dependency(self, manager: str, package_name: str, version: str = None) -> bool:
        """Install dependency using specified package manager"""
        if manager == 'cargo':
            return self.cargo.install_dependency(package_name, version)
        elif manager == 'npm':
            return self.npm.install_dependency(package_name, version)
        elif manager == 'go_mod':
            return self.go_mod.install_dependency(package_name, version)
        elif manager == 'maven':
            # Maven requires groupId:artifactId format
            return self.maven.install_dependency(package_name, 'artifact', version)
        elif manager == 'nuget':
            return self.nuget.install_dependency(package_name, version)
        elif manager == 'rubygems':
            return self.rubygems.install_dependency(package_name, version)
        elif manager == 'composer':
            return self.composer.install_dependency(package_name, version)
        else:
            return False


# Global package management instance
package_management = PackageManagement()


def get_package_management() -> PackageManagement:
    """Get global package management instance"""
    return package_management 