#!/usr/bin/env python3
"""
Real-Time Streaming for TuskLang Python SDK
===========================================
High-performance streaming data processing and real-time analytics

This module provides real-time streaming capabilities for the TuskLang Python SDK,
including streaming data processing, real-time analytics, event streaming, and
high-performance data pipelines.
"""

import asyncio
import json
import time
import threading
from typing import Any, Dict, List, Optional, Callable, Union, Tuple, AsyncIterator
from dataclasses import dataclass, asdict
from datetime import datetime, timedelta
from enum import Enum
import logging
import queue
import heapq
from collections import deque, defaultdict
import aiohttp
import websockets
import aioredis
import kafka
from kafka import KafkaProducer, KafkaConsumer
import avro.schema
from avro.io import DatumWriter, DatumReader
import struct


class StreamType(Enum):
    """Stream type enumeration"""
    KAFKA = "kafka"
    REDIS = "redis"
    WEBSOCKET = "websocket"
    HTTP = "http"
    CUSTOM = "custom"


class ProcessingWindow(Enum):
    """Processing window enumeration"""
    TUMBLING = "tumbling"
    SLIDING = "sliding"
    SESSION = "session"
    GLOBAL = "global"


@dataclass
class StreamEvent:
    """Stream event structure"""
    event_id: str
    timestamp: datetime
    source: str
    data: Any
    metadata: Dict[str, Any]
    partition: Optional[int] = None


@dataclass
class StreamConfig:
    """Stream configuration structure"""
    stream_id: str
    name: str
    stream_type: StreamType
    source_url: str
    batch_size: int
    window_size: int
    processing_window: ProcessingWindow
    schema: Optional[Dict[str, Any]] = None


class RealTimeStreaming:
    """Real-time streaming system for TuskLang"""
    
    def __init__(self, config: Dict[str, Any] = None):
        self.config = config or {}
        self.logger = logging.getLogger('tusklang.streaming')
        
        # Initialize components
        self.streams = {}
        self.stream_processors = {}
        self.event_handlers = {}
        self.stream_analytics = {}
        
        # Initialize streaming components
        self.kafka_manager = KafkaManager()
        self.redis_manager = RedisManager()
        self.websocket_manager = WebSocketManager()
        self.stream_processor = StreamProcessor()
        self.analytics_engine = AnalyticsEngine()
        
        # Initialize streaming
        self.streaming_active = True
        self.event_queue = asyncio.Queue()
        self.analytics_queue = asyncio.Queue()
        
        # Start background processes
        self._start_background_processes()
    
    def _start_background_processes(self):
        """Start background streaming processes"""
        # Event processor
        self.event_processor_thread = threading.Thread(target=self._event_processor_loop, daemon=True)
        self.event_processor_thread.start()
        
        # Analytics processor
        self.analytics_processor_thread = threading.Thread(target=self._analytics_processor_loop, daemon=True)
        self.analytics_processor_thread.start()
    
    def create_stream(self, stream_config: StreamConfig) -> bool:
        """Create a new stream"""
        try:
            stream_id = stream_config.stream_id
            
            if stream_config.stream_type == StreamType.KAFKA:
                self.streams[stream_id] = self.kafka_manager.create_stream(stream_config)
            elif stream_config.stream_type == StreamType.REDIS:
                self.streams[stream_id] = self.redis_manager.create_stream(stream_config)
            elif stream_config.stream_type == StreamType.WEBSOCKET:
                self.streams[stream_id] = self.websocket_manager.create_stream(stream_config)
            elif stream_config.stream_type == StreamType.HTTP:
                self.streams[stream_id] = self._create_http_stream(stream_config)
            else:
                raise ValueError(f"Unsupported stream type: {stream_config.stream_type}")
            
            self.logger.info(f"Created stream: {stream_id}")
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to create stream {stream_config.stream_id}: {e}")
            return False
    
    def _create_http_stream(self, stream_config: StreamConfig):
        """Create HTTP stream"""
        return {
            "config": stream_config,
            "type": "http",
            "active": True,
            "created_at": datetime.now()
        }
    
    async def start_stream(self, stream_id: str) -> bool:
        """Start a stream"""
        if stream_id not in self.streams:
            return False
        
        try:
            stream = self.streams[stream_id]
            stream_type = stream["config"].stream_type
            
            if stream_type == StreamType.KAFKA:
                await self.kafka_manager.start_stream(stream_id)
            elif stream_type == StreamType.REDIS:
                await self.redis_manager.start_stream(stream_id)
            elif stream_type == StreamType.WEBSOCKET:
                await self.websocket_manager.start_stream(stream_id)
            elif stream_type == StreamType.HTTP:
                await self._start_http_stream(stream_id)
            
            stream["active"] = True
            self.logger.info(f"Started stream: {stream_id}")
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to start stream {stream_id}: {e}")
            return False
    
    async def _start_http_stream(self, stream_id: str):
        """Start HTTP stream"""
        stream = self.streams[stream_id]
        config = stream["config"]
        
        # Start HTTP stream monitoring
        asyncio.create_task(self._monitor_http_stream(stream_id, config.source_url))
    
    async def stop_stream(self, stream_id: str) -> bool:
        """Stop a stream"""
        if stream_id not in self.streams:
            return False
        
        try:
            stream = self.streams[stream_id]
            stream_type = stream["config"].stream_type
            
            if stream_type == StreamType.KAFKA:
                await self.kafka_manager.stop_stream(stream_id)
            elif stream_type == StreamType.REDIS:
                await self.redis_manager.stop_stream(stream_id)
            elif stream_type == StreamType.WEBSOCKET:
                await self.websocket_manager.stop_stream(stream_id)
            
            stream["active"] = False
            self.logger.info(f"Stopped stream: {stream_id}")
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to stop stream {stream_id}: {e}")
            return False
    
    async def publish_event(self, stream_id: str, event_data: Any, 
                           metadata: Dict[str, Any] = None) -> bool:
        """Publish event to stream"""
        if stream_id not in self.streams:
            return False
        
        try:
            stream = self.streams[stream_id]
            stream_type = stream["config"].stream_type
            
            event = StreamEvent(
                event_id=str(int(time.time() * 1000)),
                timestamp=datetime.now(),
                source=stream_id,
                data=event_data,
                metadata=metadata or {}
            )
            
            if stream_type == StreamType.KAFKA:
                return await self.kafka_manager.publish_event(stream_id, event)
            elif stream_type == StreamType.REDIS:
                return await self.redis_manager.publish_event(stream_id, event)
            elif stream_type == StreamType.WEBSOCKET:
                return await self.websocket_manager.publish_event(stream_id, event)
            else:
                # Add to local event queue
                await self.event_queue.put(event)
                return True
                
        except Exception as e:
            self.logger.error(f"Failed to publish event to stream {stream_id}: {e}")
            return False
    
    async def subscribe_to_stream(self, stream_id: str, 
                                 event_handler: Callable[[StreamEvent], None]) -> bool:
        """Subscribe to stream events"""
        if stream_id not in self.streams:
            return False
        
        try:
            self.event_handlers[stream_id] = event_handler
            
            # Start consuming events
            asyncio.create_task(self._consume_stream_events(stream_id))
            
            self.logger.info(f"Subscribed to stream: {stream_id}")
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to subscribe to stream {stream_id}: {e}")
            return False
    
    async def _consume_stream_events(self, stream_id: str):
        """Consume events from stream"""
        stream = self.streams[stream_id]
        stream_type = stream["config"].stream_type
        
        try:
            if stream_type == StreamType.KAFKA:
                await self.kafka_manager.consume_events(stream_id, self.event_handlers[stream_id])
            elif stream_type == StreamType.REDIS:
                await self.redis_manager.consume_events(stream_id, self.event_handlers[stream_id])
            elif stream_type == StreamType.WEBSOCKET:
                await self.websocket_manager.consume_events(stream_id, self.event_handlers[stream_id])
            else:
                # Consume from local queue
                while stream["active"]:
                    try:
                        event = await asyncio.wait_for(self.event_queue.get(), timeout=1.0)
                        if event.source == stream_id:
                            await self.event_handlers[stream_id](event)
                    except asyncio.TimeoutError:
                        continue
                        
        except Exception as e:
            self.logger.error(f"Event consumption error for stream {stream_id}: {e}")
    
    async def _monitor_http_stream(self, stream_id: str, url: str):
        """Monitor HTTP stream for updates"""
        stream = self.streams[stream_id]
        
        while stream["active"]:
            try:
                async with aiohttp.ClientSession() as session:
                    async with session.get(url) as response:
                        if response.status == 200:
                            data = await response.json()
                            
                            event = StreamEvent(
                                event_id=str(int(time.time() * 1000)),
                                timestamp=datetime.now(),
                                source=stream_id,
                                data=data,
                                metadata={"source": "http_stream"}
                            )
                            
                            await self.event_queue.put(event)
                
                await asyncio.sleep(5)  # Poll every 5 seconds
                
            except Exception as e:
                self.logger.error(f"HTTP stream monitoring error: {e}")
                await asyncio.sleep(10)
    
    def add_stream_processor(self, stream_id: str, processor: Callable[[StreamEvent], StreamEvent]):
        """Add processor to stream"""
        if stream_id not in self.stream_processors:
            self.stream_processors[stream_id] = []
        
        self.stream_processors[stream_id].append(processor)
        self.logger.info(f"Added processor to stream: {stream_id}")
    
    def add_analytics(self, stream_id: str, analytics_config: Dict[str, Any]):
        """Add analytics to stream"""
        self.analytics_engine.add_analytics(stream_id, analytics_config)
        self.logger.info(f"Added analytics to stream: {stream_id}")
    
    async def get_stream_analytics(self, stream_id: str) -> Dict[str, Any]:
        """Get stream analytics"""
        return await self.analytics_engine.get_analytics(stream_id)
    
    def get_stream_info(self, stream_id: str) -> Optional[Dict[str, Any]]:
        """Get stream information"""
        if stream_id not in self.streams:
            return None
        
        stream = self.streams[stream_id]
        config = stream["config"]
        
        return {
            "stream_id": stream_id,
            "name": config.name,
            "type": config.stream_type.value,
            "source_url": config.source_url,
            "active": stream["active"],
            "created_at": stream["created_at"].isoformat(),
            "processors": len(self.stream_processors.get(stream_id, [])),
            "analytics": stream_id in self.analytics_engine.analytics
        }
    
    def list_streams(self) -> List[Dict[str, Any]]:
        """List all streams"""
        return [
            self.get_stream_info(stream_id)
            for stream_id in self.streams.keys()
        ]
    
    def _event_processor_loop(self):
        """Event processing background loop"""
        while self.streaming_active:
            try:
                # Process events from queue
                loop = asyncio.new_event_loop()
                asyncio.set_event_loop(loop)
                
                async def process_events():
                    while not self.event_queue.empty():
                        event = await self.event_queue.get()
                        await self._process_event(event)
                
                loop.run_until_complete(process_events())
                loop.close()
                
                time.sleep(0.1)  # Process every 100ms
                
            except Exception as e:
                self.logger.error(f"Event processor error: {e}")
                time.sleep(1)
    
    async def _process_event(self, event: StreamEvent):
        """Process a single event"""
        try:
            # Apply stream processors
            if event.source in self.stream_processors:
                for processor in self.stream_processors[event.source]:
                    event = processor(event)
            
            # Send to analytics
            await self.analytics_queue.put(event)
            
            # Call event handlers
            if event.source in self.event_handlers:
                await self.event_handlers[event.source](event)
                
        except Exception as e:
            self.logger.error(f"Event processing error: {e}")
    
    def _analytics_processor_loop(self):
        """Analytics processing background loop"""
        while self.streaming_active:
            try:
                # Process analytics from queue
                loop = asyncio.new_event_loop()
                asyncio.set_event_loop(loop)
                
                async def process_analytics():
                    while not self.analytics_queue.empty():
                        event = await self.analytics_queue.get()
                        await self.analytics_engine.process_event(event)
                
                loop.run_until_complete(process_analytics())
                loop.close()
                
                time.sleep(0.1)  # Process every 100ms
                
            except Exception as e:
                self.logger.error(f"Analytics processor error: {e}")
                time.sleep(1)


class KafkaManager:
    """Kafka stream manager"""
    
    def __init__(self):
        self.logger = logging.getLogger('tusklang.streaming.kafka')
        self.producers = {}
        self.consumers = {}
        self.streams = {}
    
    def create_stream(self, stream_config: StreamConfig):
        """Create Kafka stream"""
        try:
            producer = KafkaProducer(
                bootstrap_servers=['localhost:9092'],
                value_serializer=lambda v: json.dumps(v).encode('utf-8')
            )
            
            self.producers[stream_config.stream_id] = producer
            self.streams[stream_config.stream_id] = stream_config
            
            return {
                "config": stream_config,
                "type": "kafka",
                "active": False,
                "created_at": datetime.now()
            }
            
        except Exception as e:
            self.logger.error(f"Failed to create Kafka stream: {e}")
            raise
    
    async def start_stream(self, stream_id: str):
        """Start Kafka stream"""
        if stream_id not in self.streams:
            return
        
        config = self.streams[stream_id]
        
        try:
            consumer = KafkaConsumer(
                config.name,
                bootstrap_servers=['localhost:9092'],
                value_deserializer=lambda v: json.loads(v.decode('utf-8')),
                auto_offset_reset='latest',
                enable_auto_commit=True
            )
            
            self.consumers[stream_id] = consumer
            
        except Exception as e:
            self.logger.error(f"Failed to start Kafka stream {stream_id}: {e}")
    
    async def stop_stream(self, stream_id: str):
        """Stop Kafka stream"""
        if stream_id in self.producers:
            self.producers[stream_id].close()
            del self.producers[stream_id]
        
        if stream_id in self.consumers:
            self.consumers[stream_id].close()
            del self.consumers[stream_id]
    
    async def publish_event(self, stream_id: str, event: StreamEvent) -> bool:
        """Publish event to Kafka"""
        if stream_id not in self.producers:
            return False
        
        try:
            producer = self.producers[stream_id]
            config = self.streams[stream_id]
            
            future = producer.send(config.name, event.data)
            producer.flush()
            
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to publish to Kafka: {e}")
            return False
    
    async def consume_events(self, stream_id: str, event_handler: Callable[[StreamEvent], None]):
        """Consume events from Kafka"""
        if stream_id not in self.consumers:
            return
        
        consumer = self.consumers[stream_id]
        
        try:
            for message in consumer:
                event = StreamEvent(
                    event_id=str(int(time.time() * 1000)),
                    timestamp=datetime.now(),
                    source=stream_id,
                    data=message.value,
                    metadata={"partition": message.partition, "offset": message.offset}
                )
                
                await event_handler(event)
                
        except Exception as e:
            self.logger.error(f"Kafka consumption error: {e}")


class RedisManager:
    """Redis stream manager"""
    
    def __init__(self):
        self.logger = logging.getLogger('tusklang.streaming.redis')
        self.redis_client = None
        self.streams = {}
    
    async def create_stream(self, stream_config: StreamConfig):
        """Create Redis stream"""
        try:
            if not self.redis_client:
                self.redis_client = await aioredis.create_redis_pool('redis://localhost')
            
            self.streams[stream_config.stream_id] = stream_config
            
            return {
                "config": stream_config,
                "type": "redis",
                "active": False,
                "created_at": datetime.now()
            }
            
        except Exception as e:
            self.logger.error(f"Failed to create Redis stream: {e}")
            raise
    
    async def start_stream(self, stream_id: str):
        """Start Redis stream"""
        # Redis streams are automatically active
        pass
    
    async def stop_stream(self, stream_id: str):
        """Stop Redis stream"""
        # Redis streams don't need explicit stopping
        pass
    
    async def publish_event(self, stream_id: str, event: StreamEvent) -> bool:
        """Publish event to Redis stream"""
        if not self.redis_client or stream_id not in self.streams:
            return False
        
        try:
            config = self.streams[stream_id]
            await self.redis_client.xadd(
                config.name,
                {'data': json.dumps(event.data), 'timestamp': str(event.timestamp)}
            )
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to publish to Redis: {e}")
            return False
    
    async def consume_events(self, stream_id: str, event_handler: Callable[[StreamEvent], None]):
        """Consume events from Redis stream"""
        if not self.redis_client or stream_id not in self.streams:
            return
        
        config = self.streams[stream_id]
        
        try:
            while True:
                messages = await self.redis_client.xread([config.name], timeout=1000)
                
                for stream, message_list in messages:
                    for message_id, fields in message_list:
                        event = StreamEvent(
                            event_id=message_id.decode(),
                            timestamp=datetime.now(),
                            source=stream_id,
                            data=json.loads(fields[b'data'].decode()),
                            metadata={"redis_stream": stream.decode()}
                        )
                        
                        await event_handler(event)
                        
        except Exception as e:
            self.logger.error(f"Redis consumption error: {e}")


class WebSocketManager:
    """WebSocket stream manager"""
    
    def __init__(self):
        self.logger = logging.getLogger('tusklang.streaming.websocket')
        self.connections = {}
        self.streams = {}
    
    def create_stream(self, stream_config: StreamConfig):
        """Create WebSocket stream"""
        try:
            self.streams[stream_config.stream_id] = stream_config
            
            return {
                "config": stream_config,
                "type": "websocket",
                "active": False,
                "created_at": datetime.now()
            }
            
        except Exception as e:
            self.logger.error(f"Failed to create WebSocket stream: {e}")
            raise
    
    async def start_stream(self, stream_id: str):
        """Start WebSocket stream"""
        config = self.streams[stream_id]
        
        try:
            websocket = await websockets.connect(config.source_url)
            self.connections[stream_id] = websocket
            
        except Exception as e:
            self.logger.error(f"Failed to start WebSocket stream {stream_id}: {e}")
    
    async def stop_stream(self, stream_id: str):
        """Stop WebSocket stream"""
        if stream_id in self.connections:
            await self.connections[stream_id].close()
            del self.connections[stream_id]
    
    async def publish_event(self, stream_id: str, event: StreamEvent) -> bool:
        """Publish event to WebSocket"""
        if stream_id not in self.connections:
            return False
        
        try:
            websocket = self.connections[stream_id]
            await websocket.send(json.dumps(event.data))
            return True
            
        except Exception as e:
            self.logger.error(f"Failed to publish to WebSocket: {e}")
            return False
    
    async def consume_events(self, stream_id: str, event_handler: Callable[[StreamEvent], None]):
        """Consume events from WebSocket"""
        if stream_id not in self.connections:
            return
        
        websocket = self.connections[stream_id]
        
        try:
            async for message in websocket:
                event = StreamEvent(
                    event_id=str(int(time.time() * 1000)),
                    timestamp=datetime.now(),
                    source=stream_id,
                    data=json.loads(message),
                    metadata={"websocket": True}
                )
                
                await event_handler(event)
                
        except Exception as e:
            self.logger.error(f"WebSocket consumption error: {e}")


class StreamProcessor:
    """Stream data processor"""
    
    def __init__(self):
        self.logger = logging.getLogger('tusklang.streaming.processor')
    
    def filter_events(self, events: List[StreamEvent], condition: Callable[[StreamEvent], bool]) -> List[StreamEvent]:
        """Filter events based on condition"""
        return [event for event in events if condition(event)]
    
    def transform_events(self, events: List[StreamEvent], transformer: Callable[[StreamEvent], StreamEvent]) -> List[StreamEvent]:
        """Transform events"""
        return [transformer(event) for event in events]
    
    def aggregate_events(self, events: List[StreamEvent], aggregator: Callable[[List[StreamEvent]], Any]) -> Any:
        """Aggregate events"""
        return aggregator(events)


class AnalyticsEngine:
    """Real-time analytics engine"""
    
    def __init__(self):
        self.logger = logging.getLogger('tusklang.streaming.analytics')
        self.analytics = {}
        self.event_counts = defaultdict(int)
        self.event_timestamps = defaultdict(list)
    
    def add_analytics(self, stream_id: str, analytics_config: Dict[str, Any]):
        """Add analytics configuration"""
        self.analytics[stream_id] = analytics_config
    
    async def process_event(self, event: StreamEvent):
        """Process event for analytics"""
        stream_id = event.source
        
        # Update event counts
        self.event_counts[stream_id] += 1
        
        # Update timestamps
        self.event_timestamps[stream_id].append(event.timestamp)
        
        # Keep only recent timestamps (last hour)
        cutoff_time = datetime.now() - timedelta(hours=1)
        self.event_timestamps[stream_id] = [
            ts for ts in self.event_timestamps[stream_id]
            if ts > cutoff_time
        ]
    
    async def get_analytics(self, stream_id: str) -> Dict[str, Any]:
        """Get analytics for stream"""
        if stream_id not in self.analytics:
            return {}
        
        total_events = self.event_counts[stream_id]
        recent_events = len(self.event_timestamps[stream_id])
        
        # Calculate events per minute
        if self.event_timestamps[stream_id]:
            time_span = (datetime.now() - min(self.event_timestamps[stream_id])).total_seconds() / 60
            events_per_minute = recent_events / max(time_span, 1)
        else:
            events_per_minute = 0
        
        return {
            "stream_id": stream_id,
            "total_events": total_events,
            "recent_events": recent_events,
            "events_per_minute": events_per_minute,
            "last_event": max(self.event_timestamps[stream_id]).isoformat() if self.event_timestamps[stream_id] else None
        }


# Global streaming instance
real_time_streaming = RealTimeStreaming()


def create_stream(stream_id: str, name: str, stream_type: str, source_url: str,
                 batch_size: int = 100, window_size: int = 60,
                 processing_window: str = "tumbling") -> bool:
    """Create a new stream"""
    stream_type_enum = StreamType(stream_type.lower())
    processing_window_enum = ProcessingWindow(processing_window.lower())
    
    config = StreamConfig(
        stream_id=stream_id,
        name=name,
        stream_type=stream_type_enum,
        source_url=source_url,
        batch_size=batch_size,
        window_size=window_size,
        processing_window=processing_window_enum
    )
    
    return real_time_streaming.create_stream(config)


async def start_stream(stream_id: str) -> bool:
    """Start a stream"""
    return await real_time_streaming.start_stream(stream_id)


async def stop_stream(stream_id: str) -> bool:
    """Stop a stream"""
    return await real_time_streaming.stop_stream(stream_id)


async def publish_event(stream_id: str, event_data: Any, metadata: Dict[str, Any] = None) -> bool:
    """Publish event to stream"""
    return await real_time_streaming.publish_event(stream_id, event_data, metadata)


async def subscribe_to_stream(stream_id: str, event_handler: Callable[[StreamEvent], None]) -> bool:
    """Subscribe to stream events"""
    return await real_time_streaming.subscribe_to_stream(stream_id, event_handler)


def add_stream_processor(stream_id: str, processor: Callable[[StreamEvent], StreamEvent]):
    """Add processor to stream"""
    real_time_streaming.add_stream_processor(stream_id, processor)


def add_stream_analytics(stream_id: str, analytics_config: Dict[str, Any]):
    """Add analytics to stream"""
    real_time_streaming.add_analytics(stream_id, analytics_config)


async def get_stream_analytics(stream_id: str) -> Dict[str, Any]:
    """Get stream analytics"""
    return await real_time_streaming.get_stream_analytics(stream_id)


def get_stream_info(stream_id: str) -> Optional[Dict[str, Any]]:
    """Get stream information"""
    return real_time_streaming.get_stream_info(stream_id)


def list_streams() -> List[Dict[str, Any]]:
    """List all streams"""
    return real_time_streaming.list_streams()


if __name__ == "__main__":
    print("Real-Time Streaming for TuskLang Python SDK")
    print("=" * 50)
    
    # Test streaming capabilities
    print("\n1. Testing Stream Creation:")
    
    # Create streams
    create_stream("test-kafka", "TestKafkaStream", "kafka", "test-topic")
    create_stream("test-redis", "TestRedisStream", "redis", "test-stream")
    create_stream("test-websocket", "TestWebSocketStream", "websocket", "ws://localhost:8080")
    
    # List streams
    streams = list_streams()
    print(f"  Created streams: {len(streams)}")
    for stream in streams:
        print(f"    - {stream['name']} ({stream['type']})")
    
    # Test event publishing
    print("\n2. Testing Event Publishing:")
    
    async def test_streaming():
        # Start streams
        await start_stream("test-kafka")
        await start_stream("test-redis")
        
        # Publish events
        await publish_event("test-kafka", {"message": "Hello Kafka!"})
        await publish_event("test-redis", {"message": "Hello Redis!"})
        
        # Add analytics
        add_stream_analytics("test-kafka", {"type": "event_counting"})
        
        # Get analytics
        analytics = await get_stream_analytics("test-kafka")
        print(f"  Kafka analytics: {analytics}")
    
    # Run async test
    asyncio.run(test_streaming())
    
    print("\nReal-time streaming testing completed!") 