#!/usr/bin/env python3
"""
FUJSEN (Function Serialization) for TuskLang Python
==================================================
Cross-language function serialization and execution

Features:
- JavaScript function serialization and execution
- Python function serialization and execution
- Bash function serialization and execution
- Function caching and optimization
- Context injection and error handling
- Smart contract capabilities
"""

import json
import base64
import hashlib
import tempfile
import subprocess
import os
import sys
from typing import Any, Dict, List, Optional, Union, Callable
from datetime import datetime
import inspect
import ast
import re


class FUJSEN:
    """FUJSEN - Function Serialization Engine"""
    
    def __init__(self):
        self.cache = {}
        self.context = {}
        self.function_registry = {}
        self.smart_contracts = {}
    
    def serialize_function(self, func: Callable, language: str = "python") -> Dict[str, Any]:
        """Serialize a function to FUJSEN format"""
        if language == "python":
            return self._serialize_python_function(func)
        elif language == "javascript":
            return self._serialize_javascript_function(func)
        elif language == "bash":
            return self._serialize_bash_function(func)
        else:
            raise ValueError(f"Unsupported language: {language}")
    
    def _serialize_python_function(self, func: Callable) -> Dict[str, Any]:
        """Serialize Python function"""
        source = inspect.getsource(func)
        signature = inspect.signature(func)
        
        return {
            "type": "fujsen",
            "language": "python",
            "name": func.__name__,
            "source": source,
            "signature": str(signature),
            "module": func.__module__,
            "hash": hashlib.sha256(source.encode()).hexdigest(),
            "created_at": datetime.utcnow().isoformat()
        }
    
    def _serialize_javascript_function(self, func: Callable) -> Dict[str, Any]:
        """Serialize JavaScript function (convert Python to JS)"""
        source = inspect.getsource(func)
        js_source = self._python_to_javascript(source)
        
        return {
            "type": "fujsen",
            "language": "javascript",
            "name": func.__name__,
            "source": js_source,
            "original_source": source,
            "hash": hashlib.sha256(js_source.encode()).hexdigest(),
            "created_at": datetime.utcnow().isoformat()
        }
    
    def _serialize_bash_function(self, func: Callable) -> Dict[str, Any]:
        """Serialize Bash function (convert Python to Bash)"""
        source = inspect.getsource(func)
        bash_source = self._python_to_bash(source)
        
        return {
            "type": "fujsen",
            "language": "bash",
            "name": func.__name__,
            "source": bash_source,
            "original_source": source,
            "hash": hashlib.sha256(bash_source.encode()).hexdigest(),
            "created_at": datetime.utcnow().isoformat()
        }
    
    def _python_to_javascript(self, python_source: str) -> str:
        """Convert Python function to JavaScript"""
        # Simple Python to JavaScript conversion
        js_source = python_source
        
        # Function definition
        js_source = re.sub(r'def (\w+)\(([^)]*)\):', r'function \1(\2) {', js_source)
        
        # Indentation to braces
        lines = js_source.split('\n')
        js_lines = []
        indent_level = 0
        
        for line in lines:
            stripped = line.strip()
            if not stripped:
                continue
            
            # Handle indentation
            current_indent = len(line) - len(line.lstrip())
            if current_indent < indent_level:
                js_lines.append('  ' * indent_level + '}')
                indent_level = current_indent // 4
            
            # Convert Python syntax to JavaScript
            line = line.replace('print(', 'console.log(')
            line = line.replace('True', 'true')
            line = line.replace('False', 'false')
            line = line.replace('None', 'null')
            line = line.replace('elif ', 'else if (')
            line = line.replace('if ', 'if (')
            line = line.replace(' and ', ' && ')
            line = line.replace(' or ', ' || ')
            line = line.replace(' not ', ' !')
            
            # Add semicolons
            if line.strip() and not line.strip().endswith('{') and not line.strip().endswith('}'):
                line = line.rstrip() + ';'
            
            js_lines.append(line)
        
        # Close any remaining braces
        while indent_level > 0:
            js_lines.append('  ' * (indent_level - 1) + '}')
            indent_level -= 1
        
        return '\n'.join(js_lines)
    
    def _python_to_bash(self, python_source: str) -> str:
        """Convert Python function to Bash"""
        # Simple Python to Bash conversion
        bash_source = python_source
        
        # Function definition
        bash_source = re.sub(r'def (\w+)\(([^)]*)\):', r'\1() {', bash_source)
        
        # Convert Python syntax to Bash
        bash_source = bash_source.replace('print(', 'echo ')
        bash_source = bash_source.replace('True', 'true')
        bash_source = bash_source.replace('False', 'false')
        bash_source = bash_source.replace('None', '')
        bash_source = bash_source.replace('if ', 'if [ ')
        bash_source = bash_source.replace(' and ', ' ] && [ ')
        bash_source = bash_source.replace(' or ', ' ] || [ ')
        bash_source = bash_source.replace(' not ', ' ! ')
        
        # Add closing brackets for if statements
        bash_source = re.sub(r'if \[ ([^;]+);', r'if [ \1 ]; then', bash_source)
        
        return bash_source
    
    def execute_function(self, fujsen_data: Dict[str, Any], args: List[Any] = None, context: Dict[str, Any] = None) -> Any:
        """Execute a FUJSEN function"""
        if args is None:
            args = []
        
        if context is None:
            context = {}
        
        language = fujsen_data.get("language", "python")
        
        if language == "python":
            return self._execute_python_function(fujsen_data, args, context)
        elif language == "javascript":
            return self._execute_javascript_function(fujsen_data, args, context)
        elif language == "bash":
            return self._execute_bash_function(fujsen_data, args, context)
        else:
            raise ValueError(f"Unsupported language: {language}")
    
    def _execute_python_function(self, fujsen_data: Dict[str, Any], args: List[Any], context: Dict[str, Any]) -> Any:
        """Execute Python function"""
        try:
            # Create a temporary module
            source = fujsen_data["source"]
            module_name = f"fujsen_{fujsen_data['hash'][:8]}"
            
            # Inject context into source
            context_code = "\n".join([f"{k} = {repr(v)}" for k, v in context.items()])
            full_source = f"{context_code}\n{source}"
            
            # Compile and execute
            compiled = compile(full_source, module_name, 'exec')
            namespace = {}
            exec(compiled, namespace)
            
            # Get the function
            func_name = fujsen_data["name"]
            if func_name in namespace:
                func = namespace[func_name]
                return func(*args)
            else:
                raise ValueError(f"Function {func_name} not found in source")
                
        except Exception as e:
            raise Exception(f"Python execution error: {str(e)}")
    
    def _execute_javascript_function(self, fujsen_data: Dict[str, Any], args: List[Any], context: Dict[str, Any]) -> Any:
        """Execute JavaScript function"""
        try:
            # Create temporary JavaScript file
            source = fujsen_data["source"]
            func_name = fujsen_data["name"]
            
            # Add context and execution code
            context_code = "\n".join([f"const {k} = {json.dumps(v)};" for k, v in context.items()])
            args_json = json.dumps(args)
            
            js_code = f"""
{context_code}

{source}

// Execute function
const result = {func_name}(...{args_json});
console.log(JSON.stringify(result));
"""
            
            # Write to temporary file
            with tempfile.NamedTemporaryFile(mode='w', suffix='.js', delete=False) as f:
                f.write(js_code)
                temp_file = f.name
            
            try:
                # Execute with Node.js
                result = subprocess.run(['node', temp_file], capture_output=True, text=True, timeout=30)
                
                if result.returncode == 0:
                    return json.loads(result.stdout.strip())
                else:
                    raise Exception(f"JavaScript execution error: {result.stderr}")
                    
            finally:
                os.unlink(temp_file)
                
        except Exception as e:
            raise Exception(f"JavaScript execution error: {str(e)}")
    
    def _execute_bash_function(self, fujsen_data: Dict[str, Any], args: List[Any], context: Dict[str, Any]) -> Any:
        """Execute Bash function"""
        try:
            # Create temporary Bash file
            source = fujsen_data["source"]
            func_name = fujsen_data["name"]
            
            # Add context and execution code
            context_code = "\n".join([f"{k}={repr(v)}" for k, v in context.items()])
            args_str = " ".join([repr(str(arg)) for arg in args])
            
            bash_code = f"""#!/bin/bash

{context_code}

{source}

# Execute function
result=$({func_name} {args_str})
echo "$result"
"""
            
            # Write to temporary file
            with tempfile.NamedTemporaryFile(mode='w', suffix='.sh', delete=False) as f:
                f.write(bash_code)
                temp_file = f.name
            
            try:
                # Make executable and execute
                os.chmod(temp_file, 0o755)
                result = subprocess.run([temp_file], capture_output=True, text=True, timeout=30)
                
                if result.returncode == 0:
                    return result.stdout.strip()
                else:
                    raise Exception(f"Bash execution error: {result.stderr}")
                    
            finally:
                os.unlink(temp_file)
                
        except Exception as e:
            raise Exception(f"Bash execution error: {str(e)}")
    
    def cache_function(self, fujsen_data: Dict[str, Any]) -> str:
        """Cache a FUJSEN function"""
        func_hash = fujsen_data["hash"]
        self.cache[func_hash] = fujsen_data
        return func_hash
    
    def get_cached_function(self, func_hash: str) -> Optional[Dict[str, Any]]:
        """Get cached function by hash"""
        return self.cache.get(func_hash)
    
    def register_function(self, name: str, fujsen_data: Dict[str, Any]):
        """Register a function in the registry"""
        self.function_registry[name] = fujsen_data
    
    def get_registered_function(self, name: str) -> Optional[Dict[str, Any]]:
        """Get registered function by name"""
        return self.function_registry.get(name)
    
    def create_smart_contract(self, name: str, conditions: List[str], actions: List[str]) -> Dict[str, Any]:
        """Create a smart contract"""
        contract = {
            "type": "smart_contract",
            "name": name,
            "conditions": conditions,
            "actions": actions,
            "created_at": datetime.utcnow().isoformat(),
            "hash": hashlib.sha256(f"{name}{conditions}{actions}".encode()).hexdigest()
        }
        
        self.smart_contracts[name] = contract
        return contract
    
    def execute_smart_contract(self, name: str, context: Dict[str, Any]) -> bool:
        """Execute a smart contract"""
        if name not in self.smart_contracts:
            raise ValueError(f"Smart contract {name} not found")
        
        contract = self.smart_contracts[name]
        
        # Check conditions
        for condition in contract["conditions"]:
            try:
                # Evaluate condition
                if not eval(condition, context):
                    return False
            except Exception as e:
                print(f"Condition evaluation error: {e}")
                return False
        
        # Execute actions
        for action in contract["actions"]:
            try:
                exec(action, context)
            except Exception as e:
                print(f"Action execution error: {e}")
                return False
        
        return True
    
    def inject_context(self, context: Dict[str, Any]):
        """Inject context for function execution"""
        self.context.update(context)
    
    def get_context(self) -> Dict[str, Any]:
        """Get current context"""
        return self.context.copy()
    
    def clear_cache(self):
        """Clear function cache"""
        self.cache.clear()
    
    def export_functions(self) -> Dict[str, Any]:
        """Export all functions and contracts"""
        return {
            "functions": self.function_registry,
            "smart_contracts": self.smart_contracts,
            "cache": self.cache,
            "context": self.context,
            "exported_at": datetime.utcnow().isoformat()
        }
    
    def import_functions(self, data: Dict[str, Any]):
        """Import functions and contracts"""
        if "functions" in data:
            self.function_registry.update(data["functions"])
        if "smart_contracts" in data:
            self.smart_contracts.update(data["smart_contracts"])
        if "cache" in data:
            self.cache.update(data["cache"])
        if "context" in data:
            self.context.update(data["context"])


# Example functions for demonstration
def example_python_function(name: str, age: int) -> Dict[str, Any]:
    """Example Python function for FUJSEN"""
    return {
        "name": name,
        "age": age,
        "status": "active" if age >= 18 else "minor",
        "greeting": f"Hello, {name}! You are {age} years old."
    }


def example_calculation_function(a: float, b: float, operation: str) -> float:
    """Example calculation function"""
    if operation == "add":
        return a + b
    elif operation == "subtract":
        return a - b
    elif operation == "multiply":
        return a * b
    elif operation == "divide":
        return a / b if b != 0 else 0
    else:
        raise ValueError(f"Unknown operation: {operation}")


def example_data_processing_function(data: List[Dict[str, Any]]) -> Dict[str, Any]:
    """Example data processing function"""
    if not data:
        return {"count": 0, "total": 0, "average": 0}
    
    total = sum(item.get("value", 0) for item in data)
    count = len(data)
    average = total / count if count > 0 else 0
    
    return {
        "count": count,
        "total": total,
        "average": average,
        "min": min(item.get("value", 0) for item in data),
        "max": max(item.get("value", 0) for item in data)
    }


# Global FUJSEN instance
fujsen = FUJSEN()


def get_fujsen() -> FUJSEN:
    """Get global FUJSEN instance"""
    return fujsen


def create_fujsen_function(source: str, language: str = "python", name: str = None) -> Dict[str, Any]:
    """Create a FUJSEN function from source code"""
    if language == "python":
        # Compile and extract function
        try:
            compiled = compile(source, "<string>", "exec")
            namespace = {}
            exec(compiled, namespace)
            
            # Find the function
            if name:
                func = namespace.get(name)
                if func and callable(func):
                    return fujsen.serialize_function(func, language)
                else:
                    raise ValueError(f"Function {name} not found in source")
            else:
                # Find first function
                for key, value in namespace.items():
                    if callable(value) and not key.startswith('_'):
                        return fujsen.serialize_function(value, language)
                
                raise ValueError("No function found in source")
                
        except Exception as e:
            raise Exception(f"Python compilation error: {str(e)}")
    
    elif language == "javascript":
        # For JavaScript, we assume the source is already in JS format
        return {
            "type": "fujsen",
            "language": "javascript",
            "name": name or "anonymous",
            "source": source,
            "hash": hashlib.sha256(source.encode()).hexdigest(),
            "created_at": datetime.utcnow().isoformat()
        }
    
    elif language == "bash":
        # For Bash, we assume the source is already in Bash format
        return {
            "type": "fujsen",
            "language": "bash",
            "name": name or "anonymous",
            "source": source,
            "hash": hashlib.sha256(source.encode()).hexdigest(),
            "created_at": datetime.utcnow().isoformat()
        }
    
    else:
        raise ValueError(f"Unsupported language: {language}")


def execute_fujsen_function(fujsen_data: Dict[str, Any], args: List[Any] = None, context: Dict[str, Any] = None) -> Any:
    """Execute a FUJSEN function"""
    return fujsen.execute_function(fujsen_data, args or [], context or {})


# Initialize with example functions
fujsen.register_function("example_python", fujsen.serialize_function(example_python_function))
fujsen.register_function("example_calculation", fujsen.serialize_function(example_calculation_function))
fujsen.register_function("example_data_processing", fujsen.serialize_function(example_data_processing_function))

# Create example smart contract
fujsen.create_smart_contract(
    "age_verification",
    ["age >= 18"],
    ["print('User is verified')", "status = 'verified'"]
) 