#!/usr/bin/env python3
"""
Advanced Error Handling and Recovery System for TuskLang Python SDK
Goal 7.2 Implementation - Error Handling and Recovery System

Features:
- Circuit breaker pattern implementation
- Automatic retry with exponential backoff
- Graceful degradation strategies
- Error categorization and handling
- Recovery mechanisms and health checks
- Comprehensive error logging and reporting
"""

import time
import threading
import logging
import traceback
import functools
import asyncio
from typing import Any, Dict, List, Optional, Callable, Union, Type
from dataclasses import dataclass, field
from enum import Enum
from contextlib import contextmanager
import weakref

logger = logging.getLogger(__name__)

class ErrorSeverity(Enum):
    """Error severity levels"""
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"

class CircuitState(Enum):
    """Circuit breaker states"""
    CLOSED = "closed"      # Normal operation
    OPEN = "open"          # Failing, reject requests
    HALF_OPEN = "half_open"  # Testing if service recovered

@dataclass
class ErrorInfo:
    """Error information with metadata"""
    error_type: str
    message: str
    severity: ErrorSeverity
    timestamp: float
    context: Dict[str, Any] = field(default_factory=dict)
    stack_trace: str = ""
    recovery_attempts: int = 0
    is_recovered: bool = False

@dataclass
class CircuitBreakerConfig:
    """Circuit breaker configuration"""
    failure_threshold: int = 5
    recovery_timeout: float = 60.0  # seconds
    expected_exception: Type[Exception] = Exception
    monitor_interval: float = 10.0  # seconds

class CircuitBreaker:
    """Circuit breaker pattern implementation"""
    
    def __init__(self, name: str, config: CircuitBreakerConfig = None):
        self.name = name
        self.config = config or CircuitBreakerConfig()
        self.state = CircuitState.CLOSED
        self.failure_count = 0
        self.last_failure_time = 0
        self.last_success_time = 0
        self.lock = threading.RLock()
        
    def call(self, func: Callable, *args, **kwargs) -> Any:
        """Execute function with circuit breaker protection"""
        if not self._can_execute():
            raise Exception(f"Circuit breaker '{self.name}' is OPEN")
        
        try:
            result = func(*args, **kwargs)
            self._on_success()
            return result
        except self.config.expected_exception as e:
            self._on_failure()
            raise
    
    async def call_async(self, coro_func: Callable, *args, **kwargs) -> Any:
        """Execute async function with circuit breaker protection"""
        if not self._can_execute():
            raise Exception(f"Circuit breaker '{self.name}' is OPEN")
        
        try:
            result = await coro_func(*args, **kwargs)
            self._on_success()
            return result
        except self.config.expected_exception as e:
            self._on_failure()
            raise
    
    def _can_execute(self) -> bool:
        """Check if execution is allowed based on circuit state"""
        with self.lock:
            if self.state == CircuitState.CLOSED:
                return True
            elif self.state == CircuitState.OPEN:
                if time.time() - self.last_failure_time >= self.config.recovery_timeout:
                    self.state = CircuitState.HALF_OPEN
                    logger.info(f"Circuit breaker '{self.name}' transitioning to HALF_OPEN")
                    return True
                return False
            elif self.state == CircuitState.HALF_OPEN:
                return True
            return False
    
    def _on_success(self):
        """Handle successful execution"""
        with self.lock:
            self.failure_count = 0
            self.last_success_time = time.time()
            if self.state == CircuitState.HALF_OPEN:
                self.state = CircuitState.CLOSED
                logger.info(f"Circuit breaker '{self.name}' recovered, transitioning to CLOSED")
    
    def _on_failure(self):
        """Handle failed execution"""
        with self.lock:
            self.failure_count += 1
            self.last_failure_time = time.time()
            
            if self.failure_count >= self.config.failure_threshold:
                self.state = CircuitState.OPEN
                logger.warning(f"Circuit breaker '{self.name}' opened after {self.failure_count} failures")
    
    def get_status(self) -> Dict[str, Any]:
        """Get circuit breaker status"""
        with self.lock:
            return {
                'name': self.name,
                'state': self.state.value,
                'failure_count': self.failure_count,
                'last_failure_time': self.last_failure_time,
                'last_success_time': self.last_success_time,
                'config': {
                    'failure_threshold': self.config.failure_threshold,
                    'recovery_timeout': self.config.recovery_timeout
                }
            }

class RetryHandler:
    """Retry mechanism with exponential backoff"""
    
    def __init__(self, max_retries: int = 3, base_delay: float = 1.0, 
                 max_delay: float = 60.0, backoff_factor: float = 2.0):
        self.max_retries = max_retries
        self.base_delay = base_delay
        self.max_delay = max_delay
        self.backoff_factor = backoff_factor
    
    def retry(self, func: Callable, *args, **kwargs) -> Any:
        """Retry function with exponential backoff"""
        last_exception = None
        
        for attempt in range(self.max_retries + 1):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                last_exception = e
                
                if attempt == self.max_retries:
                    logger.error(f"Function {func.__name__} failed after {self.max_retries} retries")
                    raise last_exception
                
                delay = min(self.base_delay * (self.backoff_factor ** attempt), self.max_delay)
                logger.warning(f"Function {func.__name__} failed (attempt {attempt + 1}/{self.max_retries + 1}), "
                             f"retrying in {delay:.2f}s: {e}")
                time.sleep(delay)
    
    async def retry_async(self, coro_func: Callable, *args, **kwargs) -> Any:
        """Retry async function with exponential backoff"""
        last_exception = None
        
        for attempt in range(self.max_retries + 1):
            try:
                return await coro_func(*args, **kwargs)
            except Exception as e:
                last_exception = e
                
                if attempt == self.max_retries:
                    logger.error(f"Async function {coro_func.__name__} failed after {self.max_retries} retries")
                    raise last_exception
                
                delay = min(self.base_delay * (self.backoff_factor ** attempt), self.max_delay)
                logger.warning(f"Async function {coro_func.__name__} failed (attempt {attempt + 1}/{self.max_retries + 1}), "
                             f"retrying in {delay:.2f}s: {e}")
                await asyncio.sleep(delay)

class ErrorHandler:
    """Advanced error handling and recovery system"""
    
    def __init__(self):
        self.error_history: List[ErrorInfo] = []
        self.circuit_breakers: Dict[str, CircuitBreaker] = {}
        self.retry_handlers: Dict[str, RetryHandler] = {}
        self.error_handlers: Dict[str, Callable] = {}
        self.recovery_strategies: Dict[str, Callable] = {}
        self.health_checks: Dict[str, Callable] = {}
        self.lock = threading.RLock()
        
        # Register default error handlers
        self._register_default_handlers()
    
    def _register_default_handlers(self):
        """Register default error handlers for common scenarios"""
        self.register_error_handler("connection_error", self._handle_connection_error)
        self.register_error_handler("timeout_error", self._handle_timeout_error)
        self.register_error_handler("validation_error", self._handle_validation_error)
        self.register_error_handler("resource_error", self._handle_resource_error)
        
        # Register recovery strategies
        self.register_recovery_strategy("connection_error", self._recover_connection)
        self.register_recovery_strategy("resource_error", self._recover_resource)
    
    def register_error_handler(self, error_type: str, handler: Callable):
        """Register custom error handler"""
        self.error_handlers[error_type] = handler
    
    def register_recovery_strategy(self, error_type: str, strategy: Callable):
        """Register recovery strategy"""
        self.recovery_strategies[error_type] = strategy
    
    def register_health_check(self, service_name: str, health_check: Callable):
        """Register health check for service"""
        self.health_checks[service_name] = health_check
    
    def handle_error(self, error: Exception, context: Dict[str, Any] = None) -> ErrorInfo:
        """Handle error with appropriate strategy"""
        error_info = ErrorInfo(
            error_type=type(error).__name__,
            message=str(error),
            severity=self._determine_severity(error),
            timestamp=time.time(),
            context=context or {},
            stack_trace=traceback.format_exc()
        )
        
        with self.lock:
            self.error_history.append(error_info)
            
            # Keep only last 1000 errors
            if len(self.error_history) > 1000:
                self.error_history = self.error_history[-1000:]
        
        # Log error
        logger.error(f"Error handled: {error_info.error_type} - {error_info.message}")
        
        # Apply error handler if available
        if error_info.error_type in self.error_handlers:
            try:
                self.error_handlers[error_info.error_type](error_info)
            except Exception as e:
                logger.error(f"Error handler failed: {e}")
        
        # Attempt recovery
        self._attempt_recovery(error_info)
        
        return error_info
    
    def _determine_severity(self, error: Exception) -> ErrorSeverity:
        """Determine error severity based on error type and context"""
        error_name = type(error).__name__.lower()
        
        if any(critical in error_name for critical in ['critical', 'fatal', 'panic']):
            return ErrorSeverity.CRITICAL
        elif any(high in error_name for high in ['error', 'exception', 'failure']):
            return ErrorSeverity.HIGH
        elif any(medium in error_name for medium in ['warning', 'timeout', 'connection']):
            return ErrorSeverity.MEDIUM
        else:
            return ErrorSeverity.LOW
    
    def _attempt_recovery(self, error_info: ErrorInfo):
        """Attempt to recover from error"""
        if error_info.error_type in self.recovery_strategies:
            try:
                error_info.recovery_attempts += 1
                result = self.recovery_strategies[error_info.error_type](error_info)
                if result:
                    error_info.is_recovered = True
                    logger.info(f"Recovery successful for {error_info.error_type}")
            except Exception as e:
                logger.error(f"Recovery failed for {error_info.error_type}: {e}")
    
    def _handle_connection_error(self, error_info: ErrorInfo):
        """Handle connection errors"""
        logger.warning(f"Connection error detected: {error_info.message}")
        # Could implement connection pooling, retry logic, etc.
    
    def _handle_timeout_error(self, error_info: ErrorInfo):
        """Handle timeout errors"""
        logger.warning(f"Timeout error detected: {error_info.message}")
        # Could implement timeout adjustment, circuit breaker, etc.
    
    def _handle_validation_error(self, error_info: ErrorInfo):
        """Handle validation errors"""
        logger.warning(f"Validation error detected: {error_info.message}")
        # Could implement input sanitization, schema validation, etc.
    
    def _handle_resource_error(self, error_info: ErrorInfo):
        """Handle resource errors"""
        logger.warning(f"Resource error detected: {error_info.message}")
        # Could implement resource cleanup, alternative resources, etc.
    
    def _recover_connection(self, error_info: ErrorInfo) -> bool:
        """Recover from connection error"""
        # Implement connection recovery logic
        time.sleep(1)  # Simulate recovery time
        return True
    
    def _recover_resource(self, error_info: ErrorInfo) -> bool:
        """Recover from resource error"""
        # Implement resource recovery logic
        time.sleep(1)  # Simulate recovery time
        return True
    
    def get_circuit_breaker(self, name: str, config: CircuitBreakerConfig = None) -> CircuitBreaker:
        """Get or create circuit breaker"""
        with self.lock:
            if name not in self.circuit_breakers:
                self.circuit_breakers[name] = CircuitBreaker(name, config)
            return self.circuit_breakers[name]
    
    def get_retry_handler(self, name: str, max_retries: int = 3) -> RetryHandler:
        """Get or create retry handler"""
        with self.lock:
            if name not in self.retry_handlers:
                self.retry_handlers[name] = RetryHandler(max_retries=max_retries)
            return self.retry_handlers[name]
    
    def check_health(self, service_name: str = None) -> Dict[str, Any]:
        """Check health of services"""
        health_status = {}
        
        if service_name:
            if service_name in self.health_checks:
                try:
                    result = self.health_checks[service_name]()
                    health_status[service_name] = {'status': 'healthy', 'result': result}
                except Exception as e:
                    health_status[service_name] = {'status': 'unhealthy', 'error': str(e)}
        else:
            # Check all services
            for name, health_check in self.health_checks.items():
                try:
                    result = health_check()
                    health_status[name] = {'status': 'healthy', 'result': result}
                except Exception as e:
                    health_status[name] = {'status': 'unhealthy', 'error': str(e)}
        
        return health_status
    
    def get_error_statistics(self) -> Dict[str, Any]:
        """Get error statistics"""
        with self.lock:
            if not self.error_history:
                return {'total_errors': 0, 'error_types': {}, 'severity_distribution': {}}
            
            error_types = {}
            severity_distribution = {}
            
            for error in self.error_history:
                # Count by type
                error_types[error.error_type] = error_types.get(error.error_type, 0) + 1
                
                # Count by severity
                severity_distribution[error.severity.value] = severity_distribution.get(error.severity.value, 0) + 1
            
            return {
                'total_errors': len(self.error_history),
                'error_types': error_types,
                'severity_distribution': severity_distribution,
                'recovery_rate': sum(1 for e in self.error_history if e.is_recovered) / len(self.error_history)
            }

# Global error handler instance
error_handler = ErrorHandler()

def handle_errors(error_types: List[str] = None, retry: bool = False, circuit_breaker: str = None):
    """Decorator for error handling"""
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            try:
                if circuit_breaker:
                    cb = error_handler.get_circuit_breaker(circuit_breaker)
                    return cb.call(func, *args, **kwargs)
                elif retry:
                    retry_handler = error_handler.get_retry_handler(func.__name__)
                    return retry_handler.retry(func, *args, **kwargs)
                else:
                    return func(*args, **kwargs)
            except Exception as e:
                error_handler.handle_error(e, {
                    'function': func.__name__,
                    'args': str(args),
                    'kwargs': str(kwargs)
                })
                raise
        return wrapper
    return decorator

def handle_errors_async(error_types: List[str] = None, retry: bool = False, circuit_breaker: str = None):
    """Decorator for async error handling"""
    def decorator(func):
        @functools.wraps(func)
        async def wrapper(*args, **kwargs):
            try:
                if circuit_breaker:
                    cb = error_handler.get_circuit_breaker(circuit_breaker)
                    return await cb.call_async(func, *args, **kwargs)
                elif retry:
                    retry_handler = error_handler.get_retry_handler(func.__name__)
                    return await retry_handler.retry_async(func, *args, **kwargs)
                else:
                    return await func(*args, **kwargs)
            except Exception as e:
                error_handler.handle_error(e, {
                    'function': func.__name__,
                    'args': str(args),
                    'kwargs': str(kwargs)
                })
                raise
        return wrapper
    return decorator 