#!/usr/bin/env python3
"""
Unit tests for AI Code Generator (g8.1)
Tests the AI-powered code generation system
"""

import unittest
import sys
import os
import tempfile
import json
from unittest.mock import Mock, patch

# Add parent directory to path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from ai_code_generator import (
    AICodeGenerator, 
    TuskLangTemplateEngine,
    CodeGenerationRequest,
    CodeGenerationType,
    CodeGenerationResult,
    generate_tsk_code
)

class TestTuskLangTemplateEngine(unittest.TestCase):
    """Test the TuskLang template engine"""
    
    def setUp(self):
        self.template_engine = TuskLangTemplateEngine()
    
    def test_load_templates(self):
        """Test template loading"""
        self.assertIn("operator", self.template_engine.templates)
        self.assertIn("function", self.template_engine.templates)
        self.assertIn("class", self.template_engine.templates)
        self.assertIn("script", self.template_engine.templates)
    
    def test_load_operator_patterns(self):
        """Test operator pattern loading"""
        self.assertIn("math", self.template_engine.operator_patterns)
        self.assertIn("string", self.template_engine.operator_patterns)
        self.assertIn("list", self.template_engine.operator_patterns)
        self.assertIn("dict", self.template_engine.operator_patterns)
    
    def test_generate_operator(self):
        """Test operator generation"""
        code = self.template_engine.generate_operator(
            "test_operator",
            "Test operator description",
            "math"
        )
        
        self.assertIn("@operator", code)
        self.assertIn("test_operator", code)
        self.assertIn("Test operator description", code)
        self.assertIn("def test_operator_operator", code)
    
    def test_generate_function(self):
        """Test function generation"""
        code = self.template_engine.generate_function(
            "test_function",
            "Test function description",
            "param1, param2",
            "str",
            "return param1 + param2"
        )
        
        self.assertIn("def test_function", code)
        self.assertIn("Test function description", code)
        self.assertIn("param1, param2", code)
        self.assertIn("str", code)
        self.assertIn("return param1 + param2", code)

class TestAICodeGenerator(unittest.TestCase):
    """Test the AI code generator"""
    
    def setUp(self):
        self.generator = AICodeGenerator()
    
    def test_initialization(self):
        """Test generator initialization"""
        self.assertIsNotNone(self.generator.template_engine)
        self.assertIsNotNone(self.generator.code_patterns)
        self.assertIsNotNone(self.generator.safety_validator)
        self.assertIsNotNone(self.generator.context_analyzer)
    
    def test_load_code_patterns(self):
        """Test code pattern loading"""
        patterns = self.generator.code_patterns
        self.assertIn("data_processing", patterns)
        self.assertIn("api_integration", patterns)
        self.assertIn("file_operations", patterns)
        self.assertIn("database_operations", patterns)
        self.assertIn("security", patterns)
        self.assertIn("performance", patterns)
    
    def test_generate_code_operator(self):
        """Test operator code generation"""
        request = CodeGenerationRequest(
            description="Create an operator that adds two numbers",
            generation_type=CodeGenerationType.OPERATOR,
            context={"language": "python"},
            requirements=["Must handle numeric input"],
            target_complexity="simple"
        )
        
        result = self.generator.generate_code(request)
        
        self.assertIsInstance(result, CodeGenerationResult)
        self.assertIsNotNone(result.code)
        self.assertGreater(result.confidence, 0.0)
        self.assertLessEqual(result.confidence, 1.0)
        self.assertIn("@operator", result.code)
    
    def test_generate_code_function(self):
        """Test function code generation"""
        request = CodeGenerationRequest(
            description="Create a function that processes data",
            generation_type=CodeGenerationType.FUNCTION,
            context={"language": "python", "data_type": "list"},
            requirements=["Must handle list input"],
            target_complexity="medium"
        )
        
        result = self.generator.generate_code(request)
        
        self.assertIsInstance(result, CodeGenerationResult)
        self.assertIsNotNone(result.code)
        self.assertGreater(result.confidence, 0.0)
        self.assertLessEqual(result.confidence, 1.0)
        self.assertIn("def ", result.code)
    
    def test_generate_code_class(self):
        """Test class code generation"""
        request = CodeGenerationRequest(
            description="Create a class for data processing",
            generation_type=CodeGenerationType.CLASS,
            context={"language": "python"},
            requirements=["Must have init method"],
            target_complexity="medium"
        )
        
        result = self.generator.generate_code(request)
        
        self.assertIsInstance(result, CodeGenerationResult)
        self.assertIsNotNone(result.code)
        self.assertGreater(result.confidence, 0.0)
        self.assertLessEqual(result.confidence, 1.0)
        self.assertIn("class ", result.code)
    
    def test_generate_code_script(self):
        """Test script code generation"""
        request = CodeGenerationRequest(
            description="Create a script for data processing",
            generation_type=CodeGenerationType.SCRIPT,
            context={"language": "python"},
            requirements=["Must have main function"],
            target_complexity="simple"
        )
        
        result = self.generator.generate_code(request)
        
        self.assertIsInstance(result, CodeGenerationResult)
        self.assertIsNotNone(result.code)
        self.assertGreater(result.confidence, 0.0)
        self.assertLessEqual(result.confidence, 1.0)
        self.assertIn("#!/usr/bin/env python3", result.code)
    
    def test_extract_operator_name(self):
        """Test operator name extraction"""
        description = "Create an operator that adds two numbers"
        name = self.generator._extract_operator_name(description)
        self.assertIsInstance(name, str)
        self.assertGreater(len(name), 0)
    
    def test_extract_function_name(self):
        """Test function name extraction"""
        description = "Create a function that processes data"
        name = self.generator._extract_function_name(description)
        self.assertIsInstance(name, str)
        self.assertGreater(len(name), 0)
    
    def test_extract_class_name(self):
        """Test class name extraction"""
        description = "Create a class for data processing"
        name = self.generator._extract_class_name(description)
        self.assertIsInstance(name, str)
        self.assertGreater(len(name), 0)
    
    def test_determine_operator_pattern(self):
        """Test operator pattern determination"""
        description = "Create an operator that adds two numbers"
        context_info = {"language": "python"}
        pattern = self.generator._determine_operator_pattern(description, context_info)
        self.assertIsInstance(pattern, str)
        self.assertIn(pattern, self.generator.template_engine.operator_patterns)
    
    def test_extract_parameters(self):
        """Test parameter extraction"""
        description = "Create a function with parameters a and b"
        context_info = {"language": "python"}
        params = self.generator._extract_parameters(description, context_info)
        self.assertIsInstance(params, str)
    
    def test_determine_return_type(self):
        """Test return type determination"""
        description = "Create a function that returns a string"
        context_info = {"language": "python"}
        return_type = self.generator._determine_return_type(description, context_info)
        self.assertIsInstance(return_type, str)
    
    def test_generate_implementation(self):
        """Test implementation generation"""
        description = "Create a function that adds two numbers"
        context_info = {"language": "python"}
        implementation = self.generator._generate_implementation(description, context_info)
        self.assertIsInstance(implementation, str)
        self.assertGreater(len(implementation), 0)
    
    def test_identify_code_patterns(self):
        """Test code pattern identification"""
        description = "Create a function that processes data and handles errors"
        patterns = self.generator._identify_code_patterns(description)
        self.assertIsInstance(patterns, list)
    
    def test_calculate_confidence(self):
        """Test confidence calculation"""
        context_info = {"clarity_score": 0.8, "complexity_score": 0.6}
        validation_result = {"valid": True, "score": 0.9}
        confidence = self.generator._calculate_confidence(context_info, validation_result)
        self.assertIsInstance(confidence, float)
        self.assertGreaterEqual(confidence, 0.0)
        self.assertLessEqual(confidence, 1.0)
    
    def test_generate_suggestions(self):
        """Test suggestion generation"""
        request = CodeGenerationRequest(
            description="Create an operator",
            generation_type=CodeGenerationType.OPERATOR
        )
        context_info = {"language": "python"}
        suggestions = self.generator._generate_suggestions(request, context_info)
        self.assertIsInstance(suggestions, list)

class TestCodeSafetyValidator(unittest.TestCase):
    """Test the code safety validator"""
    
    def setUp(self):
        from ai_code_generator import CodeSafetyValidator
        self.validator = CodeSafetyValidator()
    
    def test_initialization(self):
        """Test validator initialization"""
        self.assertIsNotNone(self.validator.dangerous_patterns)
        self.assertIsNotNone(self.validator.safety_checks)
    
    def test_validate_code_safe(self):
        """Test validation of safe code"""
        request = CodeGenerationRequest(
            description="Create a simple operator",
            generation_type=CodeGenerationType.OPERATOR
        )
        safe_code = """
@operator("safe_operator")
def safe_operator(context):
    return context.get('value', 0)
"""
        result = self.validator.validate_code(safe_code, request)
        self.assertIsNotNone(result)
    
    def test_validate_code_dangerous(self):
        """Test validation of dangerous code"""
        request = CodeGenerationRequest(
            description="Create an operator",
            generation_type=CodeGenerationType.OPERATOR
        )
        dangerous_code = """
import os
os.system('rm -rf /')
"""
        result = self.validator.validate_code(dangerous_code, request)
        self.assertIsNotNone(result)

class TestContextAnalyzer(unittest.TestCase):
    """Test the context analyzer"""
    
    def setUp(self):
        from ai_code_generator import ContextAnalyzer
        self.analyzer = ContextAnalyzer()
    
    def test_analyze(self):
        """Test context analysis"""
        request = CodeGenerationRequest(
            description="Create an operator that adds numbers",
            generation_type=CodeGenerationType.OPERATOR,
            context={"language": "python"}
        )
        result = self.analyzer.analyze(request)
        self.assertIsInstance(result, dict)
        self.assertIn("clarity_score", result)
        self.assertIn("complexity_score", result)
        self.assertIn("pattern_match_score", result)
    
    def test_calculate_clarity_score(self):
        """Test clarity score calculation"""
        description = "Create an operator that adds two numbers"
        score = self.analyzer._calculate_clarity_score(description)
        self.assertIsInstance(score, float)
        self.assertGreaterEqual(score, 0.0)
        self.assertLessEqual(score, 1.0)
    
    def test_calculate_complexity_score(self):
        """Test complexity score calculation"""
        description = "Create a complex operator with multiple parameters"
        score = self.analyzer._calculate_complexity_score(description)
        self.assertIsInstance(score, float)
        self.assertGreaterEqual(score, 0.0)
        self.assertLessEqual(score, 1.0)
    
    def test_calculate_pattern_match_score(self):
        """Test pattern match score calculation"""
        description = "Create an operator that processes data"
        score = self.analyzer._calculate_pattern_match_score(description)
        self.assertIsInstance(score, float)
        self.assertGreaterEqual(score, 0.0)
        self.assertLessEqual(score, 1.0)

class TestConvenienceFunctions(unittest.TestCase):
    """Test convenience functions"""
    
    def test_generate_tsk_code(self):
        """Test the generate_tsk_code convenience function"""
        result = generate_tsk_code(
            "Create an operator that adds numbers",
            "operator"
        )
        
        self.assertIsInstance(result, CodeGenerationResult)
        self.assertIsNotNone(result.code)
        self.assertGreater(result.confidence, 0.0)
        self.assertLessEqual(result.confidence, 1.0)
    
    def test_generate_tsk_code_with_kwargs(self):
        """Test generate_tsk_code with additional kwargs"""
        result = generate_tsk_code(
            "Create a function that processes data",
            "function",
            context={"language": "python"},
            target_complexity="medium"
        )
        
        self.assertIsInstance(result, CodeGenerationResult)
        self.assertIsNotNone(result.code)

class TestIntegration(unittest.TestCase):
    """Test integration scenarios"""
    
    def setUp(self):
        self.generator = AICodeGenerator()
    
    def test_end_to_end_generation(self):
        """Test complete end-to-end code generation"""
        request = CodeGenerationRequest(
            description="Create a TuskLang operator that validates input data",
            generation_type=CodeGenerationType.OPERATOR,
            context={"language": "python", "validation": True},
            requirements=["Must validate input", "Must handle errors"],
            target_complexity="medium"
        )
        
        result = self.generator.generate_code(request)
        
        # Verify result structure
        self.assertIsInstance(result, CodeGenerationResult)
        self.assertIsNotNone(result.code)
        self.assertGreater(result.confidence, 0.0)
        self.assertLessEqual(result.confidence, 1.0)
        
        # Verify code content
        self.assertIn("@operator", result.code)
        self.assertIn("def ", result.code)
        self.assertIn("try:", result.code)
        self.assertIn("except", result.code)
    
    def test_multiple_generation_types(self):
        """Test generation of multiple code types"""
        types = [
            CodeGenerationType.OPERATOR,
            CodeGenerationType.FUNCTION,
            CodeGenerationType.CLASS,
            CodeGenerationType.SCRIPT
        ]
        
        for gen_type in types:
            request = CodeGenerationRequest(
                description=f"Create a {gen_type.value}",
                generation_type=gen_type,
                context={"language": "python"},
                target_complexity="simple"
            )
            
            result = self.generator.generate_code(request)
            
            self.assertIsInstance(result, CodeGenerationResult)
            self.assertIsNotNone(result.code)
            self.assertGreater(result.confidence, 0.0)

if __name__ == '__main__':
    unittest.main() 