#!/usr/bin/env python3
"""
Unit tests for Predictive Optimizer (g8.2)
Tests the ML-based predictive performance optimization engine
"""

import unittest
import sys
import os
import time
import json
from unittest.mock import Mock, patch
from collections import deque

# Add parent directory to path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from predictive_optimizer import (
    PredictiveOptimizer,
    PerformancePredictor,
    TrendAnalyzer,
    AnomalyDetector,
    PerformanceMetric,
    PerformancePrediction,
    OptimizationRecommendation,
    predict_operation_performance,
    get_optimization_recommendations
)

class TestPerformanceMetric(unittest.TestCase):
    """Test the PerformanceMetric dataclass"""
    
    def test_creation(self):
        """Test PerformanceMetric creation"""
        metric = PerformanceMetric(
            timestamp=time.time(),
            operation_name="test_operation",
            duration=1.5,
            memory_usage=100.0,
            cpu_usage=50.0,
            error_count=0,
            success_count=1
        )
        
        self.assertIsInstance(metric.timestamp, float)
        self.assertEqual(metric.operation_name, "test_operation")
        self.assertEqual(metric.duration, 1.5)
        self.assertEqual(metric.memory_usage, 100.0)
        self.assertEqual(metric.cpu_usage, 50.0)
        self.assertEqual(metric.error_count, 0)
        self.assertEqual(metric.success_count, 1)
    
    def test_with_context(self):
        """Test PerformanceMetric with context"""
        context = {"input_size": 1000, "complexity": "medium"}
        metric = PerformanceMetric(
            timestamp=time.time(),
            operation_name="test_operation",
            duration=1.0,
            memory_usage=50.0,
            cpu_usage=25.0,
            error_count=0,
            success_count=1,
            context=context
        )
        
        self.assertEqual(metric.context, context)

class TestPerformancePrediction(unittest.TestCase):
    """Test the PerformancePrediction dataclass"""
    
    def test_creation(self):
        """Test PerformancePrediction creation"""
        prediction = PerformancePrediction(
            operation_name="test_operation",
            predicted_duration=2.0,
            predicted_memory=150.0,
            confidence=0.85,
            trend="improving",
            recommendations=["Optimize algorithm", "Add caching"],
            risk_level="low"
        )
        
        self.assertEqual(prediction.operation_name, "test_operation")
        self.assertEqual(prediction.predicted_duration, 2.0)
        self.assertEqual(prediction.predicted_memory, 150.0)
        self.assertEqual(prediction.confidence, 0.85)
        self.assertEqual(prediction.trend, "improving")
        self.assertEqual(len(prediction.recommendations), 2)
        self.assertEqual(prediction.risk_level, "low")

class TestOptimizationRecommendation(unittest.TestCase):
    """Test the OptimizationRecommendation dataclass"""
    
    def test_creation(self):
        """Test OptimizationRecommendation creation"""
        recommendation = OptimizationRecommendation(
            type="caching",
            description="Add result caching to improve performance",
            expected_improvement=0.3,
            implementation_effort="medium",
            priority="high"
        )
        
        self.assertEqual(recommendation.type, "caching")
        self.assertEqual(recommendation.description, "Add result caching to improve performance")
        self.assertEqual(recommendation.expected_improvement, 0.3)
        self.assertEqual(recommendation.implementation_effort, "medium")
        self.assertEqual(recommendation.priority, "high")

class TestPerformancePredictor(unittest.TestCase):
    """Test the PerformancePredictor class"""
    
    def setUp(self):
        self.predictor = PerformancePredictor(history_window=100)
    
    def test_initialization(self):
        """Test predictor initialization"""
        self.assertEqual(self.predictor.history_window, 100)
        self.assertIsInstance(self.predictor.performance_history, dict)
        self.assertIsInstance(self.predictor.prediction_models, dict)
        self.assertIsNotNone(self.predictor.trend_analyzer)
        self.assertIsNotNone(self.predictor.anomaly_detector)
    
    def test_record_performance(self):
        """Test performance recording"""
        metric = PerformanceMetric(
            timestamp=time.time(),
            operation_name="test_op",
            duration=1.0,
            memory_usage=50.0,
            cpu_usage=25.0,
            error_count=0,
            success_count=1
        )
        
        self.predictor.record_performance(metric)
        
        self.assertIn("test_op", self.predictor.performance_history)
        self.assertEqual(len(self.predictor.performance_history["test_op"]), 1)
    
    def test_predict_performance_insufficient_data(self):
        """Test prediction with insufficient data"""
        prediction = self.predictor.predict_performance("test_op", {})
        
        self.assertIsInstance(prediction, PerformancePrediction)
        self.assertEqual(prediction.operation_name, "test_op")
        self.assertEqual(prediction.confidence, 0.0)
        self.assertEqual(prediction.trend, "stable")
        self.assertIn("Collect more performance data", prediction.recommendations[0])
    
    def test_predict_performance_with_data(self):
        """Test prediction with sufficient data"""
        # Add some test data
        for i in range(15):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_op",
                duration=1.0 + (i * 0.1),
                memory_usage=50.0 + (i * 2),
                cpu_usage=25.0 + (i * 1),
                error_count=0,
                success_count=1
            )
            self.predictor.record_performance(metric)
        
        prediction = self.predictor.predict_performance("test_op", {})
        
        self.assertIsInstance(prediction, PerformancePrediction)
        self.assertEqual(prediction.operation_name, "test_op")
        self.assertGreater(prediction.confidence, 0.0)
        self.assertGreater(prediction.predicted_duration, 0.0)
        self.assertGreater(prediction.predicted_memory, 0.0)
    
    def test_predict_duration(self):
        """Test duration prediction"""
        # Create test history
        history = []
        for i in range(10):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_op",
                duration=1.0 + (i * 0.1),
                memory_usage=50.0,
                cpu_usage=25.0,
                error_count=0,
                success_count=1
            )
            history.append(metric)
        
        predicted_duration = self.predictor._predict_duration(history, {})
        
        self.assertIsInstance(predicted_duration, float)
        self.assertGreater(predicted_duration, 0.0)
    
    def test_predict_memory(self):
        """Test memory prediction"""
        # Create test history
        history = []
        for i in range(10):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_op",
                duration=1.0,
                memory_usage=50.0 + (i * 2),
                cpu_usage=25.0,
                error_count=0,
                success_count=1
            )
            history.append(metric)
        
        predicted_memory = self.predictor._predict_memory(history, {})
        
        self.assertIsInstance(predicted_memory, float)
        self.assertGreater(predicted_memory, 0.0)
    
    def test_calculate_prediction_confidence(self):
        """Test confidence calculation"""
        history = []
        for i in range(10):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_op",
                duration=1.0,
                memory_usage=50.0,
                cpu_usage=25.0,
                error_count=0,
                success_count=1
            )
            history.append(metric)
        
        trend_analysis = {"stability": 0.8, "trend_strength": 0.6}
        confidence = self.predictor._calculate_prediction_confidence(history, trend_analysis)
        
        self.assertIsInstance(confidence, float)
        self.assertGreaterEqual(confidence, 0.0)
        self.assertLessEqual(confidence, 1.0)
    
    def test_determine_trend(self):
        """Test trend determination"""
        trend_analysis = {"trend_direction": "up", "trend_strength": 0.7}
        trend = self.predictor._determine_trend(trend_analysis)
        
        self.assertIn(trend, ["improving", "stable", "degrading"])
    
    def test_generate_recommendations(self):
        """Test recommendation generation"""
        history = []
        for i in range(10):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_op",
                duration=1.0 + (i * 0.2),
                memory_usage=50.0 + (i * 5),
                cpu_usage=25.0 + (i * 2),
                error_count=0,
                success_count=1
            )
            history.append(metric)
        
        trend_analysis = {"trend_direction": "up", "trend_strength": 0.7}
        anomaly_score = 0.3
        
        recommendations = self.predictor._generate_recommendations(history, trend_analysis, anomaly_score)
        
        self.assertIsInstance(recommendations, list)
        self.assertGreater(len(recommendations), 0)
    
    def test_determine_risk_level(self):
        """Test risk level determination"""
        risk_level = self.predictor._determine_risk_level(1.0, 50.0, 0.1)
        self.assertIn(risk_level, ["low", "medium", "high", "critical"])

class TestTrendAnalyzer(unittest.TestCase):
    """Test the TrendAnalyzer class"""
    
    def setUp(self):
        self.analyzer = TrendAnalyzer()
    
    def test_analyze_trends(self):
        """Test trend analysis"""
        history = []
        for i in range(10):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_op",
                duration=1.0 + (i * 0.1),
                memory_usage=50.0 + (i * 2),
                cpu_usage=25.0 + (i * 1),
                error_count=0,
                success_count=1
            )
            history.append(metric)
        
        analysis = self.analyzer.analyze_trends(history)
        
        self.assertIsInstance(analysis, dict)
        self.assertIn("trend_direction", analysis)
        self.assertIn("trend_strength", analysis)
        self.assertIn("stability", analysis)
    
    def test_calculate_slope(self):
        """Test slope calculation"""
        x = [1, 2, 3, 4, 5]
        y = [1, 2, 3, 4, 5]
        
        slope = self.analyzer._calculate_slope(x, y)
        
        self.assertIsInstance(slope, float)
        self.assertAlmostEqual(slope, 1.0, places=5)
    
    def test_calculate_stability(self):
        """Test stability calculation"""
        values = [1.0, 1.1, 0.9, 1.0, 1.05]
        
        stability = self.analyzer._calculate_stability(values)
        
        self.assertIsInstance(stability, float)
        self.assertGreaterEqual(stability, 0.0)
        self.assertLessEqual(stability, 1.0)

class TestAnomalyDetector(unittest.TestCase):
    """Test the AnomalyDetector class"""
    
    def setUp(self):
        self.detector = AnomalyDetector(threshold_multiplier=2.0)
    
    def test_initialization(self):
        """Test detector initialization"""
        self.assertEqual(self.detector.threshold_multiplier, 2.0)
    
    def test_detect_anomalies(self):
        """Test anomaly detection"""
        history = []
        for i in range(10):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_op",
                duration=1.0,
                memory_usage=50.0,
                cpu_usage=25.0,
                error_count=0,
                success_count=1
            )
            history.append(metric)
        
        # Add an anomaly
        anomaly_metric = PerformanceMetric(
            timestamp=time.time() + 10,
            operation_name="test_op",
            duration=10.0,  # Much higher than average
            memory_usage=500.0,  # Much higher than average
            cpu_usage=90.0,  # Much higher than average
            error_count=5,
            success_count=0
        )
        history.append(anomaly_metric)
        
        anomaly_score = self.detector.detect_anomalies(history)
        
        self.assertIsInstance(anomaly_score, float)
        self.assertGreaterEqual(anomaly_score, 0.0)
        self.assertLessEqual(anomaly_score, 1.0)

class TestPredictiveOptimizer(unittest.TestCase):
    """Test the PredictiveOptimizer class"""
    
    def setUp(self):
        self.optimizer = PredictiveOptimizer()
    
    def test_initialization(self):
        """Test optimizer initialization"""
        self.assertIsNotNone(self.optimizer.predictor)
        self.assertIsNotNone(self.optimizer.optimization_engine)
        self.assertIsNotNone(self.optimizer.alert_manager)
    
    def test_optimize_performance(self):
        """Test performance optimization"""
        recommendations = self.optimizer.optimize_performance("test_op", {})
        
        self.assertIsInstance(recommendations, list)
        # Even with no data, should return some general recommendations
        self.assertGreaterEqual(len(recommendations), 0)
    
    def test_get_performance_insights(self):
        """Test performance insights"""
        insights = self.optimizer.get_performance_insights("test_op")
        
        self.assertIsInstance(insights, dict)
        self.assertIn("operation_name", insights)
        self.assertIn("data_points", insights)
        self.assertIn("trends", insights)
        self.assertIn("anomalies", insights)

class TestOptimizationEngine(unittest.TestCase):
    """Test the OptimizationEngine class"""
    
    def setUp(self):
        from predictive_optimizer import OptimizationEngine
        self.engine = OptimizationEngine()
    
    def test_generate_recommendations(self):
        """Test recommendation generation"""
        prediction = PerformancePrediction(
            operation_name="test_op",
            predicted_duration=5.0,
            predicted_memory=200.0,
            confidence=0.8,
            trend="degrading",
            recommendations=[],
            risk_level="medium"
        )
        
        recommendations = self.engine.generate_recommendations(prediction, {})
        
        self.assertIsInstance(recommendations, list)
        self.assertGreater(len(recommendations), 0)
        
        for rec in recommendations:
            self.assertIsInstance(rec, OptimizationRecommendation)
            self.assertIn(rec.type, ["caching", "algorithm", "resource", "configuration"])
            self.assertIn(rec.implementation_effort, ["low", "medium", "high"])
            self.assertIn(rec.priority, ["low", "medium", "high", "critical"])

class TestPerformanceAlertManager(unittest.TestCase):
    """Test the PerformanceAlertManager class"""
    
    def setUp(self):
        from predictive_optimizer import PerformanceAlertManager
        self.alert_manager = PerformanceAlertManager()
    
    def test_initialization(self):
        """Test alert manager initialization"""
        self.assertIsNotNone(self.alert_manager.alert_thresholds)
        self.assertIsNotNone(self.alert_manager.alert_history)
    
    def test_trigger_alert(self):
        """Test alert triggering"""
        prediction = PerformancePrediction(
            operation_name="test_op",
            predicted_duration=10.0,
            predicted_memory=500.0,
            confidence=0.9,
            trend="degrading",
            recommendations=[],
            risk_level="high"
        )
        
        # Should not raise an exception
        self.alert_manager.trigger_alert(prediction)

class TestConvenienceFunctions(unittest.TestCase):
    """Test convenience functions"""
    
    def test_predict_operation_performance(self):
        """Test predict_operation_performance convenience function"""
        prediction = predict_operation_performance("test_op", {})
        
        self.assertIsInstance(prediction, PerformancePrediction)
        self.assertEqual(prediction.operation_name, "test_op")
    
    def test_get_optimization_recommendations(self):
        """Test get_optimization_recommendations convenience function"""
        recommendations = get_optimization_recommendations("test_op", {})
        
        self.assertIsInstance(recommendations, list)
        for rec in recommendations:
            self.assertIsInstance(rec, OptimizationRecommendation)

class TestIntegration(unittest.TestCase):
    """Test integration scenarios"""
    
    def setUp(self):
        self.optimizer = PredictiveOptimizer()
    
    def test_end_to_end_optimization(self):
        """Test complete end-to-end optimization workflow"""
        # Record some performance data
        for i in range(20):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name="test_operation",
                duration=1.0 + (i * 0.05),
                memory_usage=50.0 + (i * 1),
                cpu_usage=25.0 + (i * 0.5),
                error_count=0,
                success_count=1
            )
            self.optimizer.predictor.record_performance(metric)
        
        # Get performance prediction
        prediction = self.optimizer.predictor.predict_performance("test_operation", {})
        
        # Get optimization recommendations
        recommendations = self.optimizer.optimize_performance("test_operation", {})
        
        # Verify results
        self.assertIsInstance(prediction, PerformancePrediction)
        self.assertGreater(prediction.confidence, 0.0)
        self.assertIsInstance(recommendations, list)
        self.assertGreater(len(recommendations), 0)
    
    def test_performance_monitoring(self):
        """Test performance monitoring capabilities"""
        # Simulate performance monitoring
        operation_name = "monitored_operation"
        
        # Record performance over time
        for i in range(15):
            metric = PerformanceMetric(
                timestamp=time.time() + i,
                operation_name=operation_name,
                duration=1.0 + (i * 0.1),
                memory_usage=50.0 + (i * 2),
                cpu_usage=25.0 + (i * 1),
                error_count=0,
                success_count=1
            )
            self.optimizer.predictor.record_performance(metric)
        
        # Get insights
        insights = self.optimizer.get_performance_insights(operation_name)
        
        # Verify insights
        self.assertIsInstance(insights, dict)
        self.assertEqual(insights["operation_name"], operation_name)
        self.assertGreater(insights["data_points"], 0)
        self.assertIn("trends", insights)
        self.assertIn("anomalies", insights)

if __name__ == '__main__':
    unittest.main() 