#!/usr/bin/env python3
"""
Unit tests for Syntax Intelligence (g8.3)
Tests the AI-powered syntax intelligence system
"""

import unittest
import sys
import os
import time
import json
from unittest.mock import Mock, patch

# Add parent directory to path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..'))

from syntax_intelligence import (
    AISyntaxIntelligence,
    TuskLangSyntaxAnalyzer,
    SuggestionEngine,
    ErrorDetector,
    ContextAnalyzer,
    PatternRecognizer,
    SyntaxSuggestion,
    SyntaxError,
    CodeContext,
    SuggestionType,
    SyntaxErrorType,
    analyze_code_syntax,
    validate_code_syntax,
    get_autocomplete_suggestions
)

class TestSyntaxSuggestion(unittest.TestCase):
    """Test the SyntaxSuggestion dataclass"""
    
    def test_creation(self):
        """Test SyntaxSuggestion creation"""
        suggestion = SyntaxSuggestion(
            suggestion_type=SuggestionType.AUTOCOMPLETE,
            suggestion="@operator(\"test\")",
            confidence=0.9,
            description="Complete operator decorator",
            replacement_range=(0, 10),
            context={"line": 1}
        )
        
        self.assertEqual(suggestion.suggestion_type, SuggestionType.AUTOCOMPLETE)
        self.assertEqual(suggestion.suggestion, "@operator(\"test\")")
        self.assertEqual(suggestion.confidence, 0.9)
        self.assertEqual(suggestion.description, "Complete operator decorator")
        self.assertEqual(suggestion.replacement_range, (0, 10))
        self.assertEqual(suggestion.context["line"], 1)

class TestSyntaxError(unittest.TestCase):
    """Test the SyntaxError dataclass"""
    
    def test_creation(self):
        """Test SyntaxError creation"""
        error = SyntaxError(
            error_type=SyntaxErrorType.MISSING_OPERATOR,
            message="Function should have @operator decorator",
            line_number=5,
            column=0,
            severity="error",
            suggestions=["Add @operator decorator"],
            context={"function_name": "test_func"}
        )
        
        self.assertEqual(error.error_type, SyntaxErrorType.MISSING_OPERATOR)
        self.assertEqual(error.message, "Function should have @operator decorator")
        self.assertEqual(error.line_number, 5)
        self.assertEqual(error.column, 0)
        self.assertEqual(error.severity, "error")
        self.assertEqual(len(error.suggestions), 1)
        self.assertEqual(error.context["function_name"], "test_func")

class TestCodeContext(unittest.TestCase):
    """Test the CodeContext dataclass"""
    
    def test_creation(self):
        """Test CodeContext creation"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=["import logging", "from typing import Dict, Any"],
            following_lines=["def test_operator(context: Dict[str, Any]) -> Any:"],
            variables={"context": "Dict[str, Any]"},
            functions=["test_operator"],
            operators=[],
            imports=["logging", "typing"]
        )
        
        self.assertEqual(context.current_line, "@operator(")
        self.assertEqual(context.cursor_position, 10)
        self.assertEqual(len(context.previous_lines), 2)
        self.assertEqual(len(context.following_lines), 1)
        self.assertEqual(len(context.variables), 1)
        self.assertEqual(len(context.functions), 1)
        self.assertEqual(len(context.operators), 0)
        self.assertEqual(len(context.imports), 2)

class TestTuskLangSyntaxAnalyzer(unittest.TestCase):
    """Test the TuskLang syntax analyzer"""
    
    def setUp(self):
        self.analyzer = TuskLangSyntaxAnalyzer()
    
    def test_load_operator_patterns(self):
        """Test operator pattern loading"""
        self.assertIn("math", self.analyzer.operator_patterns)
        self.assertIn("string", self.analyzer.operator_patterns)
        self.assertIn("list", self.analyzer.operator_patterns)
        self.assertIn("dict", self.analyzer.operator_patterns)
        self.assertIn("conditional", self.analyzer.operator_patterns)
        self.assertIn("loop", self.analyzer.operator_patterns)
        self.assertIn("validation", self.analyzer.operator_patterns)
        self.assertIn("transformation", self.analyzer.operator_patterns)
    
    def test_load_syntax_rules(self):
        """Test syntax rule loading"""
        self.assertIn("operator_declaration", self.analyzer.syntax_rules)
        self.assertIn("function_declaration", self.analyzer.syntax_rules)
        self.assertIn("variable_assignment", self.analyzer.syntax_rules)
        self.assertIn("import_statement", self.analyzer.syntax_rules)
    
    def test_load_common_patterns(self):
        """Test common pattern loading"""
        self.assertIn("data_processing", self.analyzer.common_patterns)
        self.assertIn("api_integration", self.analyzer.common_patterns)
        self.assertIn("file_operations", self.analyzer.common_patterns)
        self.assertIn("database_operations", self.analyzer.common_patterns)
        self.assertIn("security", self.analyzer.common_patterns)
        self.assertIn("performance", self.analyzer.common_patterns)
    
    def test_load_error_patterns(self):
        """Test error pattern loading"""
        self.assertIn("missing_operator", self.analyzer.error_patterns)
        self.assertIn("invalid_syntax", self.analyzer.error_patterns)
        self.assertIn("missing_parameter", self.analyzer.error_patterns)
        self.assertIn("type_mismatch", self.analyzer.error_patterns)
        self.assertIn("undefined_variable", self.analyzer.error_patterns)
        self.assertIn("duplicate_definition", self.analyzer.error_patterns)

class TestAISyntaxIntelligence(unittest.TestCase):
    """Test the AI syntax intelligence system"""
    
    def setUp(self):
        self.intelligence = AISyntaxIntelligence()
    
    def test_initialization(self):
        """Test intelligence system initialization"""
        self.assertIsNotNone(self.intelligence.syntax_analyzer)
        self.assertIsNotNone(self.intelligence.suggestion_engine)
        self.assertIsNotNone(self.intelligence.error_detector)
        self.assertIsNotNone(self.intelligence.context_analyzer)
        self.assertIsNotNone(self.intelligence.pattern_recognizer)
        self.assertIsNotNone(self.intelligence.usage_patterns)
        self.assertIsNotNone(self.intelligence.suggestion_history)
        self.assertIsNotNone(self.intelligence.error_history)
    
    def test_analyze_syntax(self):
        """Test syntax analysis"""
        test_code = """
@operator("add_numbers")
def add_numbers_operator(context: Dict[str, Any]) -> Any:
    \"\"\"
    Add two numbers from context
    \"\"\"
    try:
        result = context.get('value', 0) + context.get('addend', 0)
        return result
    except Exception as e:
        logger.error(f"Error in add_numbers_operator: {e}")
        raise
"""
        
        suggestions = self.intelligence.analyze_syntax(test_code)
        
        self.assertIsInstance(suggestions, list)
        # Should provide some suggestions for valid code
        self.assertGreaterEqual(len(suggestions), 0)
    
    def test_analyze_syntax_with_context(self):
        """Test syntax analysis with context"""
        test_code = "def test_function():"
        context = CodeContext(
            current_line="def test_function():",
            cursor_position=20,
            previous_lines=["import logging"],
            following_lines=["    return True"],
            variables={},
            functions=[],
            operators=[],
            imports=["logging"]
        )
        
        suggestions = self.intelligence.analyze_syntax(test_code, context)
        
        self.assertIsInstance(suggestions, list)
    
    def test_validate_syntax(self):
        """Test syntax validation"""
        test_code = """
@operator("test_operator")
def test_operator(context):
    return context.get('value', 0)
"""
        
        errors = self.intelligence.validate_syntax(test_code)
        
        self.assertIsInstance(errors, list)
        # Valid code should have no errors
        self.assertEqual(len(errors), 0)
    
    def test_validate_syntax_with_errors(self):
        """Test syntax validation with errors"""
        test_code = """
def test_function():
    return value  # undefined variable
"""
        
        errors = self.intelligence.validate_syntax(test_code)
        
        self.assertIsInstance(errors, list)
        # Should detect some errors
        self.assertGreaterEqual(len(errors), 0)
    
    def test_get_autocomplete(self):
        """Test auto-completion"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=["import logging", "from typing import Dict, Any"],
            following_lines=["def test_operator(context: Dict[str, Any]) -> Any:"],
            variables={"context": "Dict[str, Any]"},
            functions=["test_operator"],
            operators=[],
            imports=["logging", "typing"]
        )
        
        suggestions = self.intelligence.get_autocomplete(context)
        
        self.assertIsInstance(suggestions, list)
        self.assertGreaterEqual(len(suggestions), 0)
    
    def test_extract_context_from_code(self):
        """Test context extraction from code"""
        test_code = """
import logging
from typing import Dict, Any

@operator("test_operator")
def test_operator(context: Dict[str, Any]) -> Any:
    value = context.get('value', 0)
    return value
"""
        
        context = self.intelligence._extract_context_from_code(test_code)
        
        self.assertIsInstance(context, dict)
        self.assertIn("variables", context)
        self.assertIn("functions", context)
        self.assertIn("operators", context)
        self.assertIn("imports", context)
        self.assertIn("patterns", context)
    
    def test_record_usage_pattern(self):
        """Test usage pattern recording"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=[],
            following_lines=[],
            variables={},
            functions=[],
            operators=[],
            imports=[]
        )
        
        suggestions = [
            SyntaxSuggestion(
                suggestion_type=SuggestionType.AUTOCOMPLETE,
                suggestion="@operator(\"test\")",
                confidence=0.9,
                description="Test suggestion"
            )
        ]
        
        # Should not raise an exception
        self.intelligence._record_usage_pattern(context, suggestions)

class TestSuggestionEngine(unittest.TestCase):
    """Test the suggestion engine"""
    
    def setUp(self):
        self.engine = SuggestionEngine()
    
    def test_initialization(self):
        """Test engine initialization"""
        self.assertIsNotNone(self.engine.tusk_operators)
        self.assertIsNotNone(self.engine.common_suggestions)
        self.assertIsNotNone(self.engine.pattern_suggestions)
    
    def test_load_tusk_operators(self):
        """Test TuskLang operator loading"""
        operators = self.engine.tusk_operators
        self.assertIn("add", operators)
        self.assertIn("subtract", operators)
        self.assertIn("multiply", operators)
        self.assertIn("divide", operators)
        self.assertIn("concat", operators)
        self.assertIn("filter", operators)
        self.assertIn("map", operators)
        self.assertIn("reduce", operators)
    
    def test_load_common_suggestions(self):
        """Test common suggestion loading"""
        suggestions = self.engine.common_suggestions
        self.assertIn("imports", suggestions)
        self.assertIn("decorators", suggestions)
        self.assertIn("patterns", suggestions)
    
    def test_load_pattern_suggestions(self):
        """Test pattern suggestion loading"""
        patterns = self.engine.pattern_suggestions
        self.assertIn("data_processing", patterns)
        self.assertIn("error_handling", patterns)
        self.assertIn("logging", patterns)
        self.assertIn("validation", patterns)
        self.assertIn("caching", patterns)
        self.assertIn("async", patterns)
    
    def test_generate_autocomplete_suggestions(self):
        """Test auto-completion suggestion generation"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=[],
            following_lines=[],
            variables={"test_var": "str"},
            functions=["test_func"],
            operators=[],
            imports=["logging"]
        )
        
        context_info = {
            "variables": ["test_var"],
            "functions": ["test_func"],
            "operators": [],
            "imports": ["logging"]
        }
        
        suggestions = self.engine.generate_autocomplete_suggestions(context, context_info)
        
        self.assertIsInstance(suggestions, list)
        self.assertGreaterEqual(len(suggestions), 0)
    
    def test_generate_error_suggestions(self):
        """Test error suggestion generation"""
        error = SyntaxError(
            error_type=SyntaxErrorType.MISSING_OPERATOR,
            message="Function should have @operator decorator",
            line_number=1,
            column=0,
            severity="error"
        )
        
        context_info = {"language": "python"}
        
        suggestions = self.engine.generate_error_suggestions(error, context_info)
        
        self.assertIsInstance(suggestions, list)
        self.assertGreaterEqual(len(suggestions), 0)
    
    def test_generate_optimization_suggestions(self):
        """Test optimization suggestion generation"""
        test_code = """
for i in range(1000):
    result = process_data(i)
"""
        
        context_info = {"language": "python"}
        
        suggestions = self.engine.generate_optimization_suggestions(test_code, context_info)
        
        self.assertIsInstance(suggestions, list)
        self.assertGreaterEqual(len(suggestions), 0)
    
    def test_generate_pattern_suggestions(self):
        """Test pattern suggestion generation"""
        test_code = """
def process_data(data):
    return [item * 2 for item in data]
"""
        
        context_info = {"language": "python"}
        
        suggestions = self.engine.generate_pattern_suggestions(test_code, context_info)
        
        self.assertIsInstance(suggestions, list)
        self.assertGreaterEqual(len(suggestions), 0)
    
    def test_get_current_word(self):
        """Test current word extraction"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=[],
            following_lines=[],
            variables={},
            functions=[],
            operators=[],
            imports=[]
        )
        
        word = self.engine._get_current_word(context)
        self.assertEqual(word, "@operator(")
    
    def test_get_replacement_range(self):
        """Test replacement range calculation"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=[],
            following_lines=[],
            variables={},
            functions=[],
            operators=[],
            imports=[]
        )
        
        word = "@operator("
        range_tuple = self.engine._get_replacement_range(context, word)
        
        self.assertIsInstance(range_tuple, tuple)
        self.assertEqual(len(range_tuple), 2)
        self.assertIsInstance(range_tuple[0], int)
        self.assertIsInstance(range_tuple[1], int)

class TestErrorDetector(unittest.TestCase):
    """Test the error detector"""
    
    def setUp(self):
        self.detector = ErrorDetector()
    
    def test_initialization(self):
        """Test detector initialization"""
        self.assertIsNotNone(self.detector.error_patterns)
    
    def test_load_error_patterns(self):
        """Test error pattern loading"""
        patterns = self.detector.error_patterns
        self.assertIn("missing_operator", patterns)
        self.assertIn("invalid_syntax", patterns)
        self.assertIn("missing_parameter", patterns)
        self.assertIn("type_mismatch", patterns)
        self.assertIn("undefined_variable", patterns)
        self.assertIn("duplicate_definition", patterns)
    
    def test_detect_errors_valid_code(self):
        """Test error detection with valid code"""
        test_code = """
@operator("test_operator")
def test_operator(context: Dict[str, Any]) -> Any:
    return context.get('value', 0)
"""
        
        errors = self.detector.detect_errors(test_code)
        
        self.assertIsInstance(errors, list)
        # Valid code should have no errors
        self.assertEqual(len(errors), 0)
    
    def test_detect_errors_invalid_code(self):
        """Test error detection with invalid code"""
        test_code = """
def test_function():
    return undefined_variable
"""
        
        errors = self.detector.detect_errors(test_code)
        
        self.assertIsInstance(errors, list)
        # Should detect some errors
        self.assertGreaterEqual(len(errors), 0)

class TestContextAnalyzer(unittest.TestCase):
    """Test the context analyzer"""
    
    def setUp(self):
        self.analyzer = ContextAnalyzer()
    
    def test_analyze_context(self):
        """Test context analysis"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=["import logging"],
            following_lines=["def test_operator(context):"],
            variables={"context": "Dict[str, Any]"},
            functions=["test_operator"],
            operators=[],
            imports=["logging"]
        )
        
        analysis = self.analyzer.analyze_context(context)
        
        self.assertIsInstance(analysis, dict)
        self.assertIn("variables", analysis)
        self.assertIn("functions", analysis)
        self.assertIn("operators", analysis)
        self.assertIn("imports", analysis)
        self.assertIn("current_line_type", analysis)
        self.assertIn("suggestions", analysis)
    
    def test_analyze_line_type(self):
        """Test line type analysis"""
        # Test decorator
        line_type = self.analyzer._analyze_line_type("@operator(\"test\")")
        self.assertEqual(line_type, "decorator")
        
        # Test function definition
        line_type = self.analyzer._analyze_line_type("def test_function():")
        self.assertEqual(line_type, "function_definition")
        
        # Test import
        line_type = self.analyzer._analyze_line_type("import logging")
        self.assertEqual(line_type, "import")
        
        # Test assignment
        line_type = self.analyzer._analyze_line_type("value = 42")
        self.assertEqual(line_type, "assignment")
        
        # Test return
        line_type = self.analyzer._analyze_line_type("return result")
        self.assertEqual(line_type, "return")
        
        # Test conditional
        line_type = self.analyzer._analyze_line_type("if condition:")
        self.assertEqual(line_type, "conditional")
        
        # Test loop
        line_type = self.analyzer._analyze_line_type("for item in items:")
        self.assertEqual(line_type, "loop")
        
        # Test statement
        line_type = self.analyzer._analyze_line_type("pass")
        self.assertEqual(line_type, "statement")
    
    def test_generate_context_suggestions(self):
        """Test context suggestion generation"""
        context = CodeContext(
            current_line="def test_function():",
            cursor_position=20,
            previous_lines=[],
            following_lines=[],
            variables={},
            functions=[],
            operators=[],
            imports=[]
        )
        
        suggestions = self.analyzer._generate_context_suggestions(context)
        
        self.assertIsInstance(suggestions, list)
        self.assertGreaterEqual(len(suggestions), 0)

class TestPatternRecognizer(unittest.TestCase):
    """Test the pattern recognizer"""
    
    def setUp(self):
        self.recognizer = PatternRecognizer()
    
    def test_initialization(self):
        """Test recognizer initialization"""
        self.assertIsNotNone(self.recognizer.patterns)
    
    def test_load_patterns(self):
        """Test pattern loading"""
        patterns = self.recognizer.patterns
        self.assertIn("data_processing", patterns)
        self.assertIn("error_handling", patterns)
        self.assertIn("logging", patterns)
        self.assertIn("validation", patterns)
        self.assertIn("caching", patterns)
        self.assertIn("async", patterns)
    
    def test_recognize_patterns(self):
        """Test pattern recognition"""
        test_code = """
def process_data(data):
    try:
        result = [item * 2 for item in data]
        logger.info("Data processed successfully")
        return result
    except Exception as e:
        logger.error(f"Error processing data: {e}")
        raise
"""
        
        patterns = self.recognizer.recognize_patterns(test_code)
        
        self.assertIsInstance(patterns, list)
        self.assertGreaterEqual(len(patterns), 0)

class TestConvenienceFunctions(unittest.TestCase):
    """Test convenience functions"""
    
    def test_analyze_code_syntax(self):
        """Test analyze_code_syntax convenience function"""
        test_code = """
@operator("test_operator")
def test_operator(context):
    return context.get('value', 0)
"""
        
        suggestions = analyze_code_syntax(test_code)
        
        self.assertIsInstance(suggestions, list)
    
    def test_validate_code_syntax(self):
        """Test validate_code_syntax convenience function"""
        test_code = """
def test_function():
    return True
"""
        
        errors = validate_code_syntax(test_code)
        
        self.assertIsInstance(errors, list)
    
    def test_get_autocomplete_suggestions(self):
        """Test get_autocomplete_suggestions convenience function"""
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=[],
            following_lines=[],
            variables={},
            functions=[],
            operators=[],
            imports=[]
        )
        
        suggestions = get_autocomplete_suggestions(context)
        
        self.assertIsInstance(suggestions, list)

class TestIntegration(unittest.TestCase):
    """Test integration scenarios"""
    
    def setUp(self):
        self.intelligence = AISyntaxIntelligence()
    
    def test_end_to_end_syntax_analysis(self):
        """Test complete end-to-end syntax analysis"""
        test_code = """
import logging
from typing import Dict, Any

@operator("data_processor")
def data_processor(context: Dict[str, Any]) -> Any:
    \"\"\"
    Process data from context
    \"\"\"
    try:
        data = context.get('data', [])
        result = [item * 2 for item in data]
        logger.info(f"Processed {len(result)} items")
        return result
    except Exception as e:
        logger.error(f"Error processing data: {e}")
        raise
"""
        
        # Analyze syntax
        suggestions = self.intelligence.analyze_syntax(test_code)
        
        # Validate syntax
        errors = self.intelligence.validate_syntax(test_code)
        
        # Test auto-completion
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=["import logging"],
            following_lines=["def test_operator(context):"],
            variables={"context": "Dict[str, Any]"},
            functions=["data_processor"],
            operators=[],
            imports=["logging", "typing"]
        )
        
        autocomplete_suggestions = self.intelligence.get_autocomplete(context)
        
        # Verify results
        self.assertIsInstance(suggestions, list)
        self.assertIsInstance(errors, list)
        self.assertIsInstance(autocomplete_suggestions, list)
        
        # Valid code should have no errors
        self.assertEqual(len(errors), 0)
    
    def test_error_detection_and_correction(self):
        """Test error detection and correction workflow"""
        # Code with potential errors
        test_code = """
def test_function():
    return undefined_variable
"""
        
        # Detect errors
        errors = self.intelligence.validate_syntax(test_code)
        
        # Generate suggestions for each error
        all_suggestions = []
        for error in errors:
            error_suggestions = self.intelligence.suggestion_engine.generate_error_suggestions(
                error, {"language": "python"}
            )
            all_suggestions.extend(error_suggestions)
        
        # Verify results
        self.assertIsInstance(errors, list)
        self.assertIsInstance(all_suggestions, list)
        self.assertGreaterEqual(len(errors), 0)
    
    def test_autocomplete_workflow(self):
        """Test auto-completion workflow"""
        # Simulate typing an operator
        context = CodeContext(
            current_line="@operator(",
            cursor_position=10,
            previous_lines=["import logging", "from typing import Dict, Any"],
            following_lines=["def test_operator(context: Dict[str, Any]) -> Any:"],
            variables={"context": "Dict[str, Any]"},
            functions=["test_operator"],
            operators=[],
            imports=["logging", "typing"]
        )
        
        # Get auto-completion suggestions
        suggestions = self.intelligence.get_autocomplete(context)
        
        # Filter by type
        autocomplete_suggestions = [s for s in suggestions if s.suggestion_type == SuggestionType.AUTOCOMPLETE]
        
        # Verify results
        self.assertIsInstance(suggestions, list)
        self.assertGreaterEqual(len(suggestions), 0)
        self.assertGreaterEqual(len(autocomplete_suggestions), 0)

if __name__ == '__main__':
    unittest.main() 