#!/usr/bin/env python3
"""
Performance Optimizer for TuskLang Python SDK
=============================================
Advanced performance optimization and caching system

This module provides high-performance optimizations for the TuskLang Python SDK,
including intelligent caching, memory management, and execution optimization.
"""

import time
import threading
import weakref
import gc
import psutil
import hashlib
import json
from typing import Any, Dict, List, Optional, Callable, Union
from datetime import datetime, timedelta
from collections import OrderedDict, defaultdict
from functools import wraps
import logging
import asyncio
from dataclasses import dataclass
from enum import Enum


class CacheStrategy(Enum):
    """Cache strategy enumeration"""
    LRU = "lru"
    LFU = "lfu"
    TTL = "ttl"
    ADAPTIVE = "adaptive"


@dataclass
class CacheEntry:
    """Cache entry structure"""
    key: str
    value: Any
    created_at: datetime
    accessed_at: datetime
    access_count: int
    size: int
    ttl: Optional[timedelta] = None


class PerformanceOptimizer:
    """High-performance optimization system for TuskLang"""
    
    def __init__(self, config: Dict[str, Any] = None):
        self.config = config or {}
        self.caches = {}
        self.memory_monitor = MemoryMonitor()
        self.execution_tracker = ExecutionTracker()
        self.optimization_rules = OptimizationRules()
        self.logger = logging.getLogger('tusklang.performance')
        
        # Initialize caches
        self._init_caches()
        
        # Start background optimization
        self.optimization_active = True
        self.optimization_thread = threading.Thread(target=self._optimization_loop, daemon=True)
        self.optimization_thread.start()
    
    def _init_caches(self):
        """Initialize different cache types"""
        cache_configs = {
            "parsing": {
                "strategy": CacheStrategy.LRU,
                "max_size": 1000,
                "max_memory_mb": 50
            },
            "operators": {
                "strategy": CacheStrategy.LFU,
                "max_size": 500,
                "max_memory_mb": 25
            },
            "results": {
                "strategy": CacheStrategy.TTL,
                "max_size": 2000,
                "max_memory_mb": 100,
                "default_ttl": 300  # 5 minutes
            },
            "metadata": {
                "strategy": CacheStrategy.ADAPTIVE,
                "max_size": 100,
                "max_memory_mb": 10
            }
        }
        
        for cache_name, config in cache_configs.items():
            self.caches[cache_name] = self._create_cache(config)
    
    def _create_cache(self, config: Dict[str, Any]):
        """Create cache based on strategy"""
        strategy = config["strategy"]
        
        if strategy == CacheStrategy.LRU:
            return LRUCache(config)
        elif strategy == CacheStrategy.LFU:
            return LFUCache(config)
        elif strategy == CacheStrategy.TTL:
            return TTLCache(config)
        elif strategy == CacheStrategy.ADAPTIVE:
            return AdaptiveCache(config)
        else:
            return LRUCache(config)
    
    def get_cache(self, cache_name: str):
        """Get cache by name"""
        return self.caches.get(cache_name)
    
    def cache_get(self, cache_name: str, key: str) -> Optional[Any]:
        """Get value from cache"""
        cache = self.get_cache(cache_name)
        if cache:
            return cache.get(key)
        return None
    
    def cache_set(self, cache_name: str, key: str, value: Any, ttl: Optional[int] = None):
        """Set value in cache"""
        cache = self.get_cache(cache_name)
        if cache:
            cache.set(key, value, ttl)
    
    def cache_invalidate(self, cache_name: str, pattern: str = None):
        """Invalidate cache entries"""
        cache = self.get_cache(cache_name)
        if cache:
            cache.invalidate(pattern)
    
    def optimize_parsing(self, content: str) -> str:
        """Optimize parsing performance"""
        # Pre-process content for faster parsing
        optimized = content
        
        # Remove unnecessary whitespace
        optimized = re.sub(r'\s+', ' ', optimized)
        
        # Pre-compile common patterns
        optimized = self._optimize_patterns(optimized)
        
        return optimized
    
    def _optimize_patterns(self, content: str) -> str:
        """Optimize common patterns for faster parsing"""
        # Optimize array patterns
        content = re.sub(r'\[\s*\]', '[]', content)
        content = re.sub(r'\[\s*([^,\]]+)\s*\]', r'[\1]', content)
        
        # Optimize object patterns
        content = re.sub(r'{\s*}', '{}', content)
        content = re.sub(r'{\s*([^,}]+)\s*}', r'{\1}', content)
        
        # Optimize variable references
        content = re.sub(r'\$\s*([a-zA-Z_][a-zA-Z0-9_]*)', r'$\1', content)
        
        return content
    
    def track_execution(self, operation: str, func: Callable):
        """Track execution performance"""
        return self.execution_tracker.track(operation, func)
    
    def get_performance_stats(self) -> Dict[str, Any]:
        """Get comprehensive performance statistics"""
        stats = {
            "timestamp": datetime.now().isoformat(),
            "memory": self.memory_monitor.get_stats(),
            "execution": self.execution_tracker.get_stats(),
            "caches": {}
        }
        
        for cache_name, cache in self.caches.items():
            stats["caches"][cache_name] = cache.get_stats()
        
        return stats
    
    def _optimization_loop(self):
        """Background optimization loop"""
        while self.optimization_active:
            try:
                # Memory optimization
                self.memory_monitor.optimize()
                
                # Cache optimization
                for cache in self.caches.values():
                    cache.optimize()
                
                # Execution optimization
                self.execution_tracker.optimize()
                
                # Sleep for optimization interval
                time.sleep(30)  # Optimize every 30 seconds
                
            except Exception as e:
                self.logger.error(f"Optimization loop error: {e}")
                time.sleep(60)  # Wait longer on error


class BaseCache:
    """Base cache implementation"""
    
    def __init__(self, config: Dict[str, Any]):
        self.config = config
        self.max_size = config.get("max_size", 1000)
        self.max_memory_mb = config.get("max_memory_mb", 50)
        self.entries = {}
        self.stats = {
            "hits": 0,
            "misses": 0,
            "sets": 0,
            "evictions": 0
        }
    
    def get(self, key: str) -> Optional[Any]:
        """Get value from cache"""
        if key in self.entries:
            entry = self.entries[key]
            if self._is_valid(entry):
                self._update_access(entry)
                self.stats["hits"] += 1
                return entry.value
            else:
                del self.entries[key]
        
        self.stats["misses"] += 1
        return None
    
    def set(self, key: str, value: Any, ttl: Optional[int] = None):
        """Set value in cache"""
        size = self._calculate_size(value)
        
        # Check if we need to evict entries
        while len(self.entries) >= self.max_size or self._get_memory_usage() > self.max_memory_mb:
            self._evict_entry()
        
        # Create cache entry
        entry = CacheEntry(
            key=key,
            value=value,
            created_at=datetime.now(),
            accessed_at=datetime.now(),
            access_count=0,
            size=size,
            ttl=timedelta(seconds=ttl) if ttl else None
        )
        
        self.entries[key] = entry
        self.stats["sets"] += 1
    
    def invalidate(self, pattern: str = None):
        """Invalidate cache entries"""
        if pattern:
            # Pattern-based invalidation
            import re
            regex = re.compile(pattern)
            keys_to_remove = [key for key in self.entries.keys() if regex.match(key)]
        else:
            # Clear all entries
            keys_to_remove = list(self.entries.keys())
        
        for key in keys_to_remove:
            del self.entries[key]
    
    def get_stats(self) -> Dict[str, Any]:
        """Get cache statistics"""
        hit_rate = 0
        if self.stats["hits"] + self.stats["misses"] > 0:
            hit_rate = self.stats["hits"] / (self.stats["hits"] + self.stats["misses"])
        
        return {
            "size": len(self.entries),
            "max_size": self.max_size,
            "memory_usage_mb": self._get_memory_usage(),
            "max_memory_mb": self.max_memory_mb,
            "hit_rate": hit_rate,
            "hits": self.stats["hits"],
            "misses": self.stats["misses"],
            "sets": self.stats["sets"],
            "evictions": self.stats["evictions"]
        }
    
    def optimize(self):
        """Optimize cache performance"""
        # Remove expired entries
        expired_keys = []
        for key, entry in self.entries.items():
            if not self._is_valid(entry):
                expired_keys.append(key)
        
        for key in expired_keys:
            del self.entries[key]
    
    def _is_valid(self, entry: CacheEntry) -> bool:
        """Check if cache entry is valid"""
        if entry.ttl:
            return datetime.now() - entry.created_at < entry.ttl
        return True
    
    def _update_access(self, entry: CacheEntry):
        """Update access statistics"""
        entry.accessed_at = datetime.now()
        entry.access_count += 1
    
    def _calculate_size(self, value: Any) -> int:
        """Calculate approximate size of value in bytes"""
        try:
            return len(json.dumps(value, default=str))
        except:
            return 100  # Default size
    
    def _get_memory_usage(self) -> float:
        """Get current memory usage in MB"""
        total_size = sum(entry.size for entry in self.entries.values())
        return total_size / 1024 / 1024  # Convert to MB
    
    def _evict_entry(self):
        """Evict an entry from cache (to be implemented by subclasses)"""
        pass


class LRUCache(BaseCache):
    """Least Recently Used cache"""
    
    def __init__(self, config: Dict[str, Any]):
        super().__init__(config)
        self.access_order = []
    
    def _update_access(self, entry: CacheEntry):
        """Update access order for LRU"""
        super()._update_access(entry)
        
        # Move to end of access order
        if entry.key in self.access_order:
            self.access_order.remove(entry.key)
        self.access_order.append(entry.key)
    
    def _evict_entry(self):
        """Evict least recently used entry"""
        if self.access_order:
            key_to_evict = self.access_order[0]
            del self.entries[key_to_evict]
            self.access_order.pop(0)
            self.stats["evictions"] += 1


class LFUCache(BaseCache):
    """Least Frequently Used cache"""
    
    def __init__(self, config: Dict[str, Any]):
        super().__init__(config)
        self.frequency_map = defaultdict(list)
    
    def _update_access(self, entry: CacheEntry):
        """Update frequency for LFU"""
        old_count = entry.access_count
        super()._update_access(entry)
        new_count = entry.access_count
        
        # Update frequency map
        if old_count > 0:
            self.frequency_map[old_count].remove(entry.key)
            if not self.frequency_map[old_count]:
                del self.frequency_map[old_count]
        
        self.frequency_map[new_count].append(entry.key)
    
    def _evict_entry(self):
        """Evict least frequently used entry"""
        if self.frequency_map:
            min_freq = min(self.frequency_map.keys())
            key_to_evict = self.frequency_map[min_freq][0]
            del self.entries[key_to_evict]
            self.frequency_map[min_freq].pop(0)
            
            if not self.frequency_map[min_freq]:
                del self.frequency_map[min_freq]
            
            self.stats["evictions"] += 1


class TTLCache(BaseCache):
    """Time To Live cache"""
    
    def __init__(self, config: Dict[str, Any]):
        super().__init__(config)
        self.default_ttl = config.get("default_ttl", 300)  # 5 minutes
    
    def set(self, key: str, value: Any, ttl: Optional[int] = None):
        """Set value with TTL"""
        if ttl is None:
            ttl = self.default_ttl
        
        super().set(key, value, ttl)


class AdaptiveCache(BaseCache):
    """Adaptive cache that switches between strategies"""
    
    def __init__(self, config: Dict[str, Any]):
        super().__init__(config)
        self.current_strategy = "lru"
        self.strategy_performance = {
            "lru": {"hit_rate": 0.0, "switches": 0},
            "lfu": {"hit_rate": 0.0, "switches": 0}
        }
        self.switch_threshold = 0.1  # 10% improvement threshold
    
    def optimize(self):
        """Adaptive optimization"""
        super().optimize()
        
        # Calculate current performance
        current_hit_rate = self.get_stats()["hit_rate"]
        
        # Update strategy performance
        self.strategy_performance[self.current_strategy]["hit_rate"] = current_hit_rate
        
        # Consider switching strategies
        if self._should_switch_strategy():
            self._switch_strategy()
    
    def _should_switch_strategy(self) -> bool:
        """Determine if we should switch strategies"""
        current_perf = self.strategy_performance[self.current_strategy]["hit_rate"]
        
        for strategy, perf in self.strategy_performance.items():
            if strategy != self.current_strategy:
                if perf["hit_rate"] > current_perf + self.switch_threshold:
                    return True
        
        return False
    
    def _switch_strategy(self):
        """Switch to better performing strategy"""
        best_strategy = max(
            self.strategy_performance.keys(),
            key=lambda s: self.strategy_performance[s]["hit_rate"]
        )
        
        if best_strategy != self.current_strategy:
            self.current_strategy = best_strategy
            self.strategy_performance[best_strategy]["switches"] += 1


class MemoryMonitor:
    """Memory usage monitoring and optimization"""
    
    def __init__(self):
        self.memory_threshold = 80  # 80% memory usage threshold
        self.optimization_history = []
    
    def get_stats(self) -> Dict[str, Any]:
        """Get memory statistics"""
        try:
            memory = psutil.virtual_memory()
            return {
                "total_mb": memory.total / 1024 / 1024,
                "available_mb": memory.available / 1024 / 1024,
                "used_mb": memory.used / 1024 / 1024,
                "percent": memory.percent,
                "threshold": self.memory_threshold
            }
        except:
            return {"error": "Memory stats unavailable"}
    
    def optimize(self):
        """Perform memory optimization"""
        stats = self.get_stats()
        
        if "error" in stats:
            return
        
        if stats["percent"] > self.memory_threshold:
            # Trigger garbage collection
            collected = gc.collect()
            
            # Record optimization
            self.optimization_history.append({
                "timestamp": datetime.now().isoformat(),
                "memory_percent": stats["percent"],
                "collected_objects": collected,
                "action": "garbage_collection"
            })
            
            # Keep only recent history
            if len(self.optimization_history) > 100:
                self.optimization_history = self.optimization_history[-50:]


class ExecutionTracker:
    """Execution performance tracking and optimization"""
    
    def __init__(self):
        self.execution_history = defaultdict(list)
        self.performance_threshold = 1.0  # 1 second threshold
        self.slow_operations = []
    
    def track(self, operation: str, func: Callable):
        """Track execution of a function"""
        start_time = time.time()
        
        try:
            result = func()
            execution_time = time.time() - start_time
            
            # Record execution
            self.execution_history[operation].append({
                "timestamp": datetime.now().isoformat(),
                "execution_time": execution_time,
                "success": True
            })
            
            # Check for slow operations
            if execution_time > self.performance_threshold:
                self.slow_operations.append({
                    "operation": operation,
                    "execution_time": execution_time,
                    "timestamp": datetime.now().isoformat()
                })
            
            return result
            
        except Exception as e:
            execution_time = time.time() - start_time
            
            self.execution_history[operation].append({
                "timestamp": datetime.now().isoformat(),
                "execution_time": execution_time,
                "success": False,
                "error": str(e)
            })
            
            raise
    
    def get_stats(self) -> Dict[str, Any]:
        """Get execution statistics"""
        stats = {
            "operations": {},
            "slow_operations": len(self.slow_operations),
            "total_operations": sum(len(history) for history in self.execution_history.values())
        }
        
        for operation, history in self.execution_history.items():
            if history:
                successful = [h for h in history if h["success"]]
                failed = [h for h in history if not h["success"]]
                
                avg_time = sum(h["execution_time"] for h in successful) / len(successful) if successful else 0
                max_time = max(h["execution_time"] for h in successful) if successful else 0
                
                stats["operations"][operation] = {
                    "total_executions": len(history),
                    "successful": len(successful),
                    "failed": len(failed),
                    "success_rate": len(successful) / len(history) if history else 0,
                    "avg_execution_time": avg_time,
                    "max_execution_time": max_time
                }
        
        return stats
    
    def optimize(self):
        """Optimize execution performance"""
        # Analyze slow operations
        if self.slow_operations:
            # Group by operation type
            slow_by_operation = defaultdict(list)
            for op in self.slow_operations:
                slow_by_operation[op["operation"]].append(op["execution_time"])
            
            # Generate optimization recommendations
            recommendations = []
            for operation, times in slow_by_operation.items():
                avg_time = sum(times) / len(times)
                if avg_time > self.performance_threshold:
                    recommendations.append({
                        "operation": operation,
                        "avg_time": avg_time,
                        "suggestion": "Consider caching or optimization"
                    })
            
            # Clear old slow operations
            self.slow_operations = self.slow_operations[-100:]


class OptimizationRules:
    """Optimization rules and recommendations"""
    
    def __init__(self):
        self.rules = {
            "parsing": [
                "Use pre-compiled regex patterns",
                "Minimize string operations",
                "Cache parsed results"
            ],
            "operators": [
                "Cache expensive operator results",
                "Use lazy evaluation where possible",
                "Optimize data structures"
            ],
            "memory": [
                "Monitor memory usage",
                "Use weak references for large objects",
                "Implement proper cleanup"
            ],
            "execution": [
                "Profile slow operations",
                "Use async operations where appropriate",
                "Implement connection pooling"
            ]
        }
    
    def get_recommendations(self, category: str) -> List[str]:
        """Get optimization recommendations for category"""
        return self.rules.get(category, [])
    
    def add_rule(self, category: str, rule: str):
        """Add new optimization rule"""
        if category not in self.rules:
            self.rules[category] = []
        self.rules[category].append(rule)


# Global performance optimizer instance
performance_optimizer = PerformanceOptimizer()


def optimize_performance(func: Callable):
    """Decorator for performance optimization"""
    @wraps(func)
    def wrapper(*args, **kwargs):
        operation_name = f"{func.__module__}.{func.__name__}"
        return performance_optimizer.track_execution(operation_name, lambda: func(*args, **kwargs))
    return wrapper


def cache_result(cache_name: str, key_func: Callable = None, ttl: Optional[int] = None):
    """Decorator for caching function results"""
    def decorator(func: Callable):
        @wraps(func)
        def wrapper(*args, **kwargs):
            # Generate cache key
            if key_func:
                cache_key = key_func(*args, **kwargs)
            else:
                cache_key = f"{func.__name__}:{hash(str(args) + str(kwargs))}"
            
            # Try to get from cache
            cached_result = performance_optimizer.cache_get(cache_name, cache_key)
            if cached_result is not None:
                return cached_result
            
            # Execute function and cache result
            result = func(*args, **kwargs)
            performance_optimizer.cache_set(cache_name, cache_key, result, ttl)
            
            return result
        return wrapper
    return decorator


# Convenience functions
def get_performance_stats() -> Dict[str, Any]:
    """Get performance statistics"""
    return performance_optimizer.get_performance_stats()


def optimize_parsing(content: str) -> str:
    """Optimize content for parsing"""
    return performance_optimizer.optimize_parsing(content)


def cache_get(cache_name: str, key: str) -> Optional[Any]:
    """Get value from cache"""
    return performance_optimizer.cache_get(cache_name, key)


def cache_set(cache_name: str, key: str, value: Any, ttl: Optional[int] = None):
    """Set value in cache"""
    performance_optimizer.cache_set(cache_name, key, value, ttl)


if __name__ == "__main__":
    print("Performance Optimizer for TuskLang Python SDK")
    print("=" * 50)
    
    # Test caching
    print("\n1. Testing Caching:")
    cache_set("test", "key1", "value1", ttl=60)
    result = cache_get("test", "key1")
    print(f"Cache test: {result}")
    
    # Test performance tracking
    print("\n2. Testing Performance Tracking:")
    
    @optimize_performance
    def test_function():
        time.sleep(0.1)
        return "test result"
    
    result = test_function()
    print(f"Performance tracking: {result}")
    
    # Test optimization
    print("\n3. Testing Optimization:")
    optimized_content = optimize_parsing("  [ 1, 2, 3 ]  ")
    print(f"Optimized content: {optimized_content}")
    
    # Get stats
    print("\n4. Performance Statistics:")
    stats = get_performance_stats()
    print(f"Memory usage: {stats['memory'].get('percent', 0):.1f}%")
    print(f"Cache hits: {stats['caches']['test']['hits']}")
    
    print("\nPerformance optimization testing completed!") 