"""
TuskLang Python SDK - Dynamic Query Builder (g8.2)
Object-oriented query building with type safety and validation
"""

from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Any, Union, Tuple, Callable
from enum import Enum
import re
import json


class JoinType(Enum):
    INNER = "INNER JOIN"
    LEFT = "LEFT JOIN"
    RIGHT = "RIGHT JOIN"
    FULL = "FULL OUTER JOIN"


class ComparisonOperator(Enum):
    EQ = "="
    NE = "!="
    GT = ">"
    GTE = ">="
    LT = "<"
    LTE = "<="
    LIKE = "LIKE"
    ILIKE = "ILIKE"
    IN = "IN"
    NOT_IN = "NOT IN"
    IS_NULL = "IS NULL"
    IS_NOT_NULL = "IS NOT NULL"
    BETWEEN = "BETWEEN"


class LogicalOperator(Enum):
    AND = "AND"
    OR = "OR"
    NOT = "NOT"


@dataclass
class Column:
    """Represents a database column"""
    name: str
    table: Optional[str] = None
    alias: Optional[str] = None
    
    def __str__(self) -> str:
        full_name = f"{self.table}.{self.name}" if self.table else self.name
        return f"{full_name} AS {self.alias}" if self.alias else full_name


@dataclass
class Condition:
    """Represents a WHERE condition"""
    column: str
    operator: ComparisonOperator
    value: Any = None
    logical: LogicalOperator = LogicalOperator.AND
    
    def to_sql(self) -> Tuple[str, List[Any]]:
        """Convert condition to SQL with parameters"""
        if self.operator in [ComparisonOperator.IS_NULL, ComparisonOperator.IS_NOT_NULL]:
            return f"{self.column} {self.operator.value}", []
        elif self.operator == ComparisonOperator.BETWEEN:
            return f"{self.column} BETWEEN %s AND %s", [self.value[0], self.value[1]]
        elif self.operator in [ComparisonOperator.IN, ComparisonOperator.NOT_IN]:
            placeholders = ", ".join(["%s"] * len(self.value))
            return f"{self.column} {self.operator.value} ({placeholders})", list(self.value)
        else:
            return f"{self.column} {self.operator.value} %s", [self.value]


@dataclass
class Join:
    """Represents a table join"""
    table: str
    join_type: JoinType
    on_condition: str
    alias: Optional[str] = None
    
    def to_sql(self) -> str:
        table_ref = f"{self.table} AS {self.alias}" if self.alias else self.table
        return f"{self.join_type.value} {table_ref} ON {self.on_condition}"


@dataclass
class OrderBy:
    """Represents an ORDER BY clause"""
    column: str
    direction: str = "ASC"
    
    def to_sql(self) -> str:
        return f"{self.column} {self.direction}"


@dataclass
class GroupBy:
    """Represents a GROUP BY clause"""
    columns: List[str]
    having: Optional[str] = None
    
    def to_sql(self) -> str:
        sql = f"GROUP BY {', '.join(self.columns)}"
        if self.having:
            sql += f" HAVING {self.having}"
        return sql


class QueryBuilder(ABC):
    """Abstract query builder base class"""
    
    @abstractmethod
    def build(self) -> Tuple[str, List[Any]]:
        """Build the final SQL query with parameters"""
        pass
    
    @abstractmethod
    def reset(self):
        """Reset the query builder"""
        pass


class SelectQueryBuilder(QueryBuilder):
    """SELECT query builder"""
    
    def __init__(self):
        self.reset()
    
    def reset(self):
        """Reset all query components"""
        self._columns: List[Union[str, Column]] = []
        self._from_table: Optional[str] = None
        self._table_alias: Optional[str] = None
        self._joins: List[Join] = []
        self._conditions: List[Condition] = []
        self._group_by: Optional[GroupBy] = None
        self._order_by: List[OrderBy] = []
        self._limit: Optional[int] = None
        self._offset: Optional[int] = None
        self._distinct: bool = False
    
    def select(self, *columns: Union[str, Column]) -> 'SelectQueryBuilder':
        """Add columns to SELECT clause"""
        self._columns.extend(columns)
        return self
    
    def distinct(self) -> 'SelectQueryBuilder':
        """Add DISTINCT to SELECT"""
        self._distinct = True
        return self
    
    def from_table(self, table: str, alias: Optional[str] = None) -> 'SelectQueryBuilder':
        """Set FROM table"""
        self._from_table = table
        self._table_alias = alias
        return self
    
    def join(self, table: str, on_condition: str, join_type: JoinType = JoinType.INNER, 
             alias: Optional[str] = None) -> 'SelectQueryBuilder':
        """Add JOIN clause"""
        self._joins.append(Join(table, join_type, on_condition, alias))
        return self
    
    def inner_join(self, table: str, on_condition: str, alias: Optional[str] = None) -> 'SelectQueryBuilder':
        """Add INNER JOIN"""
        return self.join(table, on_condition, JoinType.INNER, alias)
    
    def left_join(self, table: str, on_condition: str, alias: Optional[str] = None) -> 'SelectQueryBuilder':
        """Add LEFT JOIN"""
        return self.join(table, on_condition, JoinType.LEFT, alias)
    
    def where(self, column: str, operator: ComparisonOperator, value: Any = None) -> 'SelectQueryBuilder':
        """Add WHERE condition"""
        self._conditions.append(Condition(column, operator, value))
        return self
    
    def where_eq(self, column: str, value: Any) -> 'SelectQueryBuilder':
        """Add WHERE column = value"""
        return self.where(column, ComparisonOperator.EQ, value)
    
    def where_in(self, column: str, values: List[Any]) -> 'SelectQueryBuilder':
        """Add WHERE column IN (values)"""
        return self.where(column, ComparisonOperator.IN, values)
    
    def where_like(self, column: str, pattern: str) -> 'SelectQueryBuilder':
        """Add WHERE column LIKE pattern"""
        return self.where(column, ComparisonOperator.LIKE, pattern)
    
    def where_between(self, column: str, start: Any, end: Any) -> 'SelectQueryBuilder':
        """Add WHERE column BETWEEN start AND end"""
        return self.where(column, ComparisonOperator.BETWEEN, [start, end])
    
    def where_null(self, column: str) -> 'SelectQueryBuilder':
        """Add WHERE column IS NULL"""
        return self.where(column, ComparisonOperator.IS_NULL)
    
    def or_where(self, column: str, operator: ComparisonOperator, value: Any = None) -> 'SelectQueryBuilder':
        """Add OR WHERE condition"""
        condition = Condition(column, operator, value, LogicalOperator.OR)
        self._conditions.append(condition)
        return self
    
    def group_by(self, *columns: str) -> 'SelectQueryBuilder':
        """Add GROUP BY clause"""
        self._group_by = GroupBy(list(columns))
        return self
    
    def having(self, condition: str) -> 'SelectQueryBuilder':
        """Add HAVING clause"""
        if self._group_by:
            self._group_by.having = condition
        return self
    
    def order_by(self, column: str, direction: str = "ASC") -> 'SelectQueryBuilder':
        """Add ORDER BY clause"""
        self._order_by.append(OrderBy(column, direction))
        return self
    
    def limit(self, count: int) -> 'SelectQueryBuilder':
        """Add LIMIT clause"""
        self._limit = count
        return self
    
    def offset(self, count: int) -> 'SelectQueryBuilder':
        """Add OFFSET clause"""
        self._offset = count
        return self
    
    def build(self) -> Tuple[str, List[Any]]:
        """Build the final SELECT query"""
        if not self._from_table:
            raise ValueError("FROM table is required")
        
        sql_parts = []
        params = []
        
        # SELECT clause
        if not self._columns:
            columns_str = "*"
        else:
            columns_str = ", ".join([str(col) for col in self._columns])
        
        select_clause = "SELECT DISTINCT" if self._distinct else "SELECT"
        sql_parts.append(f"{select_clause} {columns_str}")
        
        # FROM clause
        from_clause = f"FROM {self._from_table}"
        if self._table_alias:
            from_clause += f" AS {self._table_alias}"
        sql_parts.append(from_clause)
        
        # JOIN clauses
        for join in self._joins:
            sql_parts.append(join.to_sql())
        
        # WHERE clause
        if self._conditions:
            where_parts = []
            for i, condition in enumerate(self._conditions):
                condition_sql, condition_params = condition.to_sql()
                
                if i == 0:
                    where_parts.append(condition_sql)
                else:
                    where_parts.append(f"{condition.logical.value} {condition_sql}")
                
                params.extend(condition_params)
            
            sql_parts.append(f"WHERE {' '.join(where_parts)}")
        
        # GROUP BY clause
        if self._group_by:
            sql_parts.append(self._group_by.to_sql())
        
        # ORDER BY clause
        if self._order_by:
            order_parts = [order.to_sql() for order in self._order_by]
            sql_parts.append(f"ORDER BY {', '.join(order_parts)}")
        
        # LIMIT clause
        if self._limit is not None:
            sql_parts.append(f"LIMIT {self._limit}")
        
        # OFFSET clause
        if self._offset is not None:
            sql_parts.append(f"OFFSET {self._offset}")
        
        return " ".join(sql_parts), params


class InsertQueryBuilder(QueryBuilder):
    """INSERT query builder"""
    
    def __init__(self):
        self.reset()
    
    def reset(self):
        """Reset query components"""
        self._table: Optional[str] = None
        self._columns: List[str] = []
        self._values: List[List[Any]] = []
        self._on_conflict: Optional[str] = None
    
    def into(self, table: str) -> 'InsertQueryBuilder':
        """Set INSERT INTO table"""
        self._table = table
        return self
    
    def columns(self, *columns: str) -> 'InsertQueryBuilder':
        """Set columns for INSERT"""
        self._columns = list(columns)
        return self
    
    def values(self, *values: Any) -> 'InsertQueryBuilder':
        """Add VALUES row"""
        if self._columns and len(values) != len(self._columns):
            raise ValueError("Number of values must match number of columns")
        self._values.append(list(values))
        return self
    
    def values_dict(self, data: Dict[str, Any]) -> 'InsertQueryBuilder':
        """Add VALUES from dictionary"""
        if not self._columns:
            self._columns = list(data.keys())
        
        values = [data.get(col) for col in self._columns]
        return self.values(*values)
    
    def on_conflict_do_nothing(self) -> 'InsertQueryBuilder':
        """Add ON CONFLICT DO NOTHING (PostgreSQL)"""
        self._on_conflict = "ON CONFLICT DO NOTHING"
        return self
    
    def on_duplicate_key_update(self, **updates: Any) -> 'InsertQueryBuilder':
        """Add ON DUPLICATE KEY UPDATE (MySQL)"""
        update_parts = [f"{col} = VALUES({col})" for col in updates.keys()]
        self._on_conflict = f"ON DUPLICATE KEY UPDATE {', '.join(update_parts)}"
        return self
    
    def build(self) -> Tuple[str, List[Any]]:
        """Build INSERT query"""
        if not self._table:
            raise ValueError("Table is required for INSERT")
        if not self._values:
            raise ValueError("VALUES are required for INSERT")
        
        sql_parts = [f"INSERT INTO {self._table}"]
        params = []
        
        # Columns
        if self._columns:
            sql_parts.append(f"({', '.join(self._columns)})")
        
        # Values
        value_placeholders = []
        for row in self._values:
            placeholders = ", ".join(["%s"] * len(row))
            value_placeholders.append(f"({placeholders})")
            params.extend(row)
        
        sql_parts.append(f"VALUES {', '.join(value_placeholders)}")
        
        # Conflict resolution
        if self._on_conflict:
            sql_parts.append(self._on_conflict)
        
        return " ".join(sql_parts), params


class UpdateQueryBuilder(QueryBuilder):
    """UPDATE query builder"""
    
    def __init__(self):
        self.reset()
    
    def reset(self):
        """Reset query components"""
        self._table: Optional[str] = None
        self._updates: Dict[str, Any] = {}
        self._conditions: List[Condition] = []
    
    def table(self, table: str) -> 'UpdateQueryBuilder':
        """Set UPDATE table"""
        self._table = table
        return self
    
    def set(self, column: str, value: Any) -> 'UpdateQueryBuilder':
        """Set column value"""
        self._updates[column] = value
        return self
    
    def set_dict(self, updates: Dict[str, Any]) -> 'UpdateQueryBuilder':
        """Set multiple column values"""
        self._updates.update(updates)
        return self
    
    def where(self, column: str, operator: ComparisonOperator, value: Any = None) -> 'UpdateQueryBuilder':
        """Add WHERE condition"""
        self._conditions.append(Condition(column, operator, value))
        return self
    
    def where_eq(self, column: str, value: Any) -> 'UpdateQueryBuilder':
        """Add WHERE column = value"""
        return self.where(column, ComparisonOperator.EQ, value)
    
    def build(self) -> Tuple[str, List[Any]]:
        """Build UPDATE query"""
        if not self._table:
            raise ValueError("Table is required for UPDATE")
        if not self._updates:
            raise ValueError("SET values are required for UPDATE")
        
        sql_parts = [f"UPDATE {self._table}"]
        params = []
        
        # SET clause
        set_parts = []
        for column, value in self._updates.items():
            set_parts.append(f"{column} = %s")
            params.append(value)
        
        sql_parts.append(f"SET {', '.join(set_parts)}")
        
        # WHERE clause
        if self._conditions:
            where_parts = []
            for i, condition in enumerate(self._conditions):
                condition_sql, condition_params = condition.to_sql()
                
                if i == 0:
                    where_parts.append(condition_sql)
                else:
                    where_parts.append(f"{condition.logical.value} {condition_sql}")
                
                params.extend(condition_params)
            
            sql_parts.append(f"WHERE {' '.join(where_parts)}")
        
        return " ".join(sql_parts), params


class DeleteQueryBuilder(QueryBuilder):
    """DELETE query builder"""
    
    def __init__(self):
        self.reset()
    
    def reset(self):
        """Reset query components"""
        self._table: Optional[str] = None
        self._conditions: List[Condition] = []
    
    def from_table(self, table: str) -> 'DeleteQueryBuilder':
        """Set DELETE FROM table"""
        self._table = table
        return self
    
    def where(self, column: str, operator: ComparisonOperator, value: Any = None) -> 'DeleteQueryBuilder':
        """Add WHERE condition"""
        self._conditions.append(Condition(column, operator, value))
        return self
    
    def where_eq(self, column: str, value: Any) -> 'DeleteQueryBuilder':
        """Add WHERE column = value"""
        return self.where(column, ComparisonOperator.EQ, value)
    
    def build(self) -> Tuple[str, List[Any]]:
        """Build DELETE query"""
        if not self._table:
            raise ValueError("Table is required for DELETE")
        
        sql_parts = [f"DELETE FROM {self._table}"]
        params = []
        
        # WHERE clause
        if self._conditions:
            where_parts = []
            for i, condition in enumerate(self._conditions):
                condition_sql, condition_params = condition.to_sql()
                
                if i == 0:
                    where_parts.append(condition_sql)
                else:
                    where_parts.append(f"{condition.logical.value} {condition_sql}")
                
                params.extend(condition_params)
            
            sql_parts.append(f"WHERE {' '.join(where_parts)}")
        
        return " ".join(sql_parts), params


class QueryBuilderFactory:
    """Factory for creating query builders"""
    
    @staticmethod
    def select() -> SelectQueryBuilder:
        """Create SELECT query builder"""
        return SelectQueryBuilder()
    
    @staticmethod
    def insert() -> InsertQueryBuilder:
        """Create INSERT query builder"""
        return InsertQueryBuilder()
    
    @staticmethod
    def update() -> UpdateQueryBuilder:
        """Create UPDATE query builder"""
        return UpdateQueryBuilder()
    
    @staticmethod
    def delete() -> DeleteQueryBuilder:
        """Create DELETE query builder"""
        return DeleteQueryBuilder()


# Convenience functions
def select(*columns: Union[str, Column]) -> SelectQueryBuilder:
    """Create SELECT query builder"""
    return QueryBuilderFactory.select().select(*columns)


def insert_into(table: str) -> InsertQueryBuilder:
    """Create INSERT query builder"""
    return QueryBuilderFactory.insert().into(table)


def update(table: str) -> UpdateQueryBuilder:
    """Create UPDATE query builder"""
    return QueryBuilderFactory.update().table(table)


def delete_from(table: str) -> DeleteQueryBuilder:
    """Create DELETE query builder"""
    return QueryBuilderFactory.delete().from_table(table)


if __name__ == "__main__":
    # Example usage
    
    # SELECT query
    query, params = (
        select("id", "name", "email")
        .from_table("users", "u")
        .left_join("profiles", "u.id = p.user_id", "p")
        .where_eq("u.active", True)
        .where_like("u.name", "%John%")
        .order_by("u.name")
        .limit(10)
        .build()
    )
    
    print("SELECT Query:")
    print(query)
    print("Parameters:", params)
    print()
    
    # INSERT query
    query, params = (
        insert_into("users")
        .columns("name", "email", "active")
        .values("John Doe", "john@example.com", True)
        .values("Jane Smith", "jane@example.com", True)
        .build()
    )
    
    print("INSERT Query:")
    print(query)
    print("Parameters:", params)
    print()
    
    # UPDATE query
    query, params = (
        update("users")
        .set("name", "John Smith")
        .set("updated_at", "NOW()")
        .where_eq("id", 1)
        .build()
    )
    
    print("UPDATE Query:")
    print(query)
    print("Parameters:", params)
    print()
    
    # DELETE query
    query, params = (
        delete_from("users")
        .where_eq("active", False)
        .where("created_at", ComparisonOperator.LT, "2023-01-01")
        .build()
    )
    
    print("DELETE Query:")
    print(query)
    print("Parameters:", params)
    
    print("\ng8.2: Dynamic Query Builder - COMPLETED ✅") 