"""
TuskLang Python SDK - Reporting Engine (g12.2)
Production automated report generation with PDF/Excel export and templating
"""

import asyncio
import base64
import json
import logging
import os
import tempfile
import uuid
from collections import defaultdict
from dataclasses import dataclass, field, asdict
from datetime import datetime, timedelta
from enum import Enum
from io import BytesIO
from typing import Dict, List, Optional, Set, Any, Callable, Union, Tuple
import jinja2

try:
    from reportlab.lib import colors
    from reportlab.lib.pagesizes import letter, A4
    from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
    from reportlab.lib.units import inch
    from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, Table, TableStyle, PageBreak, Image
    from reportlab.graphics.shapes import Drawing
    from reportlab.graphics.charts.linecharts import HorizontalLineChart
    from reportlab.graphics.charts.barcharts import VerticalBarChart
    from reportlab.graphics.charts.piecharts import Pie
    PDF_AVAILABLE = True
except ImportError:
    PDF_AVAILABLE = False

try:
    import openpyxl
    from openpyxl.chart import LineChart, BarChart, PieChart, Reference
    from openpyxl.styles import Font, PatternFill, Alignment
    EXCEL_AVAILABLE = True
except ImportError:
    EXCEL_AVAILABLE = False

try:
    import matplotlib.pyplot as plt
    import matplotlib.dates as mdates
    import pandas as pd
    PLOT_AVAILABLE = True
except ImportError:
    PLOT_AVAILABLE = False


class ReportFormat(Enum):
    PDF = "pdf"
    EXCEL = "excel"
    HTML = "html"
    JSON = "json"
    CSV = "csv"


class ReportFrequency(Enum):
    ONCE = "once"
    DAILY = "daily"
    WEEKLY = "weekly"
    MONTHLY = "monthly"
    QUARTERLY = "quarterly"
    YEARLY = "yearly"


class ReportStatus(Enum):
    SCHEDULED = "scheduled"
    GENERATING = "generating"
    COMPLETED = "completed"
    FAILED = "failed"
    CANCELLED = "cancelled"


@dataclass
class ReportDataSource:
    """Report data source configuration"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    source_type: str = "analytics"  # analytics, database, api, file
    connection_string: str = ""
    query: str = ""
    parameters: Dict[str, Any] = field(default_factory=dict)
    
    # Data transformation
    filters: List[Dict[str, Any]] = field(default_factory=list)
    aggregations: List[Dict[str, Any]] = field(default_factory=list)
    sort_by: List[Dict[str, str]] = field(default_factory=list)


@dataclass
class ReportSection:
    """Report section definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    section_type: str = "data_table"  # data_table, chart, text, image, kpi_grid
    
    # Data source
    data_source_id: str = ""
    
    # Chart configuration (if section_type is chart)
    chart_type: str = "line"  # line, bar, pie, scatter
    x_axis_field: str = ""
    y_axis_field: str = ""
    series_field: Optional[str] = None
    
    # Styling
    title: str = ""
    description: str = ""
    show_title: bool = True
    
    # Layout
    width: float = 1.0  # Fraction of page width
    height: Optional[float] = None
    page_break_before: bool = False
    page_break_after: bool = False
    
    # Formatting
    number_format: str = "{:.2f}"
    date_format: str = "%Y-%m-%d"
    
    # Table options
    show_headers: bool = True
    show_row_numbers: bool = False
    max_rows: Optional[int] = None
    
    # Text content (if section_type is text)
    content: str = ""
    template_variables: Dict[str, str] = field(default_factory=dict)


@dataclass
class ReportTemplate:
    """Report template definition"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    description: str = ""
    
    # Layout
    page_size: str = "A4"  # A4, letter
    orientation: str = "portrait"  # portrait, landscape
    margins: Dict[str, float] = field(default_factory=lambda: {
        "top": 1.0, "bottom": 1.0, "left": 1.0, "right": 1.0
    })
    
    # Header/Footer
    header_template: str = ""
    footer_template: str = ""
    show_page_numbers: bool = True
    
    # Styling
    title: str = ""
    subtitle: str = ""
    logo_path: Optional[str] = None
    
    # Content
    sections: List[str] = field(default_factory=list)  # Section IDs
    
    # Variables
    global_variables: Dict[str, Any] = field(default_factory=dict)
    
    # Metadata
    created_at: datetime = field(default_factory=datetime.now)
    created_by: str = ""
    version: str = "1.0"


@dataclass
class ReportSchedule:
    """Report generation schedule"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    name: str = ""
    template_id: str = ""
    
    # Scheduling
    frequency: ReportFrequency = ReportFrequency.DAILY
    schedule_time: str = "09:00"  # HH:MM format
    day_of_week: int = 1  # 1=Monday for weekly reports
    day_of_month: int = 1  # Day of month for monthly reports
    
    # Output
    formats: List[ReportFormat] = field(default_factory=lambda: [ReportFormat.PDF])
    output_path: str = ""
    
    # Distribution
    email_recipients: List[str] = field(default_factory=list)
    webhook_url: Optional[str] = None
    
    # Status
    is_active: bool = True
    next_run: Optional[datetime] = None
    last_run: Optional[datetime] = None
    
    # Parameters
    parameters: Dict[str, Any] = field(default_factory=dict)


@dataclass
class ReportExecution:
    """Report execution instance"""
    id: str = field(default_factory=lambda: str(uuid.uuid4()))
    template_id: str = ""
    schedule_id: Optional[str] = None
    
    # Status
    status: ReportStatus = ReportStatus.SCHEDULED
    start_time: Optional[datetime] = None
    end_time: Optional[datetime] = None
    
    # Output
    output_files: Dict[ReportFormat, str] = field(default_factory=dict)
    file_sizes: Dict[ReportFormat, int] = field(default_factory=dict)
    
    # Parameters used
    parameters: Dict[str, Any] = field(default_factory=dict)
    
    # Results
    error_message: Optional[str] = None
    rows_processed: int = 0
    sections_generated: int = 0


class DataProcessor:
    """Data processing and transformation"""
    
    def __init__(self):
        self.logger = logging.getLogger(__name__)
    
    def process_data_source(self, data_source: ReportDataSource, 
                           parameters: Dict[str, Any] = None) -> List[Dict[str, Any]]:
        """Process data source and return data"""
        params = {**(parameters or {}), **data_source.parameters}
        
        if data_source.source_type == "analytics":
            return self._process_analytics_data(data_source, params)
        elif data_source.source_type == "mock":
            return self._generate_mock_data(data_source, params)
        else:
            raise ValueError(f"Unsupported data source type: {data_source.source_type}")
    
    def _process_analytics_data(self, data_source: ReportDataSource, 
                               parameters: Dict[str, Any]) -> List[Dict[str, Any]]:
        """Process analytics data source"""
        # In a real implementation, this would connect to the analytics engine
        # For now, return mock data
        return self._generate_mock_data(data_source, parameters)
    
    def _generate_mock_data(self, data_source: ReportDataSource, 
                           parameters: Dict[str, Any]) -> List[Dict[str, Any]]:
        """Generate mock data for testing"""
        import random
        
        # Generate sample data based on query hints
        if "sales" in data_source.query.lower():
            return self._generate_sales_data()
        elif "user" in data_source.query.lower():
            return self._generate_user_data()
        elif "performance" in data_source.query.lower():
            return self._generate_performance_data()
        else:
            return self._generate_generic_data()
    
    def _generate_sales_data(self) -> List[Dict[str, Any]]:
        """Generate sample sales data"""
        import random
        
        data = []
        for i in range(30):  # 30 days of data
            date = datetime.now() - timedelta(days=30-i)
            data.append({
                'date': date.strftime('%Y-%m-%d'),
                'sales': round(random.uniform(1000, 5000), 2),
                'orders': random.randint(10, 100),
                'revenue': round(random.uniform(10000, 50000), 2),
                'region': random.choice(['North', 'South', 'East', 'West'])
            })
        
        return data
    
    def _generate_user_data(self) -> List[Dict[str, Any]]:
        """Generate sample user data"""
        import random
        
        return [
            {
                'user_id': f"user_{i:04d}",
                'name': f"User {i}",
                'email': f"user{i}@example.com",
                'signup_date': (datetime.now() - timedelta(days=random.randint(1, 365))).strftime('%Y-%m-%d'),
                'active': random.choice([True, False]),
                'total_purchases': random.randint(0, 50),
                'lifetime_value': round(random.uniform(0, 1000), 2)
            }
            for i in range(100)
        ]
    
    def _generate_performance_data(self) -> List[Dict[str, Any]]:
        """Generate sample performance data"""
        import random
        
        data = []
        for i in range(24):  # 24 hours of data
            hour = i
            data.append({
                'hour': f"{hour:02d}:00",
                'cpu_usage': round(random.uniform(20, 80), 1),
                'memory_usage': round(random.uniform(40, 90), 1),
                'requests': random.randint(100, 1000),
                'response_time': round(random.uniform(50, 300), 1)
            })
        
        return data
    
    def _generate_generic_data(self) -> List[Dict[str, Any]]:
        """Generate generic data"""
        import random
        
        return [
            {
                'id': i,
                'value': round(random.uniform(0, 100), 2),
                'category': random.choice(['A', 'B', 'C', 'D']),
                'timestamp': datetime.now() - timedelta(hours=i)
            }
            for i in range(50)
        ]
    
    def apply_filters(self, data: List[Dict[str, Any]], 
                     filters: List[Dict[str, Any]]) -> List[Dict[str, Any]]:
        """Apply filters to data"""
        filtered_data = data
        
        for filter_config in filters:
            field = filter_config.get('field')
            operator = filter_config.get('operator', 'equals')
            value = filter_config.get('value')
            
            if not field or value is None:
                continue
            
            if operator == 'equals':
                filtered_data = [row for row in filtered_data if row.get(field) == value]
            elif operator == 'greater_than':
                filtered_data = [row for row in filtered_data if row.get(field, 0) > value]
            elif operator == 'less_than':
                filtered_data = [row for row in filtered_data if row.get(field, 0) < value]
            elif operator == 'contains':
                filtered_data = [row for row in filtered_data if value in str(row.get(field, ''))]
        
        return filtered_data
    
    def apply_sorting(self, data: List[Dict[str, Any]], 
                     sort_by: List[Dict[str, str]]) -> List[Dict[str, Any]]:
        """Apply sorting to data"""
        if not sort_by:
            return data
        
        def sort_key(row):
            keys = []
            for sort_config in sort_by:
                field = sort_config.get('field')
                direction = sort_config.get('direction', 'asc')
                
                value = row.get(field, '')
                if direction == 'desc':
                    # For descending sort, negate numeric values
                    if isinstance(value, (int, float)):
                        value = -value
                    elif isinstance(value, str):
                        # For strings, we'd need more complex handling
                        pass
                
                keys.append(value)
            
            return tuple(keys)
        
        return sorted(data, key=sort_key)


class ReportRenderer:
    """Report rendering system"""
    
    def __init__(self):
        self.data_processor = DataProcessor()
        self.template_env = jinja2.Environment(
            loader=jinja2.BaseLoader(),
            autoescape=True
        )
        self.logger = logging.getLogger(__name__)
    
    async def render_report(self, template: ReportTemplate, sections: Dict[str, ReportSection],
                           data_sources: Dict[str, ReportDataSource],
                           formats: List[ReportFormat],
                           parameters: Dict[str, Any] = None) -> ReportExecution:
        """Render complete report"""
        execution = ReportExecution(
            template_id=template.id,
            parameters=parameters or {},
            start_time=datetime.now(),
            status=ReportStatus.GENERATING
        )
        
        try:
            # Process data for all sections
            section_data = {}
            for section_id in template.sections:
                section = sections.get(section_id)
                if not section:
                    continue
                
                data_source = data_sources.get(section.data_source_id)
                if not data_source:
                    continue
                
                # Process data
                raw_data = self.data_processor.process_data_source(data_source, parameters)
                
                # Apply filters and sorting
                filtered_data = self.data_processor.apply_filters(raw_data, data_source.filters)
                sorted_data = self.data_processor.apply_sorting(filtered_data, data_source.sort_by)
                
                section_data[section_id] = sorted_data
                execution.rows_processed += len(sorted_data)
            
            # Render in requested formats
            for format_type in formats:
                try:
                    if format_type == ReportFormat.PDF and PDF_AVAILABLE:
                        file_path = await self._render_pdf(template, sections, section_data, execution)
                    elif format_type == ReportFormat.EXCEL and EXCEL_AVAILABLE:
                        file_path = await self._render_excel(template, sections, section_data, execution)
                    elif format_type == ReportFormat.HTML:
                        file_path = await self._render_html(template, sections, section_data, execution)
                    elif format_type == ReportFormat.JSON:
                        file_path = await self._render_json(template, sections, section_data, execution)
                    else:
                        self.logger.warning(f"Format {format_type.value} not supported or libraries not available")
                        continue
                    
                    execution.output_files[format_type] = file_path
                    if os.path.exists(file_path):
                        execution.file_sizes[format_type] = os.path.getsize(file_path)
                        
                except Exception as e:
                    self.logger.error(f"Error rendering {format_type.value} format: {e}")
            
            execution.sections_generated = len(template.sections)
            execution.status = ReportStatus.COMPLETED
            
        except Exception as e:
            execution.error_message = str(e)
            execution.status = ReportStatus.FAILED
            self.logger.error(f"Report rendering failed: {e}")
        
        finally:
            execution.end_time = datetime.now()
        
        return execution
    
    async def _render_pdf(self, template: ReportTemplate, sections: Dict[str, ReportSection],
                         section_data: Dict[str, List[Dict]], execution: ReportExecution) -> str:
        """Render PDF report"""
        if not PDF_AVAILABLE:
            raise RuntimeError("PDF rendering not available - install reportlab")
        
        # Create temporary file
        fd, file_path = tempfile.mkstemp(suffix='.pdf', prefix='report_')
        os.close(fd)
        
        # Create PDF document
        page_size = A4 if template.page_size == "A4" else letter
        doc = SimpleDocTemplate(file_path, pagesize=page_size)
        
        # Styles
        styles = getSampleStyleSheet()
        title_style = ParagraphStyle(
            'CustomTitle',
            parent=styles['Title'],
            fontSize=24,
            spaceAfter=30
        )
        
        story = []
        
        # Title page
        if template.title:
            story.append(Paragraph(template.title, title_style))
            story.append(Spacer(1, 0.2*inch))
        
        if template.subtitle:
            story.append(Paragraph(template.subtitle, styles['Heading2']))
            story.append(Spacer(1, 0.2*inch))
        
        # Generated date
        story.append(Paragraph(f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M')}", styles['Normal']))
        story.append(Spacer(1, 0.3*inch))
        
        # Render sections
        for section_id in template.sections:
            section = sections.get(section_id)
            if not section or section_id not in section_data:
                continue
            
            data = section_data[section_id]
            
            if section.page_break_before:
                story.append(PageBreak())
            
            # Section title
            if section.show_title and section.title:
                story.append(Paragraph(section.title, styles['Heading2']))
                story.append(Spacer(1, 0.1*inch))
            
            # Section description
            if section.description:
                story.append(Paragraph(section.description, styles['Normal']))
                story.append(Spacer(1, 0.1*inch))
            
            # Render section content
            if section.section_type == "data_table":
                table_element = self._create_pdf_table(data, section)
                story.append(table_element)
            
            elif section.section_type == "chart" and PLOT_AVAILABLE:
                chart_element = await self._create_pdf_chart(data, section)
                if chart_element:
                    story.append(chart_element)
            
            elif section.section_type == "text":
                content = self._render_template_text(section.content, section.template_variables)
                story.append(Paragraph(content, styles['Normal']))
            
            story.append(Spacer(1, 0.2*inch))
            
            if section.page_break_after:
                story.append(PageBreak())
        
        # Build PDF
        doc.build(story)
        
        return file_path
    
    def _create_pdf_table(self, data: List[Dict[str, Any]], section: ReportSection) -> Table:
        """Create PDF table"""
        if not data:
            return Table([["No data available"]])
        
        # Limit rows if specified
        display_data = data[:section.max_rows] if section.max_rows else data
        
        # Get headers
        headers = list(display_data[0].keys()) if display_data else []
        
        # Build table data
        table_data = []
        
        if section.show_headers:
            table_data.append(headers)
        
        for i, row in enumerate(display_data):
            table_row = []
            if section.show_row_numbers:
                table_row.append(str(i + 1))
            
            for header in headers:
                value = row.get(header, "")
                if isinstance(value, float):
                    value = section.number_format.format(value)
                elif isinstance(value, datetime):
                    value = value.strftime(section.date_format)
                table_row.append(str(value))
            
            table_data.append(table_row)
        
        # Create table
        table = Table(table_data)
        
        # Apply styling
        table.setStyle(TableStyle([
            ('BACKGROUND', (0, 0), (-1, 0), colors.grey),
            ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
            ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
            ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
            ('FONTSIZE', (0, 0), (-1, 0), 12),
            ('BOTTOMPADDING', (0, 0), (-1, 0), 12),
            ('BACKGROUND', (0, 1), (-1, -1), colors.beige),
            ('GRID', (0, 0), (-1, -1), 1, colors.black)
        ]))
        
        return table
    
    async def _create_pdf_chart(self, data: List[Dict[str, Any]], section: ReportSection) -> Optional[Any]:
        """Create PDF chart using matplotlib"""
        if not PLOT_AVAILABLE or not data:
            return None
        
        try:
            # Create chart
            fig, ax = plt.subplots(figsize=(8, 6))
            
            x_values = [row.get(section.x_axis_field, i) for i, row in enumerate(data)]
            y_values = [row.get(section.y_axis_field, 0) for row in data]
            
            if section.chart_type == "line":
                ax.plot(x_values, y_values)
            elif section.chart_type == "bar":
                ax.bar(x_values, y_values)
            
            ax.set_xlabel(section.x_axis_field)
            ax.set_ylabel(section.y_axis_field)
            ax.set_title(section.title)
            
            # Save to temporary file
            fd, chart_path = tempfile.mkstemp(suffix='.png', prefix='chart_')
            os.close(fd)
            
            plt.savefig(chart_path, dpi=150, bbox_inches='tight')
            plt.close(fig)
            
            # Create Image element
            return Image(chart_path, width=6*inch, height=4*inch)
            
        except Exception as e:
            self.logger.error(f"Error creating chart: {e}")
            return None
    
    async def _render_excel(self, template: ReportTemplate, sections: Dict[str, ReportSection],
                           section_data: Dict[str, List[Dict]], execution: ReportExecution) -> str:
        """Render Excel report"""
        if not EXCEL_AVAILABLE:
            raise RuntimeError("Excel rendering not available - install openpyxl")
        
        # Create temporary file
        fd, file_path = tempfile.mkstemp(suffix='.xlsx', prefix='report_')
        os.close(fd)
        
        # Create workbook
        workbook = openpyxl.Workbook()
        
        # Remove default sheet
        workbook.remove(workbook.active)
        
        # Create sheets for each section
        for section_id in template.sections:
            section = sections.get(section_id)
            if not section or section_id not in section_data:
                continue
            
            data = section_data[section_id]
            
            # Create worksheet
            sheet_name = section.name[:31] if section.name else f"Section_{section_id[:8]}"
            worksheet = workbook.create_sheet(title=sheet_name)
            
            if data:
                # Write headers
                headers = list(data[0].keys())
                for col, header in enumerate(headers, 1):
                    cell = worksheet.cell(row=1, column=col, value=header)
                    cell.font = Font(bold=True)
                    cell.fill = PatternFill(start_color="CCCCCC", end_color="CCCCCC", fill_type="solid")
                
                # Write data
                for row_idx, row_data in enumerate(data, 2):
                    for col_idx, header in enumerate(headers, 1):
                        value = row_data.get(header, "")
                        worksheet.cell(row=row_idx, column=col_idx, value=value)
                
                # Auto-adjust column widths
                for column in worksheet.columns:
                    max_length = 0
                    column_letter = openpyxl.utils.get_column_letter(column[0].column)
                    
                    for cell in column:
                        try:
                            if len(str(cell.value)) > max_length:
                                max_length = len(str(cell.value))
                        except:
                            pass
                    
                    adjusted_width = min(max_length + 2, 50)
                    worksheet.column_dimensions[column_letter].width = adjusted_width
        
        # Create summary sheet if multiple sections
        if len(template.sections) > 1:
            summary_sheet = workbook.create_sheet(title="Summary", index=0)
            summary_sheet.cell(row=1, column=1, value=template.title or "Report Summary").font = Font(size=16, bold=True)
            summary_sheet.cell(row=2, column=1, value=f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M')}")
            
            row = 4
            for section_id in template.sections:
                section = sections.get(section_id)
                if section:
                    summary_sheet.cell(row=row, column=1, value=section.name or f"Section {section_id[:8]}")
                    if section_id in section_data:
                        summary_sheet.cell(row=row, column=2, value=f"{len(section_data[section_id])} rows")
                    row += 1
        
        # Save workbook
        workbook.save(file_path)
        workbook.close()
        
        return file_path
    
    async def _render_html(self, template: ReportTemplate, sections: Dict[str, ReportSection],
                          section_data: Dict[str, List[Dict]], execution: ReportExecution) -> str:
        """Render HTML report"""
        # Create temporary file
        fd, file_path = tempfile.mkstemp(suffix='.html', prefix='report_')
        os.close(fd)
        
        # Build HTML content
        html_content = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <title>{template.title or 'Report'}</title>
            <style>
                body {{ font-family: Arial, sans-serif; margin: 40px; }}
                h1 {{ color: #333; border-bottom: 2px solid #333; }}
                h2 {{ color: #666; margin-top: 30px; }}
                table {{ border-collapse: collapse; width: 100%; margin: 20px 0; }}
                th, td {{ border: 1px solid #ddd; padding: 8px; text-align: left; }}
                th {{ background-color: #f2f2f2; font-weight: bold; }}
                tr:nth-child(even) {{ background-color: #f9f9f9; }}
                .metadata {{ font-size: 0.9em; color: #666; margin-bottom: 30px; }}
            </style>
        </head>
        <body>
        """
        
        # Title
        if template.title:
            html_content += f"<h1>{template.title}</h1>"
        
        if template.subtitle:
            html_content += f"<h2>{template.subtitle}</h2>"
        
        # Metadata
        html_content += f'<div class="metadata">Generated: {datetime.now().strftime("%Y-%m-%d %H:%M")}</div>'
        
        # Render sections
        for section_id in template.sections:
            section = sections.get(section_id)
            if not section or section_id not in section_data:
                continue
            
            data = section_data[section_id]
            
            # Section title
            if section.show_title and section.title:
                html_content += f"<h2>{section.title}</h2>"
            
            # Section description
            if section.description:
                html_content += f"<p>{section.description}</p>"
            
            # Section content
            if section.section_type == "data_table" and data:
                html_content += self._create_html_table(data, section)
            elif section.section_type == "text":
                content = self._render_template_text(section.content, section.template_variables)
                html_content += f"<div>{content}</div>"
        
        html_content += "</body></html>"
        
        # Write to file
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(html_content)
        
        return file_path
    
    def _create_html_table(self, data: List[Dict[str, Any]], section: ReportSection) -> str:
        """Create HTML table"""
        if not data:
            return "<p>No data available</p>"
        
        # Limit rows if specified
        display_data = data[:section.max_rows] if section.max_rows else data
        headers = list(display_data[0].keys())
        
        html = "<table>"
        
        # Headers
        if section.show_headers:
            html += "<thead><tr>"
            if section.show_row_numbers:
                html += "<th>#</th>"
            for header in headers:
                html += f"<th>{header}</th>"
            html += "</tr></thead>"
        
        # Data rows
        html += "<tbody>"
        for i, row in enumerate(display_data):
            html += "<tr>"
            if section.show_row_numbers:
                html += f"<td>{i + 1}</td>"
            
            for header in headers:
                value = row.get(header, "")
                if isinstance(value, float):
                    value = section.number_format.format(value)
                elif isinstance(value, datetime):
                    value = value.strftime(section.date_format)
                html += f"<td>{value}</td>"
            html += "</tr>"
        
        html += "</tbody></table>"
        
        return html
    
    async def _render_json(self, template: ReportTemplate, sections: Dict[str, ReportSection],
                          section_data: Dict[str, List[Dict]], execution: ReportExecution) -> str:
        """Render JSON report"""
        # Create temporary file
        fd, file_path = tempfile.mkstemp(suffix='.json', prefix='report_')
        os.close(fd)
        
        # Build JSON structure
        report_data = {
            'metadata': {
                'template_id': template.id,
                'template_name': template.name,
                'title': template.title,
                'subtitle': template.subtitle,
                'generated_at': datetime.now().isoformat(),
                'execution_id': execution.id
            },
            'sections': {}
        }
        
        # Add section data
        for section_id in template.sections:
            section = sections.get(section_id)
            if not section or section_id not in section_data:
                continue
            
            report_data['sections'][section_id] = {
                'name': section.name,
                'title': section.title,
                'description': section.description,
                'section_type': section.section_type,
                'data': section_data[section_id]
            }
        
        # Write to file
        with open(file_path, 'w', encoding='utf-8') as f:
            json.dump(report_data, f, indent=2, default=str)
        
        return file_path
    
    def _render_template_text(self, content: str, variables: Dict[str, str]) -> str:
        """Render template text with variables"""
        try:
            template = self.template_env.from_string(content)
            return template.render(**variables)
        except Exception as e:
            self.logger.error(f"Template rendering error: {e}")
            return content


class ReportingEngine:
    """Main reporting engine"""
    
    def __init__(self):
        self.templates: Dict[str, ReportTemplate] = {}
        self.sections: Dict[str, ReportSection] = {}
        self.data_sources: Dict[str, ReportDataSource] = {}
        self.schedules: Dict[str, ReportSchedule] = {}
        self.executions: Dict[str, ReportExecution] = {}
        
        self.renderer = ReportRenderer()
        self.logger = logging.getLogger(__name__)
        
        # Scheduler
        self.is_running = False
        self.scheduler_task: Optional[asyncio.Task] = None
    
    async def start(self):
        """Start reporting engine"""
        if self.is_running:
            return
        
        self.is_running = True
        self.scheduler_task = asyncio.create_task(self._scheduler_loop())
        self.logger.info("Reporting engine started")
    
    async def stop(self):
        """Stop reporting engine"""
        if not self.is_running:
            return
        
        self.is_running = False
        
        if self.scheduler_task:
            self.scheduler_task.cancel()
        
        self.logger.info("Reporting engine stopped")
    
    def create_data_source(self, data_source: ReportDataSource) -> str:
        """Create data source"""
        self.data_sources[data_source.id] = data_source
        self.logger.info(f"Created data source: {data_source.name} ({data_source.id})")
        return data_source.id
    
    def create_section(self, section: ReportSection) -> str:
        """Create report section"""
        self.sections[section.id] = section
        self.logger.info(f"Created section: {section.name} ({section.id})")
        return section.id
    
    def create_template(self, template: ReportTemplate) -> str:
        """Create report template"""
        self.templates[template.id] = template
        self.logger.info(f"Created template: {template.name} ({template.id})")
        return template.id
    
    def create_schedule(self, schedule: ReportSchedule) -> str:
        """Create report schedule"""
        # Calculate next run time
        schedule.next_run = self._calculate_next_run(schedule)
        
        self.schedules[schedule.id] = schedule
        self.logger.info(f"Created schedule: {schedule.name} ({schedule.id})")
        return schedule.id
    
    async def generate_report(self, template_id: str, formats: List[ReportFormat],
                             parameters: Dict[str, Any] = None) -> ReportExecution:
        """Generate report immediately"""
        template = self.templates.get(template_id)
        if not template:
            raise ValueError(f"Template not found: {template_id}")
        
        execution = await self.renderer.render_report(
            template, self.sections, self.data_sources, formats, parameters
        )
        
        self.executions[execution.id] = execution
        return execution
    
    def _calculate_next_run(self, schedule: ReportSchedule) -> datetime:
        """Calculate next run time for schedule"""
        now = datetime.now()
        
        if schedule.frequency == ReportFrequency.DAILY:
            next_run = now.replace(
                hour=int(schedule.schedule_time.split(':')[0]),
                minute=int(schedule.schedule_time.split(':')[1]),
                second=0,
                microsecond=0
            )
            if next_run <= now:
                next_run += timedelta(days=1)
                
        elif schedule.frequency == ReportFrequency.WEEKLY:
            # Calculate next occurrence of the specified day
            days_ahead = schedule.day_of_week - now.weekday()
            if days_ahead <= 0:
                days_ahead += 7
            
            next_run = now + timedelta(days=days_ahead)
            next_run = next_run.replace(
                hour=int(schedule.schedule_time.split(':')[0]),
                minute=int(schedule.schedule_time.split(':')[1]),
                second=0,
                microsecond=0
            )
            
        elif schedule.frequency == ReportFrequency.MONTHLY:
            # Next month, same day
            if now.month == 12:
                next_run = now.replace(year=now.year + 1, month=1, day=schedule.day_of_month)
            else:
                next_run = now.replace(month=now.month + 1, day=schedule.day_of_month)
            
            next_run = next_run.replace(
                hour=int(schedule.schedule_time.split(':')[0]),
                minute=int(schedule.schedule_time.split(':')[1]),
                second=0,
                microsecond=0
            )
        else:
            # Default to daily
            next_run = now + timedelta(days=1)
        
        return next_run
    
    async def _scheduler_loop(self):
        """Background scheduler loop"""
        while self.is_running:
            try:
                now = datetime.now()
                
                # Check for scheduled reports
                for schedule in self.schedules.values():
                    if (schedule.is_active and 
                        schedule.next_run and 
                        schedule.next_run <= now):
                        
                        await self._execute_scheduled_report(schedule)
                
                await asyncio.sleep(60)  # Check every minute
                
            except Exception as e:
                self.logger.error(f"Scheduler error: {e}")
                await asyncio.sleep(60)
    
    async def _execute_scheduled_report(self, schedule: ReportSchedule):
        """Execute scheduled report"""
        try:
            self.logger.info(f"Executing scheduled report: {schedule.name}")
            
            execution = await self.generate_report(
                schedule.template_id,
                schedule.formats,
                schedule.parameters
            )
            
            execution.schedule_id = schedule.id
            
            # Update schedule
            schedule.last_run = datetime.now()
            schedule.next_run = self._calculate_next_run(schedule)
            
            self.logger.info(f"Scheduled report completed: {execution.id}")
            
        except Exception as e:
            self.logger.error(f"Scheduled report failed: {e}")
    
    def get_execution(self, execution_id: str) -> Optional[ReportExecution]:
        """Get report execution"""
        return self.executions.get(execution_id)
    
    def list_executions(self, template_id: Optional[str] = None, 
                       limit: int = 50) -> List[ReportExecution]:
        """List report executions"""
        executions = list(self.executions.values())
        
        if template_id:
            executions = [e for e in executions if e.template_id == template_id]
        
        executions.sort(key=lambda x: x.start_time or datetime.min, reverse=True)
        return executions[:limit]


if __name__ == "__main__":
    async def main():
        # Create reporting engine
        engine = ReportingEngine()
        
        # Create data source
        sales_data_source = ReportDataSource(
            name="Sales Data",
            source_type="mock",
            query="SELECT * FROM sales_data",
            filters=[
                {"field": "region", "operator": "equals", "value": "North"}
            ]
        )
        
        engine.create_data_source(sales_data_source)
        
        # Create sections
        sales_table_section = ReportSection(
            name="Sales Table",
            section_type="data_table",
            data_source_id=sales_data_source.id,
            title="Sales Performance",
            description="Monthly sales data by region",
            show_headers=True,
            max_rows=20
        )
        
        summary_section = ReportSection(
            name="Summary",
            section_type="text",
            title="Executive Summary",
            content="This report shows sales performance for the current period. Total revenue was {{ total_revenue }}.",
            template_variables={"total_revenue": "$125,000"}
        )
        
        engine.create_section(sales_table_section)
        engine.create_section(summary_section)
        
        # Create template
        sales_template = ReportTemplate(
            name="Monthly Sales Report",
            title="Sales Performance Report",
            subtitle="Monthly Analysis",
            sections=[summary_section.id, sales_table_section.id],
            page_size="A4",
            orientation="portrait"
        )
        
        engine.create_template(sales_template)
        
        # Generate report
        print("Generating report...")
        execution = await engine.generate_report(
            sales_template.id,
            [ReportFormat.HTML, ReportFormat.JSON]
        )
        
        print(f"Report execution: {execution.status.value}")
        print(f"Sections generated: {execution.sections_generated}")
        print(f"Rows processed: {execution.rows_processed}")
        
        if execution.output_files:
            for format_type, file_path in execution.output_files.items():
                file_size = execution.file_sizes.get(format_type, 0)
                print(f"{format_type.value.upper()}: {file_path} ({file_size} bytes)")
        
        if execution.error_message:
            print(f"Error: {execution.error_message}")
        
        print("\ng12.2: Reporting Engine with Automated Generation - COMPLETED ✅")
    
    asyncio.run(main()) 