#!/usr/bin/env python3
"""
Shell Storage Integration for TuskLang Python
============================================
Shell-based configuration storage and retrieval

Features:
- Shell variable storage and retrieval
- Environment variable integration
- Shell script generation
- Cross-platform shell compatibility
- Binary format integration
- Shell function execution
"""

import os
import subprocess
import tempfile
import json
import shlex
from typing import Any, Dict, List, Optional, Union
from pathlib import Path
import platform


class ShellStorage:
    """Shell storage integration for TuskLang"""
    
    def __init__(self):
        self.shell_type = self._detect_shell()
        self.storage_file = None
        self.export_prefix = "TUSKLANG_"
    
    def _detect_shell(self) -> str:
        """Detect the current shell type"""
        shell = os.environ.get('SHELL', '').lower()
        
        if 'bash' in shell:
            return 'bash'
        elif 'zsh' in shell:
            return 'zsh'
        elif 'fish' in shell:
            return 'fish'
        elif 'sh' in shell:
            return 'sh'
        else:
            return 'bash'  # Default to bash
    
    def set_storage_file(self, file_path: str):
        """Set the storage file for shell variables"""
        self.storage_file = file_path
    
    def export_variable(self, name: str, value: Any, persistent: bool = True) -> bool:
        """Export variable to shell environment"""
        try:
            # Convert value to string
            if isinstance(value, (dict, list)):
                value_str = json.dumps(value)
            else:
                value_str = str(value)
            
            # Export to current session
            os.environ[f"{self.export_prefix}{name}"] = value_str
            
            # Export to shell if persistent
            if persistent:
                self._export_to_shell(name, value_str)
            
            return True
        except Exception as e:
            print(f"Export error: {e}")
            return False
    
    def _export_to_shell(self, name: str, value: str):
        """Export variable to shell configuration"""
        if self.shell_type == 'bash':
            self._export_to_bash(name, value)
        elif self.shell_type == 'zsh':
            self._export_to_zsh(name, value)
        elif self.shell_type == 'fish':
            self._export_to_fish(name, value)
        else:
            self._export_to_generic(name, value)
    
    def _export_to_bash(self, name: str, value: str):
        """Export to bash configuration"""
        config_file = os.path.expanduser('~/.bashrc')
        export_line = f'export {self.export_prefix}{name}="{value}"\n'
        
        # Check if already exists
        if os.path.exists(config_file):
            with open(config_file, 'r') as f:
                content = f.read()
            
            # Remove existing export if exists
            lines = content.split('\n')
            lines = [line for line in lines if not line.startswith(f'export {self.export_prefix}{name}=')]
            lines.append(export_line)
            
            with open(config_file, 'w') as f:
                f.write('\n'.join(lines))
        else:
            with open(config_file, 'w') as f:
                f.write(export_line)
    
    def _export_to_zsh(self, name: str, value: str):
        """Export to zsh configuration"""
        config_file = os.path.expanduser('~/.zshrc')
        export_line = f'export {self.export_prefix}{name}="{value}"\n'
        
        if os.path.exists(config_file):
            with open(config_file, 'r') as f:
                content = f.read()
            
            lines = content.split('\n')
            lines = [line for line in lines if not line.startswith(f'export {self.export_prefix}{name}=')]
            lines.append(export_line)
            
            with open(config_file, 'w') as f:
                f.write('\n'.join(lines))
        else:
            with open(config_file, 'w') as f:
                f.write(export_line)
    
    def _export_to_fish(self, name: str, value: str):
        """Export to fish configuration"""
        config_file = os.path.expanduser('~/.config/fish/config.fish')
        os.makedirs(os.path.dirname(config_file), exist_ok=True)
        
        set_line = f'set -gx {self.export_prefix}{name} "{value}"\n'
        
        if os.path.exists(config_file):
            with open(config_file, 'r') as f:
                content = f.read()
            
            lines = content.split('\n')
            lines = [line for line in lines if not line.startswith(f'set -gx {self.export_prefix}{name} ')]
            lines.append(set_line)
            
            with open(config_file, 'w') as f:
                f.write('\n'.join(lines))
        else:
            with open(config_file, 'w') as f:
                f.write(set_line)
    
    def _export_to_generic(self, name: str, value: str):
        """Export to generic shell configuration"""
        config_file = os.path.expanduser('~/.profile')
        export_line = f'export {self.export_prefix}{name}="{value}"\n'
        
        if os.path.exists(config_file):
            with open(config_file, 'r') as f:
                content = f.read()
            
            lines = content.split('\n')
            lines = [line for line in lines if not line.startswith(f'export {self.export_prefix}{name}=')]
            lines.append(export_line)
            
            with open(config_file, 'w') as f:
                f.write('\n'.join(lines))
        else:
            with open(config_file, 'w') as f:
                f.write(export_line)
    
    def get_variable(self, name: str) -> Optional[Any]:
        """Get variable from shell environment"""
        try:
            value = os.environ.get(f"{self.export_prefix}{name}")
            if value is None:
                return None
            
            # Try to parse as JSON
            try:
                return json.loads(value)
            except json.JSONDecodeError:
                return value
        except Exception as e:
            print(f"Get variable error: {e}")
            return None
    
    def unset_variable(self, name: str, persistent: bool = True) -> bool:
        """Unset variable from shell environment"""
        try:
            # Remove from current session
            env_key = f"{self.export_prefix}{name}"
            if env_key in os.environ:
                del os.environ[env_key]
            
            # Remove from shell configuration if persistent
            if persistent:
                self._unset_from_shell(name)
            
            return True
        except Exception as e:
            print(f"Unset variable error: {e}")
            return False
    
    def _unset_from_shell(self, name: str):
        """Remove variable from shell configuration"""
        if self.shell_type == 'bash':
            config_file = os.path.expanduser('~/.bashrc')
        elif self.shell_type == 'zsh':
            config_file = os.path.expanduser('~/.zshrc')
        elif self.shell_type == 'fish':
            config_file = os.path.expanduser('~/.config/fish/config.fish')
        else:
            config_file = os.path.expanduser('~/.profile')
        
        if os.path.exists(config_file):
            with open(config_file, 'r') as f:
                content = f.read()
            
            lines = content.split('\n')
            lines = [line for line in lines if not line.startswith(f'export {self.export_prefix}{name}=') 
                    and not line.startswith(f'set -gx {self.export_prefix}{name} ')]
            
            with open(config_file, 'w') as f:
                f.write('\n'.join(lines))
    
    def list_variables(self) -> Dict[str, Any]:
        """List all TuskLang variables in environment"""
        variables = {}
        
        for key, value in os.environ.items():
            if key.startswith(self.export_prefix):
                name = key[len(self.export_prefix):]
                try:
                    variables[name] = json.loads(value)
                except json.JSONDecodeError:
                    variables[name] = value
        
        return variables
    
    def execute_shell_command(self, command: str, capture_output: bool = True) -> Dict[str, Any]:
        """Execute shell command and return result"""
        try:
            if capture_output:
                result = subprocess.run(
                    command,
                    shell=True,
                    capture_output=True,
                    text=True,
                    timeout=30
                )
                
                return {
                    'returncode': result.returncode,
                    'stdout': result.stdout,
                    'stderr': result.stderr,
                    'success': result.returncode == 0
                }
            else:
                result = subprocess.run(
                    command,
                    shell=True,
                    timeout=30
                )
                
                return {
                    'returncode': result.returncode,
                    'success': result.returncode == 0
                }
                
        except subprocess.TimeoutExpired:
            return {
                'returncode': -1,
                'stdout': '',
                'stderr': 'Command timed out',
                'success': False
            }
        except Exception as e:
            return {
                'returncode': -1,
                'stdout': '',
                'stderr': str(e),
                'success': False
            }
    
    def create_shell_script(self, commands: List[str], output_file: str = None) -> str:
        """Create shell script from commands"""
        if output_file is None:
            output_file = tempfile.mktemp(suffix='.sh')
        
        script_content = f"""#!/bin/bash
# TuskLang Shell Script
# Generated on {platform.platform()}

set -e  # Exit on error

"""
        
        for command in commands:
            script_content += f"{command}\n"
        
        script_content += "\n# Script completed successfully\n"
        
        with open(output_file, 'w') as f:
            f.write(script_content)
        
        # Make executable
        os.chmod(output_file, 0o755)
        
        return output_file
    
    def load_from_binary(self, binary_file: str) -> Dict[str, Any]:
        """Load configuration from binary file and export to shell"""
        try:
            import msgpack
            
            with open(binary_file, 'rb') as f:
                data = msgpack.unpackb(f.read(), raw=False)
            
            # Export all variables to shell
            for key, value in data.items():
                self.export_variable(key, value, persistent=True)
            
            return data
        except Exception as e:
            print(f"Load from binary error: {e}")
            return {}
    
    def save_to_binary(self, data: Dict[str, Any], output_file: str) -> bool:
        """Save configuration to binary file"""
        try:
            import msgpack
            
            binary_data = msgpack.packb(data, use_bin_type=True)
            
            with open(output_file, 'wb') as f:
                f.write(binary_data)
            
            return True
        except Exception as e:
            print(f"Save to binary error: {e}")
            return False
    
    def generate_shell_config(self, config: Dict[str, Any], output_file: str = None) -> str:
        """Generate shell configuration file from dictionary"""
        if output_file is None:
            output_file = tempfile.mktemp(suffix='.sh')
        
        config_content = f"""# TuskLang Shell Configuration
# Generated on {platform.platform()}

"""
        
        for key, value in config.items():
            if isinstance(value, (dict, list)):
                value_str = json.dumps(value)
            else:
                value_str = str(value)
            
            config_content += f'export {self.export_prefix}{key}="{value_str}"\n'
        
        with open(output_file, 'w') as f:
            f.write(config_content)
        
        return output_file
    
    def source_config_file(self, config_file: str) -> bool:
        """Source a configuration file in current shell"""
        try:
            if self.shell_type == 'fish':
                command = f"source {config_file}"
            else:
                command = f"source {config_file}"
            
            result = self.execute_shell_command(command)
            return result['success']
        except Exception as e:
            print(f"Source config error: {e}")
            return False
    
    def get_shell_info(self) -> Dict[str, Any]:
        """Get information about the current shell"""
        return {
            'shell_type': self.shell_type,
            'shell_path': os.environ.get('SHELL', ''),
            'platform': platform.platform(),
            'user': os.environ.get('USER', ''),
            'home': os.environ.get('HOME', ''),
            'pwd': os.getcwd(),
            'export_prefix': self.export_prefix
        }
    
    def validate_shell_environment(self) -> Dict[str, bool]:
        """Validate shell environment setup"""
        checks = {
            'shell_detected': bool(self.shell_type),
            'home_directory': bool(os.environ.get('HOME')),
            'shell_config_writable': False,
            'binary_support': False
        }
        
        # Check if shell config is writable
        if self.shell_type == 'bash':
            config_file = os.path.expanduser('~/.bashrc')
        elif self.shell_type == 'zsh':
            config_file = os.path.expanduser('~/.zshrc')
        elif self.shell_type == 'fish':
            config_file = os.path.expanduser('~/.config/fish/config.fish')
        else:
            config_file = os.path.expanduser('~/.profile')
        
        checks['shell_config_writable'] = os.access(os.path.dirname(config_file), os.W_OK)
        
        # Check binary support
        try:
            import msgpack
            checks['binary_support'] = True
        except ImportError:
            pass
        
        return checks


# Global shell storage instance
shell_storage = ShellStorage()


def get_shell_storage() -> ShellStorage:
    """Get global shell storage instance"""
    return shell_storage


def export_to_shell(name: str, value: Any, persistent: bool = True) -> bool:
    """Export variable to shell environment"""
    return shell_storage.export_variable(name, value, persistent)


def get_from_shell(name: str) -> Optional[Any]:
    """Get variable from shell environment"""
    return shell_storage.get_variable(name)


def unset_from_shell(name: str, persistent: bool = True) -> bool:
    """Unset variable from shell environment"""
    return shell_storage.unset_variable(name, persistent)


def list_shell_variables() -> Dict[str, Any]:
    """List all TuskLang variables in environment"""
    return shell_storage.list_variables()


def execute_shell_command(command: str, capture_output: bool = True) -> Dict[str, Any]:
    """Execute shell command and return result"""
    return shell_storage.execute_shell_command(command, capture_output)


def load_config_to_shell(binary_file: str) -> Dict[str, Any]:
    """Load configuration from binary file and export to shell"""
    return shell_storage.load_from_binary(binary_file)


def save_config_to_binary(data: Dict[str, Any], output_file: str) -> bool:
    """Save configuration to binary file"""
    return shell_storage.save_to_binary(data, output_file)


def generate_shell_config_file(config: Dict[str, Any], output_file: str = None) -> str:
    """Generate shell configuration file from dictionary"""
    return shell_storage.generate_shell_config(config, output_file)


def get_shell_environment_info() -> Dict[str, Any]:
    """Get information about the current shell environment"""
    return shell_storage.get_shell_info()


def validate_shell_setup() -> Dict[str, bool]:
    """Validate shell environment setup"""
    return shell_storage.validate_shell_environment() 