"""
SQLite Enhanced Operator - Advanced Embedded Database Integration
Production-ready SQLite integration with advanced features including connection pooling,
migrations, full-text search, and performance optimization.
"""

import asyncio
import json
import logging
import os
import sqlite3
import threading
import time
from concurrent.futures import ThreadPoolExecutor
from contextlib import contextmanager
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from pathlib import Path
from queue import Queue, Empty
from typing import Any, Dict, List, Optional, Union, Tuple
import hashlib
import shutil

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

@dataclass
class SQLiteConfig:
    """SQLite configuration with advanced options."""
    database_path: str = ":memory:"
    pool_size: int = 10
    pool_timeout: int = 30
    enable_wal_mode: bool = True
    enable_foreign_keys: bool = True
    synchronous_mode: str = "NORMAL"  # OFF, NORMAL, FULL
    journal_mode: str = "WAL"  # DELETE, TRUNCATE, PERSIST, MEMORY, WAL
    cache_size: int = -64000  # KB (negative for KB, positive for pages)
    busy_timeout: int = 30000  # milliseconds
    enable_fts: bool = True
    auto_vacuum: str = "INCREMENTAL"  # NONE, FULL, INCREMENTAL
    temp_store: str = "MEMORY"  # DEFAULT, FILE, MEMORY
    mmap_size: int = 268435456  # 256MB
    enable_rtree: bool = True
    enable_json1: bool = True
    backup_interval: int = 3600  # seconds
    max_backup_files: int = 10

@dataclass
class SQLiteTransaction:
    """SQLite transaction context."""
    isolation_level: str = "DEFERRED"  # DEFERRED, IMMEDIATE, EXCLUSIVE
    read_only: bool = False

@dataclass
class SQLiteQueryResult:
    """SQLite query result with metadata."""
    data: List[Dict[str, Any]]
    affected_rows: int = 0
    execution_time: float = 0.0
    query_plan: Optional[Dict[str, Any]] = None
    cache_hit: bool = False

@dataclass
class SQLiteMigration:
    """SQLite migration definition."""
    version: str
    name: str
    up_sql: str
    down_sql: str
    checksum: str = ""
    applied_at: Optional[datetime] = None

@dataclass
class SQLiteIndex:
    """SQLite index definition."""
    name: str
    table: str
    columns: List[str]
    unique: bool = False
    partial_condition: Optional[str] = None

class SQLiteConnectionPool:
    """Advanced SQLite connection pool with health monitoring."""
    
    def __init__(self, config: SQLiteConfig):
        self.config = config
        self.pool = Queue(maxsize=config.pool_size)
        self.active_connections = set()
        self.connection_stats = {
            'total_created': 0,
            'total_closed': 0,
            'active_count': 0,
            'pool_hits': 0,
            'pool_misses': 0
        }
        self._lock = threading.Lock()
        self._initialize_pool()
        
    def _initialize_pool(self):
        """Initialize connection pool."""
        for _ in range(self.config.pool_size):
            conn = self._create_connection()
            if conn:
                self.pool.put(conn)
    
    def _create_connection(self) -> Optional[sqlite3.Connection]:
        """Create optimized SQLite connection."""
        try:
            conn = sqlite3.connect(
                self.config.database_path,
                check_same_thread=False,
                timeout=self.config.busy_timeout / 1000.0
            )
            
            # Enable row factory for dict-like access
            conn.row_factory = sqlite3.Row
            
            # Apply optimizations
            cursor = conn.cursor()
            
            # Journal mode
            cursor.execute(f"PRAGMA journal_mode = {self.config.journal_mode}")
            
            # Synchronous mode
            cursor.execute(f"PRAGMA synchronous = {self.config.synchronous_mode}")
            
            # Cache size
            cursor.execute(f"PRAGMA cache_size = {self.config.cache_size}")
            
            # Foreign keys
            if self.config.enable_foreign_keys:
                cursor.execute("PRAGMA foreign_keys = ON")
            
            # Auto vacuum
            cursor.execute(f"PRAGMA auto_vacuum = {self.config.auto_vacuum}")
            
            # Temp store
            cursor.execute(f"PRAGMA temp_store = {self.config.temp_store}")
            
            # Memory map
            cursor.execute(f"PRAGMA mmap_size = {self.config.mmap_size}")
            
            # Enable extensions if available
            try:
                conn.enable_load_extension(True)
                if self.config.enable_rtree:
                    cursor.execute("SELECT load_extension('rtree')")
                if self.config.enable_json1:
                    cursor.execute("SELECT load_extension('json1')")
            except sqlite3.OperationalError:
                # Extensions not available
                pass
            
            cursor.close()
            
            with self._lock:
                self.connection_stats['total_created'] += 1
                
            return conn
            
        except Exception as e:
            logger.error(f"Error creating SQLite connection: {str(e)}")
            return None
    
    @contextmanager
    def get_connection(self):
        """Get connection from pool with automatic return."""
        conn = None
        try:
            # Try to get from pool
            try:
                conn = self.pool.get(timeout=self.config.pool_timeout)
                with self._lock:
                    self.connection_stats['pool_hits'] += 1
                    self.active_connections.add(id(conn))
                    self.connection_stats['active_count'] = len(self.active_connections)
            except Empty:
                # Create new connection if pool is empty
                conn = self._create_connection()
                with self._lock:
                    self.connection_stats['pool_misses'] += 1
                    if conn:
                        self.active_connections.add(id(conn))
                        self.connection_stats['active_count'] = len(self.active_connections)
            
            if not conn:
                raise RuntimeError("Could not obtain SQLite connection")
            
            yield conn
            
        finally:
            if conn:
                with self._lock:
                    self.active_connections.discard(id(conn))
                    self.connection_stats['active_count'] = len(self.active_connections)
                
                # Return to pool if there's space
                try:
                    self.pool.put(conn, block=False)
                except:
                    # Pool is full, close connection
                    conn.close()
                    with self._lock:
                        self.connection_stats['total_closed'] += 1
    
    def close_all(self):
        """Close all connections in pool."""
        while not self.pool.empty():
            try:
                conn = self.pool.get_nowait()
                conn.close()
                with self._lock:
                    self.connection_stats['total_closed'] += 1
            except Empty:
                break

class SQLiteMigrationManager:
    """Manages SQLite schema migrations."""
    
    def __init__(self, connection_pool: SQLiteConnectionPool):
        self.connection_pool = connection_pool
        self.migrations = []
        self._ensure_migrations_table()
    
    def _ensure_migrations_table(self):
        """Ensure migrations table exists."""
        with self.connection_pool.get_connection() as conn:
            cursor = conn.cursor()
            cursor.execute("""
                CREATE TABLE IF NOT EXISTS schema_migrations (
                    version TEXT PRIMARY KEY,
                    name TEXT NOT NULL,
                    checksum TEXT NOT NULL,
                    applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """)
            conn.commit()
    
    def add_migration(self, migration: SQLiteMigration):
        """Add migration to the list."""
        if not migration.checksum:
            migration.checksum = hashlib.sha256(
                (migration.up_sql + migration.down_sql).encode()
            ).hexdigest()
        self.migrations.append(migration)
        self.migrations.sort(key=lambda m: m.version)
    
    def get_applied_migrations(self) -> List[str]:
        """Get list of applied migration versions."""
        with self.connection_pool.get_connection() as conn:
            cursor = conn.cursor()
            cursor.execute("SELECT version FROM schema_migrations ORDER BY version")
            return [row['version'] for row in cursor.fetchall()]
    
    def apply_migrations(self) -> List[str]:
        """Apply pending migrations."""
        applied = self.get_applied_migrations()
        applied_migrations = []
        
        for migration in self.migrations:
            if migration.version not in applied:
                try:
                    with self.connection_pool.get_connection() as conn:
                        cursor = conn.cursor()
                        
                        # Execute migration
                        cursor.executescript(migration.up_sql)
                        
                        # Record migration
                        cursor.execute("""
                            INSERT INTO schema_migrations (version, name, checksum)
                            VALUES (?, ?, ?)
                        """, (migration.version, migration.name, migration.checksum))
                        
                        conn.commit()
                        applied_migrations.append(migration.version)
                        logger.info(f"Applied migration: {migration.version} - {migration.name}")
                        
                except Exception as e:
                    logger.error(f"Error applying migration {migration.version}: {str(e)}")
                    raise
        
        return applied_migrations
    
    def rollback_migration(self, version: str) -> bool:
        """Rollback specific migration."""
        migration = next((m for m in self.migrations if m.version == version), None)
        if not migration:
            raise ValueError(f"Migration {version} not found")
        
        try:
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                
                # Execute rollback
                cursor.executescript(migration.down_sql)
                
                # Remove migration record
                cursor.execute("DELETE FROM schema_migrations WHERE version = ?", (version,))
                
                conn.commit()
                logger.info(f"Rolled back migration: {version}")
                return True
                
        except Exception as e:
            logger.error(f"Error rolling back migration {version}: {str(e)}")
            raise

class SQLiteFTSManager:
    """Manages SQLite Full-Text Search (FTS5) functionality."""
    
    def __init__(self, connection_pool: SQLiteConnectionPool):
        self.connection_pool = connection_pool
        self.fts_tables = {}
    
    def create_fts_table(self, table_name: str, columns: List[str], 
                        content_table: Optional[str] = None) -> bool:
        """Create FTS5 table."""
        try:
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                
                # Check if FTS5 is available
                cursor.execute("PRAGMA compile_options")
                compile_options = [row[0] for row in cursor.fetchall()]
                if not any('FTS5' in option for option in compile_options):
                    logger.warning("FTS5 not available in this SQLite build")
                    return False
                
                # Build FTS table SQL
                columns_str = ', '.join(columns)
                fts_sql = f"CREATE VIRTUAL TABLE {table_name}_fts USING fts5({columns_str}"
                
                if content_table:
                    fts_sql += f", content='{content_table}'"
                
                fts_sql += ")"
                
                cursor.execute(fts_sql)
                conn.commit()
                
                self.fts_tables[table_name] = {
                    'columns': columns,
                    'content_table': content_table
                }
                
                logger.info(f"Created FTS5 table: {table_name}_fts")
                return True
                
        except Exception as e:
            logger.error(f"Error creating FTS table: {str(e)}")
            raise
    
    def insert_fts_data(self, table_name: str, data: Dict[str, Any]) -> bool:
        """Insert data into FTS table."""
        try:
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                
                columns = list(data.keys())
                placeholders = ', '.join(['?' for _ in columns])
                values = list(data.values())
                
                sql = f"INSERT INTO {table_name}_fts ({', '.join(columns)}) VALUES ({placeholders})"
                cursor.execute(sql, values)
                conn.commit()
                
                return True
                
        except Exception as e:
            logger.error(f"Error inserting FTS data: {str(e)}")
            raise
    
    def search_fts(self, table_name: str, query: str, limit: int = 100) -> List[Dict[str, Any]]:
        """Search FTS table."""
        try:
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                
                sql = f"""
                    SELECT *, rank FROM {table_name}_fts 
                    WHERE {table_name}_fts MATCH ? 
                    ORDER BY rank 
                    LIMIT ?
                """
                
                cursor.execute(sql, (query, limit))
                return [dict(row) for row in cursor.fetchall()]
                
        except Exception as e:
            logger.error(f"Error searching FTS: {str(e)}")
            raise

class SQLiteBackupManager:
    """Manages SQLite database backups."""
    
    def __init__(self, config: SQLiteConfig, connection_pool: SQLiteConnectionPool):
        self.config = config
        self.connection_pool = connection_pool
        self.backup_dir = Path(config.database_path).parent / "backups"
        self.backup_dir.mkdir(exist_ok=True)
    
    def create_backup(self, backup_name: Optional[str] = None) -> str:
        """Create database backup."""
        if backup_name is None:
            backup_name = f"backup_{datetime.now().strftime('%Y%m%d_%H%M%S')}.db"
        
        backup_path = self.backup_dir / backup_name
        
        try:
            with self.connection_pool.get_connection() as conn:
                # Use SQLite backup API for online backup
                backup_conn = sqlite3.connect(str(backup_path))
                conn.backup(backup_conn)
                backup_conn.close()
            
            logger.info(f"Created backup: {backup_path}")
            self._cleanup_old_backups()
            
            return str(backup_path)
            
        except Exception as e:
            logger.error(f"Error creating backup: {str(e)}")
            raise
    
    def restore_backup(self, backup_path: str) -> bool:
        """Restore database from backup."""
        try:
            if not os.path.exists(backup_path):
                raise FileNotFoundError(f"Backup file not found: {backup_path}")
            
            # Close all connections
            self.connection_pool.close_all()
            
            # Copy backup file to database location
            shutil.copy2(backup_path, self.config.database_path)
            
            # Reinitialize connection pool
            self.connection_pool._initialize_pool()
            
            logger.info(f"Restored backup from: {backup_path}")
            return True
            
        except Exception as e:
            logger.error(f"Error restoring backup: {str(e)}")
            raise
    
    def _cleanup_old_backups(self):
        """Remove old backup files."""
        try:
            backup_files = sorted(
                self.backup_dir.glob("backup_*.db"),
                key=lambda f: f.stat().st_mtime,
                reverse=True
            )
            
            for old_backup in backup_files[self.config.max_backup_files:]:
                old_backup.unlink()
                logger.debug(f"Removed old backup: {old_backup}")
                
        except Exception as e:
            logger.warning(f"Error cleaning up backups: {str(e)}")

class SQLiteQueryOptimizer:
    """SQLite query optimization and analysis."""
    
    def __init__(self, connection_pool: SQLiteConnectionPool):
        self.connection_pool = connection_pool
        self.query_cache = {}
        self.query_stats = {}
    
    def analyze_query(self, query: str, params: Optional[Tuple] = None) -> Dict[str, Any]:
        """Analyze query execution plan."""
        try:
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                
                # Get query plan
                explain_query = f"EXPLAIN QUERY PLAN {query}"
                cursor.execute(explain_query, params or ())
                plan = cursor.fetchall()
                
                # Parse plan
                plan_info = {
                    'steps': [],
                    'uses_index': False,
                    'scan_count': 0,
                    'complexity_score': 0
                }
                
                for row in plan:
                    step = dict(row)
                    plan_info['steps'].append(step)
                    
                    detail = step.get('detail', '').lower()
                    if 'index' in detail:
                        plan_info['uses_index'] = True
                    if 'scan' in detail:
                        plan_info['scan_count'] += 1
                
                # Calculate complexity score
                plan_info['complexity_score'] = len(plan_info['steps']) + plan_info['scan_count'] * 2
                
                return plan_info
                
        except Exception as e:
            logger.error(f"Error analyzing query: {str(e)}")
            return {}
    
    def suggest_indexes(self, query: str) -> List[SQLiteIndex]:
        """Suggest indexes for query optimization."""
        suggestions = []
        
        # Simple pattern matching for common cases
        query_lower = query.lower()
        
        # Look for WHERE clauses
        if 'where' in query_lower:
            # This is a simplified implementation
            # In production, you'd want more sophisticated parsing
            pass
        
        return suggestions
    
    def get_query_stats(self) -> Dict[str, Any]:
        """Get query execution statistics."""
        return self.query_stats.copy()

class SQLiteEnhancedOperator:
    """@sqlite operator implementation with enhanced production features."""
    
    def __init__(self):
        self.config: Optional[SQLiteConfig] = None
        self.connection_pool: Optional[SQLiteConnectionPool] = None
        self.migration_manager: Optional[SQLiteMigrationManager] = None
        self.fts_manager: Optional[SQLiteFTSManager] = None
        self.backup_manager: Optional[SQLiteBackupManager] = None
        self.query_optimizer: Optional[SQLiteQueryOptimizer] = None
        self.operation_stats = {
            'queries_executed': 0,
            'transactions_completed': 0,
            'migrations_applied': 0,
            'backups_created': 0,
            'fts_searches': 0,
            'cache_hits': 0,
            'query_optimizations': 0
        }
        self._executor = ThreadPoolExecutor(max_workers=5)
        self._backup_task = None
    
    async def connect(self, config: Optional[SQLiteConfig] = None) -> bool:
        """Initialize SQLite with enhanced features."""
        if config is None:
            config = SQLiteConfig()
        
        self.config = config
        
        try:
            # Initialize connection pool
            self.connection_pool = SQLiteConnectionPool(config)
            
            # Initialize managers
            self.migration_manager = SQLiteMigrationManager(self.connection_pool)
            
            if config.enable_fts:
                self.fts_manager = SQLiteFTSManager(self.connection_pool)
            
            self.backup_manager = SQLiteBackupManager(config, self.connection_pool)
            self.query_optimizer = SQLiteQueryOptimizer(self.connection_pool)
            
            # Start background backup task
            if config.backup_interval > 0:
                self._backup_task = asyncio.create_task(self._backup_scheduler())
            
            logger.info(f"SQLite Enhanced operator connected: {config.database_path}")
            return True
            
        except Exception as e:
            logger.error(f"Error connecting to SQLite: {str(e)}")
            return False
    
    async def execute_query(self, query: str, params: Optional[Tuple] = None,
                           analyze: bool = False) -> SQLiteQueryResult:
        """Execute SELECT query with optimization."""
        if not self.connection_pool:
            raise RuntimeError("Not connected to SQLite database")
        
        start_time = time.time()
        
        try:
            # Analyze query if requested
            query_plan = None
            if analyze and self.query_optimizer:
                query_plan = self.query_optimizer.analyze_query(query, params)
                self.operation_stats['query_optimizations'] += 1
            
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute(query, params or ())
                
                data = [dict(row) for row in cursor.fetchall()]
                execution_time = time.time() - start_time
                
                result = SQLiteQueryResult(
                    data=data,
                    affected_rows=len(data),
                    execution_time=execution_time,
                    query_plan=query_plan
                )
                
                self.operation_stats['queries_executed'] += 1
                return result
                
        except Exception as e:
            logger.error(f"Error executing query: {str(e)}")
            raise
    
    async def execute_command(self, command: str, params: Optional[Tuple] = None) -> SQLiteQueryResult:
        """Execute INSERT/UPDATE/DELETE command."""
        if not self.connection_pool:
            raise RuntimeError("Not connected to SQLite database")
        
        start_time = time.time()
        
        try:
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute(command, params or ())
                
                affected_rows = cursor.rowcount
                conn.commit()
                execution_time = time.time() - start_time
                
                result = SQLiteQueryResult(
                    data=[],
                    affected_rows=affected_rows,
                    execution_time=execution_time
                )
                
                self.operation_stats['queries_executed'] += 1
                return result
                
        except Exception as e:
            logger.error(f"Error executing command: {str(e)}")
            raise
    
    @contextmanager
    def transaction(self, transaction_config: Optional[SQLiteTransaction] = None):
        """Execute transaction with proper isolation."""
        if not self.connection_pool:
            raise RuntimeError("Not connected to SQLite database")
        
        if transaction_config is None:
            transaction_config = SQLiteTransaction()
        
        with self.connection_pool.get_connection() as conn:
            try:
                # Begin transaction with isolation level
                conn.execute(f"BEGIN {transaction_config.isolation_level}")
                
                yield conn
                
                conn.commit()
                self.operation_stats['transactions_completed'] += 1
                
            except Exception:
                conn.rollback()
                raise
    
    # Migration operations
    async def add_migration(self, version: str, name: str, up_sql: str, down_sql: str) -> bool:
        """Add migration."""
        if not self.migration_manager:
            raise RuntimeError("Migration manager not initialized")
        
        migration = SQLiteMigration(
            version=version,
            name=name,
            up_sql=up_sql,
            down_sql=down_sql
        )
        
        self.migration_manager.add_migration(migration)
        return True
    
    async def apply_migrations(self) -> List[str]:
        """Apply pending migrations."""
        if not self.migration_manager:
            raise RuntimeError("Migration manager not initialized")
        
        applied = self.migration_manager.apply_migrations()
        self.operation_stats['migrations_applied'] += len(applied)
        return applied
    
    async def rollback_migration(self, version: str) -> bool:
        """Rollback migration."""
        if not self.migration_manager:
            raise RuntimeError("Migration manager not initialized")
        
        return self.migration_manager.rollback_migration(version)
    
    # Full-text search operations
    async def create_fts_table(self, table_name: str, columns: List[str],
                              content_table: Optional[str] = None) -> bool:
        """Create FTS5 table."""
        if not self.fts_manager:
            raise RuntimeError("FTS manager not initialized")
        
        return self.fts_manager.create_fts_table(table_name, columns, content_table)
    
    async def fts_search(self, table_name: str, query: str, limit: int = 100) -> List[Dict[str, Any]]:
        """Search using FTS."""
        if not self.fts_manager:
            raise RuntimeError("FTS manager not initialized")
        
        results = self.fts_manager.search_fts(table_name, query, limit)
        self.operation_stats['fts_searches'] += 1
        return results
    
    # Backup operations
    async def create_backup(self, backup_name: Optional[str] = None) -> str:
        """Create database backup."""
        if not self.backup_manager:
            raise RuntimeError("Backup manager not initialized")
        
        backup_path = self.backup_manager.create_backup(backup_name)
        self.operation_stats['backups_created'] += 1
        return backup_path
    
    async def restore_backup(self, backup_path: str) -> bool:
        """Restore from backup."""
        if not self.backup_manager:
            raise RuntimeError("Backup manager not initialized")
        
        return self.backup_manager.restore_backup(backup_path)
    
    # Performance operations
    async def analyze_query(self, query: str, params: Optional[Tuple] = None) -> Dict[str, Any]:
        """Analyze query performance."""
        if not self.query_optimizer:
            raise RuntimeError("Query optimizer not initialized")
        
        return self.query_optimizer.analyze_query(query, params)
    
    async def optimize_database(self) -> Dict[str, Any]:
        """Perform database optimization."""
        if not self.connection_pool:
            raise RuntimeError("Not connected to SQLite database")
        
        results = {}
        
        try:
            with self.connection_pool.get_connection() as conn:
                cursor = conn.cursor()
                
                # ANALYZE to update statistics
                cursor.execute("ANALYZE")
                results['analyze_completed'] = True
                
                # VACUUM to optimize database
                cursor.execute("VACUUM")
                results['vacuum_completed'] = True
                
                # Get database info
                cursor.execute("PRAGMA page_count")
                results['page_count'] = cursor.fetchone()[0]
                
                cursor.execute("PRAGMA page_size")
                results['page_size'] = cursor.fetchone()[0]
                
                cursor.execute("PRAGMA freelist_count")
                results['free_pages'] = cursor.fetchone()[0]
                
                results['database_size'] = results['page_count'] * results['page_size']
                results['free_space'] = results['free_pages'] * results['page_size']
                
                conn.commit()
                
        except Exception as e:
            logger.error(f"Error optimizing database: {str(e)}")
            results['error'] = str(e)
        
        return results
    
    async def _backup_scheduler(self):
        """Background backup scheduler."""
        while True:
            try:
                await asyncio.sleep(self.config.backup_interval)
                await self.create_backup()
                logger.info("Automatic backup completed")
            except Exception as e:
                logger.error(f"Automatic backup failed: {str(e)}")
    
    def get_statistics(self) -> Dict[str, Any]:
        """Get comprehensive statistics."""
        stats = {
            'operations': self.operation_stats.copy(),
            'connected': self.connection_pool is not None
        }
        
        if self.connection_pool:
            stats['connection_pool'] = self.connection_pool.connection_stats.copy()
        
        if self.query_optimizer:
            stats['query_stats'] = self.query_optimizer.get_query_stats()
        
        return stats
    
    async def close(self):
        """Close connections and cleanup."""
        # Cancel backup task
        if self._backup_task:
            self._backup_task.cancel()
        
        # Close connection pool
        if self.connection_pool:
            self.connection_pool.close_all()
        
        # Shutdown executor
        self._executor.shutdown(wait=True)
        
        logger.info("SQLite Enhanced operator closed")

# Export the operator
__all__ = [
    'SQLiteEnhancedOperator', 'SQLiteConfig', 'SQLiteTransaction', 'SQLiteQueryResult',
    'SQLiteMigration', 'SQLiteIndex', 'SQLiteFTSManager', 'SQLiteBackupManager'
] 