"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToTranscribe = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToTranscribe class.
 */
class LambdaToTranscribe extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToTranscribe class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToTranscribeProps} props - user provided props for the construct.
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // All our tests are based upon this behavior being on, so we're setting
        // context here rather than assuming the client will set it
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        // Check props
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        // Check source bucket props
        const sourceS3Props = {
            existingBucketObj: props.existingSourceBucketObj,
            bucketProps: props.sourceBucketProps,
            loggingBucketProps: props.sourceLoggingBucketProps,
            logS3AccessLogs: props.logSourceS3AccessLogs
        };
        defaults.CheckS3Props(sourceS3Props);
        // Check destination bucket props (only if not using same bucket)
        if (!props.useSameBucket) {
            const destinationS3Props = {
                existingBucketObj: props.existingDestinationBucketObj,
                bucketProps: props.destinationBucketProps,
                loggingBucketProps: props.destinationLoggingBucketProps,
                logS3AccessLogs: props.logDestinationS3AccessLogs
            };
            defaults.CheckS3Props(destinationS3Props);
        }
        // Setup VPC if required
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.TRANSCRIBE);
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.S3);
        }
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        // Setup source S3 Bucket
        let sourceBucket;
        if (!props.existingSourceBucketObj) {
            const buildSourceBucketResponse = defaults.buildS3Bucket(this, {
                bucketProps: props.sourceBucketProps,
                loggingBucketProps: props.sourceLoggingBucketProps,
                logS3AccessLogs: props.logSourceS3AccessLogs
            }, `${id}-source-bucket`);
            this.sourceBucket = buildSourceBucketResponse.bucket;
            this.sourceLoggingBucket = buildSourceBucketResponse.loggingBucket;
            sourceBucket = this.sourceBucket;
        }
        else {
            sourceBucket = props.existingSourceBucketObj;
        }
        this.sourceBucketInterface = sourceBucket;
        // Setup destination S3 Bucket
        let destinationBucket;
        if (props.useSameBucket) {
            destinationBucket = sourceBucket;
            this.destinationBucketInterface = sourceBucket;
        }
        else {
            if (!props.existingDestinationBucketObj) {
                const buildDestinationBucketResponse = defaults.buildS3Bucket(this, {
                    bucketProps: props.destinationBucketProps,
                    loggingBucketProps: props.destinationLoggingBucketProps,
                    logS3AccessLogs: props.logDestinationS3AccessLogs
                }, `${id}-destination-bucket`);
                this.destinationBucket = buildDestinationBucketResponse.bucket;
                this.destinationLoggingBucket = buildDestinationBucketResponse.loggingBucket;
                destinationBucket = this.destinationBucket;
            }
            else {
                destinationBucket = props.existingDestinationBucketObj;
            }
            this.destinationBucketInterface = destinationBucket;
        }
        // Configure environment variables
        const sourceBucketEnvName = props.sourceBucketEnvironmentVariableName || 'SOURCE_BUCKET_NAME';
        const destinationBucketEnvName = props.destinationBucketEnvironmentVariableName || 'DESTINATION_BUCKET_NAME';
        this.lambdaFunction.addEnvironment(sourceBucketEnvName, sourceBucket.bucketName);
        this.lambdaFunction.addEnvironment(destinationBucketEnvName, destinationBucket.bucketName);
        // Grant Lambda permissions to S3 buckets
        sourceBucket.grantRead(this.lambdaFunction.grantPrincipal);
        destinationBucket.grantWrite(this.lambdaFunction.grantPrincipal);
        // Grant Lambda permissions to Transcribe service
        this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'transcribe:StartTranscriptionJob',
                'transcribe:GetTranscriptionJob',
                'transcribe:ListTranscriptionJobs'
            ],
            resources: ['*']
        }));
    }
}
exports.LambdaToTranscribe = LambdaToTranscribe;
_a = JSII_RTTI_SYMBOL_1;
LambdaToTranscribe[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-transcribe.LambdaToTranscribe", version: "2.95.0" };
//# sourceMappingURL=data:application/json;base64,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