"use strict";
const lib_1 = require("../lib");
const encoding_1 = require("../lib/private/encoding");
const intrinsic_1 = require("../lib/private/intrinsic");
const resolve_1 = require("../lib/private/resolve");
const evaluate_cfn_1 = require("./evaluate-cfn");
class Promise2 {
    constructor() {
        this.creationStack = [];
    }
    resolve() {
        return {
            Data: {
                stringProp: 'hello',
                numberProp: 1234,
            },
            Recurse: new intrinsic_1.Intrinsic(42)
        };
    }
}
class Promise1 {
    constructor() {
        this.creationStack = [];
        this.p2 = [new Promise2(), new Promise2()];
    }
    resolve() {
        return this.p2;
    }
}
class BaseDataType {
    constructor(foo) {
        this.foo = foo;
    }
}
class DataType extends BaseDataType {
    constructor() {
        super(12);
        this.goo = 'hello';
    }
}
/**
 * Return Tokens in both flavors that resolve to the given string
 */
function tokensThatResolveTo(value) {
    return [
        new intrinsic_1.Intrinsic(value),
        lib_1.Lazy.anyValue({ produce: () => value })
    ];
}
/**
 * Wrapper for resolve that creates an throwaway Construct to call it on
 *
 * So I don't have to change all call sites in this file.
 */
function resolve(x) {
    return new lib_1.Stack().resolve(x);
}
module.exports = {
    'resolve a plain old object should just return the object'(test) {
        const obj = { PlainOldObject: 123, Array: [1, 2, 3] };
        test.deepEqual(resolve(obj), obj);
        test.done();
    },
    'if a value is an object with a token value, it will be evaluated'(test) {
        const obj = {
            RegularValue: 'hello',
            LazyValue: new intrinsic_1.Intrinsic('World')
        };
        test.deepEqual(resolve(obj), {
            RegularValue: 'hello',
            LazyValue: 'World'
        });
        test.done();
    },
    'tokens are evaluated anywhere in the object tree'(test) {
        const obj = new Promise1();
        const actual = resolve({ Obj: obj });
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234
                    },
                    Recurse: 42
                },
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234
                    },
                    Recurse: 42
                }
            ]
        });
        test.done();
    },
    'tokens are evaluated recursively'(test) {
        const obj = new Promise1();
        const actual = resolve(new intrinsic_1.Intrinsic({ Obj: obj }));
        test.deepEqual(actual, {
            Obj: [
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234
                    },
                    Recurse: 42
                },
                {
                    Data: {
                        stringProp: 'hello',
                        numberProp: 1234
                    },
                    Recurse: 42
                }
            ]
        });
        test.done();
    },
    'empty arrays or objects are kept'(test) {
        test.deepEqual(resolve({}), {});
        test.deepEqual(resolve([]), []);
        const obj = {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [undefined, undefined],
                    PropD: 'Yoohoo'
                }
            }
        };
        test.deepEqual(resolve(obj), {
            Prop1: 1234,
            Prop2: {},
            Prop3: [],
            Prop4: 'hello',
            Prop5: {
                PropA: {},
                PropB: {
                    PropC: [],
                    PropD: 'Yoohoo'
                }
            }
        });
        test.done();
    },
    'if an object has a "resolve" property that is not a function, it is not considered a token'(test) {
        test.deepEqual(resolve({ a_token: { resolve: () => 78787 } }), { a_token: 78787 });
        test.deepEqual(resolve({ not_a_token: { resolve: 12 } }), { not_a_token: { resolve: 12 } });
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'if a resolvable object inherits from a class that is also resolvable, the "constructor" function will not get in the way (uses Object.keys instead of "for in")'(test) {
        test.deepEqual(resolve({ prop: new DataType() }), { prop: { foo: 12, goo: 'hello' } });
        test.done();
    },
    'isToken(obj) can be used to determine if an object is a token'(test) {
        test.ok(lib_1.isResolvableObject({ resolve: () => 123 }));
        test.ok(lib_1.isResolvableObject({ a: 1, b: 2, resolve: () => 'hello' }));
        test.ok(!lib_1.isResolvableObject({ a: 1, b: 2, resolve: 3 }));
        test.done();
    },
    'Token can be used to create tokens that contain a constant value'(test) {
        test.equal(resolve(new intrinsic_1.Intrinsic(12)), 12);
        test.equal(resolve(new intrinsic_1.Intrinsic('hello')), 'hello');
        test.deepEqual(resolve(new intrinsic_1.Intrinsic(['hi', 'there'])), ['hi', 'there']);
        test.done();
    },
    'resolving leaves a Date object in working order'(test) {
        const date = new Date('2000-01-01');
        const resolved = resolve(date);
        test.equal(date.toString(), resolved.toString());
        test.done();
    },
    'tokens can be stringified and evaluated to conceptual value'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('woof woof');
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'The dog says: woof woof');
        test.done();
    },
    'tokens stringification can be reversed'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('woof woof');
        // THEN
        test.equal(token, lib_1.Tokenization.reverseString(`${token}`).firstToken);
        test.done();
    },
    'Tokens stringification and reversing of CloudFormation Tokens is implemented using Fn::Join'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic(({ woof: 'woof' }));
        // WHEN
        const stringified = `The dog says: ${token}`;
        const resolved = resolve(stringified);
        // THEN
        test.deepEqual(resolved, {
            'Fn::Join': ['', ['The dog says: ', { woof: 'woof' }]]
        });
        test.done();
    },
    'Doubly nested strings evaluate correctly in scalar context'(test) {
        // GIVEN
        const token1 = new intrinsic_1.Intrinsic('world');
        const token2 = new intrinsic_1.Intrinsic(`hello ${token1}`);
        // WHEN
        const resolved1 = resolve(token2.toString());
        const resolved2 = resolve(token2);
        // THEN
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved1), 'hello world');
        test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved2), 'hello world');
        test.done();
    },
    'integer Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const token of tokensThatResolveTo(1)) {
            // WHEN
            const stringified = `the number is ${token}`;
            const resolved = resolve(stringified);
            // THEN
            test.deepEqual(evaluate_cfn_1.evaluateCFN(resolved), 'the number is 1');
        }
        test.done();
    },
    'intrinsic Tokens can be stringified and evaluate to conceptual value'(test) {
        // GIVEN
        for (const bucketName of tokensThatResolveTo({ Ref: 'MyBucket' })) {
            // WHEN
            const resolved = resolve(`my bucket is named ${bucketName}`);
            // THEN
            const context = { MyBucket: 'TheName' };
            test.equal(evaluate_cfn_1.evaluateCFN(resolved, context), 'my bucket is named TheName');
        }
        test.done();
    },
    'tokens resolve properly in initial position'(test) {
        // GIVEN
        for (const token of tokensThatResolveTo('Hello')) {
            // WHEN
            const resolved = resolve(`${token} world`);
            // THEN
            test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
        }
        test.done();
    },
    'side-by-side Tokens resolve correctly'(test) {
        // GIVEN
        for (const token1 of tokensThatResolveTo('Hello ')) {
            for (const token2 of tokensThatResolveTo('world')) {
                // WHEN
                const resolved = resolve(`${token1}${token2}`);
                // THEN
                test.equal(evaluate_cfn_1.evaluateCFN(resolved), 'Hello world');
            }
        }
        test.done();
    },
    'tokens can be used in hash keys but must resolve to a string'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic('I am a string');
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested in hash keys'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic(lib_1.Lazy.stringValue({ produce: () => lib_1.Lazy.stringValue({ produce: (() => 'I am a string') }) }));
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`
        };
        // THEN
        test.deepEqual(resolve(s), { 'I am a string': 'boom I am a string' });
        test.done();
    },
    'tokens can be nested and concatenated in hash keys'(test) {
        // GIVEN
        const innerToken = new intrinsic_1.Intrinsic('toot');
        const token = new intrinsic_1.Intrinsic(`${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: 'boom chicago'
        };
        // THEN
        test.deepEqual(resolve(s), { 'toot the woot': 'boom chicago' });
        test.done();
    },
    'can find nested tokens in hash keys'(test) {
        // GIVEN
        const innerToken = new intrinsic_1.Intrinsic('toot');
        const token = new intrinsic_1.Intrinsic(`${innerToken} the woot`);
        // WHEN
        const s = {
            [token.toString()]: 'boom chicago'
        };
        // THEN
        const tokens = resolve_1.findTokens(new lib_1.Stack(), () => s);
        test.ok(tokens.some(t => t === innerToken), 'Cannot find innerToken');
        test.ok(tokens.some(t => t === token), 'Cannot find token');
        test.done();
    },
    'fails if token in a hash key resolves to a non-string'(test) {
        // GIVEN
        const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
        // WHEN
        const s = {
            [token.toString()]: `boom ${token}`
        };
        // THEN
        test.throws(() => resolve(s), 'The key "${Token[TOKEN.19]}" has been resolved to {"Ref":"Other"} but must be resolvable to a string');
        test.done();
    },
    'list encoding': {
        'can encode Token to string and resolve the encoding'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const struct = {
                XYZ: lib_1.Token.asList(token)
            };
            // THEN
            test.deepEqual(resolve(struct), {
                XYZ: { Ref: 'Other' }
            });
            test.done();
        },
        'cannot add to encoded list'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const encoded = lib_1.Token.asList(token);
            encoded.push('hello');
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /Cannot add elements to list token/);
            test.done();
        },
        'cannot add to strings in encoded list'(test) {
            // GIVEN
            const token = new intrinsic_1.Intrinsic({ Ref: 'Other' });
            // WHEN
            const encoded = lib_1.Token.asList(token);
            encoded[0] += 'hello';
            // THEN
            test.throws(() => {
                resolve(encoded);
            }, /concatenate strings in/);
            test.done();
        },
        'can pass encoded lists to FnSelect'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.select(1, encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Select': [1, { Ref: 'Other' }]
            });
            test.done();
        },
        'can pass encoded lists to FnJoin'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.join('/', encoded);
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }]
            });
            test.done();
        },
        'can pass encoded lists to FnJoin, even if join is stringified'(test) {
            // GIVEN
            const encoded = lib_1.Token.asList(new intrinsic_1.Intrinsic({ Ref: 'Other' }));
            // WHEN
            const struct = lib_1.Fn.join('/', encoded).toString();
            // THEN
            test.deepEqual(resolve(struct), {
                'Fn::Join': ['/', { Ref: 'Other' }]
            });
            test.done();
        },
    },
    'number encoding': {
        'basic integer encoding works'(test) {
            test.equal(16, encoding_1.extractTokenDouble(encoding_1.createTokenDouble(16)));
            test.done();
        },
        'arbitrary integers can be encoded, stringified, and recovered'(test) {
            for (let i = 0; i < 100; i++) {
                // We can encode all numbers up to 2^48-1
                const x = Math.floor(Math.random() * (Math.pow(2, 48) - 1));
                const encoded = encoding_1.createTokenDouble(x);
                // Roundtrip through JSONification
                const roundtripped = JSON.parse(JSON.stringify({ theNumber: encoded })).theNumber;
                const decoded = encoding_1.extractTokenDouble(roundtripped);
                test.equal(decoded, x, `Fail roundtrip encoding of ${x}`);
            }
            test.done();
        },
        'arbitrary numbers are correctly detected as not being tokens'(test) {
            test.equal(undefined, encoding_1.extractTokenDouble(0));
            test.equal(undefined, encoding_1.extractTokenDouble(1243));
            test.equal(undefined, encoding_1.extractTokenDouble(4835e+532));
            test.done();
        },
        'can number-encode and resolve Token objects'(test) {
            // GIVEN
            const x = new intrinsic_1.Intrinsic(123);
            // THEN
            const encoded = lib_1.Token.asNumber(x);
            test.equal(false, lib_1.isResolvableObject(encoded), 'encoded number does not test as token');
            test.equal(true, lib_1.Token.isUnresolved(encoded), 'encoded number does not test as token');
            // THEN
            const resolved = resolve({ value: encoded });
            test.deepEqual(resolved, { value: 123 });
            test.done();
        },
    },
    'stack trace is captured at token creation'(test) {
        function fn1() {
            function fn2() {
                class ExposeTrace extends intrinsic_1.Intrinsic {
                    get creationTrace() {
                        return this.creationStack;
                    }
                }
                return new ExposeTrace('hello');
            }
            return fn2();
        }
        const token = fn1();
        test.ok(token.creationTrace.find(x => x.includes('fn1')));
        test.ok(token.creationTrace.find(x => x.includes('fn2')));
        test.done();
    },
    'newError returns an error with the creation stack trace'(test) {
        function fn1() {
            function fn2() {
                function fn3() {
                    class ThrowingToken extends intrinsic_1.Intrinsic {
                        throwError(message) {
                            throw this.newError(message);
                        }
                    }
                    return new ThrowingToken('boom');
                }
                return fn3();
            }
            return fn2();
        }
        const token = fn1();
        test.throws(() => token.throwError('message!'), /Token created:/);
        test.done();
    },
    'type coercion': (() => {
        const tests = {};
        const inputs = [
            'a string',
            1234,
            { an_object: 1234 },
            [1, 2, 3],
            false
        ];
        for (const input of inputs) {
            // GIVEN
            const stringToken = lib_1.Token.asString(new intrinsic_1.Intrinsic(input));
            const numberToken = lib_1.Token.asNumber(new intrinsic_1.Intrinsic(input));
            const listToken = lib_1.Token.asList(new intrinsic_1.Intrinsic(input));
            // THEN
            const expected = input;
            tests[`${input}<string>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(stringToken))), expected);
                test.done();
            };
            tests[`${input}<list>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(listToken))), expected);
                test.done();
            };
            tests[`${input}<number>.toNumber()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asNumber(new intrinsic_1.Intrinsic(numberToken))), expected);
                test.done();
            };
            tests[`${input}<string>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(stringToken).toString()), expected);
                test.done();
            };
            tests[`${input}<list>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(listToken).toString()), expected);
                test.done();
            };
            tests[`${input}<number>.toString()`] = (test) => {
                test.deepEqual(resolve(new intrinsic_1.Intrinsic(numberToken).toString()), expected);
                test.done();
            };
            tests[`${input}<string>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(stringToken))), expected);
                test.done();
            };
            tests[`${input}<list>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(listToken))), expected);
                test.done();
            };
            tests[`${input}<number>.toList()`] = (test) => {
                test.deepEqual(resolve(lib_1.Token.asList(new intrinsic_1.Intrinsic(numberToken))), expected);
                test.done();
            };
        }
        return tests;
    })(),
    'creation stack is attached to errors emitted during resolve'(test) {
        function showMeInTheStackTrace() {
            return lib_1.Lazy.stringValue({ produce: () => { throw new Error('fooError'); } });
        }
        const x = showMeInTheStackTrace();
        let message;
        try {
            resolve(x);
        }
        catch (e) {
            message = e.message;
        }
        test.ok(message && message.includes('showMeInTheStackTrace'));
        test.done();
    },
    'stringifyNumber': {
        'converts number to string'(test) {
            test.equal(lib_1.Tokenization.stringifyNumber(100), '100');
            test.done();
        },
        'converts tokenized number to string'(test) {
            test.equal(resolve(lib_1.Tokenization.stringifyNumber({
                resolve: () => 100
            })), '100');
            test.done();
        },
        'string remains the same'(test) {
            test.equal(lib_1.Tokenization.stringifyNumber('123'), '123');
            test.done();
        },
        'Ref remains the same'(test) {
            const val = { Ref: 'SomeLogicalId' };
            test.deepEqual(lib_1.Tokenization.stringifyNumber(val), val);
            test.done();
        },
        'lazy Ref remains the same'(test) {
            const resolvedVal = { Ref: 'SomeLogicalId' };
            const tokenizedVal = lib_1.Lazy.anyValue({
                produce: () => resolvedVal
            });
            const res = lib_1.Tokenization.stringifyNumber(tokenizedVal);
            test.notDeepEqual(res, resolvedVal);
            test.deepEqual(resolve(res), resolvedVal);
            test.done();
        },
        'tokenized Ref remains the same'(test) {
            const resolvedVal = { Ref: 'SomeLogicalId' };
            const tokenizedVal = lib_1.Token.asNumber(resolvedVal);
            const res = lib_1.Tokenization.stringifyNumber(tokenizedVal);
            test.notDeepEqual(res, resolvedVal);
            test.deepEqual(resolve(res), resolvedVal);
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,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