from hestia_earth.schema import EmissionMethodTier
from hestia_earth.utils.tools import list_sum

from hestia_earth.models.log import logger
from hestia_earth.models.utils.emission import _new_emission
from hestia_earth.models.utils.measurement import _most_relevant_measurement_value
from .utils import get_pcorr, get_p_ef_c1, get_ef_p_c2, get_practice_factor, get_water, calculate_R, calculate_A
from . import MODEL

TERM_ID = 'pErosionAllOrigins'


def _emission(value: float):
    logger.info('model=%s, term=%s, value=%s', MODEL, TERM_ID, value)
    emission = _new_emission(TERM_ID, MODEL)
    emission['value'] = [value]
    emission['methodTier'] = EmissionMethodTier.TIER_1.value
    return emission


def _run(list_of_contents_for_A: list, list_of_contents_for_R: list, list_of_contents_for_value: list):
    heavy_winter_precipitation, water = list_of_contents_for_R
    R = calculate_R(heavy_winter_precipitation, water)

    practise_factor, erodability, slope_length, pcorr, p_ef_c1, ef_p_c2 = list_of_contents_for_A
    A = calculate_A(R, practise_factor, erodability, slope_length, pcorr, p_ef_c1, ef_p_c2)

    nla_environment, P_content = list_of_contents_for_value
    logger.debug('R=%s, A=%s, nla_environment=%s, P_content=%s', R, A, nla_environment, P_content)
    value = A * nla_environment / 100 * 2 * P_content / 1000
    return [_emission(value)]


def _should_run(cycle: dict):
    end_date = cycle.get('endDate')
    site = cycle.get('site', {})
    measurements = site.get('measurements', [])

    def _get_measurement_content(term_id: str):
        return _most_relevant_measurement_value(measurements, term_id, end_date)

    nla_environment = list_sum(_get_measurement_content('nutrientLossToAquaticEnvironment'))
    soil_phosphorus_content = list_sum(_get_measurement_content('soilPhosphorusContent'))
    erodability = list_sum(_get_measurement_content('erodibility'))
    slope = list_sum(_get_measurement_content('slope'))
    slope_length = list_sum(_get_measurement_content('slopeLength'))
    heavy_winter_precipitation = list_sum(_get_measurement_content('heavyWinterPrecipitation'))

    precipitation = list_sum(_get_measurement_content('rainfallAnnual'))
    water = get_water(cycle, precipitation)

    practise_factor = get_practice_factor(site)
    pcorr = get_pcorr(slope / 100)
    p_ef_c1 = get_p_ef_c1(cycle)
    ef_p_c2 = get_ef_p_c2(cycle)

    list_of_contents_for_A = [
        practise_factor, erodability, slope_length,
        pcorr, p_ef_c1, ef_p_c2]
    list_of_contents_for_R = [heavy_winter_precipitation, water]
    list_of_contents_for_value = [nla_environment, soil_phosphorus_content]

    should_run = all(list_of_contents_for_A) \
        and all(list_of_contents_for_R) \
        and all(list_of_contents_for_value)
    logger.info('model=%s, term=%s, should_run=%s', MODEL, TERM_ID, should_run)
    return should_run, list_of_contents_for_A, list_of_contents_for_R, list_of_contents_for_value


def run(cycle):
    should_run, list_of_contents_for_A, list_of_contents_for_R, list_of_contents_for_value = _should_run(cycle)
    return _run(list_of_contents_for_A, list_of_contents_for_R, list_of_contents_for_value) if should_run else []
