"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default key'(test) {
        const stack = new core_1.Stack();
        new lib_1.Key(stack, 'MyKey');
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'default with no retention'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'TestStack');
        new lib_1.Key(stack, 'MyKey', { removalPolicy: core_1.RemovalPolicy.DESTROY });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', { DeletionPolicy: "Delete", UpdateReplacePolicy: "Delete" }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default with some permission'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey');
        const p = new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                }
            }
        }));
        test.done();
    },
    'key with some options'(test) {
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const p = new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        key.node.applyAspect(new core_1.Tag('tag1', 'value1'));
        key.node.applyAspect(new core_1.Tag('tag2', 'value2'));
        key.node.applyAspect(new core_1.Tag('tag3', ''));
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        Enabled: false,
                        EnableKeyRotation: true,
                        Tags: [
                            {
                                Key: "tag1",
                                Value: "value1"
                            },
                            {
                                Key: "tag2",
                                Value: "value2"
                            },
                            {
                                Key: "tag3",
                                Value: ""
                            }
                        ]
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                }
            }
        }));
        test.done();
    },
    'addAlias creates an alias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        const alias = key.addAlias('alias/xoo');
        test.ok(alias.aliasName);
        assert_1.expect(stack).toMatch({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        EnableKeyRotation: true,
                        Enabled: false,
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                },
                MyKeyAlias1B45D9DA: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/xoo",
                        TargetKeyId: {
                            "Fn::GetAtt": [
                                "MyKey6AB29FA6",
                                "Arn"
                            ]
                        }
                    }
                }
            }
        });
        test.done();
    },
    'grant decrypt on a key'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'Key');
        const user = new aws_iam_1.User(stack, 'User');
        // WHEN
        key.grantDecrypt(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    // This one is there by default
                    {
                        // tslint:disable-next-line:max-line-length
                        Action: ["kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*", "kms:Update*", "kms:Revoke*", "kms:Disable*", "kms:Get*", "kms:Delete*", "kms:ScheduleKeyDeletion", "kms:CancelKeyDeletion"],
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::", { Ref: "AWS::AccountId" }, ":root"]] } },
                        Resource: "*"
                    },
                    // This is the interesting one
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::GetAtt": ["User00B015A1", "Arn"] } },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Key961B73FD", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'import/export can be used to bring in an existing key'(test) {
        const stack2 = new core_1.Stack();
        const myKeyImported = lib_1.Key.fromKeyArn(stack2, 'MyKeyImported', 'arn:of:key');
        // addAlias can be called on imported keys.
        myKeyImported.addAlias('alias/hello');
        assert_1.expect(stack2).toMatch({
            Resources: {
                MyKeyImportedAliasB1C5269F: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/hello",
                        TargetKeyId: 'arn:of:key'
                    }
                }
            }
        });
        test.done();
    },
    'addToResourcePolicy allowNoOp and there is no policy': {
        'succeed if set to true (default)'(test) {
            const stack = new core_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            key.addToResourcePolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
            test.done();
        },
        'fails if set to false'(test) {
            const stack = new core_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            test.throws(() => key.addToResourcePolicy(new aws_iam_1.PolicyStatement({ resources: ['*'], actions: ['*'] }), /* allowNoOp */ false), 'Unable to add statement to IAM resource policy for KMS key: "foo/bar"');
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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